/*******************************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: JavaUtil.java,v 1.13 2010/05/11 17:09:53 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.core.util;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.preferences.DefaultScope;
import org.eclipse.core.runtime.preferences.IEclipsePreferences;
import org.eclipse.core.runtime.preferences.IEclipsePreferences.IPreferenceChangeListener;
import org.eclipse.core.runtime.preferences.IEclipsePreferences.PreferenceChangeEvent;
import org.eclipse.core.runtime.preferences.IScopeContext;
import org.eclipse.hyades.test.core.internal.resources.TestCorePluginResourceBundle;
import org.eclipse.jdt.core.IJavaElement;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.IType;
import org.eclipse.jdt.core.JavaConventions;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jdt.core.search.IJavaSearchConstants;
import org.eclipse.jdt.core.search.IJavaSearchScope;
import org.eclipse.jdt.core.search.SearchEngine;
import org.eclipse.jdt.core.search.SearchMatch;
import org.eclipse.jdt.core.search.SearchParticipant;
import org.eclipse.jdt.core.search.SearchPattern;
import org.eclipse.jdt.core.search.SearchRequestor;

/**
 * <p>Public test core utility APIs wrapping Eclipse JDT APIs.</p>
 * 
 * 
 * @author  Marcelo Paternostro
 * @author  Paul Slauenwhite
 * @version May 11, 2010
 * @since   Februrary 4, 2005
 */
public class JavaUtil{	

	private static List<String> resourcePatterns = null;
	private static String defaultResourceCopyFilter = null;
	private static IEclipsePreferences javaCoreDefaultPreferences = null;
	private static IPreferenceChangeListener resourceCopyFilterPreferenceListener = null;

	/**
	 * The current minimal supported Java level.
	 */
	public static final String CURRENT_JAVA_LEVEL = "1.5";
	
	/**
	 * <p>Adds a resource pattern to the Java Core default preference containing the 
	 * filters for controlling resource copying (see {@link JavaCore#CORE_JAVA_BUILD_RESOURCE_COPY_FILTER}
	 * or <code>Window &gt;&gt; Preferences &gt;&gt; Java &gt;&gt; Compiler &gt;&gt; Builder &gt;&gt; Output folder &gt;&gt; Filtered Resources</code>).</p>
	 * 
	 * <p>The resource pattern is the form:</p>
	 * 
	 * <code>&lt;name&gt;[,&lt;name&gt;]</code>
	 * 
	 * <p>where <code>&lt;name&gt;<code> is a file name pattern (* and ? wild-cards allowed) or the name of a folder which ends with '/'.</p>
	 * 
	 * @param resourcePattern The resource pattern to be added.
	 */
	public static void addResourcePatternToNotCopyList(String resourcePattern){

		if((resourcePattern != null) && (resourcePattern.trim().length() > 0)){

			//Initialize the list of resource patterns:
			if(resourcePatterns == null){
				resourcePatterns = new ArrayList<String>();
			}
			
			//Add the resource pattern to the list of resource patterns:
			resourcePatterns.add(resourcePattern);

			//Update the Java Core default resource copy filter preference with the list of resource patterns:
			updateResourceCopyFilterPreference();
		}
	}

	/**
	 * <p>Removes a resource pattern to the Java Core default preference containing the 
	 * filters for controlling resource copying (see {@link JavaCore#CORE_JAVA_BUILD_RESOURCE_COPY_FILTER}
	 * or <code>Window &gt;&gt; Preferences &gt;&gt; Java &gt;&gt; Compiler &gt;&gt; Builder &gt;&gt; Output folder &gt;&gt; Filtered Resources</code>).</p>
	 * 
	 * <p>The resource pattern is the form:</p>
	 * 
	 * <code>&lt;name&gt;[,&lt;name&gt;]</code>
	 * 
	 * <p>where <code>&lt;name&gt;<code> is a file name pattern (* and ? wild-cards allowed) or the name of a folder which ends with '/'.</p>
	 * 
	 * @param resourcePattern The resource pattern to be removed.
	 */
	public static void removeResourcePatternToNotCopyList(String resourcePattern){

		if((resourcePattern != null) && (resourcePattern.trim().length() > 0)){
			
			//Initialize the list of resource patterns:
			if(resourcePatterns == null){
				resourcePatterns = new ArrayList<String>();
			}

			//Remove the resource pattern to the list of resource patterns:
			resourcePatterns.remove(resourcePattern);

			//Update the Java Core default resource copy filter preference with the list of resource patterns:
			updateResourceCopyFilterPreference();
		}
	}
	
	private static void updateResourceCopyFilterPreference(){

		//Initialize the Java Core default preferences and default resource copy filter preference listener: 
		if(javaCoreDefaultPreferences == null){
			
			javaCoreDefaultPreferences = ((IScopeContext)(new DefaultScope())).getNode(JavaCore.PLUGIN_ID);
			
			if(javaCoreDefaultPreferences != null){
			
				defaultResourceCopyFilter = javaCoreDefaultPreferences.get(JavaCore.CORE_JAVA_BUILD_RESOURCE_COPY_FILTER, null);
			
				//Note: The default resource copy filter preference listener will be (re)added when the preference is updated:
				resourceCopyFilterPreferenceListener = new IPreferenceChangeListener() {

					/* (non-Javadoc)
					 * @see org.eclipse.core.runtime.preferences.IEclipsePreferences.IPreferenceChangeListener#preferenceChange(org.eclipse.core.runtime.preferences.IEclipsePreferences.PreferenceChangeEvent)
					 */
					public void preferenceChange(PreferenceChangeEvent event) {

						if(JavaCore.CORE_JAVA_BUILD_RESOURCE_COPY_FILTER.equals(event.getKey())){

							//Resolve the default resource copy filter preference:
							defaultResourceCopyFilter = ((String)(event.getNewValue())); 

							//Update the Java Core default resource copy filter preference with the list of resource patterns:
							updateResourceCopyFilterPreference();
						}
					}
				};
			}
		}

		if(javaCoreDefaultPreferences != null){

			StringBuffer newDefaultResourceCopyFilter = new StringBuffer(); 

			if((defaultResourceCopyFilter != null) && (defaultResourceCopyFilter.trim().length() > 0)){
				newDefaultResourceCopyFilter.append(defaultResourceCopyFilter);
			}
			
			Iterator<String> resourcePatternIterator = resourcePatterns.iterator();

			while (resourcePatternIterator.hasNext()) {

				String resourcePattern = resourcePatternIterator.next();

				if(newDefaultResourceCopyFilter.indexOf(resourcePattern) == -1){

					if(newDefaultResourceCopyFilter.length() > 0){
						newDefaultResourceCopyFilter.append(',');
					}

					newDefaultResourceCopyFilter.append(resourcePattern);
				}
			}

			javaCoreDefaultPreferences.removePreferenceChangeListener(resourceCopyFilterPreferenceListener);
			javaCoreDefaultPreferences.put(JavaCore.CORE_JAVA_BUILD_RESOURCE_COPY_FILTER, newDefaultResourceCopyFilter.toString());
			javaCoreDefaultPreferences.addPreferenceChangeListener(resourceCopyFilterPreferenceListener);
		}
	}
	
	/**
	 * Returns a valid Java class based on the specified name.  The name argument
	 * should not contain the package information.
	 *    
	 * @param name
	 * @param generate.  Generates a new name even if the name argument is a valid
	 * identifier. 
	 * @return the class name or <code>null</code> if the name is not valid. 
	 */
	public static String getValidClassName(String name, boolean generate)
	{
		if(name == null)
			return null;

		name = name.trim();
		if((!generate) && JavaConventions.validateIdentifier(name, CURRENT_JAVA_LEVEL, CURRENT_JAVA_LEVEL).isOK())
			return name;
		
		name = StringUtil.toProperCase(name.trim());
		name = StringUtil.replace(name, " ", "");
		if(!Character.isLetter(name.charAt(0)))
			name = "C" + name;

		if(!JavaConventions.validateIdentifier(name, CURRENT_JAVA_LEVEL, CURRENT_JAVA_LEVEL).isOK())
			return null;
			
		return name;
	}
	
	/**
	 * Returns a valid Java identifier based on the specified name.  The name argument
	 * should not contain the package information.  The identifier can be a method
	 * or an attribute.
	 *    
	 * @param name
	 * @param generate.  Generates a new name even if the name argument is a valid
	 * identifier. 
	 * @return the identifier name or <code>null</code> if the name is not valid. 
	 */
	public static String getValidIdentifierName(String name)
	{
		if(name == null) {
			return null;
		} 
		
		name = name.trim();
		if("".equals(name)) {
			return name;
		}
		if(Character.isLetter(name.charAt(0)))
		{
			String auxName = name.substring(0, 1).toLowerCase();
			if(name.length() > 1)
				auxName = auxName + name.substring(1);
			name = auxName;
		}

		if(JavaConventions.validateIdentifier(name, CURRENT_JAVA_LEVEL, CURRENT_JAVA_LEVEL).isOK())
			return name;

		name = StringUtil.toProperCase(name.trim());
		name = StringUtil.replace(name, " ", "");

		if(Character.isLetter(name.charAt(0)))
		{
			String auxName = name.substring(0, 1).toLowerCase();
			if(name.length() > 1)
				auxName = auxName + name.substring(1);
			name = auxName;
		}
		
		if(!Character.isJavaIdentifierStart(name.charAt(0)))
			name = "c" + name;

		if(!JavaConventions.validateIdentifier(name, CURRENT_JAVA_LEVEL, CURRENT_JAVA_LEVEL).isOK())
		{
			StringBuffer sb = new StringBuffer(name);
			int length = name.length();
			for(int i = 0; i < length; i++){
			   if(!Character.isJavaIdentifierPart(sb.charAt(i)))
				  sb.replace(i,i+1,"_");	
			}
			return sb.toString();
		}
		return name;
	}
	
	private static class NonPublicClassInCUCollector extends SearchRequestor {
		private IJavaElement fFound;
		private String cuName;
		public void acceptSearchMatch(SearchMatch match) throws CoreException {
			IJavaElement enclosingElement= (IJavaElement) match.getElement();
			String resourceName= match.getResource().getName();
			if ((enclosingElement instanceof IType) && (resourceName.equals(cuName)))
				fFound= enclosingElement;
		}
	}
	
	/**
	 * Returns the JavaElement matching the specified class name in a particular compilation unit
	 * of a java project.
	 * @param project A Java Project where the java element is searched.
	 * @param cuName A compilation unit name (e.g. "MyClass.java"). This argument may be null
	 * if the searched class is the public type of its compilation unit.
	 * @param className The name of the searched class.
	 * @return A Java Element, or null if not found.
	 * @throws CoreException
	 */
	public static IJavaElement findElement(IJavaProject project, String cuName, String className) throws CoreException {
		IJavaElement element= project.findType(className);
		
		//fix for bug 37333
		if (element == null && cuName != null) {
			SearchPattern pattern=	SearchPattern.createPattern(className, IJavaSearchConstants.TYPE, IJavaSearchConstants.DECLARATIONS,
					SearchPattern.R_EXACT_MATCH | SearchPattern.R_CASE_SENSITIVE | SearchPattern.R_ERASURE_MATCH);
			IJavaSearchScope scope= SearchEngine.createJavaSearchScope(new IJavaElement[] { project }, false);
			NonPublicClassInCUCollector requestor= new NonPublicClassInCUCollector();
			requestor.cuName = cuName;

			SearchEngine searchEngine= new SearchEngine();
			searchEngine.search(pattern, new SearchParticipant[] {SearchEngine.getDefaultSearchParticipant()},
					scope, requestor, new NullProgressMonitor());
			
			element= requestor.fFound;
		}
		
		return element;
	}

	/**
	 * <p>Validates a source folder using the following criteria:</p>
	 * 
	 * <ul>
	 * <li>The source folder exists.</li>
	 * <li>The source folder is a project or folder.</li>
	 * <li>The source folder is not the workspace root.</li>
	 * <li>The source folder's project is open.</li>
	 * </ul>
	 * 
	 * @param sourceFolder The source folder to be validated.
	 * @return The error message is the source folder is invalid, otherwise <code>null</code>.
	 */
	public static String validateSourceFolder(String sourceFolder){
	 
		try {
			
			IPath path = new Path(sourceFolder);
			
			if(path.segmentCount() == 1) {
				return (validateSourceFolder(ResourcesPlugin.getWorkspace().getRoot().getProject(path.toString())));
			} 
	
			return (validateSourceFolder(ResourcesPlugin.getWorkspace().getRoot().getFolder(path)));
		} 
		catch (Exception e) {
			//Ignore and return error message.
		}
		
		return (TestCorePluginResourceBundle.JavaUtil_INVALID_SOURCE_FOLDER);
	}
	
	/**
	 * <p>Validates a source folder using the following criteria:</p>
	 * 
	 * <ul>
	 * <li>The source folder exists.</li>
	 * <li>The source folder is a project or folder.</li>
	 * <li>The source folder is not the workspace root.</li>
	 * <li>The source folder's project is open.</li>
	 * </ul>
	 * 
	 * @param sourceFolder The source folder to be validated.
	 * @return The error message is the source folder is invalid, otherwise <code>null</code>.
	 */
	public static String validateSourceFolder(IContainer sourceFolder){
		
		String errorMessage = null;
		
		if(sourceFolder == null){
			errorMessage = TestCorePluginResourceBundle.JavaUtil_INVALID_SOURCE_FOLDER;
		}
		else{
	
			IProject project = sourceFolder.getProject();
	
			//Assumption: The workspace root returns null.
			if (project == null){
				errorMessage = TestCorePluginResourceBundle.JavaUtil_INVALID_SOURCE_FOLDER;
			} 
	
			//Assumption: A project is not open if it is closed or does not exist.
			else if (!project.isOpen()) {
	
				if(!project.exists()){
					errorMessage = TestCorePluginResourceBundle.JavaUtil_INVALID_SOURCE_FOLDER;					
				}
				else{
					errorMessage = TestCorePluginResourceBundle.JavaUtil_CLOSED_SOURCE_FOLDER;
				}
			}
			else if(!sourceFolder.exists()){
				errorMessage = TestCorePluginResourceBundle.JavaUtil_INVALID_SOURCE_FOLDER;					
			}
		}
		
		return errorMessage;
	}
	
	/**
	 * <p>Validates a package name using the following criteria:</p>
	 * 
	 * <ul>
	 * <li>The package name is not <code>null</code>.</li>
	 * <li>The package name is empty (default package name) or valid for a 1.5 or higher source and compliance level (see {@link JavaConventions#validatePackageName(String, String, String)}).</li>
	 * </ul>
	 * 
	 * @param packageName The package name to be validated.
	 * @return The error message is the package name is invalid, otherwise <code>null</code>.
	 * @see JavaConventions#validatePackageName(String, String, String)
	 */
	public static String validatePackageName(String packageName){
		
		String errorMessage = null;
		
		if(packageName == null){
			errorMessage = TestCorePluginResourceBundle.JavaUtil_MISSING_PACKAGE_NAME;
		}
		
		//Assumption: Default package is empty.
		else if(packageName.length() > 0) {
	
			IStatus returnStatus = JavaConventions.validatePackageName(packageName, CURRENT_JAVA_LEVEL, CURRENT_JAVA_LEVEL); //$NON-NLS-1$ //$NON-NLS-2$
	
			if (returnStatus.getSeverity() == IStatus.ERROR) {                        
	
				//Assumption: The JavaConventions error messages do not contain a period.
				errorMessage = (returnStatus.getMessage() + "."); //$NON-NLS-1$
			}
		}
	
		return errorMessage;
	}
	
	/**
	 * <p>Validates a class name using the following criteria:</p>
	 * 
	 * <ul>
	 * <li>The class name is not <code>null</code> and not empty.</li>
	 * <li>The class name is valid for a 1.5 or higher source and compliance level (see {@link JavaConventions#validateClassName(String, String, String)}).</li>
	 * </ul>
	 * 
	 * @param className The class name to be validated.
	 * @return The error message is the class name is invalid, otherwise <code>null</code>.
	 * @see JavaConventions#validateClassName(String, String, String)
	 */
	public static String validateClassName(String className){
		
		String errorMessage = null;
		
		if((className == null) || (className.length() == 0)) {
			errorMessage = TestCorePluginResourceBundle.JavaUtil_MISSING_CLASS_NAME;
		}
		else{
	
			IStatus returnStatus = JavaConventions.validateJavaTypeName(className, CURRENT_JAVA_LEVEL, CURRENT_JAVA_LEVEL); //$NON-NLS-1$ //$NON-NLS-2$
	
			if (returnStatus.getSeverity() == IStatus.ERROR) {     
				
				//Assumption: The JavaConventions error messages do not contain a period.
				errorMessage = (returnStatus.getMessage() + "."); //$NON-NLS-1$
			}
		}
	
		return errorMessage;
	}
}
