/*******************************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: LocationUtil.java,v 1.5 2010/04/12 12:38:44 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.core.util;
import java.io.File;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.hyades.models.common.configuration.CFGComparableProperty;
import org.eclipse.hyades.models.common.configuration.CFGLocation;
import org.eclipse.hyades.models.common.configuration.CFGPropertyGroup;
import org.eclipse.hyades.models.common.configuration.Common_ConfigurationFactory;
import org.eclipse.hyades.models.common.configuration.util.ConfigurationUtil;
import org.eclipse.hyades.models.common.interactions.BVRProperty;
import org.eclipse.hyades.models.common.util.ICommonConstants;

/**
 * <p>This class provides a central place for utility functions used to create or handle 
 * {@link org.eclipse.hyades.models.common.configuration.CFGLocation} resources.</p>
 * 
 * 
 * @author  Bianca Xue Jiang
 * @author  Jerome Bozier
 * @author  Paul Slauenwhite
 * @version April 12, 2010
 * @since   February 8, 2005
 */
public class LocationUtil 
{
	public static final String PROPERTY_NAME_ROOTDIR = Common_ConfigurationFactory.eINSTANCE.createHyadesRootDirectroyCategory().getName(); //"ROOTDIR"
	
	/**
	 * This method creates a new location EObject and persists the EObject in a resource
	 * sepecified by <i>file</i>.
	 *  
	 * @param file
	 * @param name
	 * @param description
	 * @return
	 * @throws Exception
	 */
	public static void createNew(IFile file, String name, String description, String hostname)
	throws Exception
	{
		Resource resource = createResource(file);
		CFGLocation location = createLocation(name, description, hostname);
		resource.getContents().add(location);
		EMFUtil.save(resource);
	}
	
	/**
	 * Creates the CFGLocation setting the attributes based on the specified information and
	 * initialize this new location object with basic properties.
	 * @return a not null EObject
	 */
	public static CFGLocation createLocation(String name, String description, String hostname)
	{
		CFGLocation location = ConfigurationUtil.createLocation(name, description, hostname);
		
		// bugzilla 61137 enhancement, no longer call setLocationRootDir here. 
		// now the root directory is determined dynamically at test launch time 
		// from the environment variable in IExecutionEnvironment which specifies
		// the location of the temp directory
		return location;
	}

	/**
	 * Creates the EMF resource in which the object is added to.
	 * @return a not null Resource
	 */
	public static Resource createResource(IFile file)
	{
		URI uri = URI.createPlatformResourceURI(file.getFullPath().toString(), false);
		Resource.Factory factory = EMFUtil.getResourceFactory(ICommonConstants.LOCATION_FILE_EXTENSION);
		return factory.createResource(uri);
	}

	public static String getFileExtension()
	{
		return ICommonConstants.LOCATION_FILE_EXTENSION;
	}
	
	public static void setLocationRootDir(CFGLocation location)
	{
	    CFGPropertyGroup propGroup = ConfigurationUtil.searchPropertyGroupById(location.getPropertyGroups(), ConfigurationUtil.ATTRS_PROP_GROUP_ID_LOCATION);
	    if(propGroup == null)
	    {
	        propGroup = Common_ConfigurationFactory.eINSTANCE.createCFGPropertyGroup();
	        propGroup.setPropertyGroupID(ConfigurationUtil.ATTRS_PROP_GROUP_ID_LOCATION);
	        location.getPropertyGroups().add(propGroup);
	    }
	    
	    BVRProperty[] properties = ConfigurationUtil.searchPropertiesByName(propGroup.getProperties(), PROPERTY_NAME_ROOTDIR, false);
	    CFGComparableProperty property = null;
	    if(properties == null || properties.length < 1)
	    {
	        property = Common_ConfigurationFactory.eINSTANCE.createCFGComparableProperty();
	        property.setName(PROPERTY_NAME_ROOTDIR);
	        property.setOperator("=");
	        propGroup.getProperties().add(property);	
	    }
	    else
	        property = (CFGComparableProperty)properties[0];
	    
	    if(property.getValue() == null)
	    {
	        String workspaceRootDir = getWorkspaceRootDir();
	        property.setValue(workspaceRootDir);        
	    }
	}
	
	protected static String getWorkspaceRootDir()
    {
        String workspaceRoot = ResourcesPlugin.getWorkspace().getRoot().getLocation().toString();  
        if(workspaceRoot.endsWith(File.separator))
            workspaceRoot = workspaceRoot.substring(0, (workspaceRoot.length() - File.separator.length()));
        
        return workspaceRoot;
    }

}
