/*******************************************************************************
 * Copyright (c) 2005, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ResourceBundleManager.java,v 1.6 2007/05/03 16:58:24 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.core.util;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.MissingResourceException;
import java.util.ResourceBundle;

/**
 * This class provides a mechanism to allow the reuse of instances of 
 * {@link java.util.ResourceBundle}.
 * 
 * <p>The {@link #getString(String)} and 
 * {@link #getString(String, String[])} methods return the value associated to a 
 * given key by looking for it in a list of resource bundles that are added to this
 * managers through the <code>add</code> methods. 
 * 
 * @author marcelop
 * @since 0.0.1
 * @deprecated As of TPTP V4.4, use the plug-in's resource bundle (e.g. {@link org.eclipse.hyades.test.core.internal.resources.TestCorePluginResourceBundle} for this plug-in) for resolving resources. 
 */
public class ResourceBundleManager
{
	private List resourceBundles;
	
	/**
	 * Constructor for ResourceBundleManager
	 */
	public ResourceBundleManager()
	{
		resourceBundles = createResourceBundleList(); 
	}
	
	/**
	 * @see org.eclipse.hyades.ui.util.IDisposable#dispose()
	 */
	public void dispose()
	{
		resourceBundles.clear();
	}

	/**
	 * Returns the list that stores the resource bundle.  Clients may
	 * overwrite.
	 * @return List
	 */
	protected List createResourceBundleList()
	{
		return new ArrayList();
	}
	
	/**
	 * Adds a resource bundle to the end of this manager's list.
	 * @param resourceBundle
	 * @return <code>true</code> if the resource bundle was added or 
	 * <code>false</code> otherwise.
	 */
	public boolean add(ResourceBundle resourceBundle)
	{
		if(resourceBundle == null)
			return false;
			
		return resourceBundles.add(resourceBundle);
	}
	
	/**
	 * Adds a list of resource bundle to the end of this manager's list.
	 * @param bundles
	 * @return <code>true</code> if the resource bundle was added or 
	 * <code>false</code> otherwise.
	 */
	public boolean add(ResourceBundle[] bundles)
	{
		if(bundles == null)
			return false;
			
		boolean result = true;
		for(int i = 0; i < bundles.length; i++)
		{
			if(resourceBundles.add(bundles[i]) == false)
				result = false;
		}
		return result;
	}
	
	/**
	 * Adds a resource bundle to a given index of this manager's list.
	 * @param index
	 * @param resourceBundle
	 * @return <code>true</code> if the resource bundle was added or 
	 * <code>false</code> otherwise.
     * @throws IllegalArgumentException if some aspect of the specified
     * element prevents it from being added to this list.
     * @throws IndexOutOfBoundsException if the index is out of range
     * (index &lt; 0 || index &gt; size()).
	 */
	public boolean add(int index, ResourceBundle resourceBundle)
	{
		if(resourceBundle == null)
			return false;
			
		resourceBundles.add(index, resourceBundle);
		return true;
	}
	
	/**
	 * Removes a resource bundle from this manager's list. 
	 * @param resourceBundle
	 * @return <code>true</code> if the resource bundle was added or 
	 * <code>false</code> otherwise.
     * @throws IndexOutOfBoundsException if the index is out of range (index
     * &lt; 0 || index &gt;= size()).
	 */
	public boolean remove(ResourceBundle resourceBundle)
	{
		return resourceBundles.remove(resourceBundle);
	}

	/**
	 * Removes a resource bundle located in a specific index of this 
	 * manager's list. 
	 * @param resourceBundleIndex
	 * @return the element previously at the specified position.
	 */
	public ResourceBundle remove(int resourceBundleIndex)
	{
		return (ResourceBundle)resourceBundles.remove(resourceBundleIndex);
	}
	
	/**
	 * Removes all the resource bundles from this manager's list.
	 */
	public void clear()
	{
		resourceBundles.clear();
	}
	
	/**
	 * Returns the iterator of the resource bundles in this 
	 * manager's list.
	 * @return Iterator
	 */
	public Iterator iterator()
	{
		return resourceBundles.iterator();
	}
	
	/**
	 * Returns the number of resource bundles added in this 
	 * manager's list.
	 * @return int
	 */
	public int size()
	{
		return resourceBundles.size();
	}
	
	/**
	 * Returns the string value associate to a given key.  The key is passed to
	 * each resource bundle in the order they are retrieve by the 
	 * {@link #iterator()} method.
	 * @param key
	 * @return String
	 * @throws java.util.MissingResourceException if the key is not in the file
	 * @throws NullPointerException if key is null
	 */
	public String getString(String key)
	throws NullPointerException, MissingResourceException
	{
		MissingResourceException missingResourceException = null;
		for(Iterator i= iterator(); i.hasNext();)
		{
			try
			{
				return ((ResourceBundle)i.next()).getString(key);
			}
			catch (MissingResourceException e)
			{
				missingResourceException = e;
			}
		}
		
		throw missingResourceException;
	}
	
	/**
	 * Returns the string value associate to a given key.  The key is passed to
	 * each resource bundle in the order they are retrieve by the 
	 * {@link #iterator()} method.
	 * 
	 * <p>The <code>arg</code> string defined replaces the %1 
	 * variable defined in the file's values.
	 * 
	 * <p>Example: If the value associated to the key <code>"a"</code> is 
	 * <code>"%0 %1 %2 %3 %4"</code> and arg is <code>"v1"</code>,
	 * the return of this method is <code>"%0 v1  %2 %3 %4"</code>.
	 * 
	 * @param key
	 * @param arg
	 * @return String
	 * @throws java.util.MissingResourceException if the key is not in the file
	 * @throws NullPointerException if key is null
	 */
	public String getString(String key, String arg)
	throws NullPointerException, MissingResourceException
	{
		return StringUtil.replace(getString(key), "%1", arg);
	}	

	/**
	 * Returns the string value associate to a given key.  The key is passed to
	 * each resource bundle in the order they are retrieve by the 
	 * {@link #iterator()} method.
	 * 
	 * <p>The strings defined in <code>args</code> replaces the %n (where n>=1) 
	 * variables defined in the file's values.
	 * 
	 * <p>Example: If the value associated to the key <code>"a"</code> is 
	 * <code>"%0 %1 %2 %3 %4"</code> and args is <code>{"v1", null, "v3"}</code>,
	 * the return of this method is <code>"%0 v1  v3 %4"</code>.
	 * 
	 * @param key
	 * @param args
	 * @return String
	 * @throws java.util.MissingResourceException if the key is not in the file
	 * @throws NullPointerException if key is null
	 */
	public String getString(String key, String[]args)
	throws NullPointerException, MissingResourceException
	{
		String ret = getString(key);
		for (int i = args.length; i > 0; i--)
			ret = StringUtil.replace(ret, ("%"+i), args[i-1]);
		
		return ret;
	}	
}
