/*******************************************************************************
 * Copyright (c) 2005, 2007 IBM Corporation, Intel Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    IBM Corporation - Initial API and implementation
 *    Vishnu K Naikawadi,Intel - Initial API and implementation
 *
 * $Id: IAgent.java,v 1.30 2007/04/23 03:35:47 gnagarajan Exp $ 
 *******************************************************************************/ 

package org.eclipse.tptp.platform.execution.client.agent;

import java.util.Vector;
import org.eclipse.tptp.platform.execution.exceptions.*;
import org.eclipse.tptp.platform.execution.util.*;
import org.eclipse.tptp.platform.execution.client.agent.internal.AgentConfiguration;
import org.eclipse.tptp.platform.execution.client.core.IAgentController;
import org.eclipse.tptp.platform.execution.client.core.IDataProcessor;
import org.eclipse.tptp.platform.execution.client.core.IProcess;

/**
 * This interface defines the services available to the client to interact with the Agent.
 * IAgent is the consumer and controller of the agent that is monitoring the target process.
 * @see org.eclipse.tptp.platform.execution.datacollection.IBaseAgent  
 * <BR>Client operations include launch an application (process), attaching and monitoring the process etc.)</BR> 
 * <BR>
 * <B>Agent Instantiation</B>
 * Agents are instantiated by IAgentController. An agent instance is obtained by the "query" methods or the "getAgent" methods 
 * of the IAgentController interface.  
 * </BR>
 * <BR><B>Agent Access</B>
 * To communicate with an Agent, the client should obtain an agent reference. There are two modes of 
 * agent reference i)Observer ii) Controller. 
 * @see TPTPAgentAccess  
 * </BR>
 * <BR><B>Agent control and communication</B>
 * Messages for control and communication can be passed to the agent by the using the command protocol 
 * defined in the overview. For standard operations like startMonitoring, request access to the agents,
 * interface methods are provided. Customized command messages can be sent to the agents. See the command protocol definition in the overview and
 * the sendCommand method provided by the interface.  
 * </BR>
 * <BR><B>Agent monitoring</B>
 * Clients can send and receive data to/from the agent by providing a dataprocessor and registering a listener.
 * </BR>
 * <BR><B>Interface</B>
 * A token used as an identifier to logically group services. 
 * Example: An interface serviced by the Agent controller is "org.eclipse.tptp.agentManager", this token groups together services provided by the AC.
 * </BR>
 * <BR><B>Command</B>
 * An instruction to the recipient (client, agent or the AC) calling for an action. 
 * A command is the primary mechanism for message passing in the AC and is represented using XML elements. See the command protocol changes for further information.
 * @since tptp 4.0
 * @provisional
 * </BR>
 */

public interface IAgent  
{
   
    /**
     * Detach from the agent
     * @deprecated since TPTP4.0
     * @see org.eclipse.tptp.platform.execution.client.core.IAgentController#getAgent(String, long)
     * @see #releaseControl()
     * @throws InactiveAgentException
     * @throws InactiveProcessException
     */
    //public void detach() throws InactiveAgentException, InactiveProcessException;

    /**
     * Indicator whether this agent is attached to a client 
     * @deprecated since TPTP 4.0
     * @see #getAgentMode()
     */		
	//public boolean isAttached() ;

	/**
	 * Returns the agent version
	 * @deprecated since TPTP 4.0
	 * @return Agent version
	 */
	//public String getVersion();
	
	/**
	 * Start the data collection operation using the DataProcessor provided.
	 * @param processor DataProcessor to receive data
	 * @see #startMonitoring(TPTPDataPath, IDataProcessor)
	 */
	public void startMonitoring(IDataProcessor processor) throws InactiveAgentException ;
	
	/**
	 * Start the data collection operation using the DataProcessor
	 * @param dataPathDirection - Data exchange mode between the Agent and Client @see TPTPDataPath
	 * @see #startMonitoring(TPTPDataPath, IDataProcessor)
	 */
	public void startMonitoring(TPTPDataPath dataPathDirection) throws InactiveAgentException ;
	
	/**
	 * Start the data collection operation using the DataProcessor
	 * @param dataPathDirection - Data exchange mode between the Agent and Client @see TPTPDataPath
	 * @param processor Data Processor to process the received data
	 */
	public void startMonitoring(TPTPDataPath dataPathDirection, IDataProcessor processor) throws InactiveAgentException ;
	
	/**
	 * Stop data collection.
	 * @throws InactiveAgentException
	 */
	public void stopMonitoring() throws InactiveAgentException ;
	
	/**
	 * SendData to the Agent
	 * @param buffer - Data to be sent to the agent 
	 * @throws DataChannelConnectionException
	 */
	public void sendData(byte[] buffer) throws DataChannelConnectionException;
	
	/**
	 * Add a data listener to process incoming data
	 * @see #startMonitoring(TPTPDataPath, IDataProcessor)	 
	 * @param dataProcessor to process the data 
	 * @return int	
	 */
	public int addDataListener(IDataProcessor dataProcessor);
	
	/**
	 * Release Agent reference
	 * @throws InactiveAgentException
	 * @throws NotConnectedException
	 */
	public void releaseAccess() throws InactiveAgentException, NotConnectedException;
	
	/**
	 * Request agent reference to communicate and or control an agent.
	 * See TPTPAgentAccess for different modes.
	 * @param accessMode - Obtain agent reference by specifying a mode
	 * @return boolean
	 * @throws InactiveAgentException
	 * @throws NotConnectedException
	 */
	public boolean requestAccess(TPTPAgentAccess accessMode) throws InactiveAgentException, NotConnectedException;
	
	/**
	 * Receive data from the agent
	 * @param buffer - Incoming data
	 * @return int - Success or failure
	 * @throws InactiveAgentException
	 * @throws NotConnectedException
	 */
	public int receiveData(char[] buffer) throws InactiveAgentException, NotConnectedException;
	
	/**
	 * Request Agent Control, this call is used when a client is referencing an 
	 * agent as an "Observer" and seeks the "Controller" access mode for controlling the agent. 
	 * @param accessMode - Obtain agent control by specifying a mode
	 * @return boolean
	 */
	public boolean requestControl() throws InactiveAgentException, NotConnectedException;
	
	/**
	 * Release Agent Control.This releases the agent reference access mode as a "Controller".
	 * The client should call releaseAccess to release other references to an Agent. 
	 * @throws InactiveAgentException
	 * @throws NotConnectedException
	 */
	public void releaseControl() throws InactiveAgentException, NotConnectedException;	
	/**
	 * Retrieve the token id of this agent.
	 * @return AgentTokenID 
	 */
	//public int getAgentTokenId();
		 
    /**
     * Set the Agent token Id
     * @param AgentTokenID
     */
	//public void setAgentTokenId(int id) ;	
	/**
	 * Retrieve the Agent type
	 * @return The type of the agent if known, null otherwise.
	 * @deprecated since TPTP 4.0
	 * @see #getSupportedInterfaces()
	 */
    //public String getType();

    /**
     * Change the type of the agent
      * @param type 
      * @deprecated since TPTP 4.0 
     */
    //public void setType(String type) ;
	
    /**
     * Retrieve the name of this agent.
     * @return Agent Name
     */
    public String getName();
	
    /**
     * Change the name of the agent
     * @param name of the new agent
     */
	//public void setName(String name) ;
	
	/**
	 * Retrieve the UUID of this agent.
	 * @deprecated since TPTP4.0
	 * @return the uuid of the agent if it is known, null otherwise.
	 */
	//public String getUUID();
//	used in RACAgentimpl
    /**
     * Change the agent's uuid
     * @deprecated since TPTP 4.0
     * @param uuid the new uuid's value 
     */
	//public void setUUID(String uuid) ;
//	used in RACAgentimpl
	/**
	 * Retrieve the instance id of this agent.
	 * @return the id of the agent if it is known, negative if it is not assigned yet
	 */
	//public int getAgentInstanceId();
	 
    /**
     * Change the agent instance's id
     * @param AgentInstanceId 
     */
	//public void setAgentInstanceId(int id) ;	
    	
	
	 
    /**
     * Set the Agent version
     * @param version  the new version number 
     */
	//public void setVersion(String version) ;
    
    /**
     * Set the Process associated with the Agent
     * @param set the Process object of the Agent 
     */
	public void setProcess(IProcess process);
    
    /**
     * Get the Process object associated with the Agent
     * @return IProcess - the process object associated with the agent 
     */
	public IProcess getProcess();

    /**
     * Add a listener for agent events
     * @param interfaceID
     * @param listener 
     */
    public long addEventListener(String interfaceID, ICommandHandler listener) ;

    /**
     * Remove a listener for agent events
     * @param interfaceID
     * @param listener 
     */
    public void removeEventListener(String interfaceID, long listenerID) ;
	
	/**
	 * Send a Command to the AgentController - the AgentController 
	 * forwards or consumes the command on the basis of the value set in the command 
	 * @param outCommand - the command element object to sent
	 * @param respHandler - CommandHandler for processing the response of the command
	 */
	public void sendCommand(ICommandElement outCommand, ICommandHandler respHandler)  throws InactiveAgentException, NotConnectedException;

	/**
	 * Send a command to the Agent. See the overview, for description of the 
	 * command protocol. 
	 * @param outCommand 
	 * @param respHandler - CommandHandler for processing the response of the command
	 * @throws InactiveAgentException
	 * @throws NotConnectedException
	 */
	public void sendCommand(String outCommand, ICommandHandler respHandler) throws InactiveAgentException, NotConnectedException;
	
	/**
	 * Retrieve the AgentConfiguration
	 * @return AgentConfiguration
	 * @deprecated since TPTP 4.0
	 * See Variable interface
	 */
	//public AgentConfiguration getConfiguration();
	
	/**
	 * Set the AgentConfiguration
	 * @param AgentConfiguration
	 * @deprecated since TPTP 4.0
	 * See Variable interface
	 */
	//public void setConfiguration(AgentConfiguration config);
	/**
	 * Publish the AgentConfiguration
	 * @deprecated since TPTP 4.0
	 * See Variable interface
	 */
	//public void publishConfiguration() throws InactiveAgentException;
	
	/**
	 * Interface: A token used as an identifier to logically group services provided by an Agent. 
	 * A list of interfaces provides information about all the services provided by the Agent.  
	 * @return list of supported interfaces
	 */
	public String[] getSupportedInterfaces();
	
	/**
	 * Return the list of Agent Listeners
	 */
	public Vector getAgentListener();
	
	/**
	 * Return the Agent meta data. The Agent MetaData is returned as a
	 * XML document. Consuming clients should process the string for obtaining 
	 * element values.
	 * @return Agent Meta data
	 * @throws NotConnectedException
	 */
	public String getAgentMetaData() throws NotConnectedException;
	
	/**
	 * Get the Agent reference mode - Observer or Controller
	 * @return int - TPTPAgentAccess
	 */
	public int getAgentMode();
	
	/**
	 * Return the Agent Monitoring status
	 * @return isMonitored
	 */
	public boolean isMonitored();
	
	
	/**
	 * Add a Agent listener to process incoming data
	 */
	public void addAgentListener(IAgentListener listener);
}
