/*******************************************************************************
 * Copyright (c) 2007 Intel Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Ruslan Scherbakov, Intel - initial API and implementation
 *
 * $Id: AgentStatePool.java,v 1.2 2007/06/17 05:17:33 gnagarajan Exp $
 *
 *******************************************************************************/ 

package org.eclipse.tptp.platform.execution.client.agent.internal;

import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.Hashtable;

import org.eclipse.tptp.platform.execution.client.core.ConnectionInfo;

/**
 * This class accumulates states of all agents created by different requests to
 * AC.
 */
public class AgentStatePool {

	private static Object lock = "";
	private static String localHostName = "";
	private static Hashtable agentStateTable = new Hashtable();

	/**
	 * Creates unified key uniquely identifying running agent.
	 * 
	 * @param host - host name where agent resides
	 * @param port - port on which AC is listening
	 * @param pid - ID of the process which agent corresponds to
	 * @return
	 */
	static public String getAgentKey(String host, int port, long pid) {
		host = getUnifiedHostName(host);
		//String key = host + ":" + port + ":" + pid;
		String key = host + ":" + pid;
		return key;
	}

	/**
	 * Gets agent state.
	 * 
	 * @param host - host name where agent resides
	 * @param port - port on which AC is listening
	 * @param pid - ID of the process which agent corresponds to
	 * @return
	 */
	static public AgentState getAgentInfo(String host, int port, long pid) {
		String key = getAgentKey(host, port, pid);
		AgentState info = null;
		synchronized (lock) {
			if (agentStateTable.containsKey(key)) {
				info = (AgentState) agentStateTable.get(key);
			}
		}
		return info;
	}

	/**
	 * Gets agent state.
	 * 
	 * @param con - connection info for the host where agent resides
	 * @param pid - ID of the process which agent corresponds to
	 * @return
	 */
	static public AgentState getAgentInfo(ConnectionInfo con, long pid) {
		return getAgentInfo(con.getHostName(), con.getPort(), pid);
	}

	/**
	 * Gets existing agent state or creates new if firstly requested.
	 * 
	 * @param con - connection info for the host where agent resides
	 * @param pid - ID of the process which agent corresponds to
	 * @return
	 */
	static public AgentState createAgentInfo(ConnectionInfo con, long pid) {
		return createAgentInfo(con.getHostName(), con.getPort(), pid);
	}

	/**
	 * Gets existing agent state or creates new if firstly requested.
	 * 
	 * @param host - host name where agent resides
	 * @param port - port on which AC is listening
	 * @param pid - ID of the process which agent corresponds to
	 * @return
	 */
	static public AgentState createAgentInfo(String host, int port, long pid) {
		String key = getAgentKey(host, port, pid);
		AgentState info = null;
		synchronized (lock) {
			if (agentStateTable.containsKey(key)) {
				info = (AgentState) agentStateTable.get(key);
			} else {
				info = new AgentState();
				agentStateTable.put(key, info);
			}
		}
		return info;
	}

	static public void setAgentInfo(String host, int port, long pid, AgentState info) {
		String key = getAgentKey(host, port, pid);
		synchronized (lock) {
			agentStateTable.put(key, info);
		}
	}

	/**
	 * The methods checks wether the host name can have several representation
	 * for local name and provides common name for it.
	 * 
	 * @param hostName - input host name
	 * @return <code>String</code> corrected unified host name
	 */
	static public String getUnifiedHostName(String hostName) {
		hostName = hostName.toLowerCase();
		try {
			if (localHostName.length() == 0) {
				localHostName = InetAddress.getLocalHost().getHostName();
				localHostName = localHostName.toLowerCase();
			}
		} catch (UnknownHostException e) {
			localHostName = "localhost";
			e.printStackTrace();
		}
		if (hostName.compareTo(localHostName) == 0
				|| hostName.compareTo("localhost") == 0
				|| hostName.startsWith("127.0.0.")) {
			hostName = localHostName;
		}
		return hostName;
	}
}
