/*******************************************************************************
 * Copyright (c) 2005, 2010 Intel Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Guru Nagarajan,Intel - Initial API and implementation
 *
 * $Id: IAgentController.java,v 1.22 2010/12/23 20:35:03 jwest Exp $ 
 *******************************************************************************/ 
package org.eclipse.tptp.platform.execution.client.core;

import java.util.Enumeration;

import org.eclipse.tptp.platform.execution.client.agent.IAgent;
import org.eclipse.tptp.platform.execution.exceptions.NotConnectedException;
import org.eclipse.tptp.platform.execution.security.User;
import org.eclipse.tptp.platform.execution.util.ICommandHandler;
import org.eclipse.tptp.platform.execution.util.TPTPAgentAccess;
/**
 * This interface represents the Agent Controller running on the target machine(either local or remote machine). 
 * The Agent Controller, provides services to launch new processes, and attach to agents that coexist within processes. 
 * It also provides extensible agent architecture for control capabilities and data collection. 
 * It provides capabilities for performing operations as described in this interface. 
 * Agent hosted within the Agent Controller may communicate data back to monitoring clients (e.g. TPTP workbench). 
 * Multiple agent controllers can reside on a target host (will service different ports) and a client can connect to multiple agent controllers on the target host. 
 * This interface is the source for instantiating IAgent and IProcess objects. 
 * @provisonal
 * @since tptp 4.0
 */
public interface IAgentController
{
	/**
	 * Retrieve the instance of INode associated with the Agent Controller 
	 * @return INode
	 */
	public INode getNode();
	
	/**
	 * Retrieve the AgentController version number
	 * @return version 
	 */
	//public String getVersion();
	
	/**
	 * Get the Next Context ID for sending commands
	 * @return long
	 */
	public long getNextContextId();
	
	/**
	 * Disconnect from Agent Controller
	 * @return long
	 */
	public void disconnect();
	
	/** 
	 * Retrieve an existing Agent instance, This method will create a new instance if there is no existing agent instance available
	 * @return IAgent
	 * @param agentName
	 * @param agentClassName 
	 * @throws NotConnectedException
	 */
	public IAgent getAgent(String agentName, String agentClassName) throws NotConnectedException;
	
	/** 
	 * Retrieve an Agent instance.
	 * @param agentName 
	 * @param agentClassName 
	 * @param accessMode see TPTPAgentAccess 
	 * @return IAgent 
	 * @throws NotConnectedException
	 */
	public IAgent getAgent(String agentName, String agentClassName, TPTPAgentAccess accessMode) throws NotConnectedException;
	
	/**
	 * Retrieve an Agent instance that is bound to the input process id of the target process. 
	 * @param agentName
	 * @param processId
	 * @return IAgent
	 * @throws NotConnectedException
	 */
	public IAgent getAgent(String agentName, long processId) throws NotConnectedException;
	/** 
	 * Retrieve an Agent instance.
	 * @param agentName 
	 * @param agentClassName 
	 * @param processId
	 * @return IAgent 
	 * @throws NotConnectedException
	 */
	public IAgent getAgent(String agentName, String agentClassName, long processId) throws NotConnectedException;
    /**
     * Enumerate all the active processes on this node.
     * @return Enumeration
     * @throws NotConnectedException
     */
    public Enumeration listProcesses() throws NotConnectedException;
    
    /**
     * Enumerate all the active agents on this node.
     * @return Enumeration
     * @throws NotConnectedException
     */
    public Enumeration listAgents() throws NotConnectedException;

    /**
     * Retrieve a specific Process on this Node
     * @param processId
     * @return IProcess
     * @throws NotConnectedException
     */
    public IProcess getProcess(long processId) throws NotConnectedException;
    
    /**
     * Blocks calling thread until the process is retrieved, if the process
     * cannot be retrieved within the timeout period, null is returned for the
     * process return value.
     *
     * @param processIdentity
     *            the identity of the process to wait for
     * @param timeout
     *            the maximum time to wait for the process to be retrieved
     * @return the process retrieved or null if the timeout is reached before
     *         the process is found
     * @throws NotConnectedException,InactiveProcessException
     * @deprecated since TPTP 4.0
     */
    /*public IProcess waitForProcess(String processIdentity, int timeout)
            throws NotConnectedException, InactiveProcessException;
*/

    /**
     * Retrieve the list of available agents on the Agent Controller.
     * @return Agent list
     * @throws NotConnectedException
     */
	public String[] queryAvailableAgents() throws NotConnectedException;
	
	/**
	 * @param interfaceID
	 * @return Agent list
	 * @throws NotConnectedException
	 */
	public String[] queryAvailableAgents(String[] interfaceID) throws NotConnectedException;
	
	/**
	 * Returns the list of running agents
	 * @return list of running agents
	 * @throws NotConnectedException
	 */
	public IAgent[] queryRunningAgents() throws NotConnectedException;
	/** 
	 * Returns the list of running agents filtered by interface ID
	 * @return list of running agents
     * @throws NotConnectedException 
	 */
	public IAgent[] queryRunningAgents(String[] interfaceID) throws NotConnectedException;
	/** 
	 * Returns the agent metadata information
	 * @param agentName 
	 * @return java.lang.String 
     * @throws NotConnectedException
	 */
	//public String getAgentMetadata(String agentName)throws NotConnectedException;
	
	/** 
	 * Transfers the file from target machine to this machine 
	 */
	//public int copyFileFromTarget(String sourceFilePath, String destinationFilePath);
	
	/* Transfers the file from local to target machine */
	//public int copyFileToTarget(String sourceFilePath, String destinationFilePath);
	
	/**
	 * Returns instance of File Transfer Manager that provides file 
	 * transfer services like Get File, Put File and Delete File
	 * @return IFileTransferManager 
	 */
	public IFileTransferManager getFileTransferManager();
	
	/**
	 * Add the Event Listener to listen to the events (specified by the interfaceID) from the Agent Controller
	 * @param interfaceID
	 * @param eventHandler
	 * @return int
	 * 
	 */
	public int addEventListener(String interfaceID, ICommandHandler eventHandler);
	
	/**
	 * Remove the Event Listener - Stop listening to the events (specified by interfaceID)
	 * @param interfaceID
	 * @param eventHandler
	 */
	public void removeEventListener(String interfaceID, ICommandHandler eventHandler);
	
	/** 
	 * Send a command to the specified destination
	 * @param Command String 
	 * @param Destination ID
	 * @param ICommandHandler 
     * @throws NotConnectedException 
	 */
	//public void sendCommand(String cmd,  int destID, ICommandHandler handler) throws InactiveProcessException, IOException;
	
	/** 
	 * Send a command to the specified destination 
	 * @param IControlMessage 
	 * @param ICommandHandler 
     * @throws NotConnectedException 
	 */
	//public void sendMessage(IControlMessage message, ICommandHandler handler) throws InactiveProcessException, IOException;
	
	/** 
	 * Send data over the data channel 
	 * @param dataConnID
	 * @param buffer
	 * @param bufferLength 
     * @throws throws DataChannelConnectionException; 
	 */
	//public int sendData(int dataConnID, byte[] buffer, int bufferLength) throws DataChannelConnectionException;
	
	/**
	 * Create the Process object to store process related information
	 * @return IProcess
	 */
	public IProcess createProcess();

	/**
	 * Create the Process and set the executable attirbute of the process object
	 * @param executable
	 */
	public IProcess createProcess(String executable);

	/**
	 * Create the Process and set the executable attirbute of the process object, and parameters
	 * @param executable
	 * @param parameters
	 * @return IProcess
	 */
	public IProcess createProcess(String executable, String parameters);
	
	
    /**
    * Retrieve the user for this node.
    * @return User
    */
    //public User getUser();

    /**
     * Update the user for this node.
     * @param User
     */
    //public void setUser(User user);

    /**
     * Set the Security Parameters required for making a secured connection 
     * to Agent Controller.
     * @param ISecureClientParameters
     */
    //public void setSecurityParameters(ISecureClientParameters manager);

    /** 
     * Retrieve the Security Parameters
     * @return ISecureClientParameters
     */
    //public ISecureClientParameters getSecurityParameters();
    
    /** 
     * Get the Connection Infomation - protocol, port number, secured connection info( for secured connection)
     */
    public ConnectionInfo getConnectionInfo();
    
	
	/*
	 * Authenticate the User Credentials with AC
	 *
	 */
	boolean authenticateUser(User user) throws NotConnectedException;
     
	boolean isAuthenticated();
	
	/**
	 * Get a Agent Reference to communicate with the agent
	 * @param agentName
	 * @param accessMode
	 * @return AgentID
	 * <Pre> 
	 * Access Modes can be
	 * TPTP_CONTROLLER_ACCESS - Request Control Access on the Agent
	 * TPTP_OBSERVER_ACCESS - Request Observer Access on the Agent
	 * TPTP_CREATE_INSTANCE - Creates a new instance even if an existing instance is available
	 * TPTP_RUNNING_AGENTS_ONLY - Request only the running agents
	 * TPTP_LOCK_AGENT - Request exclusive usage and others can't share this agent
	 * </Pre>
	 */
    //public int getAgent(String agentName, TPTPAgentAccess accessMode) throws NotConnectedException;
    /**
     * Retrieve an agent instance
     * @param agentName
     * @param accessMode
     * @return IAgent
     * @throws NotConnectedException
     */
    public IAgent getAgent(String agentName, TPTPAgentAccess accessMode) throws NotConnectedException;

	/** 
	 * Query Agent by a processid, agent name and interface id.
	 * @param agentName
	 * @param pid
	 * @return IAgent
	 */
	public IAgent[] queryRunningAgents(String agentName, String[] interfaceID, long pid) throws NotConnectedException;

	/**
	 * Get the Agent By the Agent Token returned by the Agent Controller when an agent is queried
	 * @param AgentTokenId
	 * @param accessMode
	 * @return AgentID
	 * <Pre>
	 * 	TPTP_CONTROLLER_ACCESS - Request Control Access on the Agent
	 * 	TPTP_OBSERVER_ACCESS - Request Observer Access on the Agent
	 * 	TPTP_LOCK_AGENT - Request exclusive usage and others can't share this agent
	 * </Pre>
	 */
	//public int getAgentByToken(int token, int accessMode) throws NotConnectedException;

	/** 
	 * Get the Agent Reference by using an Agent ID.  
	 * @param agentID
	 * @param accessMode
	 * @return AgentID 
	 * <Pre>
	 * TPTP_CONTROLLER_ACCESS - Request Control Access on the Agent
	 * TPTP_OBSERVER_ACCESS - Request Observer Access on the Agent
	 * TPTP_LOCK_AGENT - Request exclusive usage and others can't share this agent
	 * </Pre>
	 */
	//public int getAgentByID(int agentID, TPTPAgentAccess accessMode) throws NotConnectedException;
	/**
	 * Release an Agent reference. This reference is used by the Agent Controller 
	 * to manage the agents and provide agent information to the client.  
	 * @param agentName
	 * @throws NotConnectedException
	 */
	//public void releaseAgent(String agentName)throws NotConnectedException;
	
	/**
	 * Request agent control to perform specific agent operations
	 * that are associated with the relevant access modes.
	 * @param agentID
	 * @param accessMode
	 * @return
	 * @throws NotConnectedException
	 */
	//public boolean requestAgentControl(int agentID, int accessMode)throws NotConnectedException;
	
	/**
	 * Release an agent control. A Client obtains an agent control access mode by requesting 
	 * a agent reference with Controller access.
	 * @param agentID
	 * @throws NotConnectedException
	 */
	//public void releaseAgentControl(int agentID)throws NotConnectedException;
	
	/**
	 * BUG 307787
	 * 
	 * Ask the agent controller where it is installed on the remote machine. We can use this
	 * when generating the JVM args to use -agentpath parameter, removing the need for the user
	 * to set environment variables.
	 * 
	 * @return
	 * Returns null if the agent controller does not support this query.
	 * @throws NotConnectedException 
	 */
	public ACPathInfo queryACPath() throws NotConnectedException;
	
	/**
	 * BUG 307787
	 * 
	 * Ask the agent controller whether or not it supports the -agentpath parameter in lieu
	 * of -agentlib. Generally will be used as a conditional before using .queryACPath(). 
	 * 
	 * @return true if -agentpath is supported, otherwise false
	 * 
	 * @throws NotConnectedException
	 */
	public boolean isAgentPathSupported() throws NotConnectedException;
	

	public enum AC_PLATFORM {	WIN_X86, WIN_X64, WIN_IPF, 
		LINUX_X86, LINUX_X64, LINUX_IPF, LINUX_390_32, LINUX_390_64,
		AIX_PPC32, AIX_PPC64, 
		SOLARIS_X86, SOLARIS_X64, SOLARIS_SPARC32, SOLARIS_SPARC64, 
		ZOS_31, ZOS_64 }; 
	
	public AC_PLATFORM getACPlatform()  throws NotConnectedException;
	
	public String getACHostTime()  throws NotConnectedException;
}
