/*******************************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation, Intel Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    IBM Corporation - Initial API and implementation
 *    Vishnu K Naikawadi, Intel - Initial API and implementation
 *
 * $Id$ 
 *******************************************************************************/

package org.eclipse.tptp.platform.execution.client.core.internal;

import java.io.IOException;
import java.io.InputStream;
import java.io.InterruptedIOException;
import java.net.InetAddress;
import java.net.ServerSocket;
import java.net.Socket;
import java.net.SocketException;
import java.util.Enumeration;
import java.util.Vector;

import org.eclipse.tptp.platform.execution.client.core.IDataProcessor;
import org.eclipse.tptp.platform.execution.util.internal.Constants;
import org.eclipse.tptp.platform.execution.util.internal.DimeHeader;

public final class ACTCPDataServer {

	private ServerSocket _sock = null;
	private Socket _incommingConnection = null;
	private TCPDataProcessor _server = null;
	private ACBufferFlusher _flusher = null;
	private int _port;

	/*
	 * Buffers for getting incomming data off the socket reader and onto a dispatch thread
	 */
	private static final short NUM_BUFFERS = 32;
	
	/** '_sh' variables constitute variables shared by both the reader and the flusher. All other variables not labelled as such
	 * are kept separate between the two (i.e. not shared). */
	private SingleBuffer[] _shBufferArray;
	private short _shCurrentFullBuffers = 0;
	
	/** Lock on shBufferArray and shCurrentFullBuffers. Was previously _bufferArray[0] */
	private Object _sharedLock = new Object(); 

	/* Different types of data */
	public static final byte BINARY_DATA = 0;
	public static final byte UTF8_STRING_DATA = 1;
	public static final byte UNICODE_STRING_DATA = 2;

	private Vector _dataServerListeners = new Vector();

	class SingleBuffer implements Constants {
		public InetAddress addr;
		public int length = 0;
		public int size = MAX_MESSAGE_LENGTH;
		public byte[] data = new byte[MAX_MESSAGE_LENGTH];

	}

	class TCPDataProcessor extends Thread implements Constants {
		ServerSocket _socket;
		Socket incommingConnection;
		boolean _processing = true;
		boolean _shutdown = false; /* 9707 */
		private short _currentFillerBuffer = 0;

		public void setSocket(ServerSocket sock) {
			_socket = sock;
		}

		public void setIncommingConnection(Socket sock) {
			incommingConnection = sock;
		}

		/* 9707 */
		public void resumeProcessing() {
			synchronized (this) {
				_processing = true;
			}
		}

		/* 9707 */
		public void pauseProcessing() {
			synchronized (this) {
				_processing = false;
			}
		}

		/* 9707 */
		public boolean isProcessing() {
			synchronized (this) {
				return _processing;
			}
		}

		/* 9707 */
		public void shutdown() {
			_shutdown = true;
		}

		public void run() {
			/* Run forever */
			// Check if it is AC - AC has created the socket already
			if (incommingConnection != null) {

				outer: while (!_shutdown || (_shCurrentFullBuffers != 0)) {
				
					if (isProcessing()) {
						InputStream is = null;
						try {
							is = incommingConnection.getInputStream();
						} catch (SocketException e) {
							/* The server socket is toast, stop processing */
							pauseProcessing();
							continue;
						} catch (IOException e) {
							System.out.println("The IOException is " + e);
							continue;
						}

						while (true) {
							/* If all the buffers are full wait for the first one to be emptied */
							while (_shCurrentFullBuffers == NUM_BUFFERS) {
								synchronized (_sharedLock) {
									try {
										_sharedLock.wait();
									} catch (InterruptedException e) {
									}
								}
							}

							/* Fill the next buffer */
							_shBufferArray[_currentFillerBuffer].addr = incommingConnection.getInetAddress();
							
							try {
								_shBufferArray[_currentFillerBuffer].length = is.read(_shBufferArray[_currentFillerBuffer].data);
								/* System.out.println("---Read "+_bufferArray[_currentFillerBuffer].length+" bytes"); */
							} catch (InterruptedIOException e) {
								/* Read timeout, don't want to wait too long */

								/* This is used to terminate the thread if the process is killed abnormally */
								if (_shutdown && (_shCurrentFullBuffers == 0)) {
									pauseProcessing();
									try {
										incommingConnection.close(); /* 9707 */
									} catch (IOException e1) {
										// Handle IOException
									}
									break outer;
								}
							} catch (IOException e) {
								/* Socket is toast, we are done */
								pauseProcessing();
								try {
									incommingConnection.close(); /* 9707 */
								} catch (IOException e1) {
									// Handle IOException
								}
								break outer;
							}

							/* Is the connection closed? */
							if (_shBufferArray[_currentFillerBuffer].length < 0) {
								pauseProcessing();
								try {
									/* Will hit here when detaching agent */
									incommingConnection.close();
								} catch (IOException e) {
								}
								break outer;
							}
							synchronized (_sharedLock) {
								if (_shBufferArray[_currentFillerBuffer].length > 0) {
									/* Move on to the next buffer */
									_currentFillerBuffer++;
									if (_currentFillerBuffer == NUM_BUFFERS) {
										_currentFillerBuffer = 0;
									}
									_shCurrentFullBuffers++;

									/* Is this the first buffer filled? */
									if (_shCurrentFullBuffers == 1) {
										_sharedLock.notifyAll();
									}
								}

							}
						}
					} else {

						try {
							/* Monitoring is stopped, keep this thread in sleep state */
							sleep(1000); /* 9707 */
						} catch (InterruptedException e) {
						}
					}
				}
			}
		}

	} /* end class TCPDataProcessor */

	/* BufferFlusher for AC */
	class ACBufferFlusher extends Thread implements Constants {
		/**
		 * Increased from 8192 for bug 298837, due to problems with thread
		 * contention messages being larger than 8192.
		 */
		private final static int BINARY_FORWARD_BUFFER_SIZE = 32768;
		private Vector/* <IDataProcessor> */_processorList = new Vector();

		private byte[] _binaryForwardBuffer = new byte[BINARY_FORWARD_BUFFER_SIZE];
		private byte[] _messageHeader = new byte[MAX_MESSAGE_LENGTH];

		private short _currentFlusherBuffer = 0;
		private int _currentHeaderOffset = 0;
		private int _bytesWritten = 0;
		private DimeHeader _dimeHeader = null;

		public void addDataProcessor(IDataProcessor processor) {
			_processorList.add(processor);
		}

		public void removeDataProcessor(IDataProcessor processor) {
			_processorList.remove(processor);
		}

		/**
		 * Load the _messageHeader from a buffer of data
		 * 
		 * @param data
		 *            - the byte[] holding the raw data.
		 * @param offset
		 *            - where the header starts in the raw data buffer.
		 * @param length
		 *            - the length of the raw data buffer.
		 * @returns - the new offset in the raw data buffer to continue reading from.
		 */
		private int loadMessageHeader(byte[] data, int offset, int limit) {
			
			/* Load all we can into the header */
			while (offset < limit && _currentHeaderOffset < DIME_HEADER_LEN) {
				_messageHeader[_currentHeaderOffset++] = data[offset++];
			}
			_dimeHeader = DimeHeader.getDIMEHeader(_messageHeader);
			return offset;
		}

		private int loadMessageHeaderDetails(byte[] data, int offset, int limit) {
			
			/* Load all we can into the header */
			while (offset < limit && _currentHeaderOffset < (DIME_HEADER_LEN + _dimeHeader.getIDLength() + _dimeHeader.getOptionsLength() + _dimeHeader.getTypeLength())) {
				_messageHeader[_currentHeaderOffset++] = data[offset++];
			}
			return offset;
		}

		/**
		 * Get the length of the current message
		 */
		protected long getMessageLength() {
			return _dimeHeader.getDataLength();
		}

		/**
		 * Check the message magic number. If the magic number is incorrect we need to go into recovery mode
		 */
		private boolean checkMessageMagic() {
			return true;
		}

		/**
		 * Recursively process a buffer of raw data.
		 */
		private int processData(byte[] data, int offset, int limit, InetAddress addr) {
			long messageLength;

			/** Current is a pointer to the next bytes in data; it begins at offset, and goes to limit */
			int current;

			current = offset;

			// System.out.println("The ProcessData() method called with - " + data);
			/* Is there data to process */
			if (offset >= limit) {
				return limit;
			}

			/* Is this a new message? */
			if (_currentHeaderOffset < DIME_HEADER_LEN) {
				/* Load the message header */
				current = this.loadMessageHeader(data, current, limit);

				/* Did we get the entire header, if not return */
				if (current == limit) {
					return current;
				}
				// System.arraycopy(_messageHeader, 0, _binaryForwardBuffer, 0, Constants.DIME_HEADER_LEN);
			}

			if (_currentHeaderOffset < (DIME_HEADER_LEN + _dimeHeader.getIDLength() + _dimeHeader.getOptionsLength() + _dimeHeader.getTypeLength())) {
				/* Load the message header */
				current = this.loadMessageHeaderDetails(data, current, limit);

				/* Did we get the entire header, if not return */
				if (current == limit && _currentHeaderOffset < (DIME_HEADER_LEN + _dimeHeader.getIDLength() + _dimeHeader.getOptionsLength() + _dimeHeader.getTypeLength())) {
					return current;
				}
				if (_dimeHeader.getDataLength() > 0 && current == limit) {
					return current;
				}
				// System.arraycopy(_messageHeader, 0, _binaryForwardBuffer, 0, Constants.DIME_HEADER_LEN);
			}

			/*
			 * Validate the message header, if we are in recovery mode try and look at the next offset
			 */
			if (!checkMessageMagic()) {
				System.out.println("Corrupt data");
				_currentHeaderOffset = 0;
				return processData(data, offset + 1, limit, addr);
			}

			/* Read the total message length from the DIME header */
			messageLength = getMessageLength();

			if (messageLength == 0) {
				Enumeration processorList = _processorList.elements();
				while (processorList.hasMoreElements()) {
					IDataProcessor currentProcessor = (IDataProcessor) processorList.nextElement();

					if (TPTPMessageDebug.TPTP_MSG_DEBUG && TPTPMessageDebug.TPTP_MSG_DEBUG_PRINT_DATA_IN) {
						TPTPMessageDebug.writeDebugMessage("data received - procMessages1: [" + new String(_binaryForwardBuffer, 0, _bytesWritten) + "] (dp:" + currentProcessor.getClass() + ") (isDime:" + (currentProcessor instanceof IDataProcessorWithDime) + ")");
					}

					if (currentProcessor instanceof IDataProcessorWithDime) {
						((IDataProcessorWithDime) currentProcessor).incomingData(_binaryForwardBuffer, _bytesWritten, addr, _messageHeader);
					} else if (currentProcessor instanceof IDataProcessor) {
						currentProcessor.incomingData(_binaryForwardBuffer, _bytesWritten, addr);
					}
				}
			}

			/* Process the entire buffer */
			while (current < limit) {
				// While there are still bytes to process.....

				// Copy as many bytes as possible into the forwarding buffer:
				// - While data remains to be copied, and while we haven't written more than the total message length
				while (current < limit && _bytesWritten < messageLength) {
					_binaryForwardBuffer[_bytesWritten++] = data[current++];
				}
				/* Are we at the end of the message? If so forward to the handler */
				if (_bytesWritten == messageLength) {
					Enumeration processorList = _processorList.elements();
					while (processorList.hasMoreElements()) {
						IDataProcessor currentProcessor = (IDataProcessor) processorList.nextElement();

						if (TPTPMessageDebug.TPTP_MSG_DEBUG) {
							boolean debugIsConsole = false;

							// This is not the most accurate way of determining
							// whether or not it is console, but this is only
							// debug code.
							debugIsConsole = currentProcessor.getClass().getName().indexOf("IOProxy") != -1;

							if (debugIsConsole) {
								if (TPTPMessageDebug.TPTP_MSG_DEBUG_PRINT_CONSOLE_IN) {
									TPTPMessageDebug.writeDebugMessage("Console data received [" + (new String(_binaryForwardBuffer, 0, _bytesWritten)).trim() + "] (dp:" + currentProcessor.getClass() + ") (isDime:" + (currentProcessor instanceof IDataProcessorWithDime) + ")");
								}

							} else {

								if (TPTPMessageDebug.TPTP_MSG_DEBUG_PRINT_DATA_IN) {
									TPTPMessageDebug.writeDebugMessage("Data received - procMessages2 [" + (new String(_binaryForwardBuffer, 0, _bytesWritten)).trim() + "] (dp:" + currentProcessor.getClass() + ") (isDime:" + (currentProcessor instanceof IDataProcessorWithDime) + ")");
								}
							}
						}

						if (currentProcessor instanceof IDataProcessorWithDime) {
							((IDataProcessorWithDime) currentProcessor).incomingData(_binaryForwardBuffer, _bytesWritten, addr, _messageHeader);
						} else if (currentProcessor instanceof IDataProcessor) {
							currentProcessor.incomingData(_binaryForwardBuffer, _bytesWritten, addr);
						}
					}

					_bytesWritten = 0;
					_currentHeaderOffset = 0;
					/* Continue processing this data buffer */
					current = this.processData(data, current, limit, addr);
				}
			}
			return current;
		}

		public void run() {

			outer: while (isProcessing() || (_shCurrentFullBuffers != 0)) { /* 237169 make sure buffer is empty before exiting */
				/* If there are no current buffers to empty wait */
				if (_shCurrentFullBuffers == 0) {
					Enumeration processorList = _processorList.elements();
					while (processorList.hasMoreElements()) {
						IDataProcessor currentProcessor = (IDataProcessor) processorList.nextElement();
						currentProcessor.waitingForData();
					}
					do {
						synchronized (_sharedLock) {
							try {
								_sharedLock.wait(1000);
							} catch (InterruptedException e) {
							}
						}
						if (!isProcessing() && _shCurrentFullBuffers == 0) {
							break outer;
						}
					} while (_shCurrentFullBuffers == 0);
				}

				/* Empty the current buffer */
				if (_shBufferArray[_currentFlusherBuffer].length > 0) {
					// System.out.println("---- Flushing "+ _bufferArray[_currentFlusherBuffer].length+" bytes");
					processData(_shBufferArray[_currentFlusherBuffer].data, 0, _shBufferArray[_currentFlusherBuffer].length, _shBufferArray[_currentFlusherBuffer].addr);
					/* Mark the buffer as empty */
					_shBufferArray[_currentFlusherBuffer].length = 0;

				}

				synchronized (_sharedLock) {

					_shCurrentFullBuffers--;

					/* Increment the flusher to the next buffer */
					_currentFlusherBuffer++;
					if (_currentFlusherBuffer == NUM_BUFFERS) {
						_currentFlusherBuffer = 0;
					}

					/*
					 * If the buffers were half full before this flush notify
					 * the filler it can continue. Generally this could be
					 * NUMBUFFERS but not as efficient as more thread switches
					 * happen
					 */
					if (_shCurrentFullBuffers == 0) {
						_sharedLock.notifyAll();
					}
				}
			}

			/* Notify that the flusher is exiting */
			// if(_processor instanceof DataServerListener) {
			// ((DataServerListener)_processor).dataServerExited();
			// }
		}

	} /* end class RingBufferFiller */

	/**
	 * TCPDataServer constructor comment.
	 */
	public ACTCPDataServer() {
		super();
	}

	public int getPort() {
		return _port;
	}

	public InetAddress getServerAddress() {
		return _sock.getInetAddress();
	}

	public void startServer(IDataProcessor processor, int port) throws SocketException, IOException {
		/* Create the buffering mechanism */
		_shBufferArray = new SingleBuffer[NUM_BUFFERS];
		for (int i = 0; i < NUM_BUFFERS; i++)
			_shBufferArray[i] = new SingleBuffer();

		/* Start the server */
		_sock = new ServerSocket(port);

		_port = _sock.getLocalPort();

		startDataBuffers(processor);
	}

	public void startServer(IDataProcessor processor, Socket dataSocket) throws SocketException, IOException {
		/* Create the buffering mechanism */
		_shBufferArray = new SingleBuffer[NUM_BUFFERS];
		for (int i = 0; i < NUM_BUFFERS; i++)
			_shBufferArray[i] = new SingleBuffer();

		_port = dataSocket.getLocalPort();
		_incommingConnection = dataSocket;

		startDataBuffers(processor);
	}

	private void startDataBuffers(IDataProcessor processor) {
		_server = new TCPDataProcessor();
		_server.setSocket(_sock);
		_server.setIncommingConnection(_incommingConnection);
		_server.setName("TCPDataFiller");

		/* Set up the data flusher */
		_flusher = new ACBufferFlusher();
		if (processor != null) {
			_flusher.addDataProcessor(processor);
		}
		_flusher.setName("TCPDataFlusher");

		/* Don't block exit of the VM */
		_server.setDaemon(true);
		_flusher.setDaemon(true);

		/*
		 * RKD: Because these two threads intensely contend for the same
		 * resources we need to increase the priority of the flusher, otherwise
		 * I have noticed that the flusher doesn't run often enough and the data
		 * sits in the buffers far too long. A relitive increase of 3 seems to
		 * much better
		 */

		_flusher.setPriority(_server.getPriority() + 3);

		_server.start();
		_flusher.start();
	}

	public void startServer(IDataProcessor processor) throws SocketException, IOException {
		startServer(processor, 0);

	}

	public void addDataprocessor(IDataProcessor processor) {
		_flusher.addDataProcessor(processor);
	}

	public void removeDataprocessor(IDataProcessor processor) {
		_flusher.removeDataProcessor(processor);
	}

	public boolean isProcessing() {
		return _server.isAlive();
	}

	public void stopServer() {
		_server.pauseProcessing(); /* 9707 */
	}

	/* 9707 */
	public void resumeServer() {
		_server.resumeProcessing();
	}

	/* 9707 */
	public void resumeServer(IDataProcessor processor) {
		_flusher.addDataProcessor(processor);
		_server.resumeProcessing();
	}

	/* 9707 */
	public void shutdownServer() {
		_server.shutdown();
	}
}
