/*******************************************************************************
 * Copyright (c) 2005, 2010 IBM, Intel Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Guru Nagarajan, Intel - initial API and implementation
 *    IBM - Hyades/TPTP 3.0 - Initial Java API and implementation
 *
 * $Id$
 *
 *******************************************************************************/ 
package org.eclipse.tptp.platform.execution.client.core.internal;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.Vector;

import org.eclipse.tptp.platform.execution.client.agent.IAgent;
import org.eclipse.tptp.platform.execution.client.agent.internal.AgentImpl;
import org.eclipse.tptp.platform.execution.client.agent.internal.CollectorImpl;
import org.eclipse.tptp.platform.execution.client.core.ACPathInfo;
import org.eclipse.tptp.platform.execution.client.core.ConnectionInfo;
import org.eclipse.tptp.platform.execution.client.core.IAgentController;
import org.eclipse.tptp.platform.execution.client.core.IDataProcessor;
import org.eclipse.tptp.platform.execution.client.core.IFileTransferManager;
import org.eclipse.tptp.platform.execution.client.core.INode;
import org.eclipse.tptp.platform.execution.client.core.IProcess;
import org.eclipse.tptp.platform.execution.client.core.internal.commands.AgentDetailsCommand;
import org.eclipse.tptp.platform.execution.client.core.internal.commands.GetACHostTimeCommand;
import org.eclipse.tptp.platform.execution.client.core.internal.commands.GetACPlatformCommand;
import org.eclipse.tptp.platform.execution.client.core.internal.commands.QueryAvailableAgntCommand;
import org.eclipse.tptp.platform.execution.exceptions.DataChannelConnectionException;
import org.eclipse.tptp.platform.execution.exceptions.InactiveAgentException;
import org.eclipse.tptp.platform.execution.exceptions.InactiveProcessException;
import org.eclipse.tptp.platform.execution.exceptions.NotConnectedException;
import org.eclipse.tptp.platform.execution.exceptions.NotSupportedException;
import org.eclipse.tptp.platform.execution.exceptions.ProcessActiveException;
import org.eclipse.tptp.platform.execution.exceptions.TimeoutException;
import org.eclipse.tptp.platform.execution.security.ISecureClientParameters;
import org.eclipse.tptp.platform.execution.security.User;
import org.eclipse.tptp.platform.execution.util.ICommandElement;
import org.eclipse.tptp.platform.execution.util.ICommandHandler;
import org.eclipse.tptp.platform.execution.util.TPTPAgentAccess;
import org.eclipse.tptp.platform.execution.util.Variable;
import org.eclipse.tptp.platform.execution.util.internal.CommandFragment;
import org.eclipse.tptp.platform.execution.util.internal.Constants;
import org.eclipse.tptp.platform.execution.util.internal.TPTPString;
import org.eclipse.tptp.platform.execution.util.internal.TPTPXMLParse;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

public class AgentController implements IAgentController, ICommandHandler {
		
	private ConnectionInfo _connectioninfo 	= null;
	private INode 		   _node 			= null;
	private ISecureClientParameters _secureClientParameters = null;
	private User		   _user 			= null;
	
	private Hashtable		_processList 	= new Hashtable();
	private Hashtable		_agentList 		= new Hashtable();
	
	private boolean			TPTP_AC 		= true;
	private IConnection		_connection 	= null;
	private int 			_sourceID 		= 0;
	private int				_consoleDataConnID 	= -1;
	private ConsoleDataProcessor	_consoleMapper = null;
	private String 			DELIMIT_TOKEN = " ";
	private FileTransferManagerImpl _fileTransferManager = null;
	
	final private static String FILE_TRANSFER_AGENT = "org.eclipse.tptp.FileTransferAgent";
	//generic context used for ac events that do not have a ctxt to respond to 
	
	public AgentController(INode node,ConnectionInfo conninfo)
	{
		this._node = node;
		this._connectioninfo = conninfo;
		TPTP_AC = conninfo.isAgentControllerNew();
		_consoleDataConnID = -1;
		_consoleMapper = null;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.tptp.platform.execution.core.IAgentController#getNode()
	 */
	public INode getNode() 
	{
		return this._node;
	}
	/* (non-Javadoc)
	 * @see org.eclipse.tptp.platform.execution.core.IAgentController#setNewACType()
	 */
	public void setNewACType(boolean newAC)
	{
		this.TPTP_AC = newAC;
	}

	//223385
	public boolean isTPTP_AC()
	{
		return TPTP_AC;
	}
	/* (non-Javadoc)
	 * @see org.eclipse.tptp.platform.execution.core.IAgentController#getVersion()
	 */
	public String getVersion() 
	{
		//TODO:Need to verify how the same is supported in Hyades - Implementation Pending
		return null;
	}
	
	public long getNextContextId()
	{
		return _connection.getNextContextId();
	}
	
	public void disconnect() {
		if (_connection != null) {
			_connection.disconnect();
			_connection = null;
		}
	}

	/*
	 * Authenticate the User Credentials with AC
	 *
	 */
	public boolean authenticateUser(User user) throws NotConnectedException {
		if(!isConnected()) throw new NotConnectedException(Constants.TPTP_PLATFORM_EXEC_MSG6);

		return ((ConnectionImpl)_connection).authenticateUser(user);
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.tptp.platform.execution.core.IAgentController#getAgent(java.lang.String, java.lang.String)
	 */
	public IAgent getAgent(String agentName, String agentClassName)
			throws NotConnectedException {
		return getAgent(agentName, agentClassName, TPTPAgentAccess.TPTP_CONTROLLER_ACCESS);
	}
	
	/**
	 * (non-Javadoc)
	 * @throws NotConnectedException 
	 * @see org.eclipse.tptp.platform.execution.core.IAgentController#queryACPath()
	 */
	public ACPathInfo queryACPath() throws NotConnectedException {
		
		if(!isConnected()) throw new NotConnectedException(Constants.TPTP_PLATFORM_EXEC_MSG3);
		if(!TPTP_AC) throw new NotSupportedException(Constants.ERR_NOT_SUPPORTED);
		
		final GenericCommandHandler response = new GenericCommandHandler();

		synchronized(response) {
				
			try {
				// Send the command...
				org.eclipse.tptp.platform.execution.client.core.internal.commands.QueryACPathCommand command= new org.eclipse.tptp.platform.execution.client.core.internal.commands.QueryACPathCommand();
				this.sendCommand(command.buildCommand(), Constants.AC_DEST_ID, new ICommandHandler() {
	 		   		public void incomingCommand(INode node, ICommandElement command) {
			 		   			
	 		   			// Notify the queryACPath thread that we have a result
	 		   			synchronized( response ) {
		 		   			response.setCommandElement(command);
	 		   				response.notifyAll();
	 		   			}
		 		   			
	 		   		}		 		   		
				});
 						
				// ...and wait for the result
				int retries = Constants.PROCESS_LAUNCH_TIMEOUT_TRY_COUNT;
				while( response.getCommandElement() == null && retries > 0 ) {
					try {
						response.wait( Constants.TIMEOUT_PERIOD );
						retries--;
					} catch( InterruptedException e ) {
						// Satisfy javac...
					}
				}
			} catch( IOException e ) {
				e.printStackTrace();
			}
		}
		
	 	if (response.getCommandElement() == null) {
     		throw new NotConnectedException(Constants.TPTP_PLATFORM_EXEC_MSG44);
	 	}

	 	// Pull the result out of the response
	 	String commandStr = ((CommandFragment)response.getCommandElement()).getCommandData();
		TPTPXMLParse resultParser = new TPTPXMLParse();
		resultParser.setParser(commandStr);
		
		final Hashtable results = resultParser.getHashTable();
		final String acPath = (String)results.get("acPath");
		final String libPrefix = (String)results.get("libPrefix");
		final String libSuffix = (String)results.get("libSuffix");
		final String fileSeparator = (String)results.get("fileSeparator");
		
		// This would indicate that the AC does not support this command
		if( acPath == null || libPrefix == null || libSuffix == null || fileSeparator == null )
			return null;
		
		return new ACPathInfo( acPath, libPrefix, libSuffix, fileSeparator );
			
	}

	
	/** Returns date/time of the remote AC in the form of mm/dd/yy hh:mm:ss */
	public String getACHostTime()  throws NotConnectedException {
		if(!isConnected()) throw new NotConnectedException(Constants.TPTP_PLATFORM_EXEC_MSG3);
		if(!TPTP_AC) throw new NotSupportedException(Constants.ERR_NOT_SUPPORTED);

		final GenericCommandHandler response = new GenericCommandHandler();

		synchronized(response) {
				
			try {
				// Send the command...
				GetACHostTimeCommand command = new GetACHostTimeCommand();
				this.sendCommand(command.buildCommand(), Constants.AC_DEST_ID, new ICommandHandler() {
	 		   		public void incomingCommand(INode node, ICommandElement command) {
			 		   			
	 		   			// Notify the queryACPath thread that we have a result
	 		   			synchronized( response ) {
		 		   			response.setCommandElement(command);
	 		   				response.notifyAll();
	 		   			}
		 		   			
	 		   		}		 		   		
				});
 						
				// ...and wait for the result
				int retries = Constants.PROCESS_LAUNCH_TIMEOUT_TRY_COUNT;
				while( response.getCommandElement() == null && retries > 0 ) {
					try {
						response.wait( Constants.TIMEOUT_PERIOD );
						retries--;
					} catch( InterruptedException e ) {
						// Satisfy javac...
					}
				}
			} catch( IOException e ) {
				e.printStackTrace();
			}
		}
		
		// No result? Just return false; we can't be sure the AC supports this call 
	 	if (response.getCommandElement() == null) {
     		throw new NotConnectedException(Constants.TPTP_PLATFORM_EXEC_MSG44);
	 	}

	 	// Pull the result out of the response
	 	String commandStr = ((CommandFragment)response.getCommandElement()).getCommandData();
		TPTPXMLParse resultParser = new TPTPXMLParse();
		resultParser.setParser(commandStr);
		
		String result = (String)resultParser.getHashTable().get("getACHostTimeResponse");
		return result;
	}

	
	
	public AC_PLATFORM getACPlatform()  throws NotConnectedException {
		if(!isConnected()) throw new NotConnectedException(Constants.TPTP_PLATFORM_EXEC_MSG3);
		if(!TPTP_AC) throw new NotSupportedException(Constants.ERR_NOT_SUPPORTED);

		final GenericCommandHandler response = new GenericCommandHandler();

		synchronized(response) {
				
			try {
				// Send the command...
				GetACPlatformCommand command = new GetACPlatformCommand();
				this.sendCommand(command.buildCommand(), Constants.AC_DEST_ID, new ICommandHandler() {
	 		   		public void incomingCommand(INode node, ICommandElement command) {
			 		   			
	 		   			// Notify the queryACPath thread that we have a result
	 		   			synchronized( response ) {
		 		   			response.setCommandElement(command);
	 		   				response.notifyAll();
	 		   			}
		 		   			
	 		   		}		 		   		
				});
 						
				// ...and wait for the result
				int retries = Constants.PROCESS_LAUNCH_TIMEOUT_TRY_COUNT;
				while( response.getCommandElement() == null && retries > 0 ) {
					try {
						response.wait( Constants.TIMEOUT_PERIOD );
						retries--;
					} catch( InterruptedException e ) {
						// Satisfy javac...
					}
				}
			} catch( IOException e ) {
				e.printStackTrace();
			}
		}
		
		// No result? Just return false; we can't be sure the AC supports this call 
	 	if (response.getCommandElement() == null) {
     		throw new NotConnectedException(Constants.TPTP_PLATFORM_EXEC_MSG44);
	 	}

	 	// Pull the result out of the response
	 	String commandStr = ((CommandFragment)response.getCommandElement()).getCommandData();
		TPTPXMLParse resultParser = new TPTPXMLParse();
		resultParser.setParser(commandStr);
		
		String result = (String)resultParser.getHashTable().get("getACPlatformResponse");
		if(result != null) {
			if(result.equalsIgnoreCase("WIN_X86")) {
				return AC_PLATFORM.WIN_X86;
			}
			if(result.equalsIgnoreCase("WIN_X64")) {
				return AC_PLATFORM.WIN_X64;
			}
			if(result.equalsIgnoreCase("WIN_IPF")) {
				return AC_PLATFORM.WIN_IPF;
			}
			if(result.equalsIgnoreCase("LINUX_X86")) {
				return AC_PLATFORM.LINUX_X86;
			}
			if(result.equalsIgnoreCase("LINUX_X64")) {
				return AC_PLATFORM.LINUX_X64;
			}
			if(result.equalsIgnoreCase("LINUX_IPF")) {
				return AC_PLATFORM.LINUX_IPF;
			}
			if(result.equalsIgnoreCase("LINUX_390_32)")) {
				return AC_PLATFORM.LINUX_390_32;
			}
			if(result.equalsIgnoreCase("LINUX_390_64")) {
				return AC_PLATFORM.LINUX_390_64;
			}
			if(result.equalsIgnoreCase("AIX_PPC32")) {
				return AC_PLATFORM.AIX_PPC32;
			}
			if(result.equalsIgnoreCase("AIX_PPC64")) {
				return AC_PLATFORM.AIX_PPC64;
			}
			if(result.equalsIgnoreCase("SOLARIS_X86")) {
				return AC_PLATFORM.SOLARIS_X86;
			}
			if(result.equalsIgnoreCase("SOLARIS_X64")) {
				return AC_PLATFORM.SOLARIS_X64;
			}
			if(result.equalsIgnoreCase("SOLARIS_SPARC32")) {
				return AC_PLATFORM.SOLARIS_SPARC32;
			}
			if(result.equalsIgnoreCase("SOLARIS_SPARC64")) {
				return AC_PLATFORM.SOLARIS_SPARC64;
			}
			
			if(result.equalsIgnoreCase("ZOS_31")) {
				return AC_PLATFORM.ZOS_31;
			}
			if(result.equalsIgnoreCase("ZOS_64")) {
				return AC_PLATFORM.ZOS_64;
			}
 			
			// If no match, return null
			return null;	
		} else {
			return null;
		}
				
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.tptp.platform.execution.client.core.IAgentController#isAgentPathSupported()
	 */
	public boolean isAgentPathSupported() throws NotConnectedException {

		if(!isConnected()) throw new NotConnectedException(Constants.TPTP_PLATFORM_EXEC_MSG3);
		if(!TPTP_AC) throw new NotSupportedException(Constants.ERR_NOT_SUPPORTED);
		
		final GenericCommandHandler response = new GenericCommandHandler();

		synchronized(response) {
				
			try {
				// Send the command...
				org.eclipse.tptp.platform.execution.client.core.internal.commands.AgentPathSupportedCommand command= new org.eclipse.tptp.platform.execution.client.core.internal.commands.AgentPathSupportedCommand();
				this.sendCommand(command.buildCommand(), Constants.AC_DEST_ID, new ICommandHandler() {
	 		   		public void incomingCommand(INode node, ICommandElement command) {
			 		   			
	 		   			// Notify the queryACPath thread that we have a result
	 		   			synchronized( response ) {
		 		   			response.setCommandElement(command);
	 		   				response.notifyAll();
	 		   			}
		 		   			
	 		   		}		 		   		
				});
 						
				// ...and wait for the result
				int retries = Constants.PROCESS_LAUNCH_TIMEOUT_TRY_COUNT;
				while( response.getCommandElement() == null && retries > 0 ) {
					try {
						response.wait( Constants.TIMEOUT_PERIOD );
						retries--;
					} catch( InterruptedException e ) {
						// Satisfy javac...
					}
				}
			} catch( IOException e ) {
				e.printStackTrace();
			}
		}
		
		// No result? Just return false; we can't be sure the AC supports this call 
	 	if (response.getCommandElement() == null) {
     		throw new NotConnectedException(Constants.TPTP_PLATFORM_EXEC_MSG44);
	 	}

	 	// Pull the result out of the response
	 	String commandStr = ((CommandFragment)response.getCommandElement()).getCommandData();
		TPTPXMLParse resultParser = new TPTPXMLParse();
		resultParser.setParser(commandStr);
		
		String result = (String)resultParser.getHashTable().get("agentPathSupported");
		if( result != null && result.toLowerCase().equals("true") )
			return true;
		
		return false;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.tptp.platform.execution.core.IAgentController#getAgent(java.lang.String, java.lang.String, boolean)
	 */
	public IAgent getAgent(String agentName, String agentClassName,	TPTPAgentAccess accessMode) throws NotConnectedException {
		if(!isConnected()) throw new NotConnectedException(Constants.TPTP_PLATFORM_EXEC_MSG7);
		
		final Object _launcherLock 	= new Object();
		
		final GenericCommandHandler genCmdHandler = new GenericCommandHandler();
		IAgent agent = null;
		
		if(TPTP_AC)
		{
	 		try
			{
	 		  	if (agentClassName != null && agentClassName.equals("org.eclipse.tptp.platform.execution.client.agent.ICollector"))
	 			{
	 				agent = new CollectorImpl(agentName);
	 			}
	 			else
	 			{
	 				agent = new AgentImpl(agentName);
	 			}
	 			try
				{
					synchronized(_launcherLock) 
					{
					    
			 		   try 
					   { 	
			 		   	org.eclipse.tptp.platform.execution.client.core.internal.commands.AttachToAgentCommand command= new org.eclipse.tptp.platform.execution.client.core.internal.commands.AttachToAgentCommand(Constants.GETAGENT_TAG, agentName,-1,-1, accessMode);
			 		   	this.sendCommand(command.buildCommand(), Constants.AC_DEST_ID, new ICommandHandler()
			 		   	{
			 		   		public void incomingCommand(INode node, ICommandElement command)
			 		   		{
			 		   			genCmdHandler.setCommandElement(command);
			 		   		}
			 		   		
			 		   	});
			 		   try{_launcherLock.notifyAll();}
			   			catch(Exception e){e.printStackTrace();}
					   }
		 			   	catch(Exception e){e.printStackTrace();}
					} //synchronized
				 }
			 	 //Throw the respective exception
			 	 catch(Exception e){e.printStackTrace();}
			 	synchronized(_launcherLock)
			 	{
	 				int processLaunchTimeOut = 0;
			 	    while(genCmdHandler.getCommandElement() == null )
			 	    {
			 	        try
			 	        {		 	        	
			 	        	_launcherLock.wait(Constants.TIMEOUT_PERIOD);
			 	        	if (processLaunchTimeOut == Constants.PROCESS_LAUNCH_TIMEOUT_TRY_COUNT)
			 	        	{
			 	        		throw new TimeoutException(Constants.TPTP_PLATFORM_EXEC_MSG33);		
			 	        	}
			 	        	processLaunchTimeOut++;
			 	        	if (Constants.TPTP_DEBUG) System.out.println("The Process Launch Timeout count" + processLaunchTimeOut);
			 	        }
			 	        catch(TimeoutException e)
			 	        {
			 	        	_launcherLock.notifyAll();
			 	        	break;
			 	        }
			 	        catch(Exception e)
			 	        {
			 	        	_launcherLock.notifyAll();
			 	        	//e.printStackTrace();
			 	        	break;
			 	        }
			 	    }
			 	}	 	    
			 	if (genCmdHandler.getCommandElement() == null) {return null;}
		 	    String commandStr = ((CommandFragment)genCmdHandler.getCommandElement()).getCommandData();
		 	
		 	    TPTPXMLParse ParseObj = new TPTPXMLParse();
		 	    ParseObj.setParser(commandStr);
		 	    Vector	PVector = ParseObj.getVector();
		 	    boolean agentexists = false;
		 	    long	processId = -1;
		 	    int	    agentId = -1;
		 	    IProcess process = null;
		 	    
		 	    for(int i=0; i < PVector.size(); i++)
		 	    {
		 	        if((((Variable)(PVector.elementAt(i))).getName()).equals("agentID"))
		 	        {
		 	            agentId = Integer.parseInt((String)((Variable)(PVector.elementAt(i))).getValue());
		 	            agentexists = true;		 	            
		 	        }
		 	        if((((Variable)(PVector.elementAt(i))).getName()).equals("agentPID"))
		 	        {
		 	            processId = Long.parseLong((String)((Variable)(PVector.elementAt(i))).getValue());
		 	         
		 	            process = getProcess(processId);
		 	            if(process == null)
		 	            {
		 	            	process = createProcess();
			 	            process.setProcessId(processId);
		 	            }
		 	            ((AgentImpl)agent).setProcess(process);
		 	            attachProcess(process, false);
		 	            agentexists = true;
		 	        }
		 	    }
		 	   if(agentexists) {
		 		   addAgentToList(processId, agent);
		 	       ((AgentImpl)agent).setAgentController(this);
		 	       ((AgentImpl)agent).setName(agentName);
		 	       ((AgentImpl)agent).setAgentMode(accessMode);
		 	       ((AgentImpl)agent).setAgentInstanceId(agentId);
		 	       process.setAgent(agent);
		 	   }
			}
	 		catch(Exception e)
			{e.printStackTrace();}
		}
		else
		{
			throw new NotSupportedException(Constants.ERR_NOT_SUPPORTED);
		}
			
		if(((AgentImpl)agent).getAgentInstanceId() == -1) {return null;}
		else { return agent; }
	}
	/* TPTP CLIENT API
	 * Clients should call this method to get the list of deployed agents
	 * returns the array of agent names
	 */
	public IAgent[] queryDeployedAgents(String interfaceID) throws NotConnectedException {
		if(!isConnected()) throw new NotConnectedException(Constants.TPTP_PLATFORM_EXEC_MSG8);

		final Object _launcherLock 	= new Object();
		
		IAgent[] retagent = null;
		final GenericCommandHandler genCmdHandler = new GenericCommandHandler();
		if(TPTP_AC)
		{
			try
			{
				synchronized(_launcherLock) 
				{
		 		   try 
				   { 	
		  		   	org.eclipse.tptp.platform.execution.client.core.internal.commands.QueryRunningAgentsCommand command= new org.eclipse.tptp.platform.execution.client.core.internal.commands.QueryRunningAgentsCommand();
		  		   	this.sendCommand(command.buildCommand(), Constants.AC_DEST_ID, new ICommandHandler()
		 		   	{
		 		   		public void incomingCommand(INode node, ICommandElement command)
		 		   		{
		 		   			genCmdHandler.setCommandElement(command);
		 		   		}
		 		   	});
		  		  try{_launcherLock.notifyAll();}
		  		  catch(Exception e){e.printStackTrace();}	
				   }
	 			   	catch(Exception e){e.printStackTrace();}
	 			 
				} //synchronized
			 }
		 	 //Throw the respective exception
	 	    catch(Exception e){e.printStackTrace();}
	 	   synchronized(_launcherLock)
		 	{
				int processLauchTimeOut = 0;
		 	    while(genCmdHandler.getCommandElement() == null )
		 	    {
		 	        try
		 	        {		 	        	
		 	        	_launcherLock.wait(Constants.TIMEOUT_PERIOD);
		 	        	if (processLauchTimeOut == Constants.PROCESS_LAUNCH_TIMEOUT_TRY_COUNT)
		 	        	{
		 	        		throw new TimeoutException(Constants.TPTP_PLATFORM_EXEC_MSG33);		
		 	        	}
		 	        	processLauchTimeOut++;
		 	        	if (Constants.TPTP_DEBUG) System.out.println("The Process Launch Timeout count" + processLauchTimeOut);
		 	        }
		 	        catch(Exception e)
		 	        {
		 	        	_launcherLock.notifyAll();
		 	        	e.printStackTrace();
		 	        	break;
		 	        }
		 	    }
		 	}	
		 	if (genCmdHandler.getCommandElement() == null) {return null;}
		 	String commandStr = ((CommandFragment)genCmdHandler.getCommandElement()).getCommandData();
			TPTPXMLParse ParseObj = new TPTPXMLParse();
			ParseObj.setParser(commandStr);
			Vector retVector = new Vector();
			Vector	PVector = ParseObj.getVector();
			
			for(int i=0; i < PVector.size(); i++)
			{
				if((((Variable)(PVector.elementAt(i))).getName()).equals("agentID"))
				{
					IAgent agent = new AgentImpl();
					int	   agentId = Integer.parseInt((String)((Variable)(PVector.elementAt(i))).getValue());
					((AgentImpl)agent).setAgentInstanceId(agentId);
					retVector.add(agent);
				}		
			}
			
			retagent = new AgentImpl[retVector.size()];
			retVector.toArray(retagent);
		}

		return retagent;
	}

	/* TPTP CLIENT API
	 * Clients should call this method to get the agentID
	 * Takes the agent name as parameter.
	 * Flags can be a combination (only if it makes sense) of the following values.
	 * TPTP_CONTROLLER_ACCESS - Request Control Access on the Agent
	 * TPTP_OBSERVER_ACCESS - Request Observer Access on the Agent
	 * TPTP_CREATE_INSTANCE - Creates a new instance even if an existing instance is available
	 * TPTP_RUNNING_AGENTS_ONLY - Request only the running agents
	 * TPTP_LOCK_AGENT - Request exclusive usage and others can't share this agent
	 */
	public IAgent getAgent(String agentName, TPTPAgentAccess accessMode)throws NotConnectedException
	{
	    return getAgent(agentName, null,accessMode);
	}


	/* 
	 * TPTP CLIENT API
	 * Get the Agent By the Process ID
	 * Flags can be a combination (only if it makes sense) of the following values.
	 * TPTP_CONTROLLER_ACCESS - Request Control Access on the Agent
	 * TPTP_OBSERVER_ACCESS - Request Observer Access on the Agent
	 * TPTP_LOCK_AGENT - Request exclusive usage and others can't share this agent
	 */
	private int queryAgentByProcessID(String agentName, long pid) throws NotConnectedException {
		if(!isConnected()) throw new NotConnectedException(Constants.TPTP_PLATFORM_EXEC_MSG9);

		final Object _launcherLock 	= new Object();
		
		int token =-1;
		final GenericCommandHandler genCmdHandler = new GenericCommandHandler();
		if(TPTP_AC)
		{
			try
			{
				synchronized(_launcherLock) 
				{
		 		   try 
				   { 	
		 		   	org.eclipse.tptp.platform.execution.client.core.internal.commands.QueryRunningAgentsCommand command= new org.eclipse.tptp.platform.execution.client.core.internal.commands.QueryRunningAgentsCommand();
		 		   	sendCommand(command.buildCommand(), Constants.AC_DEST_ID, new ICommandHandler() {
		 		   		public void incomingCommand(INode node, ICommandElement command) {	
		 		   			genCmdHandler.setCommandElement(command);
		 		   		}
		 		   	});
		 		   try{_launcherLock.notifyAll();}
		 		   catch(Exception e){e.printStackTrace();}
		 		    }
	 			   	catch(Exception e){e.printStackTrace();}
				} //synchronized
			 }
		 	 //Throw the respective exception
		 	 catch(Exception e){e.printStackTrace();}
		 	synchronized(_launcherLock)
		 	{
 				int processLauchTimeOut = 0;
		 	    while(genCmdHandler.getCommandElement() == null )
		 	    {
		 	        try
		 	        {		 	        	
		 	        	_launcherLock.wait(Constants.TIMEOUT_PERIOD);
		 	        	if (processLauchTimeOut == Constants.PROCESS_LAUNCH_TIMEOUT_TRY_COUNT)
		 	        	{
		 	        		throw new TimeoutException(Constants.TPTP_PLATFORM_EXEC_MSG33);		
		 	        	}
		 	        	processLauchTimeOut++;
		 	        	if (Constants.TPTP_DEBUG) System.out.println("The Process Launch Timeout count" + processLauchTimeOut);
		 	        }
		 	        catch(Exception e)
		 	        {
		 	        	_launcherLock.notifyAll();
		 	        	e.printStackTrace();
		 	        	return -1;
		 	        }
		 	    }
		 	}
		 	 	
		 	String commandStr = ((CommandFragment)genCmdHandler.getCommandElement()).getCommandData();
			TPTPXMLParse ParseObj = new TPTPXMLParse();
			NodeList nl = ParseObj.getNodes(commandStr, "agentInfo");
			if (nl == null) return -1;
			
			for(int i = 0; i<nl.getLength(); i++) {
				String procid = ParseObj.getAttribValue(nl.item(i), "pid");
				String agentname = ParseObj.getAttribValue(nl.item(i), "name");
				if( (Long.parseLong((String)procid) == pid) && agentname.equals(agentName)) {
					String temp = ParseObj.getAttribValue(nl.item(i),"token");
				    token = (Integer.parseInt(temp));
				    break;
				}
			}
		}
		
		return token;
	}
	
	public void releaseAgent(int agentID)throws NotConnectedException {
		if(!isConnected()) throw new NotConnectedException(Constants.TPTP_PLATFORM_EXEC_MSG10);
		
		if(TPTP_AC)
		{
			try
			{
	 		   	org.eclipse.tptp.platform.execution.client.core.internal.commands.DetachFromAgentCommand command= new org.eclipse.tptp.platform.execution.client.core.internal.commands.DetachFromAgentCommand(Constants.releaseAgent_Cmd, agentID);
	 		   	this.sendCommand(command.buildCommand(), Constants.AC_DEST_ID, new ICommandHandler()
	 		   	{
	 		   		public void incomingCommand(INode node, ICommandElement command)
	 		   		{ /* DO Nothing Here - No response for this command */ }
	 			});
	 		   	
	 		}
			
		   	catch(Exception e){e.printStackTrace();}
		}
		else
		{
			throw new NotSupportedException(Constants.ERR_NOT_SUPPORTED);
		}
		return;
	}
	
	public boolean requestAgentControl(int agentID, int accessMode)throws NotConnectedException	{
		if(!isConnected()) throw new NotConnectedException(Constants.TPTP_PLATFORM_EXEC_MSG11);
		
		final Object _launcherLock 	= new Object();
		
		final GenericCommandHandler genCmdHandler = new GenericCommandHandler();
		boolean controlgranted = false;
		if(TPTP_AC)
		{
			try
			 {
				org.eclipse.tptp.platform.execution.client.core.internal.commands.RequestAgentControlCommand command= new org.eclipse.tptp.platform.execution.client.core.internal.commands.RequestAgentControlCommand( agentID, accessMode);
				
				synchronized(_launcherLock) 
				{
		 		   try 
				   { 	
		 		   	this.sendCommand(command.buildCommand(),Constants.AC_DEST_ID,new ICommandHandler()
					{
						public void incomingCommand(INode node, ICommandElement command)
						{
							genCmdHandler.setCommandElement(command);
						}
					});

		 		   try{_launcherLock.notifyAll();}
		 		   catch(Exception e){e.printStackTrace();}
				   }
	 			   	catch(Exception e){e.printStackTrace();}
				} //synchronized
			 }
		 	 catch(Exception e){e.printStackTrace();}
		 	synchronized(_launcherLock)
		 	{
 				int processLauchTimeOut = 0;
		 	    while(genCmdHandler.getCommandElement() == null )
		 	    {
		 	        try
		 	        {		 	        	
		 	        	_launcherLock.wait(Constants.TIMEOUT_PERIOD);
		 	        	if (processLauchTimeOut == Constants.PROCESS_LAUNCH_TIMEOUT_TRY_COUNT)
		 	        	{
		 	        		throw new TimeoutException(Constants.TPTP_PLATFORM_EXEC_MSG33);		
		 	        	}
		 	        	processLauchTimeOut++;
		 	        	if (Constants.TPTP_DEBUG) System.out.println("The Process Launch Timeout count" + processLauchTimeOut);
		 	        }
		 	        catch(Exception e)
		 	        {
		 	        	_launcherLock.notifyAll();
		 	        	e.printStackTrace();
		 	        	break;
		 	        }
		 	    }
		 	}	
		 	if (genCmdHandler.getCommandElement() == null) {return false;}
		 	String commandStr = ((CommandFragment)genCmdHandler.getCommandElement()).getCommandData();
		 	TPTPXMLParse ParseObj = new TPTPXMLParse();
			ParseObj.setParser(commandStr);
			Vector	PVector = ParseObj.getVector(); 
					
			for(int i=0; i < PVector.size(); i++)
			{
			  if((((Variable)(PVector.elementAt(i))).getName()).equals(Constants.agentControlGranted_Cmd))
			  {
			  	controlgranted = true;
			  }		
			}
			return controlgranted;
		}
		else
		{
			throw new NotSupportedException(Constants.ERR_NOT_SUPPORTED);
		}
	}
	
	public void releaseAgentControl(int agentID)throws NotConnectedException {
		if(!isConnected()) throw new NotConnectedException(Constants.TPTP_PLATFORM_EXEC_MSG12);

		if(TPTP_AC)	{
			try	{
	 		   	org.eclipse.tptp.platform.execution.client.core.internal.commands.DetachFromAgentCommand command= new org.eclipse.tptp.platform.execution.client.core.internal.commands.DetachFromAgentCommand(Constants.releaseAgentControl_Cmd, agentID);
	 		   	this.sendCommand(command.buildCommand(), Constants.AC_DEST_ID, new ICommandHandler()
	 		   	{
	 		   		public void incomingCommand(INode node, ICommandElement command)
	 		   		{ /* DO Nothing Here - No response for this command */ }
	 			});
	 		}
		   	catch(Exception e){e.printStackTrace();}
		}
		else
		{
			throw new NotSupportedException(Constants.ERR_NOT_SUPPORTED);
		}
		return;
	}
	/* 
	 * TPTP CLIENT API
	 * Get the Agent By the Agent Token passed by the Agent Controller previously
	 * Flags can be a combination (only if it makes sense) of the following values.
	 * TPTP_CONTROLLER_ACCESS - Request Control Access on the Agent
	 * TPTP_OBSERVER_ACCESS - Request Observer Access on the Agent
	 * TPTP_LOCK_AGENT - Request exclusive usage and others can't share this agent
	 */
	public int getAgentByToken(int token, TPTPAgentAccess accessMode) throws NotConnectedException
	{
		final Object _launcherLock 	= new Object();
		
		final GenericCommandHandler genCmdHandler = new GenericCommandHandler();
		int	agentRefID = -1;
		
		if(TPTP_AC)
		{
			try
			 {
				org.eclipse.tptp.platform.execution.client.core.internal.commands.AttachToAgentCommand command= new org.eclipse.tptp.platform.execution.client.core.internal.commands.AttachToAgentCommand(Constants.GETAGENT_TOK_TAG, null,-1,token, accessMode);
				synchronized(_launcherLock) 
				{
		 		   try 
				   { 	
		 		    this.sendCommand(command.buildCommand(),Constants.AC_DEST_ID,new ICommandHandler()
					{
						public void incomingCommand(INode node, ICommandElement command)
						{
							genCmdHandler.setCommandElement(command);
						}
					});
		 		   try{_launcherLock.notifyAll();}
		 		   catch(Exception e){e.printStackTrace();}
					}
	 			   	catch(Exception e){e.printStackTrace();}
	 			   
				} //synchronized
			 }
		 	 catch(Exception e){e.printStackTrace();}
		 	synchronized(_launcherLock)
		 	{
 				int processLauchTimeOut = 0;
		 	    while(genCmdHandler.getCommandElement() == null )
		 	    {
		 	        try
		 	        {		 	        	
		 	        	_launcherLock.wait(Constants.TIMEOUT_PERIOD);
		 	        	if (processLauchTimeOut == Constants.PROCESS_LAUNCH_TIMEOUT_TRY_COUNT)
		 	        	{
		 	        		throw new TimeoutException(Constants.TPTP_PLATFORM_EXEC_MSG33);		
		 	        	}
		 	        	processLauchTimeOut++;
		 	        	if (Constants.TPTP_DEBUG) System.out.println("The Process Launch Timeout count" + processLauchTimeOut);
		 	        }
		 	        catch(Exception e)
		 	        {
		 	        	_launcherLock.notifyAll();
		 	        	e.printStackTrace();
		 	        	break;
		 	        }
		 	    }
		 	}	 
		 	if (genCmdHandler.getCommandElement() == null) {return agentRefID;}
		 	String commandStr = ((CommandFragment)genCmdHandler.getCommandElement()).getCommandData();
			TPTPXMLParse ParseObj = new TPTPXMLParse();
			ParseObj.setParser(commandStr);
			Vector	PVector = ParseObj.getVector(); 
						
			for(int i=0; i < PVector.size(); i++)
			{
			  if((((Variable)(PVector.elementAt(i))).getName()).equals("agentID"))
			  {
			  	agentRefID = Integer.parseInt((String)((Variable)(PVector.elementAt(i))).getValue());
			  }		
			}
			
		 	return agentRefID;
		}		
		return agentRefID;
	}

	/* 
	 * TPTP CLIENT API
	 * Get the Agent By the Agent ID
	 * Flags can be a combination (only if it makes sense) of the following values.
	 * TPTP_CONTROLLER_ACCESS - Request Control Access on the Agent
	 * TPTP_OBSERVER_ACCESS - Request Observer Access on the Agent
	 * TPTP_LOCK_AGENT - Request exclusive usage and others can't share this agent
	 */
	public int getAgentByID(int agentID, TPTPAgentAccess accessMode) throws NotConnectedException
	{
		final Object _launcherLock 	= new Object();
	
		final GenericCommandHandler genCmdHandler = new GenericCommandHandler();
		int	agentRefID = -1;
		
		if(TPTP_AC)
		{
			try
			 {
				org.eclipse.tptp.platform.execution.client.core.internal.commands.AttachToAgentCommand command= new org.eclipse.tptp.platform.execution.client.core.internal.commands.AttachToAgentCommand(Constants.GETAGENT_ID_TAG, null,agentID,-1, accessMode);
				synchronized(_launcherLock) 
				{
		 		   try 
				   { 	
		 		   this.sendCommand(command.buildCommand(),Constants.AC_DEST_ID,new ICommandHandler()
					{
						public void incomingCommand(INode node, ICommandElement command)
						{
							genCmdHandler.setCommandElement(command);
						}
					});
		 		   	try{_launcherLock.notifyAll();}
		   			catch(Exception e){e.printStackTrace();}
				   }
	 			   	catch(Exception e){e.printStackTrace();}
				} //synchronized
			 }
		 	 catch(Exception e){e.printStackTrace();}
		 	synchronized(_launcherLock)
		 	{
 				int processLauchTimeOut = 0;
		 	    while(genCmdHandler.getCommandElement() == null )
		 	    {
		 	        try
		 	        {		 	        	
		 	        	_launcherLock.wait(Constants.TIMEOUT_PERIOD);
		 	        	if (processLauchTimeOut == Constants.PROCESS_LAUNCH_TIMEOUT_TRY_COUNT)
		 	        	{
		 	        		throw new TimeoutException(Constants.TPTP_PLATFORM_EXEC_MSG33);		
		 	        	}
		 	        	processLauchTimeOut++;
		 	        	if (Constants.TPTP_DEBUG) System.out.println("The Process Launch Timeout count" + processLauchTimeOut);
		 	        }
		 	        catch(Exception e)
		 	        {
		 	        	_launcherLock.notifyAll();
		 	        	e.printStackTrace();
		 	        	break;
		 	        }
		 	    }
		 	}
		 	if (genCmdHandler.getCommandElement() == null) {return agentRefID;}
		 	String commandStr = ((CommandFragment)genCmdHandler.getCommandElement()).getCommandData();
			TPTPXMLParse ParseObj = new TPTPXMLParse();
			ParseObj.setParser(commandStr);
			Vector	PVector = ParseObj.getVector(); 
						
			for(int i=0; i < PVector.size(); i++)
			{
			  if((((Variable)(PVector.elementAt(i))).getName()).equals("agentID"))
			  {
			  	agentRefID = Integer.parseInt((String)((Variable)(PVector.elementAt(i))).getValue());
			  }		
			}
			
		 	return agentRefID;
		}		
		return agentRefID;
	}


	/**
	 * Returns the instance of IAgent matching the parameter processId
	 * @return
	 * @throws NotConnectedException
	 */
	public IAgent getAgent(String agentName, long processId) throws NotConnectedException
	{
		return getAgent(agentName, null, processId);
	}
	
	private IAgent cachedAgent=null; 

	public IAgent getAgent(String agentName, String agentClassName, long processId) throws NotConnectedException {
		if(!this.isConnected()) throw new NotConnectedException(Constants.TPTP_PLATFORM_EXEC_MSG7);

		IAgent agent = null;
        if (cachedAgent != null) {
        	try { 
               if (cachedAgent.getProcess().getProcessId() == processId && cachedAgent.getName().equals(agentName)) {
                   agent = cachedAgent;
               }
           } catch (Exception e) {}
        }
        
		if (agent != null) {
			return agent;
		}

		if (agentClassName != null && agentClassName.equals("org.eclipse.tptp.platform.execution.client.agent.ICollector"))	{
			agent = new CollectorImpl(agentName);
		}
		else {
			agent = new AgentImpl(agentName);
		}

		IProcess process = getProcess(processId);
		if (process == null) {
			process = this.createProcess();
			process.setProcessId(processId);
		}
		
		if(TPTP_AC)
		{
			try
			 {
				int agenttoken = this.queryAgentByProcessID(agentName, processId);
				if(agenttoken == -1) return null;
				//Using the AgentToken get the AgentRef. and set values
				int agentRefId = this.getAgentByToken(agenttoken, TPTPAgentAccess.TPTP_OBSERVER_ACCESS);
				if(agentRefId == -1) return null;
	
				addAgentToList(processId, agent);
	
				((AgentImpl)agent).setAgentInstanceId(agentRefId);
				((AgentImpl)agent).setAgentController(this);
				((AgentImpl)agent).setName(agentName);
				((AgentImpl)agent).setAgentMode(TPTPAgentAccess.TPTP_OBSERVER_ACCESS);			
		        //is there process info for this process id if yes, then assign the values to the agent proc object
	
				attachProcess(process, false);
				((AgentImpl)agent).setProcess(process);
				process.setAgent(agent);
	            cachedAgent = agent;
			 }
			catch(Exception e){e.printStackTrace();	return null;}
	    }
	
		return agent;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.tptp.platform.execution.core.IAgentController#listProcesses()
	 */
	public Enumeration listAgents() throws NotConnectedException {
		if(!isConnected()) throw new NotConnectedException(Constants.TPTP_PLATFORM_EXEC_MSG13);
		return _agentList.elements();
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.tptp.platform.execution.core.IAgentController#listProcesses()
	 */
	public Enumeration listProcesses() throws NotConnectedException	{
		if(!isConnected()) throw new NotConnectedException(Constants.TPTP_PLATFORM_EXEC_MSG14);
		return _processList.elements();
	}

	/* (non-Javadoc)
	 * @see org.eclipse.tptp.platform.execution.core.IAgentController#getProcess(java.lang.String)
	 */
	public IProcess getProcess(long processId) throws NotConnectedException {
		if (!isConnected())	throw new NotConnectedException(Constants.TPTP_PLATFORM_EXEC_MSG15);

		return (IProcess) _processList.get(new Long(processId));
	}
	
	public void attachProcess(IProcess process, boolean overwrite) {
		try {
			Long pKey = new Long(process.getProcessId());
			boolean doAttach = _processList.containsKey(pKey) ? overwrite : true;
			if (doAttach) {
				_processList.put(pKey, process);
			}
		} catch (InactiveProcessException ignored) {
		}
	}

	public void attachProcess(IProcess process) {
		attachProcess(process, true);
	}

	public void removeProcess(long processId) throws NotConnectedException {
		_processList.remove(new Long(processId));
	}

	/* (non-Javadoc)
	 * @see org.eclipse.tptp.platform.execution.core.IAgentController#waitForProcess(java.lang.String, int)
	 */
	public IProcess waitForProcess(String processIdentity, int timeout)
			throws NotConnectedException, InactiveProcessException 
	{
		// TODO Auto-generated method stub
		return null;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.tptp.platform.execution.core.IAgentController#queryAvailableAgents()
	 */
	public String[] queryAvailableAgents(String[] interfaceID) throws NotConnectedException {
		if(!TPTP_AC) return null;
		if(!isConnected()) throw new NotConnectedException(Constants.TPTP_PLATFORM_EXEC_MSG16);

		final Object responseLock = new Object();
		final GenericCommandHandler genCmdHandler = new GenericCommandHandler();
		
		QueryAvailableAgntCommand command= new QueryAvailableAgntCommand();
		if (interfaceID != null) command.setAgentInterfaceID(interfaceID.length, interfaceID);

		Exception sendMsgEx = null;
		try { 	
		 	sendCommand(command.buildCommand(),Constants.AC_DEST_ID,new ICommandHandler() {
		 		public void incomingCommand(INode node, ICommandElement command) {
		 			genCmdHandler.setCommandElement(command);
		 			synchronized(responseLock) { responseLock.notifyAll(); }
				}
			});
		} catch(Exception e) {
			sendMsgEx = e;
		}
	 			   		
		if (sendMsgEx != null) throw new NotConnectedException(sendMsgEx.getMessage()); 

		synchronized(responseLock) {
	 	    if (genCmdHandler.getCommandElement() == null) {
	 	    	try {
	 	    		responseLock.wait(Constants.PROCESS_LAUNCH_TIMEOUT_TRY_COUNT*Constants.TIMEOUT_PERIOD);
	 	    	} catch (Exception e) {}
	 	    }
		}

	 	if (genCmdHandler.getCommandElement() == null) {
     		throw new NotConnectedException(Constants.TPTP_PLATFORM_EXEC_MSG33);		
	 	}
		
	 	String commandStr = ((CommandFragment)genCmdHandler.getCommandElement()).getCommandData();
		TPTPXMLParse ParseObj = new TPTPXMLParse();
		ParseObj.setParser(commandStr);
		Vector	pVector = ParseObj.getVector();
		if (pVector == null) return null;
		
		String response = null;
		Iterator it = pVector.iterator();
		while (it.hasNext()) {
			Variable var = (Variable) it.next();
			if ("agentNames".equals(var.getName())) {
				response = (String) var.getValue();
				break;
			}
		}
		
		return (response == null) ? null : TPTPString.tokenizeString(DELIMIT_TOKEN, response);
	}

	public String[] queryAvailableAgents() throws NotConnectedException	{
		return queryAvailableAgents(null);
	}
	
	private boolean addAgentToList(long pid, IAgent agent) {
		String pkey = String.valueOf(pid);
		Collection arr = (Collection) _agentList.get(pkey);
		if (arr == null) {
			arr = new ArrayList();
			_agentList.put(pkey, arr);
		}
		return arr.add(agent);
	}	
	
	public IAgent[] queryRunningAgents(String agentName, String[] interfaceID, long pid) throws NotConnectedException {
		if(!isConnected()) throw new NotConnectedException(Constants.TPTP_PLATFORM_EXEC_MSG17);

		final Object _launcherLock 	= new Object();
		IAgent[] retagent = null;
		final GenericCommandHandler genCmdHandler = new GenericCommandHandler();
		if(TPTP_AC)
		{
			try
			{
				synchronized(_launcherLock) 
				{
		 		   try 
				   { 	
		 		   	org.eclipse.tptp.platform.execution.client.core.internal.commands.QueryRunningAgentsCommand command= new org.eclipse.tptp.platform.execution.client.core.internal.commands.QueryRunningAgentsCommand();
		 		   	if(interfaceID != null )command.setAgentInterfaceID(interfaceID.length, interfaceID);
		 		    if(agentName != null )command.setAgentName(agentName);
		 		    if(pid != 0 )command.setProcessId(pid);

		 		   	this.sendCommand(command.buildCommand(), Constants.AC_DEST_ID, new ICommandHandler()
		 		   	{
		 		   		public void incomingCommand(INode node, ICommandElement command)
		 		   		{
		 		   			genCmdHandler.setCommandElement(command);
		 		   		}
		 		   	});
		 		   	try{_launcherLock.notifyAll();}
		   			catch(Exception e){e.printStackTrace();}
				   }
	 			   	catch(Exception e){e.printStackTrace();}
				} //synchronized
			 }
		 	 //Throw the respective exception
		 	 catch(Exception e){e.printStackTrace();}
		 	synchronized(_launcherLock)
		 	{
 				int processLauchTimeOut = 0;
		 	    while(genCmdHandler.getCommandElement() == null )
		 	    {
		 	        try
		 	        {		 	        	
		 	        	_launcherLock.wait(Constants.TIMEOUT_PERIOD);
		 	        	if (processLauchTimeOut == Constants.PROCESS_LAUNCH_TIMEOUT_TRY_COUNT)
		 	        	{
		 	        		throw new TimeoutException(Constants.TPTP_PLATFORM_EXEC_MSG33);		
		 	        	}
		 	        	processLauchTimeOut++;
		 	        	if (Constants.TPTP_DEBUG) System.out.println("The Process Launch Timeout count" + processLauchTimeOut);
		 	        }
		 	        catch(Exception e)
		 	        {
		 	        	_launcherLock.notifyAll();
		 	        	e.printStackTrace();
		 	        	break;
		 	        }
		 	    }
		 	    _launcherLock.notifyAll();
		 	}

		 	if (genCmdHandler.getCommandElement() == null) {return null;} 
		 	String commandStr = ((CommandFragment)genCmdHandler.getCommandElement()).getCommandData();
			
			TPTPXMLParse ParseObj = new TPTPXMLParse();
			NodeList nl = ParseObj.getNodes(commandStr, "agentInfo");
			if (nl == null) return null;
			
			Collection agents = new LinkedList();
			
			for(int i = 0; i<nl.getLength(); i++) {
				Node node = nl.item(i);
				String name = ParseObj.getAttribValue(node, "name");
				if (name == null) continue;

				String s = ParseObj.getAttribValue(node, "pid");

				long processId;
				if (s == null || s.trim().length() <= 0)
					processId = -1;
				else
					processId = Long.parseLong(s);
				
				
				s = ParseObj.getAttribValue(node, "token");
				
				int token;
				if (s == null || s.trim().length() <= 0)
					token = -1;
				else
					token = Integer.parseInt(s);
				
				AgentImpl agent = new AgentImpl();
				agent.setName(name);
				agent.setAgentTokenId(token);
				agent.setAgentInstanceId(getAgentByToken(token, TPTPAgentAccess.TPTP_OBSERVER_ACCESS));
				agent.setAgentController(this);

				addAgentToList(processId, agent);
			
				IProcess proc = getProcess(processId);
				if (proc == null) {
					proc = createProcess();
		 	    	proc.setProcessId(processId);
				}
				
	 	    	proc.setAgent(agent);
	 	    	agent.setProcess(proc);
	 	    	attachProcess(proc, false);

	 	    	agents.add(agent);
			}

			retagent = new AgentImpl[agents.size()];
			agents.toArray(retagent);
		}
		
		return retagent;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.tptp.platform.execution.core.IAgentController#queryRunningAgentsByType(String)
	 */
	public IAgent[] queryRunningAgents(String[] interfaceID) throws NotConnectedException 
	{
	    return queryRunningAgents(null, interfaceID,0);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.tptp.platform.execution.core.IAgentController#queryRunningAgents()
	 */
	public IAgent[] queryRunningAgents() throws NotConnectedException {
	   return queryRunningAgents(null);
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.tptp.platform.execution.core.IAgentController#getAgentMetadata(java.lang.String)
	 */
	public String getAgentMetadata(String agentName) throws NotConnectedException {
		if(!isConnected()) throw new NotConnectedException(Constants.TPTP_PLATFORM_EXEC_MSG18);
		
		final Object _launcherLock 	= new Object();
		String retagentMetadata = "";
		
		final GenericCommandHandler genCmdHandler = new GenericCommandHandler();
		if(TPTP_AC)
		{
			try
			{
				synchronized(_launcherLock) 
				{
		 		   try 
				   { 	
		 		    AgentDetailsCommand command= new AgentDetailsCommand(agentName);
		 		   	
		 		    this.sendCommand(command.buildCommand(), Constants.AC_DEST_ID, new ICommandHandler()
			 		   	{
			 		   		public void incomingCommand(INode node, ICommandElement command)
			 		   		{
			 		   		    genCmdHandler.setCommandElement(command);
			 		   		}
			 		   	});
		 		   try{_launcherLock.notifyAll();}
		   			catch(Exception e){e.printStackTrace();}
				   }
	 			   	catch(Exception e){e.printStackTrace();}
				 } //synchronized
				}
				catch(Exception e){e.printStackTrace();}
				synchronized(_launcherLock)
			 	{
	 				int processLauchTimeOut = 0;
			 	    while(genCmdHandler.getCommandElement() == null )
			 	    {
			 	        try
			 	        {		 	        	
			 	        	_launcherLock.wait(Constants.TIMEOUT_PERIOD);
			 	        	if (processLauchTimeOut == Constants.PROCESS_LAUNCH_TIMEOUT_TRY_COUNT)
			 	        	{
			 	        		throw new TimeoutException(Constants.TPTP_PLATFORM_EXEC_MSG33);		
			 	        	}
			 	        	processLauchTimeOut++;
			 	        	if (Constants.TPTP_DEBUG) System.out.println("The Process Launch Timeout count" + processLauchTimeOut);
			 	        }
			 	       catch(Exception e)
			 	        {
			 	        	_launcherLock.notifyAll();
			 	        	e.printStackTrace();
			 	        	break;
			 	        }
			 	    }
			 	}	 

				if (genCmdHandler.getCommandElement() == null) {return null;}
				String commandStr = ((CommandFragment)genCmdHandler.getCommandElement()).getCommandData();
				
				TPTPXMLParse ParseObj = new TPTPXMLParse();
				ParseObj.setParser(commandStr);
				Hashtable CommandHash = ParseObj.getHashTable();
				if(CommandHash.containsKey("metadata"))
				{
					retagentMetadata = (String)CommandHash.get("metadata");
				}
			
		 	return retagentMetadata;
		}
		return null;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.tptp.platform.execution.core.IAgentController#getFileTransferManager()
	 */
	public IFileTransferManager getFileTransferManager() 
	{
		if(TPTP_AC)
		{	
			try
			{
				_fileTransferManager = new FileTransferManagerImpl(FILE_TRANSFER_AGENT);
				IAgent fileTransferManagerAgent = getAgent(FILE_TRANSFER_AGENT, "org.eclipse.tptp.platform.execution.agent.IAgent");			
				((AgentImpl)_fileTransferManager).setAgentInstanceId(((AgentImpl)fileTransferManagerAgent).getAgentInstanceId());
				((AgentImpl)_fileTransferManager).setAgentController(this);
				_fileTransferManager.createFileDataConnection();
			}
			catch(NotConnectedException exp)
			{
				
			}

		}
		else 
		{ 
			//FileManagerImpl filemanager = new FileManagerImpl();
		}
		return _fileTransferManager;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.tptp.platform.execution.core.IAgentController#addEventListener(java.lang.String, org.eclipse.tptp.platform.execution.core.ICommandHandler)
	 */
	public int addEventListener(String interfaceID, ICommandHandler eventHandler) {
		// TODO: Add generic event listener
		return 0;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.tptp.platform.execution.core.IAgentController#removeEventListener(java.lang.String, org.eclipse.tptp.platform.execution.core.ICommandHandler)
	 */
	public void removeEventListener(String interfaceID,	ICommandHandler eventHandler) {
		// TODO Auto-generated method stub
	}

	/* (non-Javadoc)
	 * @see org.eclipse.tptp.platform.execution.core.IAgentController#sendCommand(java.lang.String, int, org.eclipse.tptp.platform.execution.core.ICommandHandler)
	 */
	public void sendCommand(String cmd, int destID, ICommandHandler handler) throws IOException {
		ControlMessage message = new ControlMessage();
		message.setMessageType(Constants.TPTP_AC_MESSAGE);
		message.setMagicNumber(Constants.AC_MAGIC_NUMBER);
		message.setFlags(0);
			
		CommandFragment cmdFrag = new CommandFragment();
		cmdFrag.setDestination(destID);
		cmdFrag.setSource(this._sourceID);
		cmdFrag.setContext(_connection.getNextContextId());
		cmdFrag.setCommand(cmd);
		cmdFrag.buildCommand();//Build the command in ConnectionImpl::sendMessage as we don't know the context at this place
		message.appendCommand(cmdFrag);
			
		if (handler == null) {
			//this.getConnection().sendMessage(message, this.acCommandHandler);
		}
		else {
			getConnection().sendMessage(message, handler);
		}
	}
	
	/** 
	 * Sends a command to the specified destination, the command requires a control message object
	 * and a commandhandler 
	 */
	
	public void sendMessage(IControlMessage mesg, ICommandHandler handler) throws IOException 
	{
		this.getConnection().sendMessage(mesg,handler);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.tptp.platform.execution.core.IAgentController#sendData(int, char[], int)
	 */
	public int sendData(int dataConnID, byte[] buffer, int bufferLength)
			throws DataChannelConnectionException 
	{
		// Data transfer from the client side to the AC
		this._connection.sendData(dataConnID, buffer, bufferLength);
		return 0;
	}
	
	
	public int createDataConnection(int direction)
	{
		int dataConnID = -1;
		try
		{
			dataConnID = ((ConnectionImpl)_connection).createDataConnection(direction);
		}
		catch(IOException exp)
		{
			System.out.println("Error while creating the data connection IOEXP "+exp);
		}
		catch(Exception exp)
		{
			System.out.println("Error while creating  the data connection"+exp);
		}
		return dataConnID;	
	}
	
	public int addDataListener(int dataConnID, IDataProcessor dataProcessor)
	{
		return ((ConnectionImpl)_connection).addDataListener(dataConnID, dataProcessor);
	}

	public int removeDataListener(int dataConnID, IDataProcessor dataProcessor)
	{
		return ((ConnectionImpl)_connection).removeDataListener(dataConnID, dataProcessor);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.tptp.platform.execution.core.IAgentController#createProcess()
	 */
	public IProcess createProcess() 
	{
		IProcess process = new ProcessImpl(this);
		
		return process;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.tptp.platform.execution.core.IAgentController#createProcess(java.lang.String)
	 */
	public IProcess createProcess(String executable) 
	{ 
		IProcess process = new ProcessImpl(this);
		try {process.setExecutable(executable);}
		catch(ProcessActiveException e){e.printStackTrace(); return null;}
			
		return process;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.tptp.platform.execution.core.IAgentController#createProcess(java.lang.String, java.lang.String)
	 */
	public IProcess createProcess(String executable, String parameters) 
	{
		IProcess process = new ProcessImpl(this);
		try { 
				process.setExecutable(executable);
				process.setParameter(parameters);	
			}
		catch(ProcessActiveException e){e.printStackTrace(); return null;}
				
		return process;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.tptp.platform.execution.core.IAgentController#getUser()
	 */
	public User getUser() 
	{
		return this._user;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.tptp.platform.execution.core.IAgentController#setUser(org.eclipse.tptp.platform.execution.security.User)
	 */
	public void setUser(User user) 
	{
		this._user = user;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.tptp.platform.execution.core.IAgentController#setSecurityParameters(org.eclipse.tptp.platform.execution.security.ISecureClientParameters)
	 */
	public void setSecurityParameters(ISecureClientParameters manager) 
	{
		this._secureClientParameters = manager;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.tptp.platform.execution.core.IAgentController#getSecurityParameters()
	 */
	public ISecureClientParameters getSecurityParameters() 
	{
		return this._secureClientParameters;
	}
	/* (non-Javadoc)
	 * @see org.eclipse.tptp.platform.execution.core.IAgentController#getConnectionInfo()
	 */
	public ConnectionInfo getConnectionInfo() 
	{
		return this._connectioninfo;
	}
	public IConnection getConnection() 
	{
		return this._connection;
	}
	
	public void setConnection(IConnection conn) 
	{
		_connection = conn;
		this._sourceID = ((ConnectionImpl)conn).getConnectionId();
		//Add a generic listener after the connection is complete
		try
		{
		    addGenericListener();
		}
		catch(Exception e){e.printStackTrace();}
		
	}
	
	public void incomingCommand(INode node, ICommandElement command) {
	}
	
	/*
	 * Create Console Data Connection
	 */
	public int getConsoleDataConnection(int direction)
	{
		//if (_consoleDataConnID == -1)
		//{
			_consoleDataConnID = this.createDataConnection(direction);
			if (_consoleMapper == null)
			{
				_consoleMapper = new ConsoleDataProcessor();
			}
			//addDataListener(_consoleDataConnID, _consoleDataProcessor);
		//}
		return _consoleDataConnID;
	}
	//Add a generic listener for capturing non-context command responses
	private void addGenericListener()
	{
	    ((ConnectionImpl)(this.getConnection())).getContextMapper().addContext(-1, new ICommandHandler(){
	   		public void incomingCommand(INode node, ICommandElement command)
	   		{
	   			//throw new InactiveAgentException(((CommandFragment)command).getCommandData());
	   			TPTPXMLParse ParseObj = new TPTPXMLParse();
				ParseObj.setParser(((CommandFragment)command).getCommandData());
				
				Hashtable	ch = ParseObj.getHashTable();
				int	   agentID  = -1;
				if(ch.containsKey("agentID") && (ch.containsKey("agentProcessExited")))
				{
				    agentID = Integer.parseInt((String)ch.get("agentID"));
				}
	   			try
	   			{
	   				synchronized(_agentList) 
	   				{
	   					Enumeration agents=listAgents();
	   					while(agents.hasMoreElements()) 
	   					{
	   						Collection c = (Collection) agents.nextElement();
	   						Iterator it = c.iterator();
	   						while (it.hasNext()) {
	   							IAgent agentvar =(IAgent) it.next();

	   							try
	   							{
	   								if(((AgentImpl)agentvar).getAgentInstanceId() == agentID)
	   								{
	   									IProcess proc = agentvar.getProcess();
	   									if (proc != null) {
	   										((ProcessImpl) proc).handleCommand(command);
	   									}

	   									//agentvar.releaseAccess();
	   								}
	   							}catch(Exception e){
	   								throw new InactiveAgentException(Constants.TPTP_PLATFORM_EXEC_MSG20);
	   							}
	   						}
	   					}
	   				} //synchronize
	   			}
	   			catch(Exception e){}
	   			//throw new InactiveAgentException(Constants.TPTP_PLATFORM_EXEC_MSG20);
	   		}});
	}
		
	//This method has to be updated - the logic is to MultiPlex the 
	//Console - a ConsoleDataProcessor - basically implementing the DataProcessor
	//will store a Map of PID + DataProcessors 
	public ConsoleDataProcessor getConsoleMapper()
	{
		//return _consoleMapper;
		return (new ConsoleDataProcessor());
	}
	
	public boolean isConnected() {
		return _connection != null && _connection.isActive();
	}
	
	public int getProtocolVersion() {
		if (_connection != null && _connection instanceof ConnectionImpl) {
			return ((ConnectionImpl) _connection).getProtocolVersion();
		}
		
		return -1;
	}
	
	public boolean isAuthenticated() {
		return _connection != null && ((ConnectionImpl)_connection).isAuthenticated(); 
	}
}
