/*******************************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation, Intel Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    IBM Corporation - Initial API and implementation
 *    Vishnu K Naikawadi, Intel - Initial API and implementation
 *    Andy Kaylor, Intel - revised reconnect implementation, removed RAC support
 *
 * $Id$ 
 *******************************************************************************/

package org.eclipse.tptp.platform.execution.client.core.internal;

import java.io.BufferedInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InterruptedIOException;
import java.io.OutputStream;
import java.net.InetAddress;
import java.net.Socket;
import java.net.SocketException;
import java.util.Enumeration;
import java.util.Vector;
import java.util.Hashtable;

import org.eclipse.tptp.platform.execution.client.core.*;
import org.eclipse.tptp.platform.execution.client.core.internal.commands.AuthenticateCommand;
import org.eclipse.tptp.platform.execution.client.core.internal.commands.DataConnectionCommand;
import org.eclipse.tptp.platform.execution.security.User;
import org.eclipse.tptp.platform.execution.util.*;
import org.eclipse.tptp.platform.execution.util.internal.CommandFragment;
import org.eclipse.tptp.platform.execution.util.internal.Constants;
import org.eclipse.tptp.platform.execution.util.internal.Message;
import org.eclipse.tptp.platform.execution.util.internal.TPTPMessageUtil;
import org.eclipse.tptp.platform.execution.util.internal.TPTPXMLParse;
import org.eclipse.tptp.platform.execution.exceptions.*;

public class ConnectionImpl implements IConnection {
	protected Socket _socket;
	protected INode _node;
	protected int _port;
	protected InetAddress inetAddress = null;
	
	private int _connectionId = 0;
	private int protoVersion = 0;

	private static long _context = 1;
	private ContextMapper _contextMapper = null;
	private Hashtable _dataConnectionMap = null;
	private ICommandHandler _cmdHandler = null;
	private boolean _isComplete = false;

	private final Vector _listeners = new Vector();

	private static final Object writeLock = new Object();
	private static final Object writeDataLock = new Object();
	private static final Object contextLock = new Object();
	private static final Object authLock = new Object();
	private boolean _isInitialized = false;
	private boolean isAuthenticated = false;

	private int _soTimeout = Constants.TIMEOUT_PERIOD;

	class DataConnection {				// it is better to move all processing of data connections to AC or Node
		private Socket _dataSocket;
		private ACTCPDataServer _dataServer;
		private int dataConnectionId;
		
		public void setDataSocket(Socket dataSocket) { _dataSocket = dataSocket; }
		public void setDataServer(ACTCPDataServer dataServer) { _dataServer = dataServer; }
		public Socket getDataSocket() { return _dataSocket; }
		public ACTCPDataServer getDataServer() { return _dataServer; }
		public int getDataConnectionId() { return dataConnectionId; }
		public void setDataconnectionId(int dataConnectionId) { this.dataConnectionId = dataConnectionId; }
	}
	
	public ConnectionImpl() {
	}

	public int getConnectionId() {
		return _connectionId;
	}
	
	public boolean isNewAC() {
		return true;
	}
	
	public long getNextContextId() {
		synchronized(contextLock){
			return _context++;
		}
	}
	
	public void addContext(long contextID, ICommandHandler handler)	{
		synchronized(contextLock){
			this._contextMapper.addContext(contextID, handler);
		}
	}

	protected Socket connectSocket () throws IOException {
		if (_node == null) return null;
		return connectSocket(_node.getAllInetAddresses(), _port);
	}
	
	protected Socket connectSocket (InetAddress addrs[], int port) throws IOException {
		if (addrs == null || addrs.length <= 0) return null;

		Socket socket = null;
		
		for (int i=0; i<addrs.length; i++) {
			if(Constants.TPTP_DEBUG) System.out.println("Connecting to " + addrs[i] + " at port " + _port);
			socket = new Socket(addrs[i], port);
			socket.setSoTimeout(_soTimeout);
			socket.setTcpNoDelay(true);
			inetAddress = addrs[i];
			break;
		}

		return socket;
	}
	
	public void connect(INode node, ConnectionInfo connInfo) throws IOException, SecureConnectionRequiredException, 
		LoginFailedException, UntrustedAgentControllerException, ReconnectRequestedException {
		
		_soTimeout = connInfo.getSoTimeout();
		if (_soTimeout > 0) _soTimeout *= Constants.CONNECT_TIMEOUT_TRY_COUNT;
		
		_node = node;
		_port = connInfo.getPort();
		
		_socket = connectSocket(node.getAllInetAddresses(), _port); 
		if (_socket == null) throw new IOException();

		init();
		
		// We expected to have a connectionID by now.  If we don't, it's because the wait above timed out
		if (_connectionId == 0) {
       		throw new LoginFailedException("Timeout while waiting for connection to complete");
		}
	}
	
	public void connect(INode node, int port) throws IOException, SecureConnectionRequiredException, LoginFailedException, UntrustedAgentControllerException, ReconnectRequestedException {
		ConnectionInfo connInfo = new ConnectionInfo();
		connInfo.setPort(port);
		connect(node, connInfo);
	}

	protected void init() throws SecureConnectionRequiredException, LoginFailedException, IOException, ReconnectRequestedException {
		_contextMapper = new ContextMapper();
		_dataConnectionMap = new Hashtable();
		
		_cmdHandler = new ICommandHandler() {
			public void incomingCommand(INode node, ICommandElement command) {
				handleACCommands(command);
			}
		};

		sendControlCommand(Constants.CONNECT);

		byte rbuf[] = new byte[512];
		BufferedInputStream inStream = new BufferedInputStream(_socket.getInputStream());
		int offset = 0;

		while (true) {
			int bytesRead = inStream.read(rbuf, offset, rbuf.length-offset);

			if(bytesRead < 0) break;

	        if (offset > 0) {
	        	bytesRead += offset;
				offset = 0;
	        }
			
			 // If we've tried to connect to a backward compatibility
			 // layer of the AC, we'll immediately get a response in 
			 // RAC format.  If we ignore this, the BC layer will
			 // respond to our first message by giving us the port
			 // to connect to the regular socket TL.
			 //
	        
			int v = checkForRACMessage(rbuf, offset);
			if(v >= 0) {
				// We read the header, the next four bytes will be the length 
				// (including the header) of the RAC message
				int len = (int) TPTPMessageUtil.readTPTPLongFromBuffer(rbuf, v);
				if (len == bytesRead) {
					offset = 0;
					continue;
				}
				
				if (len > bytesRead) {
					inStream.skip(len - bytesRead);
					offset = 0;
					continue;
				}

				offset = len;
			}

				// First we make sure we have at least enough read for the message header.
				  // If not, compress and re-read.
			//
			if (bytesRead-offset < Constants.AC_MESSAGE_HEADER_SIZE) {
				System.arraycopy(rbuf, offset, rbuf, 0, bytesRead-offset);
				offset = bytesRead - offset;
				continue;
			}
			
				// We have the header information, now lets try and get the entire message.  Once
				 //  again the same error conditions can occur.
				//
				//System.out.println("Offset xx::"+new String(buffer));

			ControlMessage msg=new ControlMessage();
			msg.setMessageType(Constants.TPTP_AC_MESSAGE);
			int current;
				
			try {
				if (rbuf.length == bytesRead + offset) {
					current = msg.readFromBuffer(rbuf, offset);
				}
				else {
					// length is not passed to readFromBuffer so we should ensure 
					// that readFromBuffer will not read trash from the end of the buffer
					// and no exception will be thrown.
					// ControlMessage.readFromBuffer(buffer, offset, length) method
					// is needed in order to prevent extra memory allocations in this case.
					byte[] buffer2 = new byte[bytesRead];
					System.arraycopy(rbuf, offset, buffer2, 0, bytesRead);
					current = msg.readFromBuffer(buffer2, 0);
				}
			} catch (Exception e) {
				throw new IOException();
			}
			
			long flags = msg.getFlags();
			if ((flags & Constants.CONNECTION_COMPLETE) != 0) {
				isAuthenticated = (flags & Constants.AUTHENTICATION_FAILED) == 0;
				
	    		int cmdOffset = 0;
	    		_connectionId = extractConnectionId(rbuf, current+cmdOffset);
	    		cmdOffset += 4;

		    	int uuidLen = (int) TPTPMessageUtil.readTPTPLongFromBuffer(rbuf, current+cmdOffset);
	    		cmdOffset += 4 + uuidLen + 1;	// including trailing null
		    		
	    		if ((cmdOffset + 4) <= msg.getLength()) { 	// check if protoVersion is available
	    			protoVersion = (int) TPTPMessageUtil.readTPTPLongFromBuffer(rbuf, current+cmdOffset);
	    		}
				
	    		if (Constants.TPTP_DEBUG) System.out.println("Connection complete response - " + _connectionId + ", protoVer=" + protoVersion);
	    		
				break;
			}								// equals - since CONNECTION_RECONNECT_REQUEST has two bits set
			else if ((flags & Constants.CONNECTION_REFUSED) != 0) {
				if ((flags & Constants.SECURITY_REQUIRED) != 0) {
					throw new SecureConnectionRequiredException();
				}
				else if((flags & Constants.CONNECTION_RECONNECT_REQUEST) == Constants.CONNECTION_RECONNECT_REQUEST) {
					int portreq = -1;
	    		
					try	{
						close();
						CommandFragment cmd = (CommandFragment)msg.getCommand(0);
						String strToParse = cmd.getCommandData();
						TPTPXMLParse ParseObj = new TPTPXMLParse();
						ParseObj.setParser(strToParse.trim());
						Hashtable CommandHash = ParseObj.getHashTable();
						if(CommandHash.containsKey("port")) {
							portreq = Integer.parseInt((String)ParseObj.getHashTable().get("port"));
						}
					}
					catch(Exception e){}
    	    	 			
					if(portreq == -1) {
						// This is a bad situation.  The AC should have given us a port
						// Since it didn't (unexpectedly), let's try a default
						portreq = 10006;
					}
					
					// Tell the caller how to reconnect
					throw new ReconnectRequestedException(portreq);
				}
				
				throw new IOException("Connection Refused");
			}
			
			throw new IOException("Unknown command");
		}
		
		_isComplete = false;
		//209165 start Thread as a daemon thread
    	Thread socketThread = new SocketReaderThread();
    	socketThread.setDaemon(true);
    	socketThread.start();
    	
    	_isInitialized = true;
	}
	
	private void handleACCommands(ICommandElement command) {
		long contextId=command.getContext();
		if(Constants.TPTP_DEBUG)System.out.println("The context of the returned command:" + contextId);
		
		//Find the command handler associated with this contextId and
		// forward the message appropriately.
		ICommandHandler ch=_contextMapper.getHandler(contextId);
		if(ch != null) {
			if(Constants.TPTP_DEBUG)System.out.println("Forwarding to command handler");
			ch.incomingCommand(_node, command);
		}	
	}
	
	private int extractConnectionId(byte[] buffer, int offset) {
		return (int) TPTPMessageUtil.readTPTPLongFromBuffer(buffer, offset);
	}
	
	/**
	 *********************************************************
	 *                                                       
	 * @brief                                                
	 *    send in the CONNECT command
	 *
	 * @return
	 *    0 if success
	 *    nonzero if error
	 *
	 *********************************************************/
	private void sendCONNECT_DATACommand(Socket datasock, int direction) throws IOException	{
		long flags = 0 ;

		/* build the CONNECT command string */
		flags = flags | Constants.CONNECT_DATA  ;		
		ControlMessage connectMessage = new ControlMessage();
		connectMessage.setMessageType(Constants.TPTP_AC_MESSAGE);
		connectMessage.setMagicNumber(Constants.AC_MAGIC_NUMBER);
		connectMessage.setFlags(flags);

		connectMessage.appendCommand(new DataConnectionCommand(direction));

		int count=connectMessage.getSize();
		byte[] buffer=new byte[count];
		connectMessage.writeToBuffer(buffer, 0);
		
		OutputStream stream = datasock.getOutputStream();
		stream.write(buffer);		
		stream.flush();
	}
	
	
	/**
	 *********************************************************
	 *                                                       
	 * @brief                                                
	 *    send in the control command
	 *
	 * @return
	 *    0 if success
	 *    nonzero if error
	 *
	 *********************************************************/
	private void sendControlCommand(long commandFlags) throws IOException  {
		long flags = 0 ;

		/* build the CONNECT command string */
		flags = flags | commandFlags ;

		ControlMessage connectMessage = new ControlMessage();
		connectMessage.setMessageType(Constants.TPTP_AC_MESSAGE);
		connectMessage.setMagicNumber(Constants.AC_MAGIC_NUMBER);
		connectMessage.setFlags(flags);
		
		if(Constants.TPTP_DEBUG)System.out.println("sending connect message... ");
		sendMessage(connectMessage, _cmdHandler);
	}	

	int createDataConnection(int direction) throws IOException, SecureConnectionRequiredException {
		Socket datasock = connectSocket();
		if (datasock == null) throw new IOException();

		return initDataConnection(datasock, direction);
	}
	
	protected int initDataConnection(Socket datasock, int direction) throws IOException, SecureConnectionRequiredException {
		if (datasock == null || !datasock.isConnected()) return -1;
		
		// CONNECT_DATA command 
		sendCONNECT_DATACommand(datasock, direction);

		// Get the InputStream for this socket so we can read some data 
		byte[] buffer = new byte[256];
		InputStream inStream = datasock.getInputStream();
		int d = inStream.read(buffer, 0, buffer.length);
		
		ControlMessage DataConnectionResponseMessage = new ControlMessage();
		DataConnectionResponseMessage.setMessageType(Constants.TPTP_AC_MESSAGE);
		int masterOffset = DataConnectionResponseMessage.readFromBuffer(buffer, 0);
		long flags = DataConnectionResponseMessage.getFlags();
		if ((flags & Constants.CONNECTION_REFUSED) != 0) {
			if ((flags & Constants.SECURITY_REQUIRED) != 0) {
				throw new SecureConnectionRequiredException();
			}
			
			throw new IOException("Connection Refused");
		}
		
		if ((flags & Constants.DATA_CONNECTION_COMPLETE) <= 0) return -1;
		
		int dataConnectionId = extractConnectionId(buffer, masterOffset);
		if (dataConnectionId <= 0) return -1;
		
		DataConnection dataConnectionInfo = new DataConnection();
		ACTCPDataServer dataServer = new ACTCPDataServer();
			//dataConnectionId = ::getConnectionId(pMsg) ;
		if(Constants.TPTP_DEBUG)System.out.println("The Data Channel ConnectionID: " + dataConnectionId);
		dataConnectionInfo.setDataSocket(datasock);
		dataConnectionInfo.setDataServer(dataServer);
		dataConnectionInfo.setDataconnectionId(dataConnectionId);
		_dataConnectionMap.put(new Integer(dataConnectionId), dataConnectionInfo);				
		
		try {
			dataServer.startServer(null, datasock);
		}
		catch(SocketException sockExp) {			
		}
		catch(IOException Exp) {			
		}
		
		return dataConnectionId;
	}

	public int addDataListener(int dataConnectionId, IDataProcessor dataProcessor) {
		if(dataConnectionId == -1) return -1;
		
		DataConnection dataConnectionInfo = (DataConnection)_dataConnectionMap.get(new Integer(dataConnectionId));
		ACTCPDataServer dataServer = dataConnectionInfo.getDataServer();
		if(dataServer != null) { 
			dataServer.addDataprocessor(dataProcessor);
			return 0;
		}
		
		return -1;
	}
	
	public int removeDataListener(int dataConnectionId, IDataProcessor dataProcessor) {
		try	{
			DataConnection dataConnectionInfo = (DataConnection)_dataConnectionMap.get(new Integer(dataConnectionId));
			ACTCPDataServer dataServer = dataConnectionInfo.getDataServer();
			dataServer.removeDataprocessor(dataProcessor);
			dataServer.shutdownServer();
		} catch(Exception e) {}
		
		return 0;
	}	
	
	/*
	 * Destroy Data Connection
	 */
	public void destroyDataConnection(int dataConnID)
	{
		closeDataConnection(dataConnID);
		// Remove it from the collection
		_dataConnectionMap.remove(new Integer(dataConnID));
	}
	
	private void closeDataConnection(int dataConnID) {
		DataConnection dataConnectionInfo = (DataConnection)_dataConnectionMap.get(new Integer(dataConnID));
		
		try {
			if (dataConnectionInfo != null) {
				// Send DISCONNECT command for data channel
				sendControlCommand(Constants.DISCONNECT); 
	
				// Stop the TCP Data Server
				dataConnectionInfo.getDataServer().stopServer();
	
				dataConnectionInfo.getDataSocket().close();
			}
		}
		catch(Exception exp) {
			if (Constants.TPTP_DEBUG) System.out.println("Error destroying the data channel - " + exp.getMessage());
		}	
	}
	
	public void sendData(int dataConnectionId, byte[] buffer, int bufferLength) {
		try {
			DataConnection dataConnectionInfo = (DataConnection)_dataConnectionMap.get(new Integer(dataConnectionId));

			synchronized (writeDataLock) {
				OutputStream stream=dataConnectionInfo._dataSocket.getOutputStream();
				stream.write(buffer);
				stream.flush();
				
				if(TPTPMessageDebug.TPTP_MSG_DEBUG && TPTPMessageDebug.TPTP_MSG_DEBUG_PRINT_CONSOLE_OUT)  {
					TPTPMessageDebug.writeDebugMessage("Console data sent to AC ["+ (new String(buffer, 0, buffer.length)).trim()+"]");
				} 
				
			}
		}
		catch(IOException exp) {
		}
		catch(Exception e) {
		}			
	}

	public void sendMessage(IControlMessage msg, ICommandHandler handler) throws IOException {
		try	{
			if (handler != null) {
				int commandCount = msg.getCommandCount();
				for (int i=0; i < commandCount; i++) {
					_contextMapper.addContext(msg.getCommand(i).getContext(), handler);
				}
			}
			
			byte[] buffer=new byte[msg.getSize()];
			msg.writeToBuffer(buffer, 0);
	
			synchronized (writeLock) {
				OutputStream stream=_socket.getOutputStream();
				
				if(TPTPMessageDebug.TPTP_MSG_DEBUG && TPTPMessageDebug.TPTP_MSG_DEBUG_PRINT_MESSAGES)  {
					TPTPMessageDebug.writeDebugMessage("Message sent to AC ["+(new String(buffer, 0, buffer.length)).trim()+"]");
				} 

				stream.write(buffer);
				stream.flush();
			}
		}
		catch(IOException exp) {
			if (Constants.TPTP_DEBUG)System.out.println("SendMessage error - " + exp);
			throw exp;
		}
		catch(Exception exp) {
			if (Constants.TPTP_DEBUG)System.out.println("SendMessage error - " + exp);
		}
	}

	public void disconnect() {
			if(!_isComplete) {
				_isComplete=true;

				try 
				{
					// Destroy Data Connections
					Enumeration DataConnections = _dataConnectionMap.keys();
					while(DataConnections.hasMoreElements())
					{
						Integer ConnID = (Integer)DataConnections.nextElement();
						this.closeDataConnection(ConnID.intValue());						
					}
					
					// Clean up the data connection map
					_dataConnectionMap.clear();
					
					// Send DISCONNECT command
					sendControlCommand(Constants.DISCONNECT);
					
					if (Constants.TPTP_DEBUG) System.out.println("Successfully disconnected.");
				}
				catch(IOException e) {
					/* We can ignore this */
					if (Constants.TPTP_DEBUG) System.out.println("Exception while disconnecting ... " + e);
				}
				
				close();

				/* If this is a nodeimpl clear the keystore spec so it tries an insecure connect next time */
				//Commented by GN since security not supported yet in the new 
				//	if(_node instanceof INode) {
				//		((INode)_node).getAgentController(this._port).setSecurityParameters(null);
				//	}
				
				/* Inform each of the listeners that the connection is closed. */
				/*
				synchronized(_listeners) {
					Enumeration e=_listeners.elements();
					while(e.hasMoreElements()) {
						//ConnectionListener listener=(ConnectionListener)e.nextElement();
						//listener.connectionClosed(this);
					}
				}
				*/			
			}
	}

	public void close () {
		if (_socket != null) {
			try { 
				_socket.close(); 
			} catch (Exception e) {}
		}
		
		_isComplete = true;
	}
	
	public INode getNode() {
		return _node;
	}

	public boolean isActive() {
		return _isInitialized && !_isComplete;
	}
	
	public static int checkForRACMessage(byte[] buffer, int offset) {
		Message tempMessage=new Message();
		tempMessage.readFromBuffer(buffer, offset);
		if(tempMessage.getMagicNumber() == Constants.RA_MAGIC) {
			return Constants.RAC_MESSAGE_HEADER_SIZE;
		} 
		return -1;
	}

	public int getPort() {
		return _port;
	}
	
	public InetAddress getInetAddress() {
		return inetAddress;
	}
	
	protected ContextMapper getContextMapper()
	{
	    return this._contextMapper;
	}

	protected int processControlMessage(byte[] buffer, int offset, int length) {
		if(_cmdHandler == null) return -1;
		
		ControlMessage msg=new ControlMessage();
		msg.setMessageType(Constants.TPTP_AC_MESSAGE);

		int current=-1;
		try {
			if(Constants.TPTP_DEBUG)System.out.println("Processing the Message.");
			if (buffer.length == length + offset) {
				current = msg.readFromBuffer(buffer, offset);
			}
			else {
				// length is not passed to readFromBuffer so we should ensure 
				// that readFromBuffer will not read trash from the end of the buffer
				// and no exception will be thrown.
				// ControlMessage.readFromBuffer(buffer, offset, length) method
				// is needed in order to prevent extra memory allocations in this case.
				byte[] buffer2 = new byte[length];
				System.arraycopy(buffer, offset, buffer2, 0, length);
				current = msg.readFromBuffer(buffer2, 0);
				if (current >= 0)
					current += offset;
			}
			/* If we read more bytes then were valid, return -1 */
			if(current>offset+length) {
				return -1;
			}
		}
		catch(IndexOutOfBoundsException e) {  
			return -1;
		}
		catch(Exception e) {
			System.out.println("Exception while processing the message" + e);
	        return -1;
		}

	        /* If we have parsed the message successfully  Then process it */
		int len = msg.getSize();
			
	    if (current == len+offset) {
			/* Valid pass on each command */
//	    		if(Constants.TPTP_DEBUG)System.out.println("Checking for CONNECTION_COMPLETE response");
	   		long flags = msg.getFlags();
   			
	   		if ((flags & Constants.AUTHENTICATION_FAILED) != 0) {
	   			synchronized (authLock) {
	   				isAuthenticated = false;
	   				current += msg.getLength();
	   				authLock.notifyAll();
	   			}
	   		}
    		else if ((flags & Constants.AUTHENTICATION_SUCCESSFUL) != 0) {
	   			synchronized (authLock) {
	    			isAuthenticated = true;
		   			current += msg.getLength();
	   				authLock.notifyAll();
	   			}
    		}
	    	else {
	    		int count=msg.getCommandCount();
	    		for(int i=0; i<count; i++) {
	    			_cmdHandler.incomingCommand(_node, msg.getCommand(i));
	    		}
	    	}
	   	}

	    return current; 
	}
	
	/**
	 * @see Connection#addConnectionListener(ConnectionListener)
	 */
	public void addConnectionListener(IConnectionListener listener) {
		synchronized(_listeners) {
			if(!_listeners.contains(listener)) {
				_listeners.add(listener);
			}
		}
	}
	
	/**
	 * @see Connection#removeConnectionListener(ConnectionListener)
	 */
	public void removeConnectionListener(IConnectionListener listener) {
		synchronized(_listeners) {
			if(_listeners.contains(listener)) {
				_listeners.remove(listener);
			}
		}
	}

	public void setSoTimeout(int timeout) {
		_soTimeout = timeout;
	}
	
	public int getProtocolVersion() {
		return protoVersion;
	}
	
	public Socket getSocket() {
		return _socket;
	}
	
	public void setSocket(Socket socket) {
		_socket = socket;
		_isComplete = true; 
	}

	class SocketReaderThread extends Thread implements Constants {
		public void run() {
			byte[] buffer = new byte[MAX_MESSAGE_LENGTH];
			int masterOffset=0;
			boolean incompleteMsg;

			// Get the InputStream for this socket so we can read some data 
			BufferedInputStream inStream;
			try {
				inStream = new BufferedInputStream(_socket.getInputStream());
			} catch (IOException e1) {
				// If we couldn't get the input stream, we can't read
				e1.printStackTrace();
				disconnect();
				return;
			}

			while(!_isComplete) {
				incompleteMsg = false;  // 185463 
				try	{
					int bytesRead = inStream.read(buffer, masterOffset, buffer.length-masterOffset);
					if(bytesRead == -1) break;
				
					if(TPTPMessageDebug.TPTP_MSG_DEBUG && TPTPMessageDebug.TPTP_MSG_DEBUG_PRINT_MESSAGES) {
						TPTPMessageDebug.writeDebugMessage("Message data received on connection ["+ new String(buffer, masterOffset, bytesRead).trim()+"]");
					} 
					
			        if (masterOffset > 0) {
			        	bytesRead += masterOffset;
						masterOffset = 0;
			        }
			      
					//
					// If we've tried to connect to a backward compatibility
					// layer of the AC, we'll immediately get a response in 
					// RAC format.  If we ignore this, the BC layer will
					// respond to our first message by giving us the port
					// to connect to the regular socket TL.
					//
					int vret = checkForRACMessage(buffer, masterOffset);
					if (vret != -1) {
						// We read the header, the next four bytes will be the length 
						// (including the header) of the RAC message
						long length = TPTPMessageUtil.readTPTPLongFromBuffer(buffer, vret);
						masterOffset = (int)length;
					}
	
					while(masterOffset<bytesRead) {
						int newOffset=0;
						// First we make sure we have at least enough read for the message header.
						 //  If not, compress and re-read.
						//
						if ( bytesRead-masterOffset < AC_MESSAGE_HEADER_SIZE ) {
							System.arraycopy( buffer, masterOffset, buffer, 0, bytesRead-masterOffset );
							masterOffset=bytesRead-masterOffset;
							incompleteMsg = true;
							break;
						}
						
						// We have the header information, now lets try and get the entire message.  Once
						 //  again the same error conditions can occur.
						//
						newOffset = processControlMessage(buffer, masterOffset, bytesRead);

						// The following statement was updated to properly output the message, the original statement follows, and is commented out
						if(Constants.TPTP_DEBUG) System.out.println("Recvd Message: " + new String(buffer, masterOffset,  newOffset-masterOffset));
						// (old): if(Constants.TPTP_DEBUG)System.out.println("Recvd Message: " + buffer.toString());
						
						
						if (newOffset == -1) {
							// newOffset of -1 indicates the message is bigger than what's left in the buffer
							// If the masterOffset is zero, this means the message is bigger than the buffer 
							//   itself, so we need to grow the buffer.  Otherwise, we'll slide the message
							//   to the beginning of the buffer and try to read some more
							if(masterOffset>0) {
								System.arraycopy( buffer, masterOffset, buffer, 0, bytesRead-masterOffset );
							}
		 					else if (bytesRead == buffer.length) 
         					{
		                        int len = buffer.length * 2;
		                        byte[] tmpbuffer=new byte[len];
		                        // Copy necessary data over to new buffer 
		                        System.arraycopy(buffer, masterOffset, tmpbuffer, 0, bytesRead-masterOffset);
		                        buffer = tmpbuffer;
                 			}
							masterOffset=bytesRead-masterOffset;
                     		incompleteMsg = true;
							break;
						}
						masterOffset=newOffset;
					} // end of inner while 
	
					if (!incompleteMsg) 
					{
						masterOffset=0;
				    }
				}
				catch(InterruptedIOException e) {
				}
				catch(SocketException e) 
				{
					//e.printStackTrace();
				  break;
				}
				catch(IOException e) 
				{
					//e.printStackTrace();
					break;
				}
			} // end of outer while 
			// The server is now stopping 
			
			disconnect();
		}
	}
	
	public boolean isAuthenticated() {
		return isAuthenticated;
	}
	
	protected boolean authenticateUser(User user) {
		if (user == null || user.getName() == null || user.getPassword() == null) return false;
		
		ControlMessage authMsg = new ControlMessage();
		authMsg.setMessageType(Constants.TPTP_AC_MESSAGE);
		authMsg.setMagicNumber(Constants.AC_MAGIC_NUMBER);
		authMsg.setFlags(Constants.AUTHENTICATE);
		
		AuthenticateCommand authCmd = new AuthenticateCommand(user.getName(), user.getPassword());
		authMsg.appendCommand(authCmd);

		synchronized (authLock) {

			try { 
				sendMessage(authMsg, new ICommandHandler() {
					public void incomingCommand(INode node, ICommandElement command) {}
				});
			} catch (Exception e) {
				return false;
			}

			try { 
				// The wait method inside a synchronized block causes the object to relinquish it's synchronization lock 
				// until notify is called on authLock (or until a timeout occurs) 
				authLock.wait(Constants.WAIT_RESPONCE_TIMEOUT); 
			} catch (Exception e) {
				e.printStackTrace();
			}
		}
		
		return isAuthenticated;
	}
}
