/*******************************************************************************
 * Copyright (c) 2005, 2007 Intel Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Randy D. Smith, Intel - Initial API and Implementation (TPTP 4.0,
 *    							based upon 3.2 FileManagerImpl.java)
 * 	  IBM - Initial API and Implementation (TPTP 3.x)
 * $Id$
 *******************************************************************************/
package org.eclipse.tptp.platform.execution.client.core.internal;

import java.io.*;

import org.eclipse.tptp.platform.execution.client.agent.internal.AgentImpl;
import org.eclipse.tptp.platform.execution.client.core.*;
import org.eclipse.tptp.platform.execution.exceptions.*;
import org.eclipse.tptp.platform.execution.util.*;
import org.eclipse.tptp.platform.execution.util.internal.CommandFragment;
import org.eclipse.tptp.platform.execution.util.internal.Constants;
import org.eclipse.tptp.platform.execution.util.internal.TPTPMessageUtil;

public class FileTransferManagerImpl extends AgentImpl implements IFileTransferManager 
{
	private FileDataProcessor _fileDataProcessor = null;
	final private static String FILE_TRANSFER_AGENT = "org.eclipse.tptp.FileTransferAgent";
	
	 
	public FileTransferManagerImpl(String agentName) {
		super(agentName);
		_fileDataProcessor = new FileDataProcessor();		
	}
	
	public void createFileDataConnection()
	{
		try
		{
			startMonitoring(TPTPDataPath.DATA_PATH_TWO_WAY, _fileDataProcessor);
		}
		catch(InactiveAgentException exp)
		{
			
		}
	}
	 
	/* (non-Javadoc)
	 * @see org.eclipse.tptp.platform.execution.core.IFileTransferManager#getFile(java.lang.String, java.lang.String)
	 */
	public int getFile(String localName, String remoteName) throws IOException
	{
		//System.out.println("getting '" + remoteName + "' to '" + localName + "'");
	  
		/* We may have to create the directory if it does not exist */
		String directoryName = null;

		File file = new File(localName);
	  
		/* We try to get the directory where the file has to be written */
		directoryName = file.getParent();
		//System.out.println("directoryName is " + directoryName);
		File dirFile = new File(directoryName);

		/* We have to create the directory only if it does not exist */
		if (!dirFile.exists())
			dirFile.mkdirs(); /* we will have to create parent directories as well */

		/* Create the local file, if it does not exist, otherwise try to overwrite. */
		if (!file.exists())
			try {
				boolean result = file.createNewFile();
				if (!result) {
					//System.out.println("File creation failed!");
					return FILE_CREATE_ERROR;
				}
			}
			catch (IOException e) {
				System.out.println("IOException on create " + e);
				return FILE_CREATE_ERROR;
			}
			
		if (!file.canWrite()) {
			System.out.println("Can't write to file!");
			return FILE_CREATE_ERROR;
		}
		
		_fileDataProcessor.initializeFileTransfer(localName, file);
		
		//Send the getFile command
		//System.out.println("Sending getFile() command");
		String getCmd = "<getFile iid=\"" + FILE_TRANSFER_AGENT + "\"><localFile>" +
						localName + "</localFile><remoteFile>" + remoteName + "</remoteFile></getFile>";
		try {
				sendCommand(getCmd, new ICommandHandler() {
 				public void incomingCommand(INode node, ICommandElement command)
 				{
 					String commandStr = new String(((CommandFragment) command).getCommandData());
 					//System.out.println("The Get Agent response from the getFile Command - " + commandStr);
 				}
			});
		}
		catch(InactiveAgentException exp) {
			System.out.println("Inactive Agent Exception" + exp);		  
		}
		
		return _fileDataProcessor.waitForFileComplete(localName);
	}
	 
	/* (non-Javadoc)
	 * @see org.eclipse.tptp.platform.execution.core.IFileTransferManager#deleteFile(java.lang.String)
	 */
	public void deleteFile(String absFileName) throws IOException {
		//System.out.println("Sending deleteFile() command to delete '" + absFileName + "'");
		String deleteCmd = "<deleteFile iid=\"" + FILE_TRANSFER_AGENT + "\"><remoteFile>" + absFileName + "</remoteFile></deleteFile>";
		try {
			sendCommand(deleteCmd, new ICommandHandler() {
 				public void incomingCommand(INode node, ICommandElement command)
 				{
 					String commandStr = new String(((CommandFragment) command).getCommandData());
 					//System.out.println("The Get Agent response from the deleteFile command - " + commandStr);
 				}
			});
		}
		catch(InactiveAgentException exp) {
			System.out.println("Inactive Agent Exception" + exp);		  
		}
	}
	 
	/* (non-Javadoc)
	 * @see org.eclipse.tptp.platform.execution.core.IFileTransferManager#putFile(java.lang.String, java.lang.String)
	 */
	public int putFile(String localName, String remoteName) throws IOException {
		File file = new File(localName);
		if (!file.exists()) return FILE_NOT_FOUND;
		if (!file.canRead()) return FILE_READ_ERROR;
		
		boolean _isComplete = false;

		//Send the putFile command
		//System.out.println("Sending putFile() command");
		String putCmd = "<putFile iid=\"" + FILE_TRANSFER_AGENT + "\"><localFile>" +
						localName + "</localFile><remoteFile>" + remoteName + "</remoteFile></putFile>";
		
		try {
			sendCommand(putCmd, new ICommandHandler() {
				public void incomingCommand(INode node, ICommandElement command)
				{
					String commandStr = new String(((CommandFragment) command).getCommandData());
					//System.out.println("The Get Agent response from the putFile command - " + commandStr);
				}
			});
		}
		catch(InactiveAgentException exp) {
			System.out.println("Inactive Agent Exception" + exp);		  
		}

		long fileLength = file.length();
		byte buffer[] = new byte[Constants.MAX_DATA_LENGTH];
		long byteCount = 0;
		final int DIME_START_HEADER_WORD0 = 0x21040000;
		final int DIME_MIDDLE_HEADER_WORD0 = 0x01040000;
		final int DIME_END_HEADER_WORD0 = 0x41040000;
		final int DIME_START_END_HEADER_WORD0 = 0x61040000;
//		final int DIME_START_HEADER_WORD0 = 0x00000421;
//		final int DIME_MIDDLE_HEADER_WORD0 = 0x00000401;
//		final int DIME_END_HEADER_WORD0 = 0x00000441;
//		final int DIME_START_END_HEADER_WORD0 = 0x00000461;

		BufferedInputStream inStream = new BufferedInputStream(new FileInputStream(file));
		while(!_isComplete) {
			try {
				int bytesRead = inStream.read(buffer);
	      
				if (bytesRead == -1) {
					// if byteCount is zero, then the file was empty and no transfer occurs?
					if (byteCount > 0) {
						// David A Scott, davids@us.ibm.com, original fix
						// We have reached EOF on the input buffer file.
						// All of the contents of the file have been written to the data connection.
						// Now we should:
						//System.out.println("Finishing putFile after exact multiple of block size!");
						// Send a packet with no data; file was exact multiple of block size!
						shipIt(DIME_END_HEADER_WORD0, remoteName, buffer, 0);
					} else {		
						//System.out.println("Finishing putFile just as we were about to start!");
						// byteCount <= 0, so can't read anything? We end just as we got started
						shipIt(DIME_START_END_HEADER_WORD0, remoteName, buffer, 0);
					}
					_isComplete = true;
				} else if (bytesRead < Constants.MAX_DATA_LENGTH) {
					if (byteCount == 0) {
						//System.out.println("Starting and ending putFile all at the same time!");
						// First packet will also be our last; normal case for a file of size less than our buffer size
						shipIt(DIME_START_END_HEADER_WORD0, remoteName, buffer, bytesRead);
					} else {
						//System.out.println("Finishing putFile after multiple blocks!");
						// Normal case for reaching end of a file of size greater than our buffer size
						shipIt(DIME_END_HEADER_WORD0, remoteName, buffer, bytesRead);
					}
					_isComplete = true;
				} else if (bytesRead == Constants.MAX_DATA_LENGTH) {
					if (byteCount == 0) {
						// First packet of a file (potentially/probably) bigger than one packet
						shipIt(DIME_START_HEADER_WORD0, remoteName, buffer, bytesRead);
					} else {
						// Normal case for a middle packet (at least we think so... as long as next read doesn't return -1!)
						shipIt(DIME_MIDDLE_HEADER_WORD0, remoteName, buffer, bytesRead);
					}
				} else {
					// What gives? How did we read MORE than a buffer full??
					// What do we do here? Bail? Send a DIME_MESSAGE_END packet?? of how much?
					//System.out.println("Should not have the case of reading more than asked for in putFile!");
					return FILE_TRANSFER_ERROR;
				}
				byteCount += ((bytesRead > 0) ? bytesRead : 0);
			}
			catch(InterruptedIOException e) {
				// This happens as the read has timed out.  Just continue as the connection is still valid.
			}
		}
		if (byteCount != fileLength) {
			//System.out.println("Did not send entire file correctly!");
		}
		inStream.close();
		
		return FILE_TRANSFER_OK;
	}
	
	private void shipIt(int word0, String fileName, byte [] buffer, int dataLength) {
		int fileNameLength = fileName.length();
		int myDimeBufferSize = Constants.DIME_HEADER_LEN + fileNameLength + 1 + dataLength; 
		byte myDimeBuffer[] = new byte[myDimeBufferSize];
		int soFar = 0;
		try {
			soFar = TPTPMessageUtil.writeTPTPLongToBuffer(myDimeBuffer, soFar, word0);
			soFar = TPTPMessageUtil.writeTPTPLongToBuffer(myDimeBuffer, soFar, ((fileNameLength+1)<<16));
			soFar = TPTPMessageUtil.writeTPTPLongToBuffer(myDimeBuffer, soFar, dataLength);
			System.arraycopy(fileName.getBytes(), 0, myDimeBuffer, soFar, fileNameLength);
			soFar += fileNameLength;
			myDimeBuffer[soFar++] = 0;
			System.arraycopy(buffer, 0, myDimeBuffer, soFar, dataLength);
			sendDataWithoutDime(myDimeBuffer, myDimeBufferSize);
		}
		catch(DataChannelConnectionException e) {
			//System.out.println("Data Channel Connection gone");
		}
	}
}
