/*******************************************************************************
 * Copyright (c) 2005, 2007 IBM Corporation, Intel Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    IBM Corporation - Initial API and implementation
 *    Vishnu K Naikawadi, Intel - Initial API and implementation
 *
 * $Id: IBaseAgent.java,v 1.16 2007/12/07 12:58:49 ialelekov Exp $ 
 *******************************************************************************/ 


package org.eclipse.tptp.platform.execution.datacollection;

import org.eclipse.tptp.platform.execution.exceptions.* ;
import org.eclipse.tptp.platform.execution.util.ICommandFragment;
import org.eclipse.tptp.platform.execution.util.TPTPAgentAccess;

/**
 * IBaseAgent provides the interface for development of an Agent.
 * This agent registers with the Agent Controller and provides datacollection services to the client. 
 * @provisional 
 * @since tptp 4.0
 */

public interface IBaseAgent 
{

	/**
	 * @throws NativeLibraryNotFound
	 */
	public void init() throws NativeLibraryNotFound;
	
	/**
	 * Returns whether the agent registered successfully with Agent Controller
	 * @return boolean 
	 */
	public boolean isRegistered() ;

    /**
     * Deregister from the Agent Controller.
     * @return int
     */
    public int deregister() throws AgentControllerUnavailableException;
    
    /**
     * Register the Agent with Agent Controller.
     * @return int
     */
    public int register() throws AgentControllerUnavailableException;
    /**
     * Provides data exchange with the Agent Controller prior to Agent Registration
     * @return int
     */
    public int preRegisterInitialization() throws AgentControllerUnavailableException;
	/**
	 * Process the commands recieved by the agent
	 * The sub-classes must override this method to provide their command processing 
	 * functionality
	 * @param command
	 */
	public void processCommand(ICommandFragment command) ;
	

	/**
	 * send the given data buffer to the given destination
	 * This method is used by the Remote Agent 
	 * on the AgentController to send the data to the Client
	 * @param targetInstanceId - send data to the target connection id
	 * @param buffer - data to be sent
	 * @throws DataChannelConnectionException
	 * @return 0 successful; non-zero error
	 */
	//public int sendData(int targetInstanceId, char[] buffer, int bufferLength) throws DataChannelConnectionException, NotConnectedException;
	public int sendData(int targetInstanceId, char[] buffer) throws DataChannelConnectionException, AgentControllerUnavailableException;
	
	/**
	 * send the given data buffer to the given destination
	 * This method is used by the Remote Agent 
	 * on the AgentController to send the data to the Client
	 * @param targetInstanceId - send data to the target connection id
	 * @param buffer - data to be sent
	 * @throws DataChannelConnectionException
	 * @return 0 successful; non-zero error
	 */
	public int sendData(int targetInstanceId, byte[] buffer) throws DataChannelConnectionException, AgentControllerUnavailableException;
		
	/**
	 * send the given data buffer to the given destination
	 * This method is used by the Remote Agent 
	 * on the AgentController to send the data to the Client
	 * @param targetInstanceId - send data to the target connection id
	 * @param buffer - data to be sent
	 * @param offset - start offset in the data
	 * @param length - number of bytes to send
	 * @throws DataChannelConnectionException
	 * @return 0 successful; non-zero error
	 */
	public int sendData(int targetInstanceId, byte[] buffer, int offset, int length) throws DataChannelConnectionException, AgentControllerUnavailableException;
	
	/**
	 * ReceiveData from the Client 
	 * @param sourceId 
	 * @param buffer
	 * @param bufferLength
	 * @throws DataChannelConnectionException, NotConnectedException
	 */
	public int receiveData(int sourceId, String buffer, int bufferLength) throws DataChannelConnectionException, AgentControllerUnavailableException;
	
 	/**
	 * ReceiveData from the Client 
	 * @param sourceId 
	 * @param buffer
	 * @param bufferLength
	 * @throws DataChannelConnectionException, NotConnectedException
	 */
	public int receiveData(int sourceId, byte[] buffer, int bufferLength) throws DataChannelConnectionException, AgentControllerUnavailableException;
	
	/**
	 * This method is invoked by the Agent Controller if there are no referencing clients
	 */
	public void waitForTermination();
	
	/**
	 * Return the agent process id
	 * @return long agent process id
	 */
	public long getProcessID();
	
	/**
	 * Return the agent configuration information.
	 * This information is defined in the agent configuration file
	 * @return configuration value
	 */
	public String getConfigValue();
	
	/**
	 * Return the agent name
	 * @return agent name
	 */
	public String getAgentName();
	
	/**
	 * Return the agent connection id
	 * @return agent connection id
	 */
	public int getAgentID();
	
	/**
	 * @param eventsIID
	 * @param listenerID
	 * @param replyDest
	 * @param replyContext
	 * @return int
	 */
	public int addEventListener(String eventsIID, int listenerID, int replyDest, int replyContext);

	/**
	 * @param eventsIID
	 * @param listenerID
	 * @param replyDest
	 * @param replyContext
	 * @return success or failure
	 */
	public int removeEventListener(String eventsIID, int listenerID, int replyDest, int replyContext);
	
	/**
	 * Return the list of interfaces serviced by the Agent
	 * @return list of interfaces serviced by the agent
	 */
	public String[] getSupportedInterfaces();
	
	/**
	 * Set the list of interfaces supported by the Agent, subclasses 
	 * should override this method to customize their characteristics
	 * @param interfaceID set the interfaces supported by the agent
	 */
	public void setSupportedInterfaces(String[] interfaceID);
	
	/**
	 * 
	 * @param cmd containing the command for event notification
	 * @throws AgentControllerUnavailableException
	 */
	public int sendEventNotifications(ICommandFragment cmd) throws AgentControllerUnavailableException;

	/**
	 * Sub classes should override this class for additional functionality
	 * @param cmd
	 * @return int
	 */
	public int processEventProviderCommands(ICommandFragment cmd);
	/**
	 * Sub classes should override this class for additional functionality
	 * @param cmd
	 * @return int
	 */
	public int processDataProviderCommands(ICommandFragment cmd);
	/**
	 * Add the clients that are referecing this agent 
	 * @param clientConnId - Add the client to the listener list
	 * @param accessMode - Agent Reference mode of the client
	 */
	public void addClient(int clientConnId, TPTPAgentAccess accessMode);
	
	/**
	 * @param clientConnId
	 */
	public void removeClient(int clientConnId);
	
	/**
	 * @param outCommand
	 * @param respHandler
	 * @throws AgentControllerUnavailableException
	 */
	public void sendCommand(String outCommand) throws AgentControllerUnavailableException;
	
	/**
	 * @param outCommand
	 * @param respHandler - Command handler to process the response
	 * @throws AgentControllerUnavailableException
	 */
	public void sendCommand(ICommandFragment outCommand) throws AgentControllerUnavailableException;
	
	/**
	 * @return Agent UUID
	 */
	public String getUUID(); 
	
	public void processCommandLine(String argv[]);
}

