/*******************************************************************************
 * Copyright (c) 2007 Intel Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Igor Alelekov, Intel - initial API and implementation
 *    
 * $Id: AgentControllerFactory.java,v 1.2 2007/09/06 11:49:39 jkubasta Exp $
 *
 *******************************************************************************/ 
package org.eclipse.tptp.platform.execution.util.internal;

import org.eclipse.tptp.platform.execution.client.core.ConnectionInfo;
import org.eclipse.tptp.platform.execution.client.core.IAgentController;
import org.eclipse.tptp.platform.execution.client.core.INode;
import org.eclipse.tptp.platform.execution.client.core.NodeFactory;
import org.eclipse.tptp.platform.execution.exceptions.AgentControllerUnavailableException;
import org.eclipse.tptp.platform.execution.exceptions.LoginFailedException;
import org.eclipse.tptp.platform.execution.exceptions.SecureConnectionRequiredException;
import org.eclipse.tptp.platform.execution.security.User;

public class AgentControllerFactory implements IAgentControllerFactory {
	/**
	 * The incremental wait times
	 */
	public static final int INCREMENTAL_WAIT = 100;

	/**
	 * The total incremental wait
	 */
	public static final int TOTAL_WAIT = 100;

	public IAgentController createConnection(String hostName, int portNumber) throws Exception {
		IAgentController ac = null;
		EstablishConnection establishConnection = new EstablishConnection(hostName, portNumber);
		String userName = null;
		boolean lookingForUser = true;
		int retry = 0;
		
		do {
			Thread ect = new Thread(establishConnection);
			ect.start();
			/* Wait for a maximum of 10 seconds before bailing out */
			ect.join(TOTAL_WAIT*INCREMENTAL_WAIT);
			if (ect.isAlive()) ect.interrupt();
			
			Exception error = establishConnection.getException();
			if (error != null) throw error;
			
			ac = establishConnection.getConnection();
			if (ac == null) {
				throw new AgentControllerUnavailableException("");
			}
			
			if (ac.isAuthenticated()) break;
			
			User user = null;
			if (lookingForUser) {
				user = findUser(hostName);
				lookingForUser = false;
				if (user != null) {
					userName = user.getName();
					establishConnection.setUser(user);
					continue;		// try it
				}
			}
			
			user = promptAuthentication(hostName, userName);
			if(user == null) break;

			userName = user.getName();
			establishConnection.setUser(user);
		} while (retry++ < 3);

		if (!ac.isAuthenticated()) {
			try { ac.disconnect(); } catch (Exception e) {}
			throw new LoginFailedException("Authentication failed");
		}
		
		return ac;
	}

	/*
	 * The establishment of the connection should be done in a separate
	 * thread.
	 */
	class EstablishConnection implements Runnable {
		private String host;
		private int port;
		private Exception e;
		private IAgentController ac;
		private User user;
		private INode node;

		public EstablishConnection (String host, int port) {
			this.host = host;
			this.port = port;
			
			node = null;
		}
		
		public void run() {
			ac = null;
			e = null;

			if (node == null) {
				try {
					node = NodeFactory.createNode(host);
				} catch (Exception ex) {
					node = null;
				}
			
				if (node == null) return;
			}
			
			ConnectionInfo connInfo = new ConnectionInfo();

			connInfo.setHostName(host);
			connInfo.setPort(port);
			
			try {
				if (user == null)
					ac = node.connect(connInfo);
				else
					ac = node.connect(connInfo, user);

				/* Notify that the connection is done */
			} catch (LoginFailedException e) {
			} catch (SecureConnectionRequiredException e) {
			} catch (Exception e) {
				/*
				 * For some reason Sun JDK 1.5_06 has problems resolving the
				 * hostname "localhost". As a workaround change it to
				 * 127.0.0.1
				 */

				this.e = e;
				
				if ("localhost".equals(connInfo.getHostName())) {
					connInfo.setHostName("127.0.0.1");
				
					try {
						ac = node.connect(connInfo);
						this.e = null;
					} catch (Exception ex) {}
				}
			}
		}

		public Exception getException() {
			return e;
		}

		public IAgentController getConnection() {
			return ac;
		}
		
		public void setUser(User user) {
			this.user = user;
		}
	}

	// 	to prevent multiple User registration, to be substituted by descendants
	public User findUser(String hostName) {
		return null;
	}
					
	public User promptAuthentication(String hostName, String userName) {
		return null;
	}
}
