/**********************************************************************
 * Copyright (c) 2007 IBM Corporation, Intel Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.tptp.platform.execution.util.internal;

import java.util.Hashtable;

import org.eclipse.tptp.platform.execution.client.agent.internal.AgentStatePool;
import org.eclipse.tptp.platform.execution.client.core.ConnectionInfo;
import org.eclipse.tptp.platform.execution.client.core.IAgentController;
import org.eclipse.tptp.platform.execution.client.core.INode;
import org.eclipse.tptp.platform.execution.client.core.NodeFactory;
import org.eclipse.tptp.platform.execution.exceptions.AgentControllerUnavailableException;
import org.eclipse.tptp.platform.execution.exceptions.LoginFailedException;
import org.eclipse.tptp.platform.execution.exceptions.NotConnectedException;
import org.eclipse.tptp.platform.execution.client.core.internal.AgentController;
import org.eclipse.tptp.platform.execution.exceptions.SecureConnectionRequiredException;
import org.eclipse.tptp.platform.execution.security.User;

/**
 * This delegate is used to keep track of connections with Agent Controller.
 * Only one connection should be established with any one host. Establishing
 * connections is expensive and should only be done once. This is a singleton
 * class.
 * 
 * @author Ali Mehregani
 */
public class AgentControllerPool {	// may be inherited to provide UI
	/**
	 * The incremental wait times
	 */
	public static final int INCREMENTAL_WAIT = 100;

	/**
	 * The total incremental wait
	 */
	public static final int TOTAL_WAIT = 100;

	/* The instance of this class */
	protected static AgentControllerPool instance;

	private IAgentControllerFactory acFactory = null;
	/*
	 * Keeps track of the connections to Agent Controller KEY = hostname VALUE =
	 * The connection
	 */
	private static Hashtable connectionPool = new Hashtable();

	/*
	 * Keeps track of the port mappings in case when AC is run on different port instead of requested.
	 * Key: host:port
	 */
	private static Hashtable portMappings = new Hashtable();
	
	/* Indicates if the conneciton is still alive */
	private boolean isConnectionAlive;

	/**
	 * Hide the construtor
	 */
	protected AgentControllerPool() {
	}

	/**
	 * Return the instance of this singleton class 
	 * 
	 * @return The instance of this class
	 */
	public static AgentControllerPool getInstance() {
		if (instance == null) instance = new AgentControllerPool();
		return instance;
	}

	public IAgentController getConnection(String hostName, int portNumber)
			throws Exception {
		return getConnection(hostName, portNumber, true);
	}

	/**
	 * Return the connection to the hostname and the port number
	 * 
	 * @param hostName
	 *            The hostname that Agent Controller is running on
	 * @param portNumber
	 *            The port number that Agent Controller is running on
	 * @param reuseExistingConnection
	 *            Reuses an existing connection if one already exists
	 * 
	 * @return An Agent Controller connection
	 * @throws Exception
	 *             If any errors occur.
	 */
	public IAgentController getConnection(String hostName, int portNumber,
			boolean reuseExistingConnection) throws Exception {
		
		String hostKeyName = AgentStatePool.getUnifiedHostName(hostName);
	
		/* Attempt to lookup the connection */
		IAgentController ac = (IAgentController) connectionPool.get(hostKeyName);
		if (ac != null && ac instanceof AgentController) {
			if (!((AgentController)ac).isConnected()) {
				connectionPool.remove(hostKeyName);
				ac = null;
			}
		}

		if (reuseExistingConnection && ac != null) {
			ConnectionInfo connectionInfo = ac.getConnectionInfo();
			portNumber = getPortMapping(hostKeyName, portNumber);

			/* We found an existing match */
			if (portNumber == connectionInfo.getPort()) {
				/* Poke the connection to make sure that it's still alive */
				isConnectionAlive = false;
				
				try {
					final IAgentController agentController = ac;
					Thread thread = new Thread(new Runnable() {
						public void run() {
							try {
								agentController.queryAvailableAgents();
								isConnectionAlive = true;
							} catch (NotConnectedException e) {
								/* Doesn't need to be handled */
							}
						}
					});

					thread.start();
					thread.join(TOTAL_WAIT*INCREMENTAL_WAIT);
				} catch (Exception e) {}

				if (isConnectionAlive) return ac;

				/* Connection is dead - remove it from the connection pool. */
				connectionPool.remove(hostKeyName);
			}
		}

		/*
		 * A match was not found in our connection pool. Establish a new
		 * connection
		 */

		ac = createConnection(hostName, portNumber);
		if (ac == null)	{
			throw new AgentControllerUnavailableException();
		}
		
		/* Store and return the connection */
		connectionPool.put(hostKeyName, ac);
		setPortMapping(hostKeyName, portNumber, ac.getConnectionInfo().getPort());

		return ac;
	}
	
	private int getPortMapping(String host, int port) {
		String key = host + ":" + port;
		if (portMappings.containsKey(key)) {
			Integer portI = (Integer) portMappings.get(key);
			port = portI.intValue();
		}
		return port;
	}
	
	private void setPortMapping(String host, int port, int newPort) {
		String key = host + ":" + port;
		portMappings.put(key, new Integer(newPort));
	}
	
	public void setACFactory(IAgentControllerFactory acFactory) {
		this.acFactory = acFactory;
	}

	public static void setAgentControllerFactory(IAgentControllerFactory acFactory) {
		getInstance().setACFactory(acFactory);
	}
	
	private IAgentController createConnection (String hostName, int portNumber) 
		throws Exception {
		
		if (acFactory == null) acFactory = new AgentControllerFactory();
		return acFactory.createConnection(hostName, portNumber);
	}
}
