/*******************************************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and Intel Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Vishnu K Naikawadi,Intel - Initial API and implementation
 *
 * $Id$ 
 *******************************************************************************/ 


package org.eclipse.tptp.platform.execution.util.internal;


import java.net.InetAddress;
import java.net.UnknownHostException;


/**
 * Utility class
 * @author vnaikawa
 *
 * 
 */
public class TPTPMessageUtil {

	/**
	 * Read a long from the input buffer
	 */
	public static long readTPTPLongFromBuffer(byte[] buffer, int offset)
	{
		return (long)(buffer[0+offset]<<24 & 0xff000000)
		  |(long)(buffer[1+offset]<<16 & 0x00ff0000)
		  |(long)(buffer[2+offset]<<8  & 0x0000ff00)
		  |(long)(buffer[3+offset]     & 0x000000ff);
	}
	
	
	/**
	 * Read an integer from the input buffer
	 */
	public static int readTPTPIntFromBuffer(byte[] buffer, int offset)
	{
		return (int)(buffer[0+offset]  & 0x000000ff)
		  |(int)(buffer[1+offset]<<8     & 0x0000ff00);
	}
	

	/**
	 * Read a String from the input buffer 
	 */
	public static int readTPTPStringFromBuffer(byte[] buffer, int offset, TPTPString rastring)
	{
		long length=readTPTPLongFromBuffer(buffer, offset);
		String data;
		try {
			data=new String(buffer, offset+4, (int)length, "UTF-8");
		}
		catch(Throwable e) {
			data=new String(buffer, 0, offset+4, (int)length);
		}
		rastring.setData(data);

		return offset+rastring.getSize();
	}
	
	/**
	 * Read a String from the input buffer - This method doesn't read the string length value before reading the string. 
	 * It used the </Cmd> string to calculte the end of the string.
	 */
	public static int readTPTPCmdStringFromBuffer(byte[] buffer, int offset, TPTPString rastring)
	{		
		 String data;
		 String endTag = "</Cmd>";
		 // Read everything into a String, starting at the offset
		 String temp;
		 try
		 {
			 temp = new String(buffer, offset, buffer.length-offset, "UTF-8");
		 }
		 catch(Exception e)
		 {
			 temp = new String(buffer, offset, buffer.length-offset);
		 }
		 // Find the end-tag for the Cmd in the String
		 int indexOfEndTag = temp.indexOf( endTag );

		 // If the tag wasn't there, return empty-handed
		 if ( indexOfEndTag == -1 ) 
		 {
			 rastring.setData("");
		     return offset;
		 }
	    // Get the substring ending at the end tag
		data = temp.substring( 0, indexOfEndTag+endTag.length() );

		// Set the data in the TPTPString
		rastring.setData(data);

		// Return the new offset
		int ret_off = 0;
		try
		{
			ret_off = data.getBytes("UTF-8").length;;
		}
		catch(Exception e)
		{
			//You will not have UTF-8 unsupported. 
			ret_off = data.getBytes().length;
		}
		return offset + ret_off;
	}

	/**
	 * Read a binary array from the buffer
	 */
	public static int readTPTPBinaryArrayFromBuffer(byte[] buffer, int offset, TPTPBinaryArray tptpArray)
	{
		long length=readTPTPLongFromBuffer(buffer, offset);
		tptpArray.setData(buffer, offset+4, (int)length);
		return offset+tptpArray.getSize();
	}

	/**
	 * read the INET Address from the buffer
	 */
	public static int readTPTPInetAddressFromBuffer(byte[] buffer, int offset, TPTPInetAddress ipaddr)
	{
		byte length=buffer[0+offset];
		byte[] address=new byte[length];
		for(int i=0; i<length; i++) {
			address[i]=buffer[1+i];
		}

		ipaddr.setLength((int)length);
		ipaddr.setData(address);

		return offset+Constants.sizeofByte+length;
	}

	/**
	 * Write a long to the buffer
	 */
	public static int writeTPTPLongToBuffer(byte[] buffer, int offset, long ralong) 
	{
		buffer[0+offset]=(byte)(ralong>>24 & 0x000000ff);
		buffer[1+offset]=(byte)(ralong>>16 & 0x000000ff);
		buffer[2+offset]=(byte)(ralong>>8  & 0x000000ff);
		buffer[3+offset]=(byte)ralong;
		return offset+4;
	}
	
	/**
	 * Write a String to the buffer
	 */
	public static int writeTPTPStringToBuffer(byte[] buffer, int offset, TPTPString rastring) 
	{
		byte[] strData;
		try {
			strData=rastring.getData().getBytes("UTF-8");
		}
		catch(Throwable e) {
			strData=rastring.getData().getBytes();
		}
		writeTPTPLongToBuffer(buffer, offset, strData.length);
		System.arraycopy((Object)strData, 0, (Object)buffer, offset+4, strData.length);

		return offset+rastring.getSize();
	}
	
	/**
	 * Write a String to the buffer - Doesn't write the string length value into the buffer
	 */
	public static int writeTPTPCmdStringToBuffer(byte[] buffer, int offset, TPTPString rastring) 
	{
		byte[] strData;
		try {
			strData=rastring.getData().getBytes("UTF-8");
		}
		catch(Throwable e) {
			strData=rastring.getData().getBytes();
		}		
		System.arraycopy((Object)strData, 0, (Object)buffer, offset, strData.length);

		return offset+rastring.getStringSize();
	}	

	/**
	 * Write a binary array to the buffer
	 */
	public static int writeTPTPBinaryArrayToBuffer(byte[] buffer, int offset, TPTPBinaryArray tptpArray) 
	{
		if(tptpArray==null) {
			writeTPTPLongToBuffer(buffer, offset, 0);
			return offset+Constants.sizeofLong;
		}
		byte[] data=tptpArray.getData();
		writeTPTPLongToBuffer(buffer, offset, data.length);
		System.arraycopy(data, 0, buffer, offset+4, data.length);

		return offset+tptpArray.getSize();
	}

	/**
	 * Write INET address to the buffer
	 * 
	 */
	public static int writeTPTPInetAddressToBuffer(byte[] buffer, int offset,  TPTPInetAddress addr) 
	{
		if(addr==null) {
			buffer[offset]=0;
			return offset+Constants.sizeofByte;
		}
		try {
			InetAddress address=addr.getAddress();
			if(address==null) {
				buffer[offset]=0;
				return offset+Constants.sizeofByte;
			}
			byte[] bytes=address.getAddress();
			buffer[offset+0]=(byte)bytes.length;
			for(int i=0; i<bytes.length; i++) {
				buffer[offset+i+Constants.sizeofByte]=bytes[i];	
			}
			return offset+Constants.sizeofByte+bytes.length;
			
			
		}
		catch(UnknownHostException e) {
			buffer[offset]=0;
			return offset+Constants.sizeofByte;
			
		}
	}
	/**
	 * Build a command to be sent to the AgentController
	 * 
	 */
	public static String buildTPTPCommand(int srcID,int destID,int	contextID,String	interfaceID,String	commandName,String	paramList)
	{
		return new String();
	}

	/**
	 * Calculate required buffer size to save string 
	 */
	public static int getTPTPCmdStringSize(TPTPString rastring) 
	{
		int size;
		try {
			size = rastring.getData().getBytes("UTF-8").length;
		}
		catch(Throwable e) {
			size = rastring.getData().getBytes().length;
		}		
		return size;
	}	
}
