/**********************************************************************
 * Copyright (c) 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 * @author ldancy
 **********************************************************************/

package org.eclipse.tptp.platform.probekit.launch.internal;

import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Vector;

import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.hyades.probekit.ProbekitPlugin;
import org.eclipse.hyades.probekit.editor.internal.core.util.JavaUtil;
import org.eclipse.hyades.probekit.ui.internal.AddProbeNatureOperation;
import org.eclipse.hyades.trace.ui.launcher.ProfilingAttribute;
import org.eclipse.hyades.trace.ui.launcher.ProfilingSetsManagerCopy;
import org.eclipse.hyades.ui.util.GridUtil;
import org.eclipse.jdt.core.IClasspathEntry;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jface.dialogs.IDialogSettings;
import org.eclipse.jface.dialogs.ProgressMonitorDialog;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.jface.viewers.ColumnPixelData;
import org.eclipse.jface.viewers.TableLayout;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.graphics.FontData;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.TabFolder;
import org.eclipse.swt.widgets.TabItem;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.TableItem;
import org.eclipse.swt.widgets.Text;
import org.eclipse.tptp.platform.probekit.builder.ProbeNature;
import org.eclipse.tptp.platform.probekit.launch.internal.config.ProbePropertiesUI;
import org.eclipse.tptp.platform.probekit.launch.internal.config.ProbekitLaunchMessages;
import org.eclipse.tptp.platform.probekit.registry.ProbeRegistry;
import org.eclipse.tptp.platform.probekit.registry.ProbeRegistryEntry;
import org.eclipse.tptp.platform.probekit.util.InvalidProbeBundleException;
import org.eclipse.tptp.platform.probekit.util.ProbeLaunchConfigString;
import org.eclipse.tptp.trace.ui.internal.launcher.core.LauncherMessages;
import org.eclipse.tptp.trace.ui.internal.launcher.core.LauncherUtility;
import org.eclipse.ui.actions.WorkspaceModifyOperation;
import org.eclipse.ui.plugin.AbstractUIPlugin;
/**
 * A class to isolate the shared UI code for both the JVMPI and JVMTI
 * dynamic probekit profiling launch configurations. This was created in TPTP
 * 4.4 in order to support the JVMTI launch configuration without duplicating 
 * the UI code already in place for the JVMPI Probekit Profiling Type
 * 
 * @author ldancy
 * @see org.eclipse.tptp.platform.probekit.launch.internal.config.ProbekitProfilingType
 */
	public class ProbekitLaunchUI implements SelectionListener
	{ 
		
// necessary to persist the UI when there is more than one collector
		
		private int numProbesSelectedinUI =0;
		
		private Composite _form;
	
		private  Table _tblProbeSets;
	
		private Button _refreshData;
	 
		private ProbePropertiesUI _propertiesUI;
	
		private String _warning;
		
		private Text _lblTitle;
	
		private TabFolder _folder;
	
		private ProfilingSetsManagerCopy _managerCopy;
	
		private ProbeRegistry _registry  ;
	
		private IDialogSettings _settings;
		
		private String _description = ""; // Non-NLS
	
		protected HashMap _mapProbeFilters;
	
		protected Map _filterSetsCopy;
		
		private Vector selectedProbes = new Vector();
		
		private final static int PROBE_NAME_INDEX = 0;
	
		private final static int PROBE_PREFILTERED_INDEX = 1;
	
		private final static int PROBE_IMPORTED_INDEX = 2;
	
		private final static int PROBE_ID_INDEX = 3;
	
		public final static String PROBE_LAUNCH_SELECTION_ATTRIBUTE_STRING = "org.eclipse.tptp.platform.probekit.launch.internal.PROBEUI"; //$NON-NLS-1$
	
		public final static String PROBE_LAUNCH_SETOPTION_ATTRIBUTE_PREFIX = "SETOPTION_"; //$NON-NLS-1$
	
		public final String PROBE_LAUNCH_UNIQUE_PROBESPEC_SUFFIX = "PROBEUI"; 
		//$NON-NLS-1$
		
		/**
		 * Constructor to create a ProbekitLaunchUI for a JVMTI or JVMPI launch
		 * configuration. Added for enahncement 141540
		 * Liz Dancy
		 */
		public ProbekitLaunchUI()
		{
			_registry = ProbeRegistry.getRegistry();
			AbstractUIPlugin plugin = (AbstractUIPlugin) ProbekitLaunchPlugin
					.getDefault();
			IDialogSettings workbenchSettings = plugin.getDialogSettings();
			IDialogSettings section = workbenchSettings
					.getSection("ProbeLaunchDialog");//$NON-NLS-1$
			if (section == null) 
			{
				section = workbenchSettings.addNewSection("ProbeLaunchDialog");//$NON-NLS-1$
			}
			_settings = section;
			_mapProbeFilters = new HashMap();
			_filterSetsCopy = new HashMap();
			_warning = new String(ProbekitLaunchMessages._18); //$NON-NLS-1$
		}
		
		
		/**
		 * 
		 * @param parent
		 * @param config
		 * @param enabled
		 * @return
		 * @see org.eclipse.tptp.platform.probekit.launch.internal.config.ProbekitProfilingType#createControl(Composite, ProfilingSetsManagerCopy)
		 */
		public Composite CreateControl(Composite parent, ProfilingSetsManagerCopy config, boolean enabled)
		{
			_form = new Composite(parent, SWT.NULL);
			GridData data = GridUtil.createFill();
			_form.setLayoutData(data);
	
			GridLayout layout = new GridLayout();
			layout.numColumns = 2;
			_form.setLayout(layout);
	
			_lblTitle = new Text(_form, SWT.LEFT /* | SWT.BORDER */| SWT.READ_ONLY);
			GridData gd = new GridData(GridData.FILL_HORIZONTAL);
			gd.grabExcessHorizontalSpace = true;
			gd.horizontalSpan = 2;
			_lblTitle.setLayoutData(gd);
	
			Composite childTop = new Composite(_form, SWT.NONE);
			data = GridUtil.createVerticalFill();
			data.widthHint = 200;
			childTop.setLayoutData(data);
	
			layout = new GridLayout();
			layout.numColumns = 1;
			childTop.setLayout(layout);
	
			Font font = _lblTitle.getFont();
			if (font != null) 
			{
				FontData[] fonts = font.getFontData();
				if (fonts != null && fonts.length >= 1) 
				{
					// Maintain consistency with built-in Hyades profiling type UIs
					// while still inheriting system font
					_lblTitle.setFont(new Font(Display.getDefault(), fonts[0]
					    .getName(), fonts[0].getHeight() + 3, SWT.BOLD));
				}
			}
			_lblTitle.setText(ProbekitLaunchMessages._25); //$NON-NLS-1$
	
			_tblProbeSets = new Table(childTop, SWT.CHECK | SWT.SINGLE
					| SWT.FULL_SELECTION | SWT.BORDER | SWT.V_SCROLL | SWT.H_SCROLL);
			data = GridUtil.createFill();
			_tblProbeSets.setLayoutData(data);
	
			_refreshData = new Button(childTop, SWT.PUSH);
			_refreshData.setText(ProbekitLaunchMessages._1);
			_refreshData.addSelectionListener(this);
	
			TableLayout layoutTable = new TableLayout();
			layoutTable.addColumnData(new ColumnPixelData(170, true));
			_tblProbeSets.setLayout(layoutTable);
			_tblProbeSets.setHeaderVisible(true);
	
			// begin adding columns
			TableColumn tc1 = new TableColumn(_tblProbeSets, SWT.LEFT);
			tc1.setText(ProbekitLaunchMessages._21); //$NON-NLS-1$
			// end adding columns
	
			populateTableFromRegistry();
	
			_tblProbeSets.addListener(SWT.Selection, new Listener() 
			{
				
				public void handleEvent(Event event) 
				{
					if (event.item instanceof TableItem) 
					{
						ProbeRegistryEntry entry = (ProbeRegistryEntry) event.item
							.getData();
						if (entry != null) 
						{
							_propertiesUI.updateWidgetValues(entry);
							// commented out for bugzilla_145401
							// _filtersUI.updateWidgetValues(entry);
	
							int index = _tblProbeSets
								.indexOf((TableItem) event.item);
							_tblProbeSets.setSelection(index);
							// we need to reset the selections so that only those
							//present at the closing of the dialog retain their state
							//selectedProbes.removeAllElements();
						}
					}
				}
			});
	
			Composite childBottom = new Composite(_form, SWT.NONE);
			data = GridUtil.createFill();
	
			childBottom.setLayoutData(data);
		
			layout = new GridLayout();
			layout.numColumns = 1;
			layout.marginHeight = 0;
			childBottom.setLayout(layout);
	
			_folder = new TabFolder(childBottom, SWT.TOP);
			data = GridUtil.createFill();
			// bugzilla_145401 required the uncommenting of this code
			data.heightHint = 320;
			data.grabExcessHorizontalSpace = true;
			_folder.setLayoutData(data);
		
			TabItem item = new TabItem(_folder, SWT.NONE);
			item.setText(ProbekitLaunchMessages._20); //$NON-NLS-1$
			_propertiesUI = new ProbePropertiesUI();
			item.setControl(_propertiesUI.createControl(_folder));
			if (_tblProbeSets.getItemCount() == 0) 
			{
				// Display text pointing user to import
				_propertiesUI.showHint(true);
			}
	
			item.setImage(ProbekitLaunchImages.INSTANCE
				.getImage(ProbekitLaunchImages.IMG_PROP));
	
			_folder.setSelection(0);
		
			// commented out for bugzilla_145401
			// TabItem item2 = new TabItem(_folder, SWT.NONE);
			// item2.setText(ProbekitLaunchMessages._19); //$NON-NLS-1$
			// _filtersUI = new ProbeFiltersUI();
			// item2.setControl(_propertiesUI.createControl(_folder));
			// item2.setImage(ProbekitLaunchImages.INSTANCE.getImage(ProbekitLaunchImages.IMG_FILTER));
		
			if (config != null) 
			{
				_managerCopy = config;
			} 
			else 
			{
				_managerCopy = new ProfilingSetsManagerCopy();
			}
			initializeValues(config);
		
			
	
			int sel = _tblProbeSets.getSelectionIndex();
			if (sel != -1) 
			{
				ProbeRegistryEntry entry = (ProbeRegistryEntry) _tblProbeSets
					.getItem(sel).getData();
			if (entry != null) 
			{
				_propertiesUI.updateWidgetValues(entry);
			}
			}
	
			_tblProbeSets.setFocus();
			
			return _form;
		}// end method createControl
		
		/**
		 * Used to initialize the page before opening.
		 * This reads in the stored attributes for the JVMPI launch 
		 * and the presistence Vector for the JVMTI launch in order to 
		 * restore the state of the UI the last time it was closed (by
		 * checking or un-checking the selections)
		 * @param config The Profilign Set Manager Copy
		 * @see org.eclipse.tptp.platform.probekit.launch.internal.config.ProbekitProfilingType#initializeValues(ProfilingSetsManagerCopy)
		 */
		protected void initializeValues(ProfilingSetsManagerCopy config) 
		{
			int iNumProbeSets = _tblProbeSets.getItemCount();
			TableItem[] items = _tblProbeSets.getItems();
			// The we know that there are saved selections from the JVMTI
			// for which have added to the selectedprobes Vector each time
			// a probe selection is saved.
			if (!selectedProbes.isEmpty())
			{
				for (int i=0;i<selectedProbes.size(); i++)
				{
					String probeName  = (String) selectedProbes.get(i);
					ProbeLaunchConfigString probeConfig = ProbeLaunchConfigString
					.fromString(probeName);
						if (probeConfig != null) 
						{
							restoreSelections(probeConfig, iNumProbeSets, items);
						}
				}
			}
			else
			{
				// this is called from the JVMPI for backwards compatability
				Map configAttributes= config.getDefaultSet().getAttributes();
				int index = 0;
				String strName = ProbeLaunchConfigString.LAUNCH_CONFIG_NAME_PREFIX
					+ PROBE_LAUNCH_UNIQUE_PROBESPEC_SUFFIX + index;
				ProfilingAttribute attribute = (ProfilingAttribute) configAttributes
					.get(strName);
				while (attribute != null) 
				{
					String strValue = attribute.getValue();
					ProbeLaunchConfigString probeConfig = ProbeLaunchConfigString
						.fromString(strValue);
					restoreSelections(probeConfig, iNumProbeSets, items);
					index++;
					strName = ProbeLaunchConfigString.LAUNCH_CONFIG_NAME_PREFIX
						+ PROBE_LAUNCH_UNIQUE_PROBESPEC_SUFFIX + index;
					attribute = (ProfilingAttribute) configAttributes.get(strName);
				}
			}
		}// end method
		
		
		/**
		 * A helper method to reduce cloned code. This simply ietrates through the
		 * possible selection and restores the UI state.
		 * @see org.eclipse.tptp.platform.probekit.launch.internal.config.ProbekitProfilingType#restoreSelections
		 */
		
		public void restoreSelections(ProbeLaunchConfigString probeConfig, int iNumProbeSets, TableItem[] items)
		{
			if (probeConfig != null) 
			{
				String strProbeKitID = probeConfig.getRegistryId();
				for (int j = 0; j < iNumProbeSets; j++) 
				{
					ProbeRegistryEntry entry = (ProbeRegistryEntry) items[j]
							.getData();
					if (entry != null) 
					{
						String strEntryID = entry.getId();
						if (strEntryID.equals(strProbeKitID)) 
						{
							items[j].setChecked(true);

							break;
						}
					}
				}
			}
		}
		
		/**
		 * Called only from the JVMPI ProfilingType
		 * Used to validate that at least one probe has been selected
		 * @param managerCopy
		 * @return String warning message if it is not valid
		 * @see org.eclipse.tptp.platform.probekit.launch.internal.config.ProbekitProfilingType#validateConfiguration(ProfilingSetsManagerCopy)
		 */
		public String validateConfiguration(ProfilingSetsManagerCopy managerCopy) {
			// RATLC00379729 - we should warn, if Probe insertion has no probe
			// selected or has no probe import.
			if (numProbesSelectedinUI ==0){
			int iSelectedProbes = 0;
			int iAvailableProbes = _tblProbeSets.getItemCount();
			if (iAvailableProbes > 0) {
				TableItem[] items = _tblProbeSets.getItems();
				for (int i = 0; i < iAvailableProbes; i++) {
					if (items[i].getChecked()) {
						iSelectedProbes++;
					}
				}
			}
			if (iAvailableProbes == 0 || iSelectedProbes == 0) {
				return _warning;
			} else {
				return null;
			}
			}
			return null;
		}
		/**
		 * @see org.eclipse.tptp.platform.probekit.launch.internal.config.ProbekitProfilingType#widgetSelected(SelectionEvent)
		 */
		public void widgetSelected(SelectionEvent e) {
			final ArrayList list = new ArrayList();
			IProject[] projects = ResourcesPlugin.getWorkspace().getRoot()
					.getProjects();
			for (int idx = 0; idx < projects.length; idx++) {
				try {
					IProject project = projects[idx];
					if (project.isOpen() && project.hasNature(JavaCore.NATURE_ID)
							&& !project.hasNature(ProbeNature.NATURE_NAME)) {
						IJavaProject jProject = JavaCore.create(project);
						Object[] resources = jProject.getNonJavaResources();

						boolean hasProbe = false;
						int i = 0;
						while (i < resources.length && !hasProbe) {
							String extension = ((IResource) resources[i])
									.getFileExtension();
							if (extension != null && (extension.equals("probe"))) {
								hasProbe = true;
							}
							i++;

						}

						if (hasProbe) {
							list.add(project);
							continue;
						}

						List sources = JavaUtil.getSourceContainerEntries(jProject);
						i = 0;
						while (i < sources.size() && !hasProbe) {
							IPath path = ((IClasspathEntry) sources.get(i))
									.getPath();
							IFolder folder = ResourcesPlugin.getWorkspace()
									.getRoot().getFolder(path);
							resources = folder.members();
							int j = 0;
							while (!hasProbe && j < resources.length) {

								String extension = ((IResource) resources[j])
										.getFileExtension();
								if (extension != null
										&& (extension.equals("probe") || extension
												.equals("probescript"))) {
									hasProbe = true;
								}
								j++;
							}

							i++;
						}

						if (hasProbe)
							list.add(project);
					}
				} catch (Exception exc) {
				}

			}

			/*
			 * N.M. Need to populate the list before checking for list.size().
			 * bugzilla_140821
			 */
			populateTableFromRegistry();

			if (list.size() == 0)
				return;

			IRunnableWithProgress operation = new WorkspaceModifyOperation() {
				public void execute(IProgressMonitor monitor) {
					try {
						AddProbeNatureOperation addProbeNature = new AddProbeNatureOperation(
								list);
						addProbeNature.run(monitor);

					} catch (CoreException ex) {
						// TODO Auto-generated catch block
						ex.printStackTrace();
					} finally {
						monitor.done();
					}
				}
			};

			try {
				new ProgressMonitorDialog(Display.getCurrent().getActiveShell())
						.run(true, true, operation);
			} catch (InvocationTargetException exec) {
				// TODO Auto-generated catch block
				exec.printStackTrace();
			} catch (InterruptedException exec) {
				// TODO Auto-generated catch block
				exec.printStackTrace();
			}

		}
			public void widgetDefaultSelected(SelectionEvent e) 
			{
				//Does not need to be implemented
			}
			
			
			/**
			 * Reads in the values from the ProbeRegistry and uses these
			 * to populate the list of available probes for deployment.
			 * @see org.eclipse.tptp.platform.probekit.launch.internal.config.ProbekitProfilingType#ppopulateTableFromRegistry()
			 */
			
			public void populateTableFromRegistry() 
			{
				_tblProbeSets.removeAll();
				_mapProbeFilters.clear();
				_registry = null;
	
				if (_registry == null) 
				{
					_registry = ProbeRegistry.getRegistry();
				}
	
				if (_registry != null) 
				{
					ProbeRegistryEntry[] probes = getProbesSortedByName();
					for (int i = 0; i < probes.length; i++) 
					{
						ProbeRegistryEntry probeEntry = probes[i];
						TableItem item = new TableItem(_tblProbeSets, SWT.NONE);
						// Name
						item.setText(PROBE_NAME_INDEX, probeEntry.getName());
						// Pre-filtered
						if (probeEntry.hasTargets()) 
						{
							item.setText(PROBE_PREFILTERED_INDEX,
									ProbekitLaunchMessages._7); //$NON-NLS-1$
						} 
						else 
						{
							item.setText(PROBE_PREFILTERED_INDEX,
									ProbekitLaunchMessages._8); //$NON-NLS-1$					
						}
						_mapProbeFilters.put(probeEntry.getId(), null);// "D", ""
	
						// Imported
						if (probeEntry.isAuthored()) 
						{
							item.setText(PROBE_IMPORTED_INDEX,
									ProbekitLaunchMessages._8); //$NON-NLS-1$
						} else 
						{
							item.setText(PROBE_IMPORTED_INDEX,
									ProbekitLaunchMessages._7); //$NON-NLS-1$
						}
						// ID
						item.setText(PROBE_ID_INDEX, probeEntry.getId());
	
						// Data
						item.setData(probeEntry);
					}
				}
			}
			/**
			 * Returns the set of Probes sorted alphabetically by name
			 * These values are taken from the ProbeRegistry
			 * @return - ProbeRegsitryEntry[] of the probes sorted by name
			 * @see org.eclipse.tptp.platform.probekit.launch.internal.config.ProbekitProfilingType#getProbesSortedByName()
			 */
			public ProbeRegistryEntry[] getProbesSortedByName() 
			{
				Collection probes = new ArrayList();
				Iterator iter = _registry.iterator();
				while (iter.hasNext()) 
				{
					ProbeRegistryEntry entry = (ProbeRegistryEntry) iter.next();
					probes.add(entry);
				}
	
				ProbeRegistryEntry[] sortedProbes = new ProbeRegistryEntry[probes
						.size()];
				probes.toArray(sortedProbes);
				Arrays.sort(sortedProbes, new Comparator() 
				{
					public int compare(Object obj1, Object obj2) 
					{
						ProbeRegistryEntry probe1 = (ProbeRegistryEntry) obj1;
						ProbeRegistryEntry probe2 = (ProbeRegistryEntry) obj2;
						return probe1.getName().compareTo(probe2.getName());
					}
				});
	
				return sortedProbes;
			}
			
			/**
			 * Used by the JVMPI Probe insertion only.
			 * The purpose of thsi method is to save the user's selections 
			 * in the UI.
			 * @see org.eclipse.tptp.platform.probekit.launch.internal.config.ProbekitProfilingType#saveSelections()
			 */
			public void saveSelections(int probesSelected) 
			{
				numProbesSelectedinUI= probesSelected;
				int indexSelection =_tblProbeSets.getSelectionIndex();
	
				if (indexSelection >= 0) {
					TableItem tableItem = _tblProbeSets.getItem(indexSelection);
					if (tableItem != null) 
					{
						ProbeRegistryEntry entry = (ProbeRegistryEntry) tableItem
								.getData();
						if (entry != null) 
						{
							String strID = entry.getId();
							if (_settings != null) 
							{
								// Store table selection
								_settings.put("ProbeTableSelection", strID); //$NON-NLS-1$
							}
						}
					}
				} 
				else 
				{
					if (_settings != null) 
					{
						_settings.put("ProbeTableSelection", ""); //$NON-NLS-1$ //$NON-NLS-2$
					}
				}
				TabItem[] item = _folder.getSelection();
				if (item[0] != null) {
					String strName = item[0].getText();
					if (_settings != null) 
					{
						// Store folder selection
						_settings.put("ProbeFolderSelection", strName); //$NON-NLS-1$
					}
				} 
				else 
				{
					_settings.put("ProbeFolderSelection", ""); //$NON-NLS-1$ //$NON-NLS-2$
				}
				
			}
			/**
			 * Returns the user-defined Probe Description for the
			 * Description Text Box on the Edit Options Page
			 * @param config
			 * @return
			 * @see org.eclipse.tptp.platform.probekit.launch.internal.config.ProbekitProfilingType#getDescription(ProfilingSetsManagerCopy)
			 */
			public String getDescription(ProfilingSetsManagerCopy config) {
				
				String strDelim = "\n"; //$NON-NLS-1$
				String strIndent = "    "; //$NON-NLS-1$
	
				// Empty previous description
				_description = ""; //$NON-NLS-1$
	
				if (_registry == null) {
					_registry = ProbeRegistry.getRegistry();
				}
	
				if (config != null && config.getDefaultSet() != null) {
					int index = 0;
					String strName = ProbeLaunchConfigString.LAUNCH_CONFIG_NAME_PREFIX
							+ PROBE_LAUNCH_UNIQUE_PROBESPEC_SUFFIX + index;
	
					Map configAttributes = config.getDefaultSet().getAttributes();
					ProfilingAttribute attribute = (ProfilingAttribute) configAttributes
							.get(strName);
	
					while (attribute != null) {
						String strValue = attribute.getValue();
						ProbeLaunchConfigString probeString = ProbeLaunchConfigString
								.fromString(strValue);
	
						String strProbeID = probeString.getRegistryId();
	
						if (strProbeID != null && strProbeID.length() > 0) {
							if (_registry != null) {
								ProbeRegistryEntry probeEntry = _registry
										.lookupById(strProbeID);
	
								if (probeEntry != null) {
									String strProbeName = probeEntry.getName();
	
									if (strProbeName != null) {
										_description = _description + strIndent
												+ strProbeName + strDelim;
									}
								}
							}
						}
	
						index++;
						strName = ProbeLaunchConfigString.LAUNCH_CONFIG_NAME_PREFIX
								+ PROBE_LAUNCH_UNIQUE_PROBESPEC_SUFFIX + index;
						attribute = (ProfilingAttribute) configAttributes.get(strName);
					}
				}
	
				if (_description != null && _description.length() > 0) {
					// Remove extra new-line at end of string
					int iLastNewLineIndex = _description.lastIndexOf(strDelim);
					if (iLastNewLineIndex >= 0) 
					{
						_description = _description.substring(0, iLastNewLineIndex);
					}
				} 
				else 
				{
					// Return null instead of zero-length string to prevent blank line
					// _description = null;
					_description = strIndent + ProbekitLaunchMessages._28; //$NON-NLS-1$
				}
				return _description;
			}
	
			/**
			 * Accessor and mutator methods for fields in the UI
			 * Necssary so that the PI and TI-specific methods can access these fields
			 * @return
			 */
			public Table getProbeSets(){
				return _tblProbeSets;
			}
			public ProfilingSetsManagerCopy getManager(){
				return _managerCopy;
			}
			public ProbeRegistry getRegistry(){
				return _registry;
			}
			
			public IDialogSettings getSettings(){
				return _settings;
			}
			
			public void setSettings(IDialogSettings givenSettings){
				_settings = givenSettings;
			}
			public void addSelectedProbes(String selection){
				selectedProbes.add(selection);
			}
			public void clearSelectedProbes(){
				selectedProbes.removeAllElements();
			}
			
			/**
			 * This method is used to return the additional executable parameters
			 * which need to be appended to the arguments. 
			 * @param conf
			 * @return
			 * @see org.eclipse.tptp.platform.probekit.launch.internal.config.ProbekitProfilingType#getVMArguments(ILaunchConfiguration)
			 */
			public synchronized String getVMArguments(ILaunchConfiguration conf) 
			{
				try
				{
					/* Will hold the final VM argument */
					String vmArgument = "";
					
					/* Will hold the bootclasspath entry */
					String bootClassPath = "-Xbootclasspath/p:"; 
					
					/* Get the absolute path of the workspace */
					String workspaceLocation = ResourcesPlugin.getWorkspace().getRoot().getLocation().toString();	
					
					/* Logs informaiton messages if true. It can be set by using the following property: -DprobekitLogging=true */
					boolean log = Boolean.getBoolean("probekitLogging");
					
					/* If this is a Windows operating system, add quotes to the beginning and end of
					 * the value passed to bootclasspath.  This is done in case there are spaces in the
					 * path(s) specified. */
					if (System.getProperty("os.name").toLowerCase().indexOf("windows") != -1)
					{
						bootClassPath = bootClassPath + "\"";
					}
					
					/* Path separator ';' on Windows. ':' on Linux */
					String pathSeparator = System.getProperty("path.separator");
					
					/* Step through the project of each probe selected and add its CLASSPATH entries to
					 * the BOOTCLASSPATH
					 */
					Map activeProbes = getActiveProbes(conf);
					Iterator iterKeys = activeProbes.keySet().iterator();
					while (iterKeys.hasNext()) {
						String key = iterKeys.next().toString();
						String strProbeSpec = (String) activeProbes.get(key);
						ProbeLaunchConfigString probeConfigDeploy = ProbeLaunchConfigString.fromString(strProbeSpec);
						String strProbeID = probeConfigDeploy.getRegistryId();
						ProbeRegistryEntry registryEntry = (ProbeRegistryEntry) getRegistry().lookupById(strProbeID);
						IResource probeSource = null;			
						
						/* Extract the probe source file */
						try {
							probeSource = registryEntry.getSource();				
						} catch (InvalidProbeBundleException e) {
							IStatus status = new Status(IStatus.ERROR, "org.eclipse.tptp.platform.probekit.launch", 0, ProbekitLaunchMessages._29, e);
							ProbekitPlugin.getDefault().getLog().log(status);
							continue; /* Error occured, skip to the next probe */
						}
						
						/* Make sure the probe source is not NULL */
						if (probeSource == null)
						{
							IStatus status = new Status(IStatus.ERROR, "org.eclipse.tptp.platform.probekit.launch", 0, ProbekitLaunchMessages._30, null);
							ProbekitPlugin.getDefault().getLog().log(status);
							continue; /* Error occured, skip to the next probe */
						}
	
						/* Grab the Project where the probe resides */			
						IJavaProject javaProject = JavaCore.create(probeSource.getProject());
						if (javaProject != null)
						{
							String projectName = javaProject.getPath().toString();
							
							/* Log information messages if asked to */
							if (log)
							{
								/* Project Name */
								String message = NLS.bind(ProbekitLaunchMessages._36, projectName);
								IStatus status = new Status(IStatus.INFO, "org.eclipse.tptp.platform.probekit.launch", 0, message, null);
								ProbekitPlugin.getDefault().getLog().log(status);
								
								/* Probe Name */
								message = NLS.bind(ProbekitLaunchMessages._37, probeSource.toString());
								status = new Status(IStatus.INFO, "org.eclipse.tptp.platform.probekit.launch", 0, message, null);
								ProbekitPlugin.getDefault().getLog().log(status);
							}
							
							/* Bug 158019: For imported projects, 'workspaceLocation' has to be changed to refer to the real 
							 * workspace of the project.  Not the workspace that makes a reference to the real workspace. */
							String workspaceOfProbeFile = probeSource.getLocation().toString();
							if (workspaceOfProbeFile.indexOf(workspaceLocation) == -1)
							{
								workspaceLocation = workspaceOfProbeFile.substring(0, workspaceOfProbeFile.indexOf(projectName));
							}				
							/* End Bug 158019 */
							
							/* First add the output location of the project to BOOTCLASSPATH */
							try {										
								bootClassPath = bootClassPath + workspaceLocation + javaProject.getOutputLocation() + pathSeparator;
								
							} catch (JavaModelException e1) {
								IStatus status = new Status(IStatus.ERROR, "org.eclipse.tptp.platform.probekit.launch", 0, ProbekitLaunchMessages._33, e1);
								ProbekitPlugin.getDefault().getLog().log(status);
							}				
							
							IClasspathEntry[] classpathEntryList; 
							try {
								/* Extract the classpath entry of the project */
								classpathEntryList = javaProject.getRawClasspath();
							} catch (JavaModelException e) {
								IStatus status = new Status(IStatus.ERROR, "org.eclipse.tptp.platform.probekit.launch", 0, ProbekitLaunchMessages._31, e);
								ProbekitPlugin.getDefault().getLog().log(status);
								continue; /* Error occured, move on to the next probe */
							}
										
							/* Step through the classpath entries, check their type and properly add them to the BOOTCLASSPATH.
							 * There are five classpath entry types:
							 *  IClasspathEntry.CPE_SOURCE: Source folder of the project. Added as-is to the BOOTCLASSPATH
							 *  IClasspathEntry.CPE_LIBRARY: JAR library.  Added as-is to the BOOTCLASSPATH
							 *  IClasspathEntry.CPE_PROJECT: Project depends on another project in the workspace.  The 'bin'
							 *       directory of the project is added to BOOTCLASSPATH.
							 *  IClasspathEntry.CPE_VARIABLE: A classpath entry variable.  The variable is resolved and its
							 *       entry is added to BOOTCLASSPATH.
							 *  IClasspathEntry.CPE_CONTAINER: A classpath entry container. Containing source, library, and project
							 *       type entries. Container is resolved and its entries are added to BOOTCLASSPATH.  The standard
							 *       JRE container is ignored, since it's already in BOOTCLASSPATH.
							 */
							for (int i=0; i < classpathEntryList.length; i++)
							{
								/* Analyze one classpath entry at a time */
								IClasspathEntry classpathEntry = classpathEntryList[i];
								
								/* Get the classpath entry type */
								int type = classpathEntry.getEntryKind();
								
								String logMessage = "Type:";
								
								if (type == IClasspathEntry.CPE_SOURCE)
								{					
									if (classpathEntry.getPath().toString().startsWith(projectName))
										bootClassPath = bootClassPath + workspaceLocation;				
									
									bootClassPath = bootClassPath + classpathEntry.getPath().toString() + pathSeparator;
									logMessage = logMessage + "Source  Path:" + classpathEntry.getPath().toString();
								}
								else if (type == IClasspathEntry.CPE_LIBRARY)
								{											
									if (classpathEntry.getPath().toString().startsWith(projectName))
										bootClassPath = bootClassPath + workspaceLocation;				
									
									bootClassPath = bootClassPath + classpathEntry.getPath().toString() + pathSeparator;
									logMessage = logMessage + "Library  Path:" + classpathEntry.getPath().toString();
								}						
								else if (type == IClasspathEntry.CPE_PROJECT)
								{
									IProject dependentProject = ResourcesPlugin.getWorkspace().getRoot().getProject(classpathEntry.getPath().toString().substring(1));
									IJavaProject dependentJavaProject = JavaCore.create(dependentProject.getProject());
									try {					
										bootClassPath = bootClassPath + workspaceLocation + dependentJavaProject.getOutputLocation().toString() + pathSeparator;
										logMessage = logMessage + "Project  Path:" + dependentJavaProject.getOutputLocation().toString();
									} catch (JavaModelException e) {
										IStatus status = new Status(IStatus.ERROR, "org.eclipse.tptp.platform.probekit.launch", 0, ProbekitLaunchMessages._33, e);
										ProbekitPlugin.getDefault().getLog().log(status);
										continue; /* Error occured, move on to the next classpath entry */
									}						
								}
								else if (type == IClasspathEntry.CPE_VARIABLE)
								{
									if (classpathEntry.getPath().toString().startsWith(projectName))
										bootClassPath = bootClassPath + workspaceLocation;		
									
									bootClassPath = bootClassPath + JavaCore.getClasspathVariable(classpathEntry.getPath().toString()).toString() + pathSeparator;
									
									logMessage = logMessage + "Variable  Path:" + JavaCore.getClasspathVariable(classpathEntry.getPath().toString()).toString();
								}						
								else if (type == IClasspathEntry.CPE_CONTAINER)
								{					
									
									String logContainerContent = "";  /* Used for logging */						
									String containerName = classpathEntry.getPath().toString();
									
									/* Ignore if it's the JRE container.  It's already in the BOOTCLASSPATH */
									if (containerName.indexOf("JRE_CONTAINER") != -1)
									{
										logMessage = logMessage + "Container " + containerName + " is being ignored";							
									}
									else
									{
										IClasspathEntry[] containerContent;
										try
										{
											containerContent = JavaCore.getClasspathContainer(classpathEntry.getPath(), javaProject).getClasspathEntries();
										} catch (JavaModelException e) {
											IStatus status = new Status(IStatus.ERROR, "org.eclipse.tptp.platform.probekit.launch", 0, ProbekitLaunchMessages._32, e);
											ProbekitPlugin.getDefault().getLog().log(status);
											continue; /* Error occured, skip to the next classpath entry */	
										}
										
										/* Step through the content of the container and add it to the BOOTCLASSPATH */
										for (int k=0; k < containerContent.length; k++)
										{
											classpathEntry = containerContent[k];
											type = classpathEntry.getEntryKind();
											
											if ((type == IClasspathEntry.CPE_SOURCE) || (type == IClasspathEntry.CPE_LIBRARY) || (type == IClasspathEntry.CPE_PROJECT))
											{									
												if (classpathEntry.getPath().toString().startsWith(projectName))
													bootClassPath = bootClassPath + workspaceLocation;	
												
												bootClassPath = bootClassPath + classpathEntry.getPath().toString() + pathSeparator;
												logContainerContent = logContainerContent + classpathEntry.getPath().toString() + pathSeparator;
											}
											else
											{
												String message = NLS.bind(ProbekitLaunchMessages._34, new Integer(type));
												IStatus status = new Status(IStatus.ERROR, "org.eclipse.tptp.platform.probekit.launch", 0, message, null);
												ProbekitPlugin.getDefault().getLog().log(status);
											}
										}
										logMessage = logMessage + "Container  Name:" + containerName + "  Path:" + logContainerContent;
										
									}
								} // end of else if	(type == IClasspathEntry.CPE_CONTAINER)		
										
								if (log)
								{
									IStatus status = new Status(IStatus.INFO, "org.eclipse.tptp.platform.probekit.launch", 0, logMessage, null);
									ProbekitPlugin.getDefault().getLog().log(status);
								}
								
							} // end of for loop
						} // end of if (javaProject != null)
					} // end of while loop
	
					/* If this is a Windows operating system, add quotes to the beginning and end of
					 * the value passed to bootclasspath.  This is done in case there are spaces in the
					 * path(s) specified. */
					if (System.getProperty("os.name").toLowerCase().indexOf("windows") != -1)
					{
						bootClassPath = bootClassPath + "\"";
					}
					
					/* Bugzilla 177010: We'll need to add another VM argument in order for OSGi to properly load our Probe
					 * classes from the boot classloader */
					String osgiProperty = "-Dorg.osgi.framework.bootdelegation=\"*\"";
					/* End of bugzilla 177010*/
					
					/* Are we logging information messages? */
					if (log)
					{
						String message = NLS.bind(ProbekitLaunchMessages._35, bootClassPath);
						IStatus status = new Status(IStatus.INFO, "org.eclipse.tptp.platform.probekit.launch", 0, message, null);
						ProbekitPlugin.getDefault().getLog().log(status);
					}		
					
					vmArgument = bootClassPath + " " + osgiProperty;
					
					return vmArgument;
				}
			
			catch(CoreException e)
			{
				e.printStackTrace();
				return null;
			}
			
				
			}
				
			/**
			 * This method returns the activeProbe files which we have selected for
			 * launch. It returns them in Map form.
			 * taken from org.eclipse.tptp.platform.probekit.launch.internal.config.ProbekitProfilingType
			 * @param launchConfig
			 * @return
			 */
			protected Map getActiveProbes(ILaunchConfiguration launchConfig) throws CoreException {
				//org.eclipse.tptp.platform.probekit.Probespec.PROBEUI
				String strProbeDeployPrefix = ProbeLaunchConfigString.AGENT_CONFIG_NAME_PREFIX
						+ PROBE_LAUNCH_UNIQUE_PROBESPEC_SUFFIX;
				Map activeProbes = launchConfig.getAttributes();
				Map map = filterAttributes(launchConfig, strProbeDeployPrefix, activeProbes);
				return map;
			}
			
			/**
			 * A helper method to return a Map of the selected probes
			 * taken from org.eclipse.tptp.platform.probekit.launch.internal.config.ProbekitProfilingType
			 * @param launchConfig
			 * @param strPrefix
			 * @return
			 */
			
			protected Map filterAttributes(ILaunchConfiguration launchConfig, String strPrefix, Map fullSet)
			{
				Map filteredSet = new HashMap();
				Collection values = fullSet.keySet();
				Iterator iterOptions = values.iterator();
				while (iterOptions.hasNext() ) 
				{
					Object key = iterOptions.next();
					if (key instanceof String)
					{
						String probeKey = (String)key;
						if ((probeKey!= null) && (probeKey.startsWith(strPrefix))) 
						{
							String value = (String)fullSet.get(key);
							if (!value.equals(" "))
							filteredSet.put(key, value );
						}
					}					
				}
				return filteredSet;
			}
	}// end class
