/**********************************************************************
 * Copyright (c) 2006 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: CustomWizardFileSystemResourceExportPage1.java,v 1.1 2006/03/16 15:56:15 popescu Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.tptp.platform.probekit.launch.internal.wizard;

import java.io.File;
import java.util.List;

import org.eclipse.core.resources.*;
import org.eclipse.core.runtime.*;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.*;

/**
 *	Page 1 of the base resource export-to-file-system Wizard
 *	See also org.eclipse.ui.wizards.datatransfer.WizardFileSystemResourceExportPage1.
 */
public class CustomWizardFileSystemResourceExportPage1 extends CustomWizardExportResourcesPage implements Listener
{
	// widgets
	private Combo destinationNameField;
	private Button destinationBrowseButton;
	protected Button overwriteExistingFilesCheckbox;
	protected Button createDirectoryStructureButton;
	protected Button createSelectionOnlyButton;
	
	// Mask value with .zip included is "*.probekit;*.zip"
	private static final String FILE_EXPORT_MASK = "*.probekit"; //$NON-NLS-1$
	
	protected static final String SOURCE_EMPTY_MESSAGE = Messages.SourceEmpty; //$NON-NLS-1$
	protected static final String DESTINATION_EMPTY_MESSAGE = Messages.DestinationEmpty; //$NON-NLS-1$

	/**
	 *	Create an instance of this class
	 */
	protected CustomWizardFileSystemResourceExportPage1(
		String name,
		IStructuredSelection selection)
	{
		super(name, selection);
	}
	/**
	 *	Add the passed value to self's destination widget's history
	 *
	 *	@param value java.lang.String
	 */
	protected void addDestinationItem(String value) {
		destinationNameField.add(value);
	}
	/** (non-Javadoc)
	 * Method declared on IDialogPage.
	 */
	public void createControl(Composite parent)
	{
		super.createControl(parent);
		giveFocusToDestination();
	}
	/**
	 *	Create the export destination specification widgets
	 *
	 *	@param parent org.eclipse.swt.widgets.Composite
	 */
	protected void createDestinationGroup(Composite parent)
	{
		// destination specification group
		Composite destinationSelectionGroup = new Composite(parent, SWT.NONE);
		GridLayout layout = new GridLayout();
		layout.numColumns = 3;
		destinationSelectionGroup.setLayout(layout);
		destinationSelectionGroup.setLayoutData(
			new GridData(
				GridData.HORIZONTAL_ALIGN_FILL | GridData.VERTICAL_ALIGN_FILL));

		Label destinationLabel = new Label(destinationSelectionGroup, SWT.NONE);
		destinationLabel.setText(getDestinationLabel());

		// destination name entry field
		destinationNameField =
			new Combo(destinationSelectionGroup, SWT.SINGLE | SWT.BORDER);
		destinationNameField.addListener(SWT.Modify, this);
		destinationNameField.addListener(SWT.Selection, this);
		GridData data =
			new GridData(
				GridData.HORIZONTAL_ALIGN_FILL | GridData.GRAB_HORIZONTAL);
		data.widthHint = SIZING_TEXT_FIELD_WIDTH;
		destinationNameField.setLayoutData(data);

		// destination browse button
		destinationBrowseButton =
			new Button(destinationSelectionGroup, SWT.PUSH);
		destinationBrowseButton.setText(Messages.DestinationBrowseButton); //$NON-NLS-1$
		destinationBrowseButton.addListener(SWT.Selection, this);
		setButtonLayoutData(destinationBrowseButton);

		new Label(parent, SWT.NONE); // vertical spacer
	}

	/**
	 * Create the buttons in the options group.
	 */
	/*
	 * Do not expose overwrite and directory options to the user since force one value to be used in executeExportOperation
	 */
	/*
	protected void createOptionsGroupButtons(Group optionsGroup) {

		Font font = optionsGroup.getFont();
		createOverwriteExisting(optionsGroup, font);

		createDirectoryStructureOptions(optionsGroup, font);
	}
	*/
	/**
	 * Create the buttons for the group that determine if the entire or
	 * selected directory structure should be created.
	 * @param optionsGroup
	 * @param font
	 */
	/*
	 * Do not expose directory options to the user since force one value to be used in executeExportOperation
	 */
	/*
	protected void createDirectoryStructureOptions(
		Group optionsGroup,
		Font font) {
		// create directory structure radios
		createDirectoryStructureButton =
			new Button(optionsGroup, SWT.RADIO | SWT.LEFT);
		createDirectoryStructureButton.setText(Messages.CreateDirectoryStructureOption")); //$NON-NLS-1$
		createDirectoryStructureButton.setSelection(false);

		// create directory structure radios
		createSelectionOnlyButton =
			new Button(optionsGroup, SWT.RADIO | SWT.LEFT);
		createSelectionOnlyButton.setText(Messages.CreateSelectedDirectoriesOption"));//$NON-NLS-1$
		createSelectionOnlyButton.setSelection(true);
	}
	*/
	/**
	 * Create the button for checking if we should ask if we are going to
	 * overwrite existing files.
	 * @param optionsGroup
	 * @param font
	 */
	/*
	 * Do not expose overwrite option to the user since force one value to be used in executeExportOperation
	 */
	/*
	protected void createOverwriteExisting(Group optionsGroup, Font font) {
		// overwrite... checkbox
		overwriteExistingFilesCheckbox =
			new Button(optionsGroup, SWT.CHECK | SWT.LEFT);
		overwriteExistingFilesCheckbox.setText(Messages.OverwriteFilesCheckbox")); //$NON-NLS-1$
	}
	*/
	/**
	 * Attempts to ensure that the specified directory exists on the local file system.
	 * Answers a boolean indicating success.
	 *
	 * @return boolean
	 * @param directory java.io.File
	 */
	protected boolean ensureDirectoryExists(File directory) {
		if (!directory.exists()) {
			if (!queryYesNoQuestion(Messages.CreateTargetDirectory)) //$NON-NLS-1$
				return false;

			if (!directory.mkdirs()) {
				displayErrorDialog(Messages.DirectoryCreationError); //$NON-NLS-1$
				giveFocusToDestination();
				return false;
			}
		}

		return true;
	}
	/**
	 *	If the target for export does not exist then attempt to create it.
	 *	Answer a boolean indicating whether the target exists (ie.- if it
	 *	either pre-existed or this method was able to create it)
	 *
	 *	@return boolean
	 */
	protected boolean ensureTargetIsValid(File targetDirectory) {
		if (targetDirectory.exists() && !targetDirectory.isDirectory()) {
			displayErrorDialog(Messages.DirectoryExists); //$NON-NLS-1$
			giveFocusToDestination();
			return false;
		}

		return ensureDirectoryExists(targetDirectory);
	}
	/**
	 *	Answer the string to display in self as the destination type
	 *
	 *	@return java.lang.String
	 */
	protected String getDestinationLabel() {
		return Messages.ToDirectory; //$NON-NLS-1$
	}
	/**
	 *	Answer the contents of self's destination specification widget
	 *
	 *	@return java.lang.String
	 */
	protected String getDestinationValue() {
		return destinationNameField.getText().trim();
	}
	/**
	 *	Set the current input focus to self's destination entry field
	 */
	protected void giveFocusToDestination() {
		destinationNameField.setFocus();
	}
	/* (non-Javadoc)
	 * Method declared on IDialogPage. Set the selection up when it becomes visible.
	 */
	public void setVisible(boolean visible)
	{
		super.setVisible(visible);
		if (visible)
		{
			String strMessage = getMessage();
			if (strMessage != null)
			{
				if (strMessage.equals(CustomWizardFileSystemResourceExportPage1.SOURCE_EMPTY_MESSAGE))
				{
					giveFocusToSource();
				}
				else if (strMessage.equals(CustomWizardFileSystemResourceExportPage1.DESTINATION_EMPTY_MESSAGE))
				{
					giveFocusToDestination();			
				}
				else
				{
					giveFocusToTree();
				}
			}
			else
			{
				giveFocusToDestination();			
			}
		}
	}
	/**
	 *	Open a FileDialog so that the user can specify the destination
	 *	file to export to
	 */
	protected void handleDestinationBrowseButtonPressed() {
		String selectedFile = queryZipFileToExport();

		if (selectedFile != null) {
			String strCurrentDestination = destinationNameField.getText();
			if (!selectedFile.equals(strCurrentDestination))
			{
				//Be sure it is valid before we go setting any names
				//ZipFile sourceFile = getSpecifiedSourceFile(selectedFile);
				//if (sourceFile != null) {
				//	closeZipFile(sourceFile);
					setDestinationValue(selectedFile);
				//}
			}
		}
	}
	/**
	 * Opens a file selection dialog and returns a string representing the
	 * selected file, or <code>null</code> if the dialog was canceled.
	 */
	protected String queryZipFileToExport() {
		FileDialog dialog =
			new FileDialog(destinationNameField.getShell(), SWT.SAVE);
		dialog.setFilterExtensions(new String[] { FILE_EXPORT_MASK });
		dialog.setText(Messages.FileDialogText);  //$NON-NLS-1$

		String currentSourceString = destinationNameField.getText();
		int lastSeparatorIndex =
			currentSourceString.lastIndexOf(File.separator);
		if (lastSeparatorIndex != -1)
			dialog.setFilterPath(
				currentSourceString.substring(0, lastSeparatorIndex));

		return dialog.open();
	}	
	/**
	 * Handle all events and enablements for widgets in this page
	 * @param e Event
	 */
	public void handleEvent(Event event) {
		Widget source = event.widget;

		if (event.type == SWT.Selection && source == destinationBrowseButton)
		{
			handleDestinationBrowseButtonPressed();
		}

		//updatePageCompletion();
		updateWidgetEnablements();
	}
	/**
	 *	Hook method for saving widget values for restoration by the next instance
	 *	of this class.
	 */
	protected void internalSaveWidgetValues() {
	}
	/**
	 *	Hook method for restoring widget values to the values that they held
	 *	last time this wizard was used to completion.
	 */
	protected void restoreWidgetValues() {
	}
	/**
	 *	Set the contents of the receivers destination specification widget to
	 *	the passed value
	 *
	 */
	protected void setDestinationValue(String value) {
		destinationNameField.setText(value);
	}
	/**
	 *	Answer a boolean indicating whether the receivers destination specification
	 *	widgets currently all contain valid values.
	 */
	protected boolean validateDestinationGroup()
	{
		String destinationValue = getDestinationValue();
		if (destinationValue.length() == 0) {
			setMessage(destinationEmptyMessage());
			return false;
		}

		String conflictingContainer =
			getConflictingContainerNameFor(destinationValue);
		if (conflictingContainer == null)
			setErrorMessage(null);
		else {
			setErrorMessage(NLS.bind(Messages.ConflictingContainer, //$NON-NLS-1$
			new Object[] { conflictingContainer }));
			giveFocusToDestination();
			return false;
		}

		return true;
	}
	
	/**
	 * Get the message used to denote an empty destination.
	 */
	protected String destinationEmptyMessage(){
		return DESTINATION_EMPTY_MESSAGE; //$NON-NLS-1$
	}

	/**
	 * Returns the name of a container with a location that encompasses targetDirectory.
	 * Returns null if there is no conflict.
	 * 
	 * @param targetDirectory the path of the directory to check.
	 * @return the conflicting container name or <code>null</code>
	 */
	protected String getConflictingContainerNameFor(String targetDirectory)
	{
		IWorkspaceRoot root = ResourcesPlugin.getWorkspace().getRoot();
		IPath testPath = new Path(targetDirectory);

		if (root.getLocation().isPrefixOf(testPath))
			return Messages.RootName; //$NON-NLS-1$

		IProject[] projects = root.getProjects();

		for (int i = 0; i < projects.length; i++) {
			if (projects[i].getLocation().isPrefixOf(testPath))
				return projects[i].getName();
		}

		return null;
	}
	
	/**
	 *	Answer a boolean indicating whether self's source specification
	 *	widgets currently all contain valid values.
	 */
	protected boolean validateSourceGroup()
	{
		// Just use current selection from table since not supporting export of multiple probes
		//List resourcesToExport = getSelectedResources();
		//if (!(resourcesToExport.size() > 0))
		//{
		//	setMessage(SOURCE_EMPTY_MESSAGE);
		//	return false;
		//}
		//else
		//{
			List resourcesToExport = getCurrentSelection();
			if (!(resourcesToExport.size() > 0))
			{
				setMessage(SOURCE_EMPTY_MESSAGE);
				return false;
			}
		//}
		
		return true;
	}
}
