/**********************************************************************
 * Copyright (c) 2006 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ExportProbeWizardPage.java,v 1.1 2006/03/16 15:56:15 popescu Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.tptp.platform.probekit.launch.internal.wizard;

import java.io.File;
import java.lang.reflect.InvocationTargetException;
import java.util.List;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.jface.dialogs.*;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.*;
import org.eclipse.tptp.platform.probekit.launch.internal.ProbekitLaunchPlugin;
import org.eclipse.tptp.platform.probekit.util.InvalidProbeBundleException;
import org.eclipse.tptp.platform.probekit.util.ProbeBundleBuildRequiredException;
import org.eclipse.tptp.platform.probekit.util.ProbeBundleException;
import org.eclipse.tptp.platform.probekit.util.ProbeResourceBundle;
import org.eclipse.ui.PlatformUI;


/**
 *	Page 1 of the base resource export-to-zip Wizard modified for Probes.
 *	See also org.eclipse.ui.wizards.datatransfer.WizardZipFileResourceExportPage1.
 */
public class ExportProbeWizardPage extends CustomWizardFileSystemResourceExportPage1
{
	// widgets
	//protected Button compressContentsCheckbox;
	protected Button includeSourceFileCheckbox;

	// dialog store id constants
	private final static String STORE_DESTINATION_NAMES_ID = "ExportProbeWizardPage.STORE_DESTINATION_NAMES_ID"; //$NON-NLS-1$
	//private final static String STORE_CREATE_STRUCTURE_ID = "ExportProbeWizardPage.STORE_CREATE_STRUCTURE_ID"; //$NON-NLS-1$
	//private final static String STORE_COMPRESS_CONTENTS_ID = "ExportProbeWizardPage.STORE_COMPRESS_CONTENTS_ID"; //$NON-NLS-1$
	private final static String STORE_INCLUDE_SOURCE_ID = "ExportProbeWizardPage.STORE_INCLUDE_SOURCE_ID"; //$NON-NLS-1$
	
	/**
	 *	Create an instance of this class. 
	 *
	 *	@param name java.lang.String
	 */
	protected ExportProbeWizardPage(
		String name,
		IStructuredSelection selection)
	{
		super(name, selection);
		
		setTitle(name);
		setDescription(Messages.Description); //$NON-NLS-1$
	}
	
	/**
	 * Create an instance of this class
	 * @param IStructuredSelection selection
	 *//*
	public ExportProbeWizardPage(IStructuredSelection selection) {
		this("ExportProbeWizardPageName", selection); //$NON-NLS-1$
		setTitle(DMessages.getString("Export.WindowTitle")); //$NON-NLS-1$
		setDescription(Messages.getString("Export.Description")); //$NON-NLS-1$
	}*/
	
	/** (non-Javadoc)
	 * Method declared on IDialogPage.
	 */
	public void createControl(Composite parent)
	{
		super.createControl(parent);
		
		PlatformUI.getWorkbench().getHelpSystem().setHelp(
			getControl(),
			IProbekitUIHelpContextIDs.EXPORT_PROBE_WIZARD_PAGE);
		
		giveFocusToTree();
	}
	
	/**
	 *	Create the export options specification widgets.
	 *
	 */
	protected void createOptionsGroupButtons(Group optionsGroup)
	{
		// include source file checkbox
		includeSourceFileCheckbox = new Button(optionsGroup, SWT.CHECK | SWT.LEFT);
		includeSourceFileCheckbox.setText(Messages.IncludeSourceCheckbox); //$NON-NLS-1$
		
		//Font font = optionsGroup.getFont();
		/*
		 * Do not expose compress option to the user since force one value to be used in executeExportOperation
		 */ 
		// compress... checkbox
		//compressContentsCheckbox =
		//	new Button(optionsGroup, SWT.CHECK | SWT.LEFT);
		//compressContentsCheckbox.setText(Messages.getString("Export.CompressContentsCheckbox")); //$NON-NLS-1$
		/*
		 * Do not expose directory options to the user since force one value to be used in executeExportOperation
		 */
		//createDirectoryStructureOptions(optionsGroup, font);

		// initial setup
		includeSourceFileCheckbox.setSelection(true);
		//createDirectoryStructureButton.setSelection(true);
		//createSelectionOnlyButton.setSelection(false);
		//compressContentsCheckbox.setSelection(true);
	}
	
	/**
	 * Enable/Disable Finish button upon dialog change.
	 */
	protected void updateWidgetEnablements() {
		boolean pageComplete = determinePageCompletion();
		setPageComplete(pageComplete);
		if (pageComplete)
			setMessage(null);
		super.updateWidgetEnablements();
	}

	/*
	 * @see WizardDataTransferPage.determinePageCompletion.
	 */
	protected boolean determinePageCompletion() {
		return super.determinePageCompletion();
	}	
	
	/**
	 * Returns a boolean indicating whether the directory portion of the
	 * passed pathname is valid and available for use.
	 */
	protected boolean ensureTargetDirectoryIsValid(String fullPathname) {
		int separatorIndex = fullPathname.lastIndexOf(File.separator);

		if (separatorIndex == -1) // ie.- default dir, which is fine
			return true;

		return ensureTargetIsValid(
			new File(fullPathname.substring(0, separatorIndex)));
	}
	
	/**
	 * Returns a boolean indicating whether the passed File handle is
	 * is valid and available for use.
	 */
	protected boolean ensureTargetFileIsValid(File targetFile) {
		if (targetFile.exists() && targetFile.isDirectory()) {
			displayErrorDialog(Messages.MustBeFile); //$NON-NLS-1$
			giveFocusToDestination();
			return false;
		}

		if (targetFile.exists()) {
			if (targetFile.canWrite()) {
				if (!queryYesNoQuestion(Messages.AlreadyExists)) //$NON-NLS-1$
					return false;
			}
			else {
				displayErrorDialog(Messages.AlreadyExistsError); //$NON-NLS-1$
				giveFocusToDestination();
				return false;
			}
		}

		// Restrict file extension to .probekit
		String strExtension = getOutputSuffix();
		String strFileName = targetFile.getName();
		if (!strFileName.endsWith(strExtension))
		{
			displayErrorDialog(Messages.InvalidExtension); //$NON-NLS-1$
			giveFocusToDestination();
			return false;
		}
		
		return true;
	}
	
	/**
	 * Ensures that the target output file and its containing directory are
	 * both valid and able to be used.  Answer a boolean indicating validity.
	 */
	protected boolean ensureTargetIsValid() {
		String targetPath = getDestinationValue();

		if (!ensureTargetDirectoryIsValid(targetPath))
			return false;

		if (!ensureTargetFileIsValid(new File(targetPath)))
			return false;

		return true;
	}
	
	/**
	 *  Export the passed resource and recursively export all of its child resources
	 *  (iff it's a container).  Answer a boolean indicating success.
	 */
	protected boolean executeExportOperation(CustomZipFileExportOperation op)
	{
		// Always create a flat zip file (i.e., files have no directory information)
		//op.setCreateLeadupStructure(createDirectoryStructureButton.getSelection());
		op.setCreateLeadupStructure(false);
		// Always turn off compression (since files shouldn't be that big anyway)
		//op.setUseCompression(compressContentsCheckbox.getSelection());
		op.setUseCompression(false);

		try {
			getContainer().run(true, true, op);
		} catch (InterruptedException e) {
			return false;
		} catch (InvocationTargetException e) {
			displayErrorDialog(e.getTargetException());
			return false;
		}

		IStatus status = op.getStatus();
		if (!status.isOK()) {
			ErrorDialog.openError(getContainer().getShell(), getErrorDialogTitle(), null /* no special message */, status);
			return false;
		}

		return true;
	}
	
	/**
	 * The Finish button was pressed.  Try to do the required work now and answer
	 * a boolean indicating success.  If false is returned then the wizard will
	 * not close.
	 * @returns boolean
	 */
	public boolean finish()
	{
		if (!ensureTargetIsValid())
		{
			return false;
		}
		//List resourcesToExport = getWhiteCheckedResources();
		//List resourcesToExport = getSelectedResources();
		// Just use current selection from table since not supporting export of multiple probes
		List resourcesToExport = getCurrentResourceSelection();
		if (resourcesToExport.size() > 0)
		{
			// Currently only support exporting a single probe at a time
			Object obj = resourcesToExport.get(0);
			if (obj instanceof IResource)
			{
				IResource resource = (IResource) obj;
				if (resource.getType() == IResource.FILE)
				{
					IFile resourceProbeFile = (IFile) resource;
					try
					{
						ProbeResourceBundle bundle = new ProbeResourceBundle(resourceProbeFile);
						if (bundle.isComplete())
						{
							boolean bIncludeSource = includeSourceFileCheckbox.getSelection();
							if (!bIncludeSource)
							{
								resourcesToExport.remove(resourceProbeFile);
							}
							
						    IResource resourceScript = bundle.getScript();
						    resourcesToExport.add(resourceScript);
						    
						    IResource resourceInfo = bundle.getProbeInfo();
						    resourcesToExport.add(resourceInfo);
						    
							IResource[] resourcesClassFiles = bundle.getSupporting();
							for (int i = 0; i < resourcesClassFiles.length; i++)
							{
								resourcesToExport.add(resourcesClassFiles[i]);
							}
							
							//Save dirty editors if possible but do not stop if not all are saved
							saveDirtyEditors();
							// about to invoke the operation so save our state
							saveWidgetValues();

							return executeExportOperation(
								new CustomZipFileExportOperation(null, resourcesToExport, getDestinationValue()));
						}
						else
						{
						    // Pieces are missing or inaccessible. It probably means a rebuild is a good idea.
						    // This is pretty unlikely to happen in your scenario - the ctor call is going to catch most problems.
							String strMessage = NLS.bind(Messages.GeneralResourceError, new Object[] {resourceProbeFile.getName()}); //$NON-NLS-1$
							displayErrorDialog(strMessage);
						}
					}
					catch (InvalidProbeBundleException e)
					{
						// this means one or more of the resources doesn't exist or isn't accessible
						// it'll throw you back the IResource it had a problem with, inside the exception
						String strMessage = NLS.bind(Messages.GeneralResourceError, new Object[] {resourceProbeFile.getName()}); //$NON-NLS-1$
						
						String resourceMissing = e.getFileOrResourceName();
						if (resourceMissing != null && resourceMissing.length() > 0)
						{
								strMessage = NLS.bind(Messages.MissingResourceError, new Object[] {resourceMissing}); //$NON-NLS-1$
						}

						displayErrorDialog(strMessage);
					}
					catch (ProbeBundleBuildRequiredException e)
					{
						// None of the properties are set. The probe source file has  
						// never been compiled or has not been re-built since a Clean.
						String strMessage = NLS.bind(Messages.BuildRequiredError, resourceProbeFile.getName()); //$NON-NLS-1$
						String strLineSeparator = System.getProperty("line.separator"); //$NON-NLS-1$
						strMessage = strMessage + strLineSeparator
						+ NLS.bind(Messages.RebuildProjectHint, resourceProbeFile.getProject().getName()); //$NON-NLS-1$
						displayErrorDialog(strMessage);
					}
					catch (ProbeBundleException e)
					{
						// This means something else weird went wrong, such as one of property values being mal-formed
						// This does NOT throw back extra data.
						String strMessage = NLS.bind(Messages.InvalidProbeSourceError, resourceProbeFile.getName()); //$NON-NLS-1$
						String strLineSeparator = System.getProperty("line.separator"); //$NON-NLS-1$
						strMessage = strMessage + strLineSeparator + Messages.PropertyUnavailableError; //$NON-NLS-1$
						displayErrorDialog(strMessage);
					}
					catch (Exception e)
					{
						e.printStackTrace();
						String strTitle = getErrorDialogTitle();
						String strMessage = Messages.InternalError; //$NON-NLS-1$
						ProbekitLaunchPlugin.reportException(e, strTitle, strMessage);
					}
				}
				else
				{
					String strError = Messages.SourceInvalid; //$NON-NLS-1$
					this.setErrorMessage(strError);
				}
			}
			else
			{
				String strError = Messages.SourceInvalid; //$NON-NLS-1$
				this.setErrorMessage(strError);
			}
		}
		else
		{
			MessageDialog.openInformation(getContainer().getShell(),
				Messages.InformationDialogTitle, //$NON-NLS-1$
				Messages.NoneSelected); //$NON-NLS-1$
		}
		
		return false;
	}
	
	/**
	 * Display an error dialog with the specified message.
	 *
	 * @param message the error message
	 */
	protected void displayErrorDialog(String message)
	{
		// RATLC00378743 - The right-hand part of the long file path is truncated in the Import Error dialog box.
		// Override implementation in org.eclipse.ui.dialogs.WizardDataTransferPage.displayErrorDialog.
		//MessageDialog.openError(getContainer().getShell(), getErrorDialogTitle(), message); //$NON-NLS-1$
		MessageBox mbox = new MessageBox(getContainer().getShell(), SWT.ICON_ERROR | SWT.OK);
		mbox.setMessage(message);
		mbox.setText(getErrorDialogTitle());
		mbox.open();
	}
	
	/**
	 *	Answer the string to display in the receiver as the destination type
	 */
	protected String getDestinationLabel() {
		return Messages.DestinationLabel; //$NON-NLS-1$
	}
	
	/**
	 *	Answer the contents of self's destination specification widget.  If this
	 *	value does not have a suffix then add it first.
	 */
	protected String getDestinationValue() {
		String idealSuffix = getOutputSuffix();
		String destinationText = super.getDestinationValue();

        // only append a suffix if the destination doesn't already have a . in 
        // its last path segment.  
        // Also prevent the user from selecting a directory.  Allowing this will 
        // create a ".zip" file in the directory
		if (destinationText.length() != 0 && !destinationText.endsWith(File.separator)) {
            int dotIndex = destinationText.lastIndexOf('.');
            if (dotIndex != -1) {
                // the last path seperator index
                int pathSepIndex = destinationText.lastIndexOf(File.separator);
                if (pathSepIndex != -1 && dotIndex < pathSepIndex) {
                    destinationText += idealSuffix;
                }
            }
            else {
                destinationText += idealSuffix;
            }
		}

		return destinationText;
	}
	
	/**
	 *	Answer the suffix that files exported from this wizard should have.
	 *	If this suffix is a file extension (which is typically the case)
	 *	then it must include the leading period character.
	 *
	 */
	protected String getOutputSuffix() {
		return ".probekit"; //$NON-NLS-1$
		//return ".zip"; //$NON-NLS-1$
	}
	
	/**
	 *	Open an appropriate destination browser so that the user can specify a source
	 *	to import from
	 *//*
	protected void handleDestinationBrowseButtonPressed() {
		FileDialog dialog = new FileDialog(getContainer().getShell(), SWT.SAVE);
		dialog.setFilterExtensions(new String[] { "*.zip", "*.*" }); //$NON-NLS-1$ //$NON-NLS-2$
		dialog.setText(DataTransferMessages.getString("ZipExport.selectDestinationTitle")); //$NON-NLS-1$
		String currentSourceString = getDestinationValue();
		int lastSeparatorIndex =
			currentSourceString.lastIndexOf(File.separator);
		if (lastSeparatorIndex != -1)
			dialog.setFilterPath(
				currentSourceString.substring(0, lastSeparatorIndex));
		String selectedFileName = dialog.open();

		if (selectedFileName != null) {
			setErrorMessage(null);
			setDestinationValue(selectedFileName);
		}
	}*/
	
	/**
	 *	Hook method for saving widget values for restoration by the next instance
	 *	of this class.
	 */
	protected void internalSaveWidgetValues() {
		// update directory names history
		IDialogSettings settings = getDialogSettings();
		if (settings != null) {
			String[] directoryNames =
				settings.getArray(STORE_DESTINATION_NAMES_ID);
			if (directoryNames == null)
				directoryNames = new String[0];

			directoryNames =
				addToHistory(directoryNames, getDestinationValue());
			settings.put(STORE_DESTINATION_NAMES_ID, directoryNames);

			// Do not expose directory options to the user since force one value to be used in executeExportOperation
			//settings.put(
			//	STORE_CREATE_STRUCTURE_ID,
			//	createDirectoryStructureButton.getSelection());
			
			// Do not expose compress option to the user since force one value to be used in executeExportOperation
			//settings.put(
			//		STORE_COMPRESS_CONTENTS_ID,
			//		compressContentsCheckbox.getSelection());
			
			settings.put(
				STORE_INCLUDE_SOURCE_ID,
				includeSourceFileCheckbox.getSelection());
		}
	}
	
	/**
	 *	Hook method for restoring widget values to the values that they held
	 *	last time this wizard was used to completion.
	 */
	protected void restoreWidgetValues() {
		IDialogSettings settings = getDialogSettings();
		if (settings != null) {
			String[] directoryNames =
				settings.getArray(STORE_DESTINATION_NAMES_ID);
			if (directoryNames == null || directoryNames.length == 0)
				return; // ie.- no settings stored

			// destination
			setDestinationValue(directoryNames[0]);
			for (int i = 0; i < directoryNames.length; i++)
				addDestinationItem(directoryNames[i]);

			// Do not expose directory options to the user since force one value to be used in executeExportOperation
			//boolean setStructure = settings.getBoolean(STORE_CREATE_STRUCTURE_ID);
			//createDirectoryStructureButton.setSelection(setStructure);
			//createSelectionOnlyButton.setSelection(!setStructure);
			
			// Do not expose compress option to the user since force one value to be used in executeExportOperation
			//compressContentsCheckbox.setSelection(
			//		settings.getBoolean(STORE_COMPRESS_CONTENTS_ID));

			includeSourceFileCheckbox.setSelection(
				settings.getBoolean(STORE_INCLUDE_SOURCE_ID));
		}
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.ui.wizards.datatransfer.CustomWizardFileSystemResourceExportPage1#destinationEmptyMessage()
	 */
	protected String destinationEmptyMessage() {
		return Messages.DestinationEmpty; //$NON-NLS-1$
	}
}
