/**********************************************************************
 * Copyright (c) 2006 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ImportProbeWizardPage.java,v 1.1 2006/03/16 15:56:15 popescu Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.tptp.platform.probekit.launch.internal.wizard;

import java.io.File;
import java.io.IOException;
import java.util.zip.ZipException;
import java.util.zip.ZipFile;

import org.eclipse.jface.dialogs.IDialogSettings;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.*;
import org.eclipse.tptp.platform.probekit.launch.internal.ProbekitLaunchPlugin;
import org.eclipse.tptp.platform.probekit.registry.CandidateEntry;
import org.eclipse.tptp.platform.probekit.registry.ProbeRegistry;
import org.eclipse.tptp.platform.probekit.registry.ProbeRegistryEntry;
import org.eclipse.tptp.platform.probekit.registry.ProbeRegistryException;
import org.eclipse.tptp.platform.probekit.util.InvalidProbeBundleException;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.PlatformUI;


/**
 *	Page 1 of the base resource import-from-zip Wizard modified for Probes.
 *	See also org.eclipse.ui.wizards.datatransfer.WizardZipFileResourceImportPage1.
 */
class ImportProbeWizardPage
	extends CustomWizardFileSystemResourceImportPage1
	implements Listener {

	// constants
	// Mask value with .zip included is "*.probekit;*.zip"
	private static final String FILE_IMPORT_MASK = "*.probekit"; //$NON-NLS-1$
	private static final String FILE_IMPORT_EXTENSION = ".probekit"; //$NON-NLS-1$
	protected static final String SOURCE_EMPTY_MESSAGE = Messages.ISourceEmpty; //$NON-NLS-1$

	// dialog store id constants
	private final static String STORE_SOURCE_NAMES_ID =
		"ImportProbeWizardPage.STORE_SOURCE_NAMES_ID"; //$NON-NLS-1$
	private final static String STORE_OVERWRITE_EXISTING_RESOURCES_ID =
		"ImportProbeWizardPage.STORE_OVERWRITE_EXISTING_RESOURCES_ID"; //$NON-NLS-1$
	/**
	 *	Creates an instance of this class
	 * @param aWorkbench IWorkbench
	 * @param selection IStructuredSelection
	 */
	public ImportProbeWizardPage(
		IWorkbench aWorkbench,
		IStructuredSelection selection) {
		super(Messages.IPageName, aWorkbench, selection); //$NON-NLS-1$
		setTitle(Messages.PageTitle); //$NON-NLS-1$
		setDescription(Messages.Description);  //$NON-NLS-1$
	}
	/**
	 * Called when the user presses the Cancel button. Return a boolean
	 * indicating permission to close the wizard.
	 *
	 * @return boolean
	 */
	public boolean cancel() {
		return true;
	}
	/**
	 * Attempts to close the passed zip file, and answers a boolean indicating success.
	 */
	protected boolean closeZipFile(ZipFile file) {
		try {
			file.close();
		} catch (IOException e) {
			displayErrorDialog(e);
			return false;
		}

		return true;
	}
	/** (non-Javadoc)
	 * Method declared on IDialogPage.
	 */
	public void createControl(Composite parent)
	{
		super.createControl(parent);
		
		PlatformUI.getWorkbench().getHelpSystem().setHelp(
			getControl(),
			IProbekitUIHelpContextIDs.IMPORT_PROBE_WIZARD_PAGE);
	}
	/**
	 *	Create the options specification widgets. There is only one
	 * in this case so create no group.
	 *
	 *	@param parent org.eclipse.swt.widgets.Composite
	 */
	protected void createOptionsGroup(Composite parent) {

		// overwrite... checkbox
		overwriteExistingResourcesCheckbox = new Button(parent, SWT.CHECK);
		overwriteExistingResourcesCheckbox.setText(Messages.OverwriteWarningCheckbox); //$NON-NLS-1$
	}
	/**
	 *	Answer a boolean indicating whether the specified source currently exists
	 *	and is valid (ie.- proper format)
	 */
	protected boolean ensureSourceIsValid() {
		ZipFile specifiedFile = getSpecifiedSourceFile();

		if (specifiedFile == null)
			return false;

		return closeZipFile(specifiedFile);
	}
	/**
	 * The Finish button was pressed.  Try to do the required work now and answer
	 * a boolean indicating success.  If <code>false</code> is returned then the
	 * wizard will not close.
	 *
	 * @return boolean
	 */
	public boolean finish() {
		if (!ensureSourceIsValid())
			return false;
		
		ProbeRegistry registry = ProbeRegistry.getRegistry();
		
		if (registry != null)
		{
			String strSource = sourceNameField.getText();
			if (strSource != null && strSource.length() > 0)
			{
				String strLineSeparator = System.getProperty("line.separator"); //$NON-NLS-1$
				try
				{
					CandidateEntry candidate = registry.addIfNoConflict(strSource);
					if (candidate != null)
					{
						ProbeRegistryEntry conflict = candidate.getConflict();
						if (conflict != null)
						{
							if (!overwriteExistingResourcesCheckbox.getSelection())
							{
								String strConflicts = ""; //$NON-NLS-1$
								
								int iConflictDetails = candidate.getFlags();
								if ((iConflictDetails & CandidateEntry.EXISTS_IN_WORKSPACE) != 0)
								{
									strConflicts = Messages.WarningProbeExists + strLineSeparator; //$NON-NLS-1$
								}
								
								if ((iConflictDetails & CandidateEntry.VERSION_CONFLICT) != 0)
								{
									if ((iConflictDetails & CandidateEntry.CANDIDATE_VERSION_OLDER) != 0)
									{
										//strConflicts = strConflicts + "The selected Probe Kit has an older version number than one already in the Probekit Registry.\n";
										strConflicts = strConflicts + Messages.WarningOlderVersion + strLineSeparator; //$NON-NLS-1$
									}
									else
									{
										//strConflicts = strConflicts + "The selected Probe Kit has a newer version number than one already in the Probekit Registry.\n";
										strConflicts = strConflicts + Messages.WarningNewerVersion + strLineSeparator; //$NON-NLS-1$
									}
								}
								else
								{
									//strConflicts = strConflicts + "The selected Probe Kit has the same version number as one already in the Probekit Registry.\n";
									strConflicts = strConflicts + Messages.WarningSameVersion + strLineSeparator; //$NON-NLS-1$
								}
								
								//if (iConflictDetails != CandidateEntry.NO_CONFLICT)
								//{
									//String strMessage = "The selected Probe Kit conflicts with an existing Probe Kit already registered in the Probekit Registry under the same ID as follows.\n";
									///String strMessage = "The selected Probe Kit conflicts with an existing Probe Kit already registered in the Probekit Registry under the same ID.\n";
									String strMessage = Messages.WarningVersionConflict + strLineSeparator + strLineSeparator; //$NON-NLS-1$
									strMessage = strMessage + strConflicts + strLineSeparator + Messages.WarningOverwriteConfirm; //$NON-NLS-1$
									boolean bOk = MessageDialog.openConfirm(
											getContainer().getShell(),
											Messages.WarningTitle, //$NON-NLS-1$
											strMessage);
									if (bOk)
									{
										registry.commit(candidate);
									}
									else
									{
										registry.discard(candidate);
										return false;
									}
								//}
							}
							else
							{
								registry.commit(candidate);
							}
						}
						//else nothing since already committed and discarded due to conflict
					}
				}
				catch (ProbeRegistryException e) 
				{
					// ProbeRegistryException has now become more of an internal error. For example,
					// trying to commit something you already committed or discarded.
					displayErrorDialog(Messages.ProbeRegistryException); //$NON-NLS-1$
					sourceNameField.setFocus();
					return false;
				}
				catch (InvalidProbeBundleException exc)
				{
					String strMessage = NLS.bind(Messages.ErrorImporting, strSource); //$NON-NLS-1$
					strMessage = strMessage + strLineSeparator;
					String strReason = "";  //$NON-NLS-1$
					if (exc.testReason(InvalidProbeBundleException.MISSING_FILE))
					{
						if (exc.testReason(InvalidProbeBundleException.MISSING_MODEL_FILE))
						{
							strReason = strReason + Messages.ModelFileMissing + strLineSeparator; //$NON-NLS-1$
						}
						if (exc.testReason(InvalidProbeBundleException.MISSING_SCRIPT_FILE))
						{
							strReason = strReason + Messages.ScriptFileMissing + strLineSeparator; //$NON-NLS-1$
						}
						if (exc.testReason(InvalidProbeBundleException.MISSING_SUPPORT_FILE))
						{
							strReason = strReason + Messages.SupportFileMissing + strLineSeparator; //$NON-NLS-1$
						}
						if (strReason.length() == 0)
						{
							strReason = Messages.FileMissingError + strLineSeparator; //$NON-NLS-1$
						}
					}
					if (exc.testReason(InvalidProbeBundleException.INACCESSIBLE_FILE))
					{
						if (exc.testReason(InvalidProbeBundleException.INACCESSIBLE_MODEL_FILE))
						{
							strReason = strReason + Messages.ModelFileInaccessible + strLineSeparator; //$NON-NLS-1$
						}
						if (exc.testReason(InvalidProbeBundleException.INACCESSIBLE_SCRIPT_FILE))
						{
							strReason = strReason + Messages.ScriptFileInaccessible + strLineSeparator; //$NON-NLS-1$
						}
						if (exc.testReason(InvalidProbeBundleException.INACCESSIBLE_SUPPORT_FILE))
						{
							strReason = strReason + Messages.SupportFileInaccessible + strLineSeparator; //$NON-NLS-1$
						}
						if (strReason.length() == 0)
						{
							strReason = Messages.FileInaccessibleError + strLineSeparator; //$NON-NLS-1$
						}
					}
					if (exc.testReason(InvalidProbeBundleException.INVALID_MODEL))
					{
						strReason = strReason + Messages.InvalidModelError + strLineSeparator; //$NON-NLS-1$
					}
					if (exc.testReason(InvalidProbeBundleException.TOO_MANY_FILES_WITH_EXT))
					{
						String strExtension = exc.getExtension();
						strReason = strReason + NLS.bind(Messages.TooManyFiles, strExtension) + strLineSeparator; //$NON-NLS-1$
					}
					
					String strErrorMessage = strMessage + strReason;
					displayErrorDialog(strErrorMessage);
					sourceNameField.setFocus();
					return false;					
				}
				catch (IOException e)
				{
					displayErrorDialog(e);
					sourceNameField.setFocus();
					return false;
				}
				catch (Exception e)
				{
					e.printStackTrace();
					String strTitle = Messages.ErrorDialogTitle; //$NON-NLS-1$
					String strMessage = Messages.InternalError; //$NON-NLS-1$
					ProbekitLaunchPlugin.reportException(e, strTitle, strMessage);
					return false;
				}
			}
		}
		
		saveWidgetValues();

		return true;
	}
	
	/**
	 * Display an error dialog with the specified message.
	 *
	 * @param message the error message
	 */
	protected void displayErrorDialog(String message)
	{
		// RATLC00378743 - The right-hand part of the long file path is truncated in the Import Error dialog box.
		// Override implementation in org.eclipse.ui.dialogs.WizardDataTransferPage.displayErrorDialog.
		//MessageDialog.openError(getContainer().getShell(), getErrorDialogTitle(), message); //$NON-NLS-1$
		MessageBox mbox = new MessageBox(getContainer().getShell(), SWT.ICON_ERROR | SWT.OK);
		mbox.setMessage(message);
		mbox.setText(getErrorDialogTitle());
		mbox.open();
	}
	
	/**
	 *	Answer the string to display as the label for the source specification field
	 */
	protected String getSourceLabel() {
		return Messages.SourceLabel; //$NON-NLS-1$
	}
	/**
	 *	Answer a handle to the zip file currently specified as being the source.
	 *	Return null if this file does not exist or is not of valid format.
	 */
	protected ZipFile getSpecifiedSourceFile() {
		return getSpecifiedSourceFile(sourceNameField.getText());
	}
	/**
	 *	Answer a handle to the zip file currently specified as being the source.
	 *	Return null if this file does not exist or is not of valid format.
	 */
	private ZipFile getSpecifiedSourceFile(String fileName) {
		if (fileName.length() == 0)
			return null;

		try {
			return new ZipFile(fileName);
		} catch (ZipException e) {
			displayErrorDialog(e);
		} catch (IOException e) {
			displayErrorDialog(e);
		}

		sourceNameField.setFocus();
		return null;
	}
	/**
	 *	Handle all events and enablements for widgets in this dialog
	 *
	 * @param event Event
	 */
	public void handleEvent(Event event) {
		if (event.widget == sourceBrowseButton)
			handleSourceBrowseButtonPressed();
		
		super.handleEvent(event);
	}
	/**
	 *	Open a FileDialog so that the user can specify the source
	 *	file to import from
	 */
	protected void handleSourceBrowseButtonPressed() {
		String selectedFile = queryZipFileToImport();

		if (selectedFile != null) {
			if (!selectedFile.equals(sourceNameField.getText())) {
				//Be sure it is valid before we go setting any names
				ZipFile sourceFile = getSpecifiedSourceFile(selectedFile);
				if (sourceFile != null) {
					closeZipFile(sourceFile);
					setSourceName(selectedFile);
				}
			}
		}
	}
	/**
	 * Opens a file selection dialog and returns a string representing the
	 * selected file, or <code>null</code> if the dialog was canceled.
	 */
	protected String queryZipFileToImport() {
		FileDialog dialog =
			new FileDialog(sourceNameField.getShell(), SWT.OPEN);
		dialog.setFilterExtensions(new String[] { FILE_IMPORT_MASK });
		dialog.setText(Messages.QueryFileDialogTitle); //$NON-NLS-1$

		String currentSourceString = sourceNameField.getText();
		int lastSeparatorIndex =
			currentSourceString.lastIndexOf(File.separator);
		if (lastSeparatorIndex != -1)
			dialog.setFilterPath(
				currentSourceString.substring(0, lastSeparatorIndex));

		return dialog.open();
	}
	/**
	 *	Use the dialog store to restore widget values to the values that they held
	 *	last time this wizard was used to completion
	 */
	protected void restoreWidgetValues() {
		IDialogSettings settings = getDialogSettings();
		if (settings != null) {
			String[] sourceNames = settings.getArray(STORE_SOURCE_NAMES_ID);
			if (sourceNames == null)
				return; // ie.- no settings stored

			// set filenames history
			for (int i = 0; i < sourceNames.length; i++)
				sourceNameField.add(sourceNames[i]);

			// radio buttons and checkboxes	
			overwriteExistingResourcesCheckbox.setSelection(
				settings.getBoolean(STORE_OVERWRITE_EXISTING_RESOURCES_ID));
		}
	}
	/**
	 * 	Since Finish was pressed, write widget values to the dialog store so that they
	 *	will persist into the next invocation of this wizard page.
	 *
	 *	Note that this method is identical to the one that appears in the superclass.
	 *	This is necessary because proper overriding of instance variables is not occurring.
	 */
	protected void saveWidgetValues() {
		IDialogSettings settings = getDialogSettings();
		if (settings != null) {
			// update source names history
			String[] sourceNames = settings.getArray(STORE_SOURCE_NAMES_ID);
			if (sourceNames == null)
				sourceNames = new String[0];

			sourceNames = addToHistory(sourceNames, sourceNameField.getText());
			settings.put(STORE_SOURCE_NAMES_ID, sourceNames);

			settings.put(
				STORE_OVERWRITE_EXISTING_RESOURCES_ID,
				overwriteExistingResourcesCheckbox.getSelection());
		}
	}
	/**
	 *	Answer a boolean indicating whether self's source specification
	 *	widgets currently all contain valid values.
	 */
	protected boolean validateSourceGroup() {
		File sourceFile = getSourceFile();
		if (sourceFile == null)
		{
			setMessage(SOURCE_EMPTY_MESSAGE);
			return false;
		}
		
		// Restrict file extension to .probekit
		String strExtension = getInputSuffix();
		String strFileName = sourceFile.getName();
		if (!strFileName.endsWith(strExtension))
		{
			return false;
		}
		
		return true;
	}
	
	/**
	 *	Answer the suffix that files exported from this wizard should have.
	 *	If this suffix is a file extension (which is typically the case)
	 *	then it must include the leading period character.
	 *
	 */
	protected String getInputSuffix() {
		return FILE_IMPORT_EXTENSION;
	}
}
