/********************************************************************** 
 * Copyright (c) 2004, 2006 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html         
 * $Id: LaunchPad.java,v 1.1 2006/02/22 16:41:43 nmehrega Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/ 
/*
 * Created on Apr 2, 2004
 *
 * LaunchPad.java
 * class LaunchPad
 * Probe deployment and launching implementation
 * 
 * This class is intended to be used by the hyades probe launcher
 * in conjunction with the Probe Registry. The deploy method 
 * should be invoked from the Agent Listener Callback after
 * the agent was attached but before the "RESUME" command 
 * was processed.
 * 
 * In the 'Attach' scenario deployProbe can be invoked any time. 
 * 
 */
package org.eclipse.tptp.platform.probekit.launch.launchpad;

import java.io.DataInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.nio.ByteBuffer;
import java.util.Iterator;
import java.util.Vector;

import org.eclipse.emf.common.util.EList;
import org.eclipse.hyades.internal.execution.local.common.BinaryCustomCommand;
import org.eclipse.hyades.internal.execution.local.control.Agent;
import org.eclipse.hyades.internal.execution.local.control.InactiveAgentException;
import org.eclipse.hyades.loaders.util.LoadersUtils;
import org.eclipse.hyades.models.hierarchy.TRCAgentProxy;
import org.eclipse.hyades.models.hierarchy.TRCConfiguration;
import org.eclipse.hyades.models.hierarchy.TRCFilter;
import org.eclipse.hyades.models.hierarchy.TRCOption;
import org.eclipse.tptp.platform.probekit.registry.ProbeRegistry;
import org.eclipse.tptp.platform.probekit.registry.ProbeRegistryEntry;
import org.eclipse.tptp.platform.probekit.util.InvalidProbeBundleException;
import org.eclipse.tptp.platform.probekit.util.ProbeLaunchConfigString;
import org.eclipse.tptp.platform.probekit.util.ProbekitConstants;
import org.eclipse.tptp.platform.probekit.util.ProbekitDebugConfig;


/**
 * @author vhavin
 *
 * TODO To change the template for this generated type comment go to
 * Window - Preferences - Java - Code Generation - Code and Comments
 */
/**
 * @author vhavin
 *
 */
public class LaunchPad {
	//- Private Constants ------------------------------------------------------
	private static final String ENCODING_NAME_UTF8 = "UTF-8";
	private static final String OPCODE_PROBE = "PROBE";
	private static final String OPCODE_RULE = "RULE";
	private static final String OPCODE_RULE_WITH_BLANK = "RULE ";
	private static final String OPCODE_FILTER_WITH_BLANK = "FILTER ";
	private static final String RAC_DEPLOY_SCRIPT_COMMAND = "PROBE_SCRIPT";
	private static final String RAC_DEPLOY_CLASS_COMMAND = "REMOTE_CLASS";
	private static final String RAC_SEND_FILTERS_COMMAND = "PROBE_FILTERS";
	
	//- Private fields ---------------------------------------------------------
	private Agent agent;
	private Vector filters;

	//- Private methods --------------------------------------------------------

	/**
	 * @param file - File reference
	 * @return - true if the file is a Java class
	 */
	private boolean isClass(File file) 
	{
		if(file.getName().endsWith(ProbekitConstants.CLASS_FILE_EXT))
		{
			return true;
		}
		else
		{
			return false;
		}
	}

	/**
	 * Read bytes from the input file into the array of bytes.
	 * 
	 * @param file - input file 
	 * @return - array of bytes from the file (no conversion applied)
	 * @throws FileNotFoundException
	 * @throws IOException
	 * 
	 */
	private byte[] readFileIntoBuffer(File file) throws FileNotFoundException, IOException
	{
		FileInputStream fs;
		byte buffer[];
		
		fs = new FileInputStream(file);
		DataInputStream probeStream = new DataInputStream(fs);
		try {
			int bufferSize = (int)file.length();
			buffer = new byte[bufferSize];
			probeStream.read(buffer);
		} finally {
			probeStream.close();
		}
		return buffer;
	}
	
	/**
	 * This method weaves the incoming probe script
	 * with the filter set defined by user in the probe launching dialog.
	 * 
	 * By the time you get here, this.filters is an array of strings consisting
	 * of Probescript-format "FILTER" commands which represent the filters
	 * specified in the UI.
	 * 
	 * The basic goal of this function is to append the FILTER strings to the
	 * probescript that's passed in. That is all that's necessary for
	 * the agent to perform two-tier filtering: one tier is the RULEs that 
	 * are in the probescript to begin with (which came from the probe source code), 
	 * and the other tier is the patterns that come from the UI. A method
	 * will have a probe applied to it only if it is "included" by both
	 * tiers. Another way to say this is "...if it's not 'excluded'
	 * by either tier."
	 * 
	 * However, this two-tier filtering system was invented after Probekit
	 * had already shipped once. That leaves a backward compatibility situation to 
	 * address. The old BCI engine ignores FILTER statements in probescript files,
	 * using only the RULE statements it sees. When a probe is "pre-targeted"
	 * (meaning it has RULEs in it from the probe source code), those RULEs
	 * are what the old engine should see. If a probe is not pre-targeted,
	 * we need to place the UI-selected patterns into the probescript as RULEs
	 * so they are what the old engine sees.
	 * 
	 * We can't tell in advance whether we're talking to a new engine or
	 * an old one. So we need to do something in this function that will
	 * give the right result no matter which one we're talking to.
	 * 
	 * Turns out, we can. We write the UI-selected patterns into untargeted
	 * probes as RULEs, leaving the RULEs in pre-targeted probes alone.
	 * In addition, we write the UI-selected patterns as FILTERs at the
	 * end of the probescript file. 
	 * 
	 * Walk through the outcomes: an old engine with a single tier of target
	 * processing (the RULEs in the probescript, ignoring FILTER commands) will 
	 * either see the RULEs that were in the probe source to begin with (for a 
	 * pretargeted probe) or it'll see the UI patterns as RULEs. That's just like 
	 * the original version, achieving backward compatibility.
	 * 
	 * A new engine with its two tiers of target processing (both RULEs and
	 * FILTERS) will see the UI selections as FILTERs. That's one tier.
	 * The other tier is the RULEs. In the case of a pre-targeted probe, they are
	 * the RULEs from the probe source file. In the case
	 * of an untargeted probe, the RULEs are from the UI selections. 
	 * This means the UI selections appear both as RULEs and FILTERs, so
	 * both tiers have identical content. That's harmless: the outcome is the same 
	 * as if the RULE tier was empty.
	 *  
	 * @param script the original probescript
	 * @return a new probescript with filters woven in
	 * 
	 * @throws UnsupportedEncodingException
	 */
	private byte[] weaveFilters(byte script[]) throws UnsupportedEncodingException
	{
		if(filters.size() == 0) {
			// Nothing to do: just use the original filters from the script
			// This case is unlikely: it means there are no filters at all selected in the UI.
			return script;
		}
		
		// Step one: convert the original probescript into an array of Strings.
		String[] probescriptAsStrings = convertBytesToStrings(script); 

		// Step two: scan for PROBE sections without RULEs.
		// (See inside for Step three.)
		boolean seenAnyProbeLinesYet = false;
		boolean seenAnyRulesYet = false;
		Vector newProbescriptStringVector = new Vector();
		Vector filtersAsRules = createRulesStringVector();
		
		for (int i = 0; i < probescriptAsStrings.length; i++) {
			String opcode = getFirstWord(probescriptAsStrings[i]);
			if (opcode.equalsIgnoreCase(OPCODE_PROBE)) {
				// Found a PROBE line, marking the end of the previous PROBE.
				// Step three: if there was a previous PROBE and it had no RULEs, 
				// append the FILTERs as RULEs.
				if (seenAnyProbeLinesYet) {
					if (!seenAnyRulesYet) {
						newProbescriptStringVector.addAll(filtersAsRules);
					}
				}
				seenAnyProbeLinesYet = true;
				seenAnyRulesYet = false;
			}
			else if (opcode.equalsIgnoreCase(OPCODE_RULE)) {
				// Remember we've seen at least one RULE in the current PROBE.
				seenAnyRulesYet = true;
			}
			// Copy this original probescript command to the new probescript.
			newProbescriptStringVector.add(probescriptAsStrings[i]);
		}
		
		// We hit the end of the script: take care of the last PROBE section in the script.
		// If the last section didn't have any RULEs in it, append the FILTERs as RULEs.
		if (seenAnyProbeLinesYet && !seenAnyRulesYet) {
			newProbescriptStringVector.addAll(filtersAsRules);
		}
		
		// Step four: append the strings in this.filters as FILTER statements,
		// to implement two-tier filtering for pretargeted probes being sent to new BCI engines.
		newProbescriptStringVector.addAll(filters);

		// Final step: turn the new probescript string vector back into a byte array.
		
		// Compute the number of bytes needed to hold the new script
		Iterator itr = newProbescriptStringVector.iterator();
		int size = 0; 
		while(itr.hasNext())
		{
			String next = (String)itr.next();
			size += next.getBytes(ENCODING_NAME_UTF8).length + 1;
		}
		
		// Create a ByteBuffer for the new script
		ByteBuffer buf = ByteBuffer.allocate(size);
		itr = newProbescriptStringVector.iterator();
		while(itr.hasNext())
		{
			String next = (String)itr.next();
			buf.put(next.getBytes(ENCODING_NAME_UTF8));
			buf.put((byte)0);
		}
		return buf.array();
	}
	
	/**
	 * This method returns a Vector of Strings which is a copy 
	 * of the "this.filters" Vector, but with the word "RULE"
	 * instead of the word "FILTER" at the start of each one.
	 * @return
	 */
	private Vector createRulesStringVector() {
		Vector result = new Vector();
		for (int i = 0; i < this.filters.size(); i++) {
			String s = (String)this.filters.elementAt(i);
			if (s.startsWith(OPCODE_FILTER_WITH_BLANK)) {
				s = OPCODE_RULE_WITH_BLANK + s.substring(OPCODE_FILTER_WITH_BLANK.length());
			}
			else {
				// This function expects to operate on strings
				// created elsewhere in this class where every one 
				// starts with OPCODE_FILTER_WITH_BLANK ... It's a surprise
				// to find a string that doesn't start that way,
				// but we'd better leave it alone.
			}
			result.add(s);
		}
		return result;
	}
	/**
	 * Split the incoming string at newlines and null chars.
	 * It's a pity there's no regex symbol for "the set of chars considered as end-of-line chars,"
	 * but there isn't, so I took the list from the documentation for class regex.Pattern. 
	 * @param bytes
	 * @return
	 * @throws UnsupportedEncodingException
	 */
	private String[] convertBytesToStrings(byte[] bytes) throws UnsupportedEncodingException {
		String oneBigString = new String(bytes, ENCODING_NAME_UTF8);
		String[] result = oneBigString.split("[\u0000\n\r\u0085\u2028\u2029]");
		return result;
	}

	/**
	 * Extract the first word from a string. Ignores leading blanks.
	 * For "word" boundaries we use the Java regex pattern \s which
	 * matches tab, space, and other chars.  
	 * @param input the string to pick the first word from
	 * @return the first word of the (trimmed) input string
	 */
	private String getFirstWord(String input) {
		String[] splitResult = input.trim().split("\\s+", 2);
		return splitResult[0];
	}
	
	/**
	 * Derives Probe Kit filtering rules from the Hyades filters
	 * 
	 * @param filterString - Hyades filter string
	 * @return String[] - array of RULE statements for probe script
	 * 
	 */
	/*--------------------------------------------------------------------------
	 * This method was needed to match two different filter specifications.
	 * Hyades is using the following filter syntax:
	 * <filters> ::= <filter>[& <filter>]
	 * <filter> ::= <pkg_and_class>[":"<method>[":"<signature>]]":"<action>
	 * <action> ::= "INCLUDE"|"EXCLUDE"
	 * <pkg_and_class> ::= ["*"][<name>.]<name>["*"]|"*"
	 * <method> ::= <signature> ::= ["*"]<name>["*"}|"*"
	 * 
	 * whereas probe kit is using
		 * <filters> ::= <filter>[<new_line> <filter>]
	 * <filter> ::= "RULE" <ws> <package> <ws> <class> <ws> <method> <ws> <signature> <ws> <action> 
	 * <ws> ::= white space character(s)
	 * <action> ::= "INCLUDE"|"EXCLUDE"
	 * <package> ::= <global_package>|["*"]<name>["*"}|"*"
	 * <global_package> ::= "."
	 * <class> ::= <method> ::= <signature> ::=	 ["*"]<name>["*"}|"*"
	 * 
	 * The main difference in these two specification is in package handling.
	 * Probe kit specification requires named package or "." in place of global package.
	 * Hyades handles package+class as a single literal with wildcards.
	 * Transformation between these two formats is non-trivial and is briefly 
	 * described by the table below:
	 * 
	 *-------------------------------------------------------------------------- 
	 * Hyades					Probe Kit
	 * Package+class			Package			Class                   
	 *-------------------------------------------------------------------------- 
	 * foo						.				foo
	 *-------------------------------------------------------------------------- 
	 * foo.bar					foo				bar
	 *-------------------------------------------------------------------------- 
	 * *foo.bar*				*foo			bar*
	 *-------------------------------------------------------------------------- 
	 * 							*foo.bar*		*
	 *-------------------------------------------------------------------------- 
	 * foo*						foo*			*
	 * 							.				foo*
	 *-------------------------------------------------------------------------- 
	 * *foo						*				*foo
	 *-------------------------------------------------------------------------- 
	 * *foo*					*foo*			*
	 * 							*				*foo*
	 *-------------------------------------------------------------------------- 
	 * *						*				*
	 *-------------------------------------------------------------------------- 
	 * 	
	 */
	private String[] translateFilters(String filterString, String prefix) 
	{
		if (filterString == null) 
			return new String[0];
		if(filterString.endsWith("&"))
		{	// Compensate for the trailing '&'. Somtimes that comes from other modules. 
			filterString = filterString.substring(0, filterString.length()-1);
		}
		String filters[] = filterString.split("[&]");
		Vector rules = new Vector();
		for(int n = 0; n < filters.length; n++)
		{
			// Status: 0-Package&class; 1-Method; 2-Signature; 3-Action
			int nStat = 0;
			// Tokens derived from the filter parsing
			// Package/class parsing can produce one ore two tokens per entry
			StringBuffer rule = new StringBuffer(prefix);
			rule.append(' ');
			String tokenPackage1 = ".", tokenPackage2 = null;
			String tokenClass1 = "*", tokenClass2 = null;
			String tokenMethod = "*";
			String tokenSignature = "*";
			String tokenAction = "EXCLUDE";
			String tokens[] = filters[n].split("[:]");
			for(int t = 0; t < tokens.length; t++)
			{
				String token = tokens[t];
				if( // The 'action' token (always the last one)
					token.compareTo("INCLUDE") == 0
				||  token.compareTo("EXCLUDE") == 0	)
				{
					tokenAction = token;
					break;
				}
				else switch(t)
				{
					case 0:
						// Package & class. 
						// This is the most complex case (see the conversion table in the 
						// method comment
						int indDot = token.lastIndexOf('.');
						int indStar = token.indexOf('*');
						if(indDot != -1)
						{	//Dot(s) present
							// foo.bar
							tokenPackage1 = token.substring(0, indDot);
							tokenClass1 = token.substring(indDot+1);
							if(token.endsWith("*"))
							{   // *foo.bar*
								tokenPackage2 = token;
								tokenClass2 ="*";
							}
						}
						else if(indStar != -1)
						{	//Star(s)present but no dots
							if(token.startsWith("*"))
							{
								if(token.length() > 1 && token.endsWith("*"))
								{   // *foo* (and not just *)
									tokenPackage1 = token;
									tokenClass1 = "*";
									tokenPackage2 = "*";
									tokenClass2 = token;
								}
								else
								{	// *foo
									tokenPackage1 = "*";
									tokenClass1 = token;
								}
							}
							else
							{	// foo*
								tokenPackage1 = token;
								tokenClass1 = "*";
								tokenPackage2 = ".";
								tokenClass2 = token;
							}
						}
						else if(indStar == -1 && indDot == -1)
						{	// No stars and no dots
							// Foo
							tokenClass1 = token;
						}
						break;
					case 1:
						// Method name
						tokenMethod = token;
						break;
					case 2:
						// Signature
						tokenSignature = token;
						break;
					default:
						break;
				}
			}
			// Populate the rules vector
			rule.append(tokenPackage1);
			rule.append(' ');
			rule.append(tokenClass1);
			rule.append(' ');
			rule.append(tokenMethod);
			rule.append(' ');
			rule.append(tokenSignature);
			rule.append(' ');
			rule.append(tokenAction);
			rules.add(rule.toString());
			if(tokenPackage2 != null)
			{	// Second rule present
				rule = new StringBuffer(prefix);
				rule.append(' ');
				rule.append(tokenPackage2);
				rule.append(' ');
				rule.append(tokenClass2);
				rule.append(' ');
				rule.append(tokenMethod);
				rule.append(' ');
				rule.append(tokenSignature);
				rule.append(' ');
				rule.append(tokenAction);
				rules.add(rule.toString());
			}
		}
		// Convert vector to array of strings and return it
		String rulesOut[] = new String[rules.size()];
		for(int i = 0; i < rulesOut.length; i++)
		{
			rulesOut[i] = (String)rules.get(i);
		}
		return rulesOut;
	}

	/**
	 * @param configFilters EList - list of default filters from the WorkBench
	 * @return String with '&' separated filters in the same format
	 * as returned by ProbeLaunchConfigString.getFilters()
	 */
	private String processDefaultFilters(EList configFilters) 
	{
		Iterator itrFilters = configFilters.iterator();
		StringBuffer newFilter = new StringBuffer();
		boolean insertDelimiter = false;
		while(itrFilters.hasNext())
		{
			TRCFilter configFilter = (TRCFilter)itrFilters.next();
			String className = configFilter.getPattern();
			String methodName = configFilter.getOperation();
			String action = configFilter.getType();
			Boolean active = configFilter.getActive();
			if(active.booleanValue())
			{
				if(insertDelimiter)
				{
					newFilter.append('&');
				}
				newFilter.append(className);
				newFilter.append(':');
				newFilter.append(methodName);
				newFilter.append(':');
				newFilter.append(action);
				insertDelimiter = true;
			}
		}
		return newFilter.toString();
	}

	//- Public methods ---------------------------------------------------------

	/**
	 * Deploy a class into the running process over the RAC connection
	 * 
	 * @param probeScriptBuffer - byte array with raw script bytes
	 * 	
	 */
	public void deployProbeScript(byte probeScriptBuffer[]) throws LaunchPadException
	{
		trace("Deploying probescript of length " + probeScriptBuffer.length);
		try 
		{
			probeScriptBuffer = weaveFilters(probeScriptBuffer);
		} 
		catch (UnsupportedEncodingException e) {
			throw new LaunchPadException(e);
		}
		BinaryCustomCommand command = new BinaryCustomCommand();
		byte header[] = RAC_DEPLOY_SCRIPT_COMMAND.getBytes();
		int command_size 	= header.length 			// command header string 
							+ 1 						// trailing 0
							+ 4							// raw script size
							+ probeScriptBuffer.length;	// raw script itself
		// Allocate buffer
		ByteBuffer buffer = ByteBuffer.allocate(command_size);
		// Populate buffer with data
		buffer.put(header).put((byte)0);
		buffer.putInt(probeScriptBuffer.length);
		buffer.put(probeScriptBuffer);
		
		// Prepare RAC command
		command.setData(buffer.array());
		// Invoke RAC command
		try 
		{
			agent.invokeCustomCommand(command);
		} 
		catch (InactiveAgentException e) 
		{
			trace("InactiveAgentException in deployProbeScript");
			throw new LaunchPadException(e);
		}
	}
	
	/**
	 * Deploy a probe script into the running process over the RAC connection
	 * 
	 * @param probeScriptFile
	 */
	public void deployProbeScript(File probeScriptFile) throws LaunchPadException
	{
		try 
		{
			byte probeScriptBuffer[] = readFileIntoBuffer(probeScriptFile);
			deployProbeScript(probeScriptBuffer);
		} 
		catch (FileNotFoundException e) 
		{
			trace("FileNotFoundException in deployProbeScript");
			throw new LaunchPadException(e);
		} 
		catch (IOException e) 
		{
			trace("IOException in deployProbeScript");
			throw new LaunchPadException(e);
		}
	}

	/**
	 * Set the Hyades profiling agent.
	 * Probes will be later deployed to this agent. 
	 * @param agent - Hyades profiling agent
	 */
	public void setAgent(Agent agent)
	{
		this.agent = agent;
	}

	/**
	 * Clear all probe filters.
	 * If no filters are defined by the LaunchPad then the
	 * filter set from the probe script will take effect.
	 * Otherwise filters from the launcher will replace
	 * the original filters in the script.
	 */
	public void clearFilters()
	{
		filters.clear();
	}
	
	/**
	 * Add a filter to the LaunchPad filter list.
	 * These filters will override the original filters from the script.
	 * The filter string is defined in the following format:
	 * filter ::= <pkg_and_class> [":" method [":" signature ]] ":" action
	 * action ::= "INCLUDE"|"EXCLUDE"
	 * pkg_and_class ::= ["*"][name "."] name ["*"]|"*"
	 * method ::= signature ::= ["*"] name ["*"}|"*"
	 * 
	 * @param filterString
	 */
	public void addFilter(String filterString)
	{
		filters.add(filterString);
	}
	
	
	/**
	 * Send a filter to the Agent Extension for using in the agent extension 
	 * callback filtering. Filters are sent as binary custom command.
	 * The command is structered in the following way:
	 * First 4 bytes contain the total length of all filter strings 
	 * including the trailing zero byte. The length is followed by
	 * the strings in utf8 encoding separated by zeros
	 * 
	 * Note: This feature is not used as of today (03/15/05). 
	 * It is reserved for the future if we need a 'Global' filtering
	 * mechanism that applies to all probes simulteniously. 
	 * 
	 * @param filterStrings
	 */
	public void sendFilters(String filterStrings[]) throws LaunchPadException
	{
		try {
			int nFilters = filterStrings.length;
			int commandSize = RAC_SEND_FILTERS_COMMAND.length() // Command length
							+ 1 	// Trailing zero
							+ 4;	// Number of filters (int)
			for (int n = 0; n < filterStrings.length; n++)
			{	// Calculate the buffer size for UTF8 filters
				// in the custom command data
				commandSize += filterStrings[n].getBytes(ENCODING_NAME_UTF8).length + 1;	
			}
			ByteBuffer buffer = ByteBuffer.allocate(commandSize);
			buffer.put(RAC_SEND_FILTERS_COMMAND.getBytes()).put((byte)0);
			buffer.putInt(nFilters);
			for (int n = 0; n < filterStrings.length; n++)
			{
				// Convert filters and copy them into the buffer
				byte filterBytes[] = filterStrings[n].getBytes(ENCODING_NAME_UTF8);
				buffer.put(filterBytes).put((byte)0);
			}
			BinaryCustomCommand command = new BinaryCustomCommand();
			command.setData(buffer.array());
			agent.invokeCustomCommand(command);
		}
		catch (UnsupportedEncodingException e)
		{
			trace("UnsupportedEncodingException in deployClass1");
			throw new LaunchPadException(e);
		}
		catch (InactiveAgentException e) 
		{
			trace("InactiveAgentException in deployClass1");
			throw new LaunchPadException(e);
		}
	}
	
	/**
	 * Deploy a class into the running process over the RAC connection
	 * 
	 * @param classBuffer - array of bytes with the raw class data
	 * @param className - class name
	 * @throws LaunchPadException
	 */
	public void deployClass(byte classBuffer[], String className) throws LaunchPadException
	{
		try {
			byte header[] = RAC_DEPLOY_CLASS_COMMAND.getBytes();
			byte classNameBytes[] = className.getBytes(ENCODING_NAME_UTF8);
			BinaryCustomCommand command = new BinaryCustomCommand();
			int classSize = classBuffer.length;
			int commandSize 	= header.length	// command header string 
								+ 1 							// trailing 0
								+ 4								// raw class data size
								+ classSize						// raw class data
								+ classNameBytes.length   // class name
								+ 1;							// trailing 0
			// Allocate buffer
			ByteBuffer buffer = ByteBuffer.allocate(commandSize);
			int i = 0;
			// Populate buffer with data
			buffer.put(header).put((byte)0);
			buffer.putInt(classSize);
			buffer.put(classBuffer);
			buffer.put(classNameBytes).put((byte)0);
			// Prepare RAC command
			command.setData(buffer.array());
			// Invoke RAC command
			agent.invokeCustomCommand(command);
		} 
		catch (UnsupportedEncodingException e)
		{
			trace("UnsupportedEncodingException in deployClass1");
			throw new LaunchPadException(e);
		}
		catch (InactiveAgentException e) 
		{
			trace("InactiveAgentException in deployClass1");
			throw new LaunchPadException(e);
		}
		
	}
	
	/**
	 * Deploys a class from the supplied file into a running process 
	 * over the RAC connection
	 * 
	 * @param classFile - Class file
	 * @throws LaunchPadException
	 */
	public void deployClass(File classFile) throws LaunchPadException 
	{
		try 
		{
			String className;
			String fileName = classFile.getName();
			int lastSeparator = fileName.lastIndexOf(File.separator);
			className = fileName.substring(lastSeparator + 1);
			className = className.substring(0, className.indexOf(ProbekitConstants.CLASS_FILE_EXT));
			byte classBuffer[] = readFileIntoBuffer(classFile);
			trace("Deploying class file " + fileName + " of length " + classBuffer.length);
			deployClass(classBuffer, className);
		} 
		catch (FileNotFoundException e) 
		{
			trace("FileNotFoundException in deployClass2");
			throw new LaunchPadException(e);
		} 
		catch (IOException e) 
		{
			trace("IOException in deployClass2");
			throw new LaunchPadException(e);
		}
	}

	/**
	 * Constructor
	 * 
	 */
	public LaunchPad()
	{
		filters = new Vector();
	}

	
	/***
	 * This method is called by the Workbench extension point to notify
	 * the launchpad that the agent is active and is about to get resumed.
	 * At this point the launcpad is parsing the launcher options, retrieves
	 * the probe script and probe classes, prepares filters and then deploys 
	 * the script and related classes to the active agent
	 *  
	 * @param agentProxy - TRCAgentProxy delivered by the Workbench extension point
	 * @throws LaunchPadException
	 */
	public void agentActive(TRCAgentProxy agentProxy) throws LaunchPadException
	{
		agent = (Agent) LoadersUtils.locateAgentInstance(agentProxy);
		if(agent == null)
		{
			throw new LaunchPadException("Agent not found");
		}
		EList configs = agentProxy.getConfigurations();
		Iterator itrConfig = configs.iterator();
		
		String rules[];	// vector of string rules. Will be populated from the config
		String probeID;
		ProbeRegistry registry = ProbeRegistry.getRegistry();
		
		// Extract the stuff we need to launch a probe from the configuration:
		// Right now we need the probe filter list and probe registry ID.
		while(itrConfig.hasNext())
		{
			TRCConfiguration config = (TRCConfiguration) itrConfig.next();
			
			String configName = config.getName();
			EList options = config.getOptions();
			Iterator itrOptions = options.iterator();
			
			while(itrOptions.hasNext())
			{
				TRCOption option = (TRCOption) itrOptions.next(); 
				String key = option.getKey();
				if(!key.startsWith(ProbeLaunchConfigString.AGENT_CONFIG_NAME_PREFIX))
				{
					continue;
				}
				String value = option.getValue();
				String probeFilters;
				ProbeLaunchConfigString probeConfig = ProbeLaunchConfigString.fromString(value);

				if(probeConfig.getUseDefaultFilters())
				{
					probeFilters = processDefaultFilters(config.getFilters());
				}
				else
				{
					probeFilters = probeConfig.getFilters();
				}
				rules = translateFilters(probeFilters, OPCODE_FILTER_WITH_BLANK);
				clearFilters();
				for(int i = 0; i < rules.length; i++)
				{
					addFilter(rules[i]);
				}
				// sendFilters(rules);
				if(probeConfig.getType() == ProbeLaunchConfigString.TYPE_REGISTRY_ID)
				{
					probeID = probeConfig.getRegistryId();
					ProbeRegistryEntry registryEntry = registry.lookupById(probeID);
					if(null != registryEntry)
					{
						deployProbe(registryEntry);
					}
				}
				else
				{
					String script = probeConfig.getProbeScript();
					deployProbeScript(script.getBytes());
				}
			}
		}
		
	}
	
	/**
	 * Depolys a probe given a probe registry entry.
	 * This method is usually called when somebody configures an application
	 * to be launched with probes. It can be called more than once if several
	 * probes were selected from the registry in the launch configuration.
	 * 
	 * The right invokation point is the AgentActive callback (after the 
	 * agent was activated but before it processed the resume command.  
	 * 
	 * @param probeRegistryEntry - Probe Registry Entry
	 * @throws LaunchPadException - Launchpad Exception
	 * 
	 */
	public void deployProbe(ProbeRegistryEntry probeRegistryEntry) throws LaunchPadException
	{
		try {
			File probeScript = probeRegistryEntry.getProbescript();
			File probeFiles[] = probeRegistryEntry.getFiles();
			deployProbeScript(probeScript);
			for(int i = 0; i < probeFiles.length; i++)
			{
				if(isClass(probeFiles[i]))
				{
					deployClass(probeFiles[i]);
				}
			}
		} 
		catch (InvalidProbeBundleException ex)
		{
			trace("InvalidProbeBundleException in deployProbe()");
			throw new LaunchPadException("Invalid registry entry:" + probeRegistryEntry.getId());
		}
	}
	
	static void trace(String message)
	{
		if ( ProbekitDebugConfig.TRACE_LAUNCHPAD && message != null &&
				message.length() > 0) {
			System.out.println("ProbeLaunchPad: " + message);
		}
	}

}

//= End of LaunchPad.java ======================================================
