/********************************************************************** 
 * Copyright (c) 2007 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html         
 * 
 * Contributors: 
 * IBM and OCSystems- Initial API and implementation 
 **********************************************************************/ 


package org.eclipse.tptp.platform.probekit.launch.launchpad;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.Vector;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.hyades.trace.internal.ui.TraceConstants;
import org.eclipse.hyades.trace.ui.internal.core.TraceFilterManager;
import org.eclipse.hyades.trace.ui.internal.util.FilterSetElement;
import org.eclipse.hyades.trace.ui.internal.util.FilterTableElement;
import org.eclipse.tptp.platform.execution.client.agent.IAgent;
import org.eclipse.tptp.platform.execution.client.agent.ICollector;
import org.eclipse.tptp.platform.execution.client.core.INode;
import org.eclipse.tptp.platform.execution.exceptions.InactiveAgentException;
import org.eclipse.tptp.platform.execution.exceptions.NotConnectedException;
import org.eclipse.tptp.platform.execution.util.ICommandElement;
import org.eclipse.tptp.platform.execution.util.ICommandHandler;
import org.eclipse.tptp.platform.execution.util.internal.CommandFragment;
import org.eclipse.tptp.platform.probekit.launch.internal.wizard.CustomBase64Encoder;
import org.eclipse.tptp.platform.probekit.registry.ProbeRegistry;
import org.eclipse.tptp.platform.probekit.registry.ProbeRegistryEntry;
import org.eclipse.tptp.platform.probekit.util.InvalidProbeBundleException;
import org.eclipse.tptp.platform.probekit.util.ProbeLaunchConfigString;
import org.eclipse.tptp.trace.ui.internal.launcher.core.LauncherConstants;
import org.eclipse.tptp.trace.ui.internal.launcher.core.LauncherUtility;
/**
 * This class replaces LaunchPad.java for the JVMTI implementation
 * of probekit profiling. After the agent attaches but before it calls
 * RESUME there is a small window of time in which we can send commands to
 * the agent. This class is invoked during that time from the agentActive
 * call to the ProbeControlListener, notified at the start of this crucial
 * time.
 * 
 * This class is used to first retrieve the ProbeID information stored
 * as an attribute in the LaunchConfiguration which is passed in to
 * agentActive. Once we have the ProbeIDs of all the selected probes, we
 * can look up the probe information for these probes stored in the
 * Probe Registry.
 * 
 * Upon retrieving the registry entry, we have to wrap the class file
 * and probescript information as a String using a Base64 encoder so that it
 * can be sent as the commandData in the new execution framework
 * command protocol.
 * 
 * The command creation and invocation of agent.sendCommand() happens during
 * deployClasses() which uses the agent passed in to send a command
 * containing all of the necessary probe file information.
 * 
 * @provisional As of TPTP V4.4.0, this is stable provisional API (see http://www.eclipse.org/tptp/home/documents/process/development/api_contract.html).
 */
public class LaunchPadJVMTI 

{
	/* The active agent which is being used to send Commands */
	private ICollector agent;
	/* A Vector representation of the filters which have been user-defined for this launch configuration*/
	private Vector filters = new Vector();
	/* Private constants*/
	private static final String ENCODING_NAME_UTF8 = "UTF-8";
	private static final String OPCODE_PROBE = "PROBE";
	private static final String OPCODE_RULE = "RULE";
	private static final String OPCODE_RULE_WITH_BLANK = "RULE ";
	private static final String OPCODE_FILTER_WITH_BLANK = "FILTER ";
	private static final String ATTR_FILTER_SET_ID = "org.eclipse.tptp.platform.jvmti.client.ATTR_FILTERSET_ID";
	/**
	 * A method called when the agent gives control to the probekit. 
	 * The purpose of this method is to extract what we need
	 * from the given launch configuration and then use those attributes
	 * to send the correct probe files as well as any user-defined filters to
	 * the probekit agent extension
	 * @param currentAgent - the currently active Agent
	 * @param config - the Launch Configuration
	 * @throws CoreException
	 * @throws LaunchPadException
	 * 
	 */
	public void agentActive(ICollector currentAgent, ILaunchConfiguration config) throws CoreException, LaunchPadException
	{
			// to hold the filter rules
			String rules[] = new String[1000];
			agent= currentAgent;
			Vector registryEntries = new Vector();
			if(agent == null)
			// if the agent is null we want to stop right away
			{
				throw new LaunchPadException("Agent not found");
			}
			
			String probeID;
			ProbeRegistry registry = ProbeRegistry.getRegistry();
			Map m = config.getAttributes();
	
			// Extract the stuff we need to launch a probe from the configuration:
			// Right now we need the probe filter list and probe registry ID.
			String key = " ";
			
			Set keys = m.keySet();
			Iterator iter = keys.iterator();
			while (iter.hasNext())
			{
				key = (String)iter.next();
				Object value = m.get((Object)key);
				
					
				// We are looking for this key: org.eclipse.tptp.platform.probekit.Probespec
				// Initially when we stored the attributes we stored each selected probe as a
				// String with this key as the prefix.
					
				if(!key.startsWith(ProbeLaunchConfigString.AGENT_CONFIG_NAME_PREFIX)&& !(value.equals(" ")))
				{
					continue;
				}
					String probeFilters;
					// important because each time we save the selections we add an empty String to those
					// Probes which are no longer selected (as attributes)
					if (!value.equals((" ")))
					{
						ProbeLaunchConfigString probeConfig = ProbeLaunchConfigString.fromString(value.toString());
			
			
						// Filters section: Need to pass in the user-specified filters (if any).
						// With JVMTI we have the user-selected filters stored as an attribute
						
						
												String defaultFilterSet = " ";
											
												if ( !LauncherUtility.isAutoFilteringCriteria(config) ){
													defaultFilterSet = LauncherConstants.DEFAULT_FILTER_ID;
												}
												else{
													defaultFilterSet = TraceConstants.AUTO_GENERATED_FILTER_SET;
												}
						String filterSetId = config.getAttribute(ATTR_FILTER_SET_ID, defaultFilterSet);						
						FilterSetElement filterSet = (FilterSetElement)TraceFilterManager.getInstance().getFilterSet().get(filterSetId);
						ArrayList filters = filterSet.getChildren();
						probeFilters = processDefaultFilters(filters);
			
						rules = translateFilters(probeFilters, OPCODE_FILTER_WITH_BLANK);
						clearFilters();
						for(int i = 0; i < rules.length; i++)
						{
							addFilter(rules[i]);
						}
			
						if(probeConfig.getType() == ProbeLaunchConfigString.TYPE_REGISTRY_ID)
						{
							probeID = probeConfig.getRegistryId();
							ProbeRegistryEntry registryEntry = registry.lookupById(probeID);
							if(null != registryEntry)
							{
								registryEntries.add(registryEntry);
								deployProbes(registryEntry, registryEntries.size()-1);
							}
						}
					}// end if
				}// end while
				
		}// end method
	
	
	
	/**
	 * A method to translate the filters so that we can place them into the probescript
	 * @param filterString - the initial filter String from the Launch Configuration
	 * @param prefix 
	 * @return String[] - array of RULE statements for probe script
	 * @see org.eclipse.tptp.platform.probekit.launch.launchpad.LaunchPad#translateFilters(String, String)
	 */
	private String[] translateFilters(String filterString, String prefix) 
	{
		if (filterString == null) 
			return new String[0];
		if(filterString.endsWith("&"))
		{	// Compensate for the trailing '&'. Somtimes that comes from other modules. 
			filterString = filterString.substring(0, filterString.length()-1);
		}
		String filters[] = filterString.split("[&]");
		Vector rules = new Vector();
		for(int n = 0; n < filters.length; n++)
		{
			// Status: 0-Package&class; 1-Method; 2-Signature; 3-Action
			int nStat = 0;
			// Tokens derived from the filter parsing
			// Package/class parsing can produce one ore two tokens per entry
			StringBuffer rule = new StringBuffer(prefix);
			rule.append(' ');
			String tokenPackage1 = ".", tokenPackage2 = null;
			String tokenClass1 = "*", tokenClass2 = null;
			String tokenMethod = "*";
			String tokenSignature = "*";
			String tokenAction = "EXCLUDE";
			String tokens[] = filters[n].split("[:]");
			for(int t = 0; t < tokens.length; t++)
			{
				String token = tokens[t];
				if( // The 'action' token (always the last one)
					token.compareTo("INCLUDE") == 0
				||  token.compareTo("EXCLUDE") == 0	)
				{
					tokenAction = token;
					break;
				}
				else switch(t)
				{
					case 0:
						// Package & class. 
						// This is the most complex case (see the conversion table in the 
						// method comment
						int indDot = token.lastIndexOf('.');
						int indStar = token.indexOf('*');
						if(indDot != -1)
						{	//Dot(s) present
							// foo.bar
							tokenPackage1 = token.substring(0, indDot);
							tokenClass1 = token.substring(indDot+1);
							if(token.endsWith("*"))
							{   // *foo.bar*
								tokenPackage2 = token;
								tokenClass2 ="*";
							}
						}
						else if(indStar != -1)
						{	//Star(s)present but no dots
							if(token.startsWith("*"))
							{
								if(token.length() > 1 && token.endsWith("*"))
								{   // *foo* (and not just *)
									tokenPackage1 = token;
									tokenClass1 = "*";
									tokenPackage2 = "*";
									tokenClass2 = token;
								}
								else
								{	// *foo
									tokenPackage1 = "*";
									tokenClass1 = token;
								}
							}
							else
							{	// foo*
								tokenPackage1 = token;
								tokenClass1 = "*";
								tokenPackage2 = ".";
								tokenClass2 = token;
							}
						}
						else if(indStar == -1 && indDot == -1)
						{	// No stars and no dots
							// Foo
							tokenClass1 = token;
						}
						break;
					case 1:
						// Method name
						tokenMethod = token;
						break;
					case 2:
						// Signature
						tokenSignature = token;
						break;
					default:
						break;
				}
			}
			// Populate the rules vector
			rule.append(tokenPackage1);
			rule.append(' ');
			rule.append(tokenClass1);
			rule.append(' ');
			rule.append(tokenMethod);
			rule.append(' ');
			rule.append(tokenSignature);
			rule.append(' ');
			rule.append(tokenAction);
			rules.add(rule.toString());
			if(tokenPackage2 != null)
			{	// Second rule present
				rule = new StringBuffer(prefix);
				rule.append(' ');
				rule.append(tokenPackage2);
				rule.append(' ');
				rule.append(tokenClass2);
				rule.append(' ');
				rule.append(tokenMethod);
				rule.append(' ');
				rule.append(tokenSignature);
				rule.append(' ');
				rule.append(tokenAction);
				rules.add(rule.toString());
			}
		}
		// Convert vector to array of strings and return it
		String rulesOut[] = new String[rules.size()];
		for(int i = 0; i < rulesOut.length; i++)
		{
			rulesOut[i] = (String)rules.get(i);
		}
		return rulesOut;
	}

	
	/**
	 * A private helper method to process the filters that the user has selected from
	 * the Launch Configuration UI and parse them into a String
	 * which can be translated to be appended to the probescript
	 * @param configFilters - an ArrayList of the selected filters
	 * @return String with properly formatted filters
	 * @see org.eclipse.tptp.platform.probekit.launch.launchpad.LaunchPad#processDefaultFilters(ArrayList configFilters)
	 */
	private String processDefaultFilters(ArrayList configFilters) 
	{
		Iterator itrFilters = configFilters.iterator();
		StringBuffer newFilter = new StringBuffer();
		boolean insertDelimiter = false;
		while(itrFilters.hasNext())
		{
			FilterTableElement configFilter = (FilterTableElement)itrFilters.next();
			String className = configFilter.getText();
			String methodName = configFilter.getMethod();
			String action = configFilter.getVisibility();
			boolean active = true;
			if(active)
			{
				if(insertDelimiter)
				{
					newFilter.append('&');
				}
				newFilter.append(className);
				newFilter.append(':');
				newFilter.append(methodName);
				newFilter.append(':');
				newFilter.append(action);
				insertDelimiter = true;
			}
		}
		return newFilter.toString();
	}
	
	/**
	 * Begins the process of deploying all of the user-defined probes by first
	 * looping through the selected Probe Registry entries and retrieving the
	 * probe classPaths. It then sends the classpath information to be put into
	 * array format and eventually deployed.
	 * 
	 * @param RegsitryEntries - Vector of selected probes' Regsitry Entries
	 * @param probeNumber
	 * 
	 */
	public void deployProbes(ProbeRegistryEntry  probeRegistryEntry, int probeNumber) throws LaunchPadException
	{
		try 
		{
			
			Vector classPathVector = new Vector();
			
					File probeScript = probeRegistryEntry.getProbescript();
			
					File probeFiles[] = probeRegistryEntry.getFiles();
					String probeScriptPath = probeScript.getAbsolutePath();
					classPathVector.add(probeScriptPath);
			
					// add the probe script path first
			
					for(int j = 0; j < probeFiles.length; j++)
					{
						if(LaunchPadUtils.isClass(probeFiles[j]))
							// then we want to add the classPath to probeClassPath
							// so that we have the probescript followed by
							// the probe class
							// followed by the second class file
						{
							String filePath = ((File)probeFiles[j]).getAbsolutePath();
							classPathVector.add(filePath);
					
						}
				
					}
			
				
			
			prepareFileArrays(classPathVector.toArray(), agent, probeNumber);
		}
		
		catch (InvalidProbeBundleException ex)
		{
			LaunchPadUtils.trace("InvalidProbeBundleException in deployProbe()");
			throw new LaunchPadException("Invalid registry entry" );
		}
	}
	

	/**
	 * This method is used to read in the class files and create two
	 * arrays, one of the class files themselves and one of the file names
	 * to pass to the method which sends the probekit commands to the agent.
	 * 
	 * @param probeClassPath - Array of the classPaths for selected probe files
	 * @param agent- the active agent we will use to send commands
	 * @param probeNumber
	 * 
	 */
	
	public void prepareFileArrays(Object [] probeClassPath, ICollector agent, int probeNumber)throws LaunchPadException
	{
		try{
			Vector classNames = new Vector();
			Vector bytes = new Vector();
			for (int i=0; i< probeClassPath.length; i++){
				String path = (String) probeClassPath[i];
				if (path.endsWith(".probescript")){
					File scriptf = new File(path);
					byte[] script = weaveFilters(LaunchPadUtils.readFileIntoBuffer(scriptf));
					classNames.add(scriptf.getName());
					bytes.add(script);
				}
			
				else
			{
				if (probeClassPath[i] instanceof String){
					String classPath = (String) probeClassPath[i];
					// and all of the entries in probeClassPath should be Strings
					File f = new File(classPath);
					classNames.add(f.getName());
					bytes.add(LaunchPadUtils.readFileIntoBuffer(f));
				}
			}
			}
			
			deployProbeClasses(bytes, classNames, agent, probeNumber);
		}
		catch(FileNotFoundException e)
		{
			LaunchPadUtils.trace("FileNotFound in LaunchPadJVMTI");
			throw new LaunchPadException(e);
		}
		catch(IOException e)
		{
			LaunchPadUtils.trace("IOException in LaunchPadJVMTI");
			throw new LaunchPadException(e);
		}
	}// end method
	
	
	 /**
	  * A method to send the probe classes to the Agent using the new 
	  * execution framework command protocol
	  * 
	  * @param bytes - The Vector of bytes representing the probe file classes (probescript etc.)
	  * @param classNames - A Vector of the classNames for each probe
	  * @param agent - The currently active agent
	  * @throws LaunchPadException
	  * 
	  */
	 public static void deployProbeClasses(Vector bytes, Vector classNames, IAgent agent, int probeNumber) throws LaunchPadException
	 {
	   try 
	   {		 		 		   
	 		   int count = 4; // magic
	 		   for (int i = 0; i < bytes.size(); i++) 
	 		   {
	 		 		   String className = (String)classNames.get(i);
	 		 		   byte[] classBytes =(byte[]) bytes.get(i);
	 		 		   byte classNameBytes[] = className.getBytes(ENCODING_NAME_UTF8);
	 		 		   count += 4 + classNameBytes.length + 4 + classBytes.length;
	 		   }
	 		   
	 		   // Allocate buffer
	 		   ByteBuffer buffer = ByteBuffer.allocate(count); // tmp
	 		   buffer.putInt(0xDECAECC0);
	 		   
	 		   for (int i = 0; i < bytes.size(); i++) 
	 		   {
	 			   	   String className = (String)classNames.get(i);
	 		 		   byte[] classBytes =(byte[]) bytes.get(i);
	 		 		   byte classNameBytes[] = className.getBytes(ENCODING_NAME_UTF8);
	 		 		   /*
	 		 		 int commandSize =4 + // magic
	 		 		 		 		 4 + // className length
	 		 		 		 		 classNameBytes.length +
	 		 		 		 		 4 + // class size
	 		 		 		 		 classBytes.length
	 		 		 		 		 ;
	 		 		    */

	 		 		   // Populate buffer with data		 		 		 		 
	 		 		   buffer.putInt(classNameBytes.length).put(classNameBytes);
	 		 		   buffer.putInt(classBytes.length).put(classBytes);
	 		 		 
	 		   }
	 		   
	 		   // Prepare RAC command
	 		   String commandData = new CustomBase64Encoder().encode(buffer.array());
	 		 
	 		   CommandFragment command = new CommandFragment();
	 		   command.setCommand(
	 		 		 		   "<applyOptions iid='org.eclipse.tptp.jvmti'>" +
	 		 		 		   "<option><name>PROBEKIT_DATA_" + String.valueOf(probeNumber)+"</name>" + 
	 		 		 		   "<value>" + commandData + "</value></option></applyOptions>");

	 		   // Invoke RAC command- this is using the new sendCommand from the new execution framework
	 		   agent.sendCommand(command.buildCommand(), new ICommandHandler() 
	 		   {
	 		 		   public void incomingCommand(INode node, ICommandElement element) 
	 		 		   {
	 		 		 		   //no handler necessary
	 		 		   }
	 		   }
	 		   );
	   }// end try
	   catch(InactiveAgentException e)
	   {
		   LaunchPadUtils.trace("InactiveAgentException in LaunchPadJVMTI");
			throw new LaunchPadException(e);
	   }
	   catch (NotConnectedException e)
	   {
		   LaunchPadUtils.trace("NotConnectedException in LaunchPadJVMTI");
		   	throw new LaunchPadException(e);
	   }
	   catch (UnsupportedEncodingException e)
	   {
		   LaunchPadUtils.trace("UnsupportedEncodingException in LaunchPadJVMTI");
			throw new LaunchPadException(e);
	   }
	 }// end method deployProbeClasses	
	 
	 /**
	  * 
	  * @param script -the probescript
	  * @return - byte[] representing the probescript appended with the new filters
	  * @throws UnsupportedEncodingException
	  * @see org.eclipse.tptp.platform.probekit.launch.launchpad.LaunchPad#weaveFilters(byte []script)
	  */
		private byte[] weaveFilters(byte script[]) throws UnsupportedEncodingException
		{
			if(filters.size() == 0) {
				// Nothing to do: just use the original filters from the script
				// This case is unlikely: it means there are no filters at all selected in the UI.
				return script;
			}
			
			// Step one: convert the original probescript into an array of Strings.
			String[] probescriptAsStrings = convertBytesToStrings(script); 

			// Step two: scan for PROBE sections without RULEs.
			// (See inside for Step three.)
			boolean seenAnyProbeLinesYet = false;
			boolean seenAnyRulesYet = false;
			Vector newProbescriptStringVector = new Vector();
			Vector filtersAsRules = createRulesStringVector();
			
			for (int i = 0; i < probescriptAsStrings.length; i++) {
				String opcode = getFirstWord(probescriptAsStrings[i]);
				if (opcode.equalsIgnoreCase(OPCODE_PROBE)) {
					// Found a PROBE line, marking the end of the previous PROBE.
					// Step three: if there was a previous PROBE and it had no RULEs, 
					// append the FILTERs as RULEs.
					if (seenAnyProbeLinesYet) {
						if (!seenAnyRulesYet) {
							newProbescriptStringVector.addAll(filtersAsRules);
						}
					}
					seenAnyProbeLinesYet = true;
					seenAnyRulesYet = false;
				}
				else if (opcode.equalsIgnoreCase(OPCODE_RULE)) {
					// Remember we've seen at least one RULE in the current PROBE.
					seenAnyRulesYet = true;
				}
				// Copy this original probescript command to the new probescript.
				newProbescriptStringVector.add(probescriptAsStrings[i]);
			}
			
			// We hit the end of the script: take care of the last PROBE section in the script.
			// If the last section didn't have any RULEs in it, append the FILTERs as RULEs.
			if (seenAnyProbeLinesYet && !seenAnyRulesYet) {
				newProbescriptStringVector.addAll(filtersAsRules);
			}
			
			// Step four: append the strings in this.filters as FILTER statements,
			// to implement two-tier filtering for pretargeted probes being sent to new BCI engines.
			newProbescriptStringVector.addAll(filters);

			// Final step: turn the new probescript string vector back into a byte array.
			
			// Compute the number of bytes needed to hold the new script
			Iterator itr = newProbescriptStringVector.iterator();
			int size = 0; 
			while(itr.hasNext())
			{
				String next = (String)itr.next();
				size += next.getBytes(ENCODING_NAME_UTF8).length + 1;
			}
			
			// Create a ByteBuffer for the new script
			ByteBuffer buf = ByteBuffer.allocate(size);
			itr = newProbescriptStringVector.iterator();
			while(itr.hasNext())
			{
				String next = (String)itr.next();
				buf.put(next.getBytes(ENCODING_NAME_UTF8));
				buf.put((byte)0);
			}
			return buf.array();
		}
		
		private String[] convertBytesToStrings(byte[] bytes) throws UnsupportedEncodingException {
			String oneBigString = new String(bytes, ENCODING_NAME_UTF8);
			String[] result = oneBigString.split("[\u0000\n\r\u0085\u2028\u2029]");
			return result;
		}
		
		/**
		 * A helper method to create the rules from the given filters
		 * @return - Vector of properly formatted Strings
		 * @see org.eclipse.tptp.platform.probekit.launch.launchpad.LaunchPad#createRulesStringVector()
		 */
		private Vector createRulesStringVector() {
			Vector result = new Vector();
			for (int i = 0; i < this.filters.size(); i++) {
				String s = (String)this.filters.elementAt(i);
				if (s.startsWith(OPCODE_FILTER_WITH_BLANK)) {
					s = OPCODE_RULE_WITH_BLANK + s.substring(OPCODE_FILTER_WITH_BLANK.length());
				}
				else 
				{
					// This function expects to operate on strings
					// created elsewhere in this class where every one 
					// starts with OPCODE_FILTER_WITH_BLANK ... It's a surprise
					// to find a string that doesn't start that way,
					// but we'd better leave it alone.
				}
				result.add(s);
			}
			return result;
		}
		
		/**
		 * A helper method to return the first "word" of a String
		 * @param input the full String
		 * @return - String representing the first word
		 * @see org.eclipse.tptp.platform.probekit.launch.launchpad.LaunchPad#getFirstWord(String)
		 */
		private String getFirstWord(String input) {
			String[] splitResult = input.trim().split("\\s+", 2);
			return splitResult[0];
		}
		
		/**
		 * Wipes out all of the filters in the global filters Vector
		 * @see org.eclipse.tptp.platform.probekit.launch.launchpad.LaunchPad#clearFilters()
		 * 
		 */
		public void clearFilters()
		{
			filters.clear();
		}
		
		/**
		 * Add a filter to the LaunchPad filter list.
		 * These filters will override the original filters from the script.
		 *
		 * @param filterString
		 * @see org.eclipse.tptp.platform.probekit.launch.launchpad.LaunchPad#addFilter(String)
		 * 
		 */
		public void addFilter(String filterString)
		{
			filters.add(filterString);
		}
		
		/**
		 * Set the TPTP profiling agent.
		 * Probes will be later deployed to this agent. 
		 * @param agent - The profiling agent
		 */
		public void setAgent(ICollector agent)
		{
			this.agent = agent;
		}
	 
}// end class
	

			
		
		
	

