/*****************************************************************************
 * Copyright (c) 2008, 2010 Intel Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Intel Corporation - Initial API and implementation
 *    Ruslan A. Scherbakov, Intel - Initial API and implementation
 *
 * $Id: AllocationDetailsView.java,v 1.9 2010/06/25 21:05:01 jwest Exp $ 
 *****************************************************************************/

package org.eclipse.tptp.trace.jvmti.internal.client.views;

import java.text.NumberFormat;
import java.util.HashMap;
import java.util.Map;

import org.eclipse.emf.common.util.EList;
import org.eclipse.hyades.models.trace.TRCClass;
import org.eclipse.hyades.models.trace.TRCFullTraceObject;
import org.eclipse.hyades.models.trace.TRCMethod;
import org.eclipse.hyades.models.trace.TRCObjectAllocationAnnotation;
import org.eclipse.hyades.trace.ui.TraceViewerPage;
import org.eclipse.hyades.trace.views.internal.TraceUIMessages;
import org.eclipse.hyades.trace.views.util.internal.ColumnData;
import org.eclipse.hyades.ui.provisional.context.IContextLabelFormatProvider;
import org.eclipse.hyades.ui.util.GridUtil;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.graphics.FontData;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.tptp.trace.jvmti.internal.client.AllocationSite;
import org.eclipse.tptp.trace.jvmti.internal.client.context.TIContextAttributes;
import org.eclipse.tptp.trace.jvmti.internal.client.context.TIContextProvider;
import org.eclipse.tptp.trace.jvmti.internal.client.widgets.Utils;
import org.eclipse.ui.forms.IFormColors;

/**
 * Allocation Details view provides allocation info for objects
 * of specified type.
 * 
 * @author Ruslan A. Scherbakov
 *
 */
public class AllocationDetailsView extends BaseStatisticView {

	/**
	 * Column constants ids.
	 */
	static public final int AD_COL_METHOD_NAME = 0;
	static public final int AD_COL_LINE_NO = 1;
	static public final int AD_COL_CLASS_NAME = 2;
	static public final int AD_COL_PACKAGE_NAME = 3;
	static public final int AD_COL_LIVE_INST = 4;
	static public final int AD_COL_LIVE_SIZE = 5;
	static public final int AD_COL_TOTAL_INST = 6;
	static public final int AD_COL_TOTAL_SIZE = 7;
	static public final int AD_COL_AVG_AGE = 8;

	private Map _allocSites = new HashMap();
	private TRCClass _class;
	private int _nextIndex;
	private Label _label;
	AllocationSite _allocAll = new AllocationSite();
	static final Object _emptyArr[] = new Object[0];

	protected AllocationDetailsView(Composite parent, TraceViewerPage page) {
		super(parent, page);
	}

	protected String getColumnsPreferencesKey() {
		return "org.eclipse.hyades.trace.views.statistic.allocdetails04";
	}

	protected String getViewTypeStr() { 
		return "viewoption.allocdetails";
	}
	
	protected void createLabel(Composite viewContainer) {
		_label = _toolkit.createLabel(viewContainer, null);
		Font font = _label.getFont();
		FontData fd = font.getFontData()[0];
		font = new Font(null, new FontData(fd.getName(), fd.getHeight() * 3 / 2, fd.getStyle() | SWT.BOLD));
		_label.setFont(font);
		_label.setBackground(viewContainer.getBackground());
		_label.setForeground(_toolkit.getColors().getColor(IFormColors.TITLE));
		_label.setLayoutData(GridUtil.createHorizontalFill());
	}

	protected String getDefaultColumnsTemplate() {
		return ""
			+ TIContextAttributes.METHOD_NAME + ":" + AD_COL_METHOD_NAME + ":" + String.valueOf(ColumnData.NONDELETABLE | ColumnData.IS_VISIBLE | ColumnData.NONMOVABLE) + ":left:150,"
			+ TIContextAttributes.METHOD_LINE_NUMBER + ":" + AD_COL_LINE_NO + ":" + String.valueOf(ColumnData.IS_VISIBLE) + ":right:80,"
			+ TIContextAttributes.CLASS_NAME + ":" + AD_COL_CLASS_NAME + ":" + String.valueOf(ColumnData.IS_VISIBLE) + ":left:150,"
			+ TIContextAttributes.PACKAGE_NAME + ":" + AD_COL_PACKAGE_NAME + ":" + String.valueOf(ColumnData.IS_VISIBLE) + ":left:120,"
			+ TIContextAttributes.CLASS_LIVE_INST + ":" + AD_COL_LIVE_INST + ":" + String.valueOf(ColumnData.IS_VISIBLE) + ":right:80,"
			+ TIContextAttributes.CLASS_ACTIVE_SIZE + ":" + AD_COL_LIVE_SIZE + ":" + String.valueOf(ColumnData.IS_VISIBLE) + ":right:80,"
			+ TIContextAttributes.CLASS_TOTAL_INST + ":" + AD_COL_TOTAL_INST + ":" + String.valueOf(ColumnData.IS_VISIBLE) + ":right:80,"
			+ TIContextAttributes.CLASS_TOTAL_SIZE + ":" + AD_COL_TOTAL_SIZE + ":" + String.valueOf(ColumnData.IS_VISIBLE) + ":right:80,"
			+ TIContextAttributes.CLASS_AVG_AGE + ":" + AD_COL_AVG_AGE + ":" + String.valueOf(ColumnData.IS_VISIBLE) + ":right:80"
			//+ TIContextAttributes.CLASS_GENERATIONS + ":" + 9 + ":" + String.valueOf(ColumnData.IS_VISIBLE) + ":right:80"
			;
	}
	
	private NumberFormat _commaNumFormat = NumberFormat.getInstance();

	private String formatNumberWCommas(long number) {
		return _commaNumFormat.format(number);		
	}

	public String getColumnText(Object element, int columnIndex) {
		StatisticViewColumnInfo info = StatisticViewColumnInfo
				.getStatisticTableColumnInfo(getTree().getColumn(columnIndex));
		int pos = info.getColumnData().getInitalPos();
		String label = "";
		if (!(element instanceof AllocationSite))
			return label;
		boolean noperc = !isShowPercent();
		AllocationSite allocSite = (AllocationSite) element;
		switch (pos) {
		case AD_COL_METHOD_NAME:
			label = allocSite.getMethodName();
			if (null == label)
				label = "?";
			break;
		case AD_COL_CLASS_NAME:
			label = allocSite.getClassName();
			break;
		case AD_COL_PACKAGE_NAME:
			label = allocSite.getPackageName();
			if (label != null && label.length() == 0)
				label = TraceUIMessages._87;
			break;
		case AD_COL_LINE_NO:
			label = String.valueOf(allocSite._lineNo);
			break;
		case AD_COL_TOTAL_INST:
			if (noperc)
				label = formatNumberWCommas(allocSite._total);
			else
				label = Utils.formatPercent(allocSite._total, _allocAll._total);
			break;
		case AD_COL_TOTAL_SIZE:
			if (noperc)
				label = formatNumberWCommas(allocSite._totalSize); 
			else
				label = Utils.formatPercent(allocSite._totalSize, _allocAll._totalSize);
			break;
		case AD_COL_LIVE_INST:
			if (noperc) 
				label = formatNumberWCommas(allocSite._total - allocSite._collected);
			else
				label = Utils.formatPercent(allocSite._total - allocSite._collected,
						_allocAll._total - _allocAll._collected);
			break;
		case AD_COL_LIVE_SIZE:
			if (noperc)
				label = formatNumberWCommas(allocSite._totalSize - allocSite._collectedSize);
			else
				label = Utils.formatPercent(allocSite._totalSize - allocSite._collectedSize,
						_allocAll._totalSize - _allocAll._collectedSize);
			break;
		case AD_COL_AVG_AGE:
			if (noperc) {
				double age = 0;
				if (allocSite._total > 0) {
					age = (double) allocSite._sumAges / allocSite._total;
					label = String.valueOf(age);
				}
				IContextLabelFormatProvider formatter = TIContextProvider.instance().getContextLabelFormatProvider(TIContextAttributes.CALL_SITE_AVG_AGE);
				label = formatter.getDisplayStringFromElement(new Double(age), null, IContextLabelFormatProvider.MODE_COLUMN_CONTENT);
			} else {
				label = Utils.formatPercent(allocSite._sumAges, _allocAll._sumAges);
			}
			break;
		}
		if (null == label)
			label = "";
		return label;
	}

	public Image getColumnImage(Object element, int columnIndex) {
		StatisticViewColumnInfo info = StatisticViewColumnInfo.getStatisticTableColumnInfo(getTree().getColumn(columnIndex));
		int pos = info.getColumnData().getInitalPos();
		Image image = null;
		if (!(element instanceof AllocationSite))
			return image;
		AllocationSite allocSite = (AllocationSite) element;
		String name = null;
		switch (pos) {
		case AD_COL_METHOD_NAME:
			name = allocSite.getMethodName();
			if (name != null) {
				IContextLabelFormatProvider formatter = TIContextProvider.instance().getContextLabelFormatProvider(TIContextAttributes.METHOD_NAME);
				image = formatter.getDisplayImageByElement(name, null, IContextLabelFormatProvider.MODE_COLUMN_CONTENT);
			}
			break;
		case AD_COL_CLASS_NAME:
			name = allocSite.getClassName();
			if (name != null) {
				IContextLabelFormatProvider formatter = TIContextProvider.instance().getContextLabelFormatProvider(TIContextAttributes.CLASS_NAME);
				image = formatter.getDisplayImageByElement(name, null, IContextLabelFormatProvider.MODE_COLUMN_CONTENT);
			}
			break;
		case AD_COL_PACKAGE_NAME:
			name = allocSite.getPackageName();
			if (name != null) {
				IContextLabelFormatProvider formatter = TIContextProvider.instance().getContextLabelFormatProvider(TIContextAttributes.PACKAGE_NAME);
				image = formatter.getDisplayImageByElement(name, null, IContextLabelFormatProvider.MODE_COLUMN_CONTENT);
			}
			break;
		}
		return image;
	}
	
	public int compareElements(Object e1, Object e2, int pos, boolean isDelta) {
		int ret = 0;
		if (!(e1 instanceof AllocationSite && e2 instanceof AllocationSite))
			return ret;
		AllocationSite a1 = (AllocationSite) e1;
		AllocationSite a2 = (AllocationSite) e2;
		switch (pos) {
		case AD_COL_PACKAGE_NAME:
			if (0 == ret) {
				ret = Utils.compare(a1.getPackageName(), a2.getPackageName());
			}
			//break;
		case AD_COL_CLASS_NAME:
			if (0 == ret) {
				ret = Utils.compare(a1.getClassName(), a2.getClassName());
			}
			//break;
		case AD_COL_METHOD_NAME:
			if (0 == ret) {
				ret = Utils.compare(a1.getMethodName(), a2.getMethodName());
			}
			//break;
		case AD_COL_LINE_NO:
			if (0 == ret) {
				ret = a1._lineNo - a2._lineNo;
			}
			break;
		case AD_COL_TOTAL_INST:
			ret = a1._total - a2._total;
			break;
		case AD_COL_TOTAL_SIZE:
			ret = a1._totalSize - a2._totalSize;
			break;
		case AD_COL_LIVE_INST:
			ret = (a1._total - a1._collected) - (a2._total - a2._collected);
			break;
		case AD_COL_LIVE_SIZE:
			ret = (a1._totalSize - a1._collectedSize) - (a2._totalSize - a2._collectedSize);
			break;
		case AD_COL_AVG_AGE:
			ret = a1._sumAges - a2._sumAges;
			break;
		}
		return ret;
	}

	public Object[] getElements(Object inputElement) {
		updateData();
		return _allocSites.values().toArray();
	}

	public Object[] getChildren(Object parentElement) {
		return null;
	}

	public boolean hasChildren(Object element) {
		return false;
	}

	protected StatisticSorter getViewerSorterInstance() {
		return new StatisticSorter() {
			public int compare(Viewer viewer, Object e1, Object e2) {
				return _sortSequence * compareElements(e1, e2, _pos, _info.isDeltaColumn());
			}
		};
	}

	protected void showAllocDetails(Object obj) {
		if (obj == null || obj instanceof TRCClass) {
			_class = (TRCClass) obj;
			_nextIndex = 0;
			_allocSites.clear();
			_allocAll.clear();
			String text = Utils.getPackageClassName(_class);
			if (text.length() > 0)
				text = UIMessages.MEM_ALLOCATION_DETAILS_FOR + " '" + text + "'";
			_label.setText(text);
		}
		update();
	}

	void updateData() {
		_nextIndex = 0;
		_allocSites.clear();
		_allocAll.clear();
		if (null == _class)
			return;
		TRCObjectAllocationAnnotation aa = Utils.getAllocationAnnotation(_class);
		if (null == aa)
			return;
		EList listGC = _class.getPackage().getProcess().getGcEvents();
		EList lineNumbers = aa.getLineNumbers();
		EList methods = aa.getMethods();
		EList objects = _class.getObjects();
		int lineNoLen = lineNumbers.size();
		int methodsLen = methods.size();
		int objectsLen = objects.size();
		// gather alloc sites by line No
		for (int i = _nextIndex; i < lineNoLen; i++) {
			Integer lineNo = (Integer) lineNumbers.get(i);
			TRCMethod method = null;
			long id = lineNo.intValue();
			if (i < methodsLen) {
				method = (TRCMethod) methods.get(i);
				id += method.getId() << 20;
			}
			Long siteId = new Long(id);
			AllocationSite allocSite = (AllocationSite) _allocSites.get(siteId);
			if (null == allocSite) {
				int line = lineNo.intValue();
				// find corresponding method
				allocSite = new AllocationSite(line, method);
				_allocSites.put(siteId, allocSite);
			}
			int j = i + i + 1;
			if (j < objectsLen && objects.get(j) instanceof TRCFullTraceObject) {
				TRCFullTraceObject trObj = (TRCFullTraceObject) objects.get(j);
				int objSize = trObj.getSize();
				if (trObj.getCollectTime() > 0) {
					allocSite._collected++;
					allocSite._collectedSize += objSize;
					_allocAll._collected++;
					_allocAll._collectedSize += objSize;
				}
				int age = Utils.getObjAge(trObj, listGC);
				allocSite._totalSize += objSize;
				allocSite._sumAges += age;
				allocSite._total++;
				_allocAll._totalSize += objSize;
				_allocAll._sumAges += age;
				_allocAll._total++;
			}
		}
		_nextIndex = lineNoLen;
	}

	public void widgetDefaultSelected(SelectionEvent event) {
		if (event.item.getData() instanceof AllocationSite) {
			AllocationSite allocSite = (AllocationSite) event.item.getData();
			String pattern = Utils.composeMethodName(allocSite);
			Utils.openSource(pattern);
		}
	}
}
