/*****************************************************************************
 * Copyright (c) 2007, 2010 Intel Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Intel Corporation - Initial API and implementation
 *    Ruslan A. Scherbakov, Intel - Initial API and implementation
 *
 * $Id: BaseMemoryStatisticView.java,v 1.13 2010/12/21 21:41:22 mreid Exp $ 
 *****************************************************************************/

package org.eclipse.tptp.trace.jvmti.internal.client.views;

import java.util.ArrayList;
import java.util.Hashtable;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import org.eclipse.emf.common.util.EList;
import org.eclipse.hyades.models.hierarchy.extensions.SimpleSearchQuery;
import org.eclipse.hyades.models.trace.TRCClass;
import org.eclipse.hyades.models.trace.TRCFullTraceObject;
import org.eclipse.hyades.models.trace.TRCPackage;
import org.eclipse.hyades.models.trace.impl.TRCClassImpl;
import org.eclipse.hyades.models.trace.impl.TRCPackageImpl;
import org.eclipse.hyades.models.trace.impl.TRCClassImpl.TRCClassSnapshot;
import org.eclipse.hyades.models.trace.impl.TRCPackageImpl.TRCPackageSnapshot;
import org.eclipse.hyades.trace.ui.FilterTraceViewer;
import org.eclipse.hyades.trace.ui.TraceViewer;
import org.eclipse.hyades.trace.ui.TraceViewerPage;
import org.eclipse.hyades.trace.ui.internal.util.PerftraceUtil;
import org.eclipse.hyades.trace.views.adapter.internal.DynamicFilter;
import org.eclipse.hyades.trace.views.internal.FilteringUtil;
import org.eclipse.hyades.trace.views.internal.view.columnlabels.ColumnDisplayInfo;
import org.eclipse.hyades.trace.views.internal.view.columnlabels.ColumnLabelAdapter;
import org.eclipse.hyades.trace.views.util.internal.ColumnData;
import org.eclipse.hyades.ui.filters.internal.actions.FiltersEditorAction;
import org.eclipse.hyades.ui.filters.internal.util.FilterInformationManager;
import org.eclipse.hyades.ui.util.GridUtil;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.TreePath;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.graphics.FontData;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Link;
import org.eclipse.tptp.platform.common.ui.trace.internal.CommonUITraceMessages;
import org.eclipse.tptp.trace.jvmti.internal.client.context.TIContextAttributes;
import org.eclipse.tptp.trace.jvmti.internal.client.context.TIContextUpdaterHelper;
import org.eclipse.tptp.trace.jvmti.internal.client.views.columnlabels.LiveSizeLabelAdapter;
import org.eclipse.tptp.trace.jvmti.internal.client.views.columnlabels.AvgAgeLabelAdapter;
import org.eclipse.tptp.trace.jvmti.internal.client.views.columnlabels.ClassPackageNameLabelAdapter;
import org.eclipse.tptp.trace.jvmti.internal.client.views.columnlabels.LiveInstancesLabelAdapter;
import org.eclipse.tptp.trace.jvmti.internal.client.views.columnlabels.PackageNameLabelAdapter;
import org.eclipse.tptp.trace.jvmti.internal.client.views.columnlabels.TotalInstancesLabelAdapter;
import org.eclipse.tptp.trace.jvmti.internal.client.views.columnlabels.TotalSizeLabelAdapter;
import org.eclipse.tptp.trace.jvmti.internal.client.widgets.Utils;
import org.eclipse.tptp.trace.jvmti.provisional.client.extension.CustomAnalysisView;
import org.eclipse.tptp.trace.jvmti.provisional.client.extension.CustomColumn;
import org.eclipse.tptp.trace.jvmti.provisional.client.extension.ICustomAnalysisView;
import org.eclipse.ui.forms.IFormColors;

public class BaseMemoryStatisticView extends BaseStatisticView {
	static public final int CLASS_MODE = 0; // CM
	static public final int PACKAGE_MODE = 1;// PM
	// class mode
	static public final int CM_COL_NAME = 0;
	static public final int CM_COL_PACKAGE_NAME = 1;
	static public final int CM_COL_LIVE_INST = 2;
	static public final int CM_COL_LIVE_SIZE = 3;
	static public final int CM_COL_TOTAL_INST = 4;
	static public final int CM_COL_TOTAL_SIZE = 5;
	static public final int CM_COL_AVG_AGE = 6;
	//static public final int CM_COL_GENERATIONS = 7;
	// Total number of class view mode columns
	static public final int CM_COL_NUM = 7;
	// package mode
	static public final int PM_COL_NAME = 0;
	static public final int PM_COL_LIVE_INST = 1;
	static public final int PM_COL_LIVE_SIZE = 2;
	static public final int PM_COL_TOTAL_INST = 3;
	static public final int PM_COL_TOTAL_SIZE = 4;
	static public final int PM_COL_AVG_AGE = 5;
	//static public final int PM_COL_GENERATIONS = 6;
	static public final int PM_COL_NUM = 6;

	protected int _totalAge;
	protected int _totalGenerations;
	protected AllocationDetailsView _viewDetails;

	static private int V = ColumnData.IS_VISIBLE;
	static private int D = ColumnData.SUPPORTS_DELTA;

	private int _viewMode = CLASS_MODE;
	private TRCPackageSnapshot packageSnapshot = new TRCPackageImpl.TRCPackageSnapshot();
	private TRCClassSnapshot classSnapshot = new TRCClassImpl.TRCClassSnapshot();
	
	private ColumnLabelAdapter _packageModeCols[];
	private ColumnLabelAdapter _classModeCols[];
	
	private String _customClassColumnTemplate = "";
	private String _customPackageColumnTemplate = "";
	
	private Map _ageMap = new Hashtable();
	private Link setFilter;

	public BaseMemoryStatisticView(Composite parent, TraceViewerPage page) {
		super(parent, page);
		createColumnsLabelProviders();
		
		// Allow custom view's to setup
		ICustomAnalysisView[] customViewProviders = CustomAnalysisView.getContributors();		
		for( ICustomAnalysisView customView : customViewProviders ) {
			
			customView.attachView(this, page);
				
		}
	}
	
	public void dispose() {

		// Detach custom views from us
		for( ICustomAnalysisView customView : CustomAnalysisView.getContributors() )
			customView.detachView(this);
		
		super.dispose();
		
	}

	public int getViewMode() {
		return _viewMode;
	}

	public void setViewMode(int viewMode) {
		if (PACKAGE_MODE == viewMode)
			viewMode = PACKAGE_MODE;
		else
			viewMode = CLASS_MODE;
		if (viewMode == _viewMode)
			return;
		_viewMode = viewMode;
		redrawTable();
		update();
	}

	ColumnLabelAdapter getColumnsLabelProvider(int col) {
		if (PACKAGE_MODE == _viewMode)
			return _packageModeCols[col];
		else
			return _classModeCols[col];
	}

	private void createColumnsLabelProviders() {
		
		// Load custom column contributors
		final CustomColumn[] customColumns = CustomColumn.getContributors();
		final List<ColumnLabelAdapter> customClassLabels = new LinkedList<ColumnLabelAdapter>();
		final List<ColumnLabelAdapter> customPackageLabels = new LinkedList<ColumnLabelAdapter>();
		
		// Sort them into 'class' and 'package' view modes
		int customClassCount = 0; int customPackageCount = 0;
		for( int i=0; i<customColumns.length; i++ ) {
			
			if( !customColumns[i].isEnabled(_page) )
				continue;
			
			ColumnLabelAdapter labelAdapter = customColumns[i].getLabelAdapter();
			if( CLASS_MODE == customColumns[i].getViewMode() ) {
				
				customClassLabels.add( labelAdapter );
				_customClassColumnTemplate = _customClassColumnTemplate + "," +
					customColumns[i].getId() + ":" +
					(CM_COL_NUM + customClassCount) + ":" +
					customColumns[i].getStyle() + ":" + 
					customColumns[i].getAlign() + ":" +
					customColumns[i].getWidth();
				customClassCount++;
				
			} else if( PACKAGE_MODE == customColumns[i].getViewMode() ) {
			
				customPackageLabels.add( labelAdapter );
				_customPackageColumnTemplate = _customPackageColumnTemplate + "," +
					customColumns[i].getId() + ":" +
					(PM_COL_NUM + customPackageCount) + ":" +
					customColumns[i].getStyle() + ":" + 
					customColumns[i].getAlign() + ":" +
					customColumns[i].getWidth();
				customPackageCount++;
				
			}
		}

		_classModeCols = new ColumnLabelAdapter[ CM_COL_NUM + customClassLabels.size() ];
		_classModeCols[CM_COL_NAME] = new ClassPackageNameLabelAdapter();
		_classModeCols[CM_COL_PACKAGE_NAME] = new PackageNameLabelAdapter();
		_classModeCols[CM_COL_LIVE_INST] = new LiveInstancesLabelAdapter();
		_classModeCols[CM_COL_LIVE_SIZE] = new LiveSizeLabelAdapter();
		_classModeCols[CM_COL_TOTAL_INST] = new TotalInstancesLabelAdapter();
		_classModeCols[CM_COL_TOTAL_SIZE] = new TotalSizeLabelAdapter();
		_classModeCols[CM_COL_AVG_AGE] = new AvgAgeLabelAdapter(_ageMap);
		//_classModeCols[CM_COL_GENERATIONS] = new GenerationsColumnLabel();

		// Add the custom class mode columns
		for( int i=CM_COL_NUM; i<_classModeCols.length; i++ )
			_classModeCols[i] = customClassLabels.get( i - CM_COL_NUM );

		_packageModeCols = new ColumnLabelAdapter[ PM_COL_NUM + customPackageLabels.size() ];
		_packageModeCols[PM_COL_NAME] = _classModeCols[CM_COL_NAME];
		_packageModeCols[PM_COL_LIVE_INST] = _classModeCols[CM_COL_LIVE_INST];
		_packageModeCols[PM_COL_LIVE_SIZE] = _classModeCols[CM_COL_LIVE_SIZE];
		_packageModeCols[PM_COL_TOTAL_INST] = _classModeCols[CM_COL_TOTAL_INST];
		_packageModeCols[PM_COL_TOTAL_SIZE] = _classModeCols[CM_COL_TOTAL_SIZE];
		_packageModeCols[PM_COL_AVG_AGE] = _classModeCols[CM_COL_AVG_AGE];
		//_packageModeCols[PM_COL_GENERATIONS] = _classModeCols[CM_COL_GENERATIONS];

		// Add custom package mode columns
		for( int i=PM_COL_NUM; i<_packageModeCols.length; i++ )
			_packageModeCols[i] = customPackageLabels.get( i - PM_COL_NUM );

		for (int i = 0; i < CM_COL_NUM; i++) {
			if (i != CM_COL_NAME && i != CM_COL_PACKAGE_NAME && null != _classModeCols[i])
				_classModeCols[i].initialize(packageSnapshot, classSnapshot, null);
		}
	}

	protected ColumnDisplayInfo getColumnDisplayInfo(ColumnLabelAdapter col, boolean isDeltaColumn) {
		if (col == _classModeCols[CM_COL_PACKAGE_NAME])
			return TIContextUpdaterHelper.updatePackageName(col, _page.getMOFObject());
		else if (col == _classModeCols[CM_COL_TOTAL_INST])
			return TIContextUpdaterHelper.updateTotalInstances(col, isDeltaColumn, isShowPercent(), _totalInst);
		else if (col == _classModeCols[CM_COL_LIVE_INST])
			return TIContextUpdaterHelper.updateLiveInstances(col, isDeltaColumn, isShowPercent(), _activeInst);			
		else if (col == _classModeCols[CM_COL_TOTAL_SIZE])
			return TIContextUpdaterHelper.updateTotalSize(col, isDeltaColumn, isShowPercent(), _totalSize);			
		else if (col == _classModeCols[CM_COL_LIVE_SIZE])
			return TIContextUpdaterHelper.updateActiveSize(col, isDeltaColumn, isShowPercent(), _activeSize);
		else if (col == _classModeCols[CM_COL_AVG_AGE])
			return TIContextUpdaterHelper.updateAverageAge(col, isDeltaColumn, isShowPercent(), _totalAge);
		//else if (col == _classModeCols[CM_COL_GENERATIONS])
		//	return TIContextUpdaterHelper.updateAverageAge(col, isDeltaColumn, isShowPercent(), _totalGenerations);
		else
			return super.getColumnDisplayInfo(col, isDeltaColumn);
	}

	protected void showPercentUpdate() {
		if (isShowPercent()) {
			_totalInst = PerftraceUtil.getTotalInstances(_page.getMOFObject());
			if (_totalInst < 1)
				_totalInst = 1;
			_activeInst = PerftraceUtil.getActiveInstances(_page.getMOFObject());
			if (_activeInst < 1)
				_activeInst = 1;
			_totalSize = PerftraceUtil.getTotalSize(_page.getMOFObject());
			if (_totalSize < 1)
				_totalSize = 1;
			_activeSize = PerftraceUtil.getActiveSize(_page.getMOFObject());
			if (_activeSize < 1)
				_activeSize = 1;
			_collectedInst = PerftraceUtil.getTotalCollectedInstances(_page.getMOFObject());
			if (_collectedInst < 1)
				_collectedInst = 1;
			if (_totalAge < 1)
				_totalAge = 1;
		}
	}

	protected String getColumnsPreferencesKey() {
		if (PACKAGE_MODE == _viewMode)
			return PREFERENCE_KEY_PREFIX + "MemStatsPack12";
		else
			return PREFERENCE_KEY_PREFIX + "MemStatsClass12";
	}

	protected String getViewTypeStr() {
		return org.eclipse.hyades.trace.views.adapter.internal.TraceConstants.MEMORY_STATS_VIEW;
	}

	protected String getDefaultColumnsTemplate() {
		if (PACKAGE_MODE == _viewMode)
			return getPackageModeColumnsTemplate();
		else
			return getClassModeColumnsTemplate();
	}

	public String getColumnText(Object element, int columnIndex) {
		StatisticViewColumnInfo info = StatisticViewColumnInfo.getStatisticTableColumnInfo(getTree().getColumn(columnIndex));
		int pos = info.getColumnData().getInitalPos();
		String label = "placeholder(view)";
		ColumnLabelAdapter columnAdapter = getColumnsLabelProvider(pos);
		if (null != columnAdapter)
			label = getElementColumnText(element, columnAdapter, info.isDeltaColumn());
		return label;
	}

	public Image getColumnImage(Object element, int columnIndex) {
		StatisticViewColumnInfo info = StatisticViewColumnInfo.getStatisticTableColumnInfo(getTree().getColumn(columnIndex));
		int pos = info.getColumnData().getInitalPos();
		Image image = null;
		ColumnLabelAdapter columnAdapter = getColumnsLabelProvider(pos);
		if (null != columnAdapter)
			image = getElementColumnImage(element, columnAdapter, info.isDeltaColumn());
		return image;
	}

	protected StatisticSorter getViewerSorterInstance() {
		return new StatisticSorter() {
			public int compare(Viewer viewer, Object e1, Object e2) {
				ColumnLabelAdapter columnAdapter = getColumnsLabelProvider(_pos);
				int ret = 0;
				if (null != columnAdapter)
					ret = _sortSequence * compareElements(e1, e2, columnAdapter, _info.isDeltaColumn());
				return ret;
			}
		};
	}

	protected String getClassModeColumnsTemplate() {
		return ""
			+ TIContextAttributes.THREAD_CLASS_NAME + ":" + CM_COL_NAME + ":" + String.valueOf(ColumnData.NONDELETABLE | V | ColumnData.NONMOVABLE) + ":left:150,"
			+ TIContextAttributes.PACKAGE_NAME + ":" + CM_COL_PACKAGE_NAME + ":" + String.valueOf(ColumnData.NONDELETABLE | V | ColumnData.NONMOVABLE) + ":left:120,"
			+ TIContextAttributes.CLASS_LIVE_INST + ":" + CM_COL_LIVE_INST + ":" + String.valueOf(V | D) + ":right:80,"
			+ TIContextAttributes.CLASS_ACTIVE_SIZE + ":" + CM_COL_LIVE_SIZE + ":" + String.valueOf(V | D) + ":right:80,"
			+ TIContextAttributes.CLASS_TOTAL_INST + ":" + CM_COL_TOTAL_INST + ":" + String.valueOf(V | D) + ":right:80,"
			+ TIContextAttributes.CLASS_TOTAL_SIZE + ":" + CM_COL_TOTAL_SIZE + ":" + String.valueOf(V | D) + ":right:80,"
			+ TIContextAttributes.CLASS_AVG_AGE + ":" + CM_COL_AVG_AGE + ":" + String.valueOf(V) + ":right:80"
			//+ TIContextAttributes.CLASS_GENERATIONS + ":" + CM_COL_GENERATIONS + ":" + String.valueOf(V | D) + ":right:80"
			+ (null == this._customClassColumnTemplate ? "" : this._customClassColumnTemplate)
			;
	}

	protected String getPackageModeColumnsTemplate() {
		return ""
			+ TIContextAttributes.PACKAGE_NAME + ":" + PM_COL_NAME + ":" + String.valueOf(ColumnData.NONDELETABLE | V | ColumnData.NONMOVABLE) + ":left:120,"
			+ TIContextAttributes.PACKAGE_LIVE_INST + ":" + PM_COL_LIVE_INST + ":" + String.valueOf(V | D) + ":right:80,"
			+ TIContextAttributes.PACKAGE_ACTIVE_SIZE + ":" + PM_COL_LIVE_SIZE + ":" + String.valueOf(V | D) + ":right:80,"
			+ TIContextAttributes.PACKAGE_TOTAL_INST + ":" + PM_COL_TOTAL_INST + ":" + String.valueOf(V | D) + ":right:80,"
			+ TIContextAttributes.PACKAGE_TOTAL_SIZE + ":" + PM_COL_TOTAL_SIZE + ":" + String.valueOf(V | D) + ":right:80,"
			+ TIContextAttributes.PACKAGE_AVG_AGE + ":" + PM_COL_AVG_AGE + ":" + String.valueOf(V) + ":right:80"
			//+ TIContextAttributes.PACKAGE_GENERATIONS + ":" + PM_COL_GENERATIONS + ":" + String.valueOf(V | D) + ":right:80"
			+ (null == this._customPackageColumnTemplate ? "" : this._customPackageColumnTemplate)
			;
	}

	public Object[] getElements(Object inputElement) {
		Object ret[] = _empty;

		SimpleSearchQuery currentFilter = getCurrentFilter();
		if (DynamicFilter.containsDynamicFilters(currentFilter)) {
			currentFilter = DynamicFilter.updateDynamicFilters(currentFilter, _page.getMOFObject());
		}
		
		if (PACKAGE_MODE == _viewMode) {
			ret = FilteringUtil.getFilteredPackages(currentFilter, FilteringUtil.getProcessList(_page.getMOFObject())).toArray();

			if (ret != null && ret.length > 0) {
				List list = new LinkedList();
				for(int i=0; i<ret.length; i++) {
					Object obj = ret[i];
					if (obj instanceof TRCPackageImpl) {
						TRCPackageImpl p = (TRCPackageImpl) obj;
						if (p.getTotalInstances() > 0) list.add(p);
					}
					else
						list.add(obj);
				}
				
				ret = list.toArray();
			}
		}
		else if (CLASS_MODE == _viewMode) {
			// System.out.println("filter string: " +currentFilter);
			ret = FilteringUtil.getFilteredClasses(currentFilter, FilteringUtil.getProcessList(_page.getMOFObject())).toArray();
			updateAgeInfo(ret);
			
			if (ret != null && ret.length > 0) {
				List list = new LinkedList();
				for(int i=0; i<ret.length; i++) {
					Object obj = ret[i];
					if (obj instanceof TRCClassImpl) {
						TRCClassImpl c = (TRCClassImpl) obj;
						if (c.getTotalInstances() > 0) list.add(c);
					}
					else
						list.add(obj);
				}
				
				ret = list.toArray();
			}
		}
		
		return ret;
	}

	public boolean hasChildren(Object element) {
		boolean ret = false;

		// Allow the option for handling hasChildren to be customized with the customAnalysisView extension point.
		ICustomAnalysisView[] customViewProviders = CustomAnalysisView.getContributors();
		// True if an element has children elements.
		boolean customEngineHasChildren = false;
		// True if at least one provider requires using hasChildren instead of getChildren
		boolean customEngineForHasChildren = false;
		if(customViewProviders != null) {
			// Check if any implementations override how to handle the element
			for(int provider = 0; provider < customViewProviders.length; provider++) {
				// Check if the getChildren or hasChildren should be used to determine if an element has child elements.
				if(!customEngineForHasChildren)
					customEngineForHasChildren = customViewProviders[provider].getEngineForHasChildren();
				
				if(customEngineForHasChildren) {
					customEngineHasChildren = customViewProviders[provider].hasChildren(element);
				
					// At least one provider implements hasChildren
					if(customEngineHasChildren)
						break;
				}
			}
		}
		
		if (FilteringUtil.USE_ENGINE_FOR_HAS_CHILDREN && !customEngineForHasChildren)
			ret = getChildren(element).length > 0;
		else if (element instanceof TRCPackage)
			ret = true;
		else if(customEngineForHasChildren) {
			ret = customEngineHasChildren;
		}
		return ret;
	}

	public Object[] getChildren(Object parentElement) {
		Object ret[] = _empty;
		ICustomAnalysisView[] customViewProviders = CustomAnalysisView.getContributors();
		if (parentElement instanceof TRCPackage) {
			ret = FilteringUtil.getFilteredClasses(getCurrentFilter(), (TRCPackage)parentElement).toArray();
		} else if(customViewProviders != null) {
			List<Object> providerRet = new ArrayList<Object>();
			for(int provider = 0; provider < customViewProviders.length; provider++) {
				Object retArray[] = _empty;
				retArray = customViewProviders[provider].getChildren(parentElement);
				// Store the results in a temporary list to retrieve all the provider's getChildren results.
				if(retArray != null) {
					for(int i = 0; i < retArray.length; i++) {
						providerRet.add(retArray[i]);
					}
				}
			}

			ret = providerRet.toArray();
		}
		return ret;
	}

	void updateAgeInfo(Object arr[]) {
		_ageMap.clear();
		_totalAge = 1;
		EList listGC = null;
		for (int i = 0; i < arr.length; i++) {
			if (arr[i] instanceof TRCClass) {
				TRCClass cls = (TRCClass) arr[i];
				if (cls.getTotalInstances() == 0)
					continue;
				AgeInfo ageInfo = new AgeInfo();
				if (listGC == null)
					listGC = cls.getPackage().getProcess().getGcEvents();
				if (null == listGC || listGC.size() == 0)
					return;
				_totalAge = listGC.size();
				EList list = cls.getObjects();
				int len = list.size();
				for (int j = 0; j < len; j++) {
					Object obj = list.get(j);
					if (obj instanceof TRCFullTraceObject) {
						int age = Utils.getObjAge((TRCFullTraceObject)obj, listGC);
						ageInfo.sumAges += age;
						ageInfo.objCount++;
					}
				}
				_ageMap.put(cls, ageInfo);
			}
		}
	}

	static public class AgeInfo {
		public int sumAges;
		public int objCount;
	}

	public void update() {
		ICustomAnalysisView[] customViewProviders = CustomAnalysisView.getContributors();
		// True if one provider requires restoring the expanded elements
		boolean restoreExpandedItems = false;
		TreePath[] expandedItems = null;
		
		if(customViewProviders != null) {
			for(int provider = 0; provider < customViewProviders.length; provider++) {
				restoreExpandedItems = customViewProviders[provider].getRestoreExpandedItems();
				if(restoreExpandedItems)
					break;
			}
		}
		
		if(restoreExpandedItems) {
			// Store the visible expanded items by path in case they can be in more than one location.
			expandedItems = getTreeViewer().getExpandedTreePaths();
		}
		
		super.update();
		if (_page instanceof MemoryAnalysisPage) {
			((MemoryAnalysisPage) _page).updateFilterAppliedDescription();
	    	setFilter.setText(NLS.bind(CommonUITraceMessages.CHOOSE_FILTER_MSG, getCurrentFilterName()));
		}
		
		if(restoreExpandedItems && expandedItems != null) {
			// Update the view restoring the expanded items.
			getTreeViewer().setExpandedTreePaths(expandedItems);
		}
	}

	protected void defaultActionForSelection(ISelection selection) {
	}

	public void widgetDefaultSelected(SelectionEvent event) {
		if (_page instanceof MemoryAnalysisPage)
			((MemoryAnalysisPage)_page).showAllocDetails();
	}

	protected SimpleSearchQuery getCurrentFilter() {
		if (_page instanceof MemoryAnalysisPage)
			return ((MemoryAnalysisPage) _page).getCurrentFilter();
		return null;
	}
	
	protected void createLabel(Composite viewContainer) {
		Label _label = _toolkit.createLabel(viewContainer, null);
		
		_label.setBackground(viewContainer.getBackground());
    	_label.setForeground(_toolkit.getColors().getColor(IFormColors.TITLE));

    	Font font = _label.getFont();
    	FontData fd = font.getFontData()[0];
    	font = new Font(null, new FontData(fd.getName(), fd.getHeight()*3/2, fd.getStyle() | SWT.BOLD));
    	_label.setFont(font);
    	
		_label.setLayoutData(GridUtil.createHorizontalFill());
		
		_label.setText(UIMessages.MEM_STATISTICS_TITLE);

    	setFilter = new Link(viewContainer, SWT.WRAP | _toolkit.getOrientation());
    	setFilter.setBackground(_toolkit.getColors().getBackground());
    	setFilter.setForeground(_toolkit.getColors().getColor(IFormColors.TITLE));
    	font = setFilter.getFont();
    	fd = font.getFontData()[0];
    	font = new Font(null, new FontData(fd.getName(), fd.getHeight(), fd.getStyle() | SWT.BOLD));
    	setFilter.setFont(font);
    	setFilter.addSelectionListener(new SelectionListener() {
    		public void widgetDefaultSelected(SelectionEvent e) {}
    	
    		public void widgetSelected(SelectionEvent e) {
    			if (_page != null && _page instanceof MemoryAnalysisPage) {
    				TraceViewer viewer = ((MemoryAnalysisPage) _page).getTraceViewer();
    				if (viewer != null && viewer instanceof FilterTraceViewer) {
    					FilterInformationManager fim = ((FilterTraceViewer) viewer).getFilterInformationManager();
        				FiltersEditorAction feAction = new FiltersEditorAction(fim, true);
        				feAction.run();
    				}
    			}
    		}
    	});
    	
    	setFilter.setLayoutData(GridUtil.createHorizontalFill());
    	setFilter.setText(NLS.bind(CommonUITraceMessages.CHOOSE_FILTER_MSG, getCurrentFilterName()));
	}
	
	public String getCurrentFilterName() {
		String filterAppliedName;
	
		SimpleSearchQuery cf = getCurrentFilter();
		if (cf != null)
			filterAppliedName = cf.getName();
		else
			filterAppliedName = CommonUITraceMessages.ST_FLTAPN;
		
		return filterAppliedName;
	}
}
