/*****************************************************************************
 * Copyright (c) 2007, IBM Corporation, Intel Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Intel Corporation - Initial API and implementation
 *    Ruslan A. Scherbakov, Intel - Initial API and implementation
 *
 * $Id: BaseStatisticView.java,v 1.4 2007/04/20 13:06:48 ewchan Exp $ 
 *****************************************************************************/

package org.eclipse.tptp.trace.jvmti.internal.client.views;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.hyades.trace.internal.ui.PDPluginImages;
import org.eclipse.hyades.trace.ui.ProfileEvent;
import org.eclipse.hyades.trace.ui.TraceViewer;
import org.eclipse.hyades.trace.ui.TraceViewerPage;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.views.actions.internal.OpenSourceAction;
import org.eclipse.hyades.trace.views.adapter.internal.IContextViewer;
import org.eclipse.hyades.trace.views.adapter.internal.TraceConstants;
import org.eclipse.hyades.trace.views.internal.TracePluginImages;
import org.eclipse.hyades.trace.views.internal.TraceUIPlugin;
import org.eclipse.hyades.trace.views.internal.view.columnlabels.ColumnDisplayInfo;
import org.eclipse.hyades.trace.views.internal.view.columnlabels.ColumnLabelAdapter;
import org.eclipse.hyades.trace.views.internal.view.columnlabels.ColumnLabelComparator;
import org.eclipse.hyades.trace.views.internal.view.columnlabels.ContextUpdaterHelper;
import org.eclipse.hyades.trace.views.util.internal.ColumnData;
import org.eclipse.hyades.trace.views.util.internal.ColumnsDialog;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.viewers.DoubleClickEvent;
import org.eclipse.jface.viewers.IContentProvider;
import org.eclipse.jface.viewers.IDoubleClickListener;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerSorter;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.swt.widgets.TreeColumn;
import org.eclipse.swt.widgets.TreeItem;
import org.eclipse.tptp.trace.jvmti.internal.client.widgets.Utils;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.forms.widgets.ColumnLayout;

/**
 * This class was ported from StatisticView from hyades project for JVMTI trace client.
 * 
 * @see org.eclipse.hyades.trace.views.internal.StatisticView
 * 
 */
public abstract class BaseStatisticView extends BaseProfilerView implements SelectionListener {

	protected TreeViewer _treeViewer;
	protected StatisticSorter _viewerSorter;
	protected ArrayList _currentColumns = null;
	protected Action _updateAction;
	protected Action _showPercent;
	protected Action _deltaColumnsAction;
	protected Action _chooseColumnAction;
	protected Action _sortByColumnAction;
	protected Separator _separator;
	protected boolean _newSelection = false;
	protected boolean _isShowPercent = false;
	protected boolean _deltaColumns = false;
	protected boolean _refresh = false;
	protected boolean _firstTime = true;
	protected double _maxTime;
	protected int _totalInst;
	protected int _totalCalls;
	protected int _activeInst;
	protected long _totalSize;
	protected long _activeSize;
	protected int _collectedInst;

	protected abstract String getColumnsPreferencesKey();
	protected abstract String getDefaultColumnsTemplate();
	protected abstract StatisticSorter getViewerSorterInstance();
	protected abstract String getColumnText(Object element, int columnIndex);
	protected abstract Image getColumnImage(Object element, int columnIndex);
	protected abstract Object[] getElements(Object inputElement);
	protected abstract boolean hasChildren(Object element);
	protected abstract Object[] getChildren(Object parentElement);

	public class BaseStatisticContentProvider implements ITreeContentProvider {

		public Object[] getChildren(Object parentElement) {
			return BaseStatisticView.this.getChildren(parentElement);
		}

		public Object getParent(Object element) {
			return null;
		}

		public boolean hasChildren(Object element) {
			return BaseStatisticView.this.hasChildren(element);
		}

		public Object[] getElements(Object inputElement) {
			return BaseStatisticView.this.getElements(inputElement);
		}

		public void dispose() {
		}

		public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {
		}
	}

	class StatisticTreeViewer extends TreeViewer {

		public StatisticTreeViewer(Tree tree) {
			super(tree);
		}

		public void expandItem(TreeItem item) {
			item.setExpanded(true);
			createChildren(item);
		}

		public ISelection getSelection() {
			ISelection sel = super.getSelection();
			return sel;
		}
	}

	public abstract class StatisticSorter extends ViewerSorter {
		protected TreeColumn _sortColumn;
		protected int _sortSequence = -1;
		protected StatisticViewColumnInfo _info;
		protected int _pos = 0;

		public StatisticSorter() {
			super();
		}

		public void setSortedColumn(TreeColumn newSortColumn) {
			// Get the currently designated sort column and turn remove its sorted indicator
			TreeColumn currentColumn = getCurrentSortColumn(newSortColumn .getParent());
			if (currentColumn != null) {
				if (currentColumn == newSortColumn) {
					_sortSequence = -_sortSequence;
				} else {
					_sortSequence = -1;
				}
			} else {
				_sortSequence = 1;
			}

			setSortedColumn(_sortSequence, newSortColumn);
		}

		public void setSortedColumn(int sortSequence, TreeColumn newSortColumn) {
			_sortSequence = sortSequence;
			_sortColumn = newSortColumn;

			// Get the currently designated sort column and turn remove its sorted indicator
			TreeColumn currentColumn = getCurrentSortColumn(newSortColumn.getParent());
			if (currentColumn != null) {
				String text = currentColumn.getText();
				if (text.startsWith(">") || text.startsWith("<"))
					text = text.substring(1);
				currentColumn.setText(text);
				StatisticViewColumnInfo.getStatisticTableColumnInfo(
						currentColumn).setSortColumn(false);
			}

			// Add the sorted indicator to the newly sorted column
			String text = newSortColumn.getText();
			_info = StatisticViewColumnInfo.getStatisticTableColumnInfo(newSortColumn);
			_pos = _info.getColumnData().getInitalPos();
			newSortColumn.setText(((_sortSequence > 0) ? ">" : "<") + text);
			StatisticViewColumnInfo.getStatisticTableColumnInfo(newSortColumn).setSortColumn(true);
		}

		public int getSortSequence() {
			return _sortSequence;
		}

		public void setSortSequence(int sortSequence) {
			_sortSequence = sortSequence;
		}

		public TreeColumn getCurrentSortColumn(Tree tree) {
			for (int idx = 0; idx < tree.getColumnCount(); idx++) {
				if (StatisticViewColumnInfo.getStatisticTableColumnInfo(
						tree.getColumn(idx)).isSortColumn()) {
					return tree.getColumn(idx);
				}
			}
			return null;
		}

		public TreeColumn resetSortColumn(Tree tree) {
			_sortColumn = getCurrentSortColumn(tree);
			return _sortColumn;
		}
	}

	/** Update action */
	class UpdateAction extends Action {
		public UpdateAction(String name) {
			super(name);
			PlatformUI.getWorkbench().getHelpSystem().setHelp(UpdateAction.this, TraceUIPlugin.getPluginId() + ".uvew0000");
			PDPluginImages.setImageDescriptors(this, PDPluginImages.T_LCL, PDPluginImages.IMG_UPDATEVIEWS);
		}

		public void run() {
			ProfileEvent event = UIPlugin.getDefault().getRefreshViewEvent(_page.getMOFObject());
			UIPlugin.getDefault().notifyProfileEventListener(event);
		}
	}

	/** Show as percent action */
	class ShowPercentAction extends Action {
		public ShowPercentAction(String name) {
			super(name);
			PlatformUI.getWorkbench().getHelpSystem().setHelp(ShowPercentAction.this, TraceUIPlugin.getPluginId() + ".clst0003");
		}

		public void run() {
			handlePercentChanged();
		}
	}

	/** Show delta columns action */
	class DeltaColumnsAction extends Action {
		public DeltaColumnsAction(String name) {
			super(name);
			PlatformUI.getWorkbench().getHelpSystem().setHelp(DeltaColumnsAction.this, TraceUIPlugin.getPluginId() + ".ccol0000");
		}

		public void setChecked(boolean value) {
			super.setChecked(value);
		}

		public void run() {
			handleDeltaChanged();
		}
	}

	/** ChooseColumnsAction popup action */
	class ChooseColumnsAction extends Action {
		private ArrayList _cols;
		private String _key;

		public ChooseColumnsAction(String name, ArrayList cols, String key) {
			super(name);
			_cols = cols;
			_key = key;
			PlatformUI.getWorkbench().getHelpSystem().setHelp(
					ChooseColumnsAction.this,
					TraceUIPlugin.getPluginId() + ".ccol0000");
			TracePluginImages.setImageDescriptors(this,
					TracePluginImages.T_LCL,
					TracePluginImages.IMG_CHOOSECOLUMNS);
		}

		public void run() {
			ColumnsDialog dialog = new ColumnsDialog(getControl().getShell(),
					UIMessages._ChooseColumns, null, _cols,
					getDefaultColumnsTemplate(), getContextHelpId(),
					getContext());
			dialog.open();
			if (dialog.getReturnCode() == Window.OK) {
				ColumnData.setColumns(dialog.getData(), _cols, _key);
				resetColumns(_cols);
				update();
			}
		}
	}   

	/** SortByColumnsAction popup action */
	class SortByColumnsAction extends Action {
		public SortByColumnsAction(String name) {
			super(name);
			PlatformUI.getWorkbench().getHelpSystem().setHelp(
					SortByColumnsAction.this,
					TraceUIPlugin.getPluginId() + ".ccol0000");
			TracePluginImages.setImageDescriptors(this,
					TracePluginImages.T_LCL, TracePluginImages.IMG_SORTCOLUMNS);
		}

		private TreeColumn getTreeColumn(Tree tree, StatisticViewColumnInfo columnInfo) {
			for (int idx = 0; idx < tree.getColumnCount(); idx++) {
				StatisticViewColumnInfo treeColumnInfo = getColumnInfo(tree.getColumn(idx));
				if (treeColumnInfo != null
						&& columnInfo != null
						&& columnInfo.getColumnData().name().equals(treeColumnInfo.getColumnData().name())
						&& columnInfo.isDeltaColumn() == treeColumnInfo.isDeltaColumn()) {
					return tree.getColumn(idx);
				}
			}
			return null;
		}

		public void run() {
			List columnInfoList = new ArrayList();
			for (int i = 0; i < getTree().getColumnCount(); i++) {
				StatisticViewColumnInfo ci = getColumnInfo(getTree().getColumn(i));
				if (ci != null) {
					columnInfoList.add(ci);
				}
			}
			SortByColumnDlg dialog = new SortByColumnDlg(getControl().getShell(), null, columnInfoList, getViewerSorter().getSortSequence());
			if (dialog.open() == Window.OK) {
				getViewerSorter().setSortedColumn(dialog.sortSequence(),
						getTreeColumn(getTree(), dialog.sortColumn()));
				_treeViewer.refresh();
			}
		}
	}

	protected BaseStatisticView(Composite parent, TraceViewerPage page) {
		super(parent, page);
	}

	public void widgetDefaultSelected(SelectionEvent arg0) {
	}

	public void widgetSelected(SelectionEvent arg0) {
		if (arg0.widget instanceof TreeColumn) {
			_viewerSorter.setSortedColumn((TreeColumn) arg0.widget);
			_treeViewer.setSorter(_viewerSorter);
			_treeViewer.refresh();
		} else if (arg0.widget instanceof Table || arg0.widget instanceof Tree) {
			updateModelSelection();
		}
		updateButtons();
	}

	public Tree getTree() {
		return (Tree) getTreeViewer().getControl();
	}

	protected TreeViewer getTreeViewer(Tree tree) {
		return new StatisticTreeViewer(tree);
	}

	protected TreeViewer createTreeViewer(Tree tree) {
		TreeViewer tv = getTreeViewer(tree);
		tv.addDoubleClickListener(new IDoubleClickListener() {
			public void doubleClick(DoubleClickEvent event) {
				ISelection selection = event.getSelection();
				if (selection != null && !selection.isEmpty()) {
					TreeViewer tv = (TreeViewer) getTreeViewer();
					Object node = ((IStructuredSelection) selection).getFirstElement();
					if (tv.isExpandable(node)) {
						tv.setExpandedState(node, !tv.getExpandedState(node));
					} else {
						defaultActionForSelection(selection);
					}
				}
			}
		});
		return tv;
	}

	protected IContentProvider getContentProvider() {
		return new BaseStatisticContentProvider();
	}

	public LabelProvider getTableLabelProvider() {
		class StatisticLabelProvider extends LabelProvider implements ITableLabelProvider {
			public Image getColumnImage(Object element, int columnIndex) {
				return BaseStatisticView.this.getColumnImage(element, columnIndex);
			}
			public String getColumnText(Object element, int columnIndex) {
				return BaseStatisticView.this.getColumnText(element, columnIndex);
			}
		}
		return new StatisticLabelProvider();
	}

	protected void defaultActionForSelection(ISelection selection) {
		OpenSourceAction action = new OpenSourceAction();
		action.selectionChanged(action, selection);
		action.run();
	}

	protected TraceViewerPage getTraceViewerPage() {
		return _page;
	}

	public void showNewSelection() {
		if (_newSelection) {
			getTree().showSelection();
			_newSelection = false;
		}
	}

	public boolean showingDeltaColumns() {
		return _deltaColumns;
	}

	protected void setNewSelection() {
		_newSelection = true;
	}

	public Action getUpdateAction() {
		if (_updateAction == null) {
			_updateAction = new UpdateAction(UIPlugin.getResourceString("REFRESH_VIEWS"));
		}
		return _updateAction;
	}
	
	public Separator getSeparator() {
		if (_separator == null)
			_separator = new Separator();
		return _separator;
	}

	public Action getChooseColumnsAction() {
		ArrayList cols = getColumnDataList();
		String key = getColumnsPreferencesKey();
		if (_chooseColumnAction == null)
			_chooseColumnAction = new ChooseColumnsAction(UIMessages._ChooseColumns + "...", cols, key);
		return _chooseColumnAction;
	}

	protected void resetChooseColumnsAction() {
		_chooseColumnAction = null;
	}

	public Action getSortByColumnAction() {
		if (_sortByColumnAction == null)
			_sortByColumnAction = new SortByColumnsAction(UIMessages._SortByColumn + "...");
		return _sortByColumnAction;
	}

	public Action getDeltaColumnsAction() {
		if (_deltaColumnsAction == null) {
			_deltaColumnsAction = new DeltaColumnsAction(UIMessages._ShowDeltaColumns);
		}
		return _deltaColumnsAction;
	}

	public Action getShowPercentAction() {
		if (_showPercent == null) {
			_showPercent = new ShowPercentAction(UIMessages._ShowAsPercentage);
		}
		return _showPercent;
	}

	public boolean isShowPercent() {
		return _isShowPercent;
	}

	public ArrayList getColumnDataList() {
		return _currentColumns;
	}

	public TreeViewer getTreeViewer() {
		return _treeViewer;
	}

	public StatisticSorter getViewerSorter() {
		return _viewerSorter;
	}

//
//	public void controlMoved(ControlEvent arg0) {
//	}
//
//	public void controlResized(ControlEvent arg0) {
//		// Get the client area for the shell
//		Rectangle dataBounds = getViewComponent().getClientArea();
//		getDataPane().setBounds(dataBounds.x, dataBounds.y, dataBounds.width, dataBounds.height);
//		// If the table column was resized, update the ColumnData array with the
//		// new size and save the new column definitions to the preferences.
//		if (arg0.getSource() instanceof TableColumn) {
//			ColumnData.setColumns(getColumnDataList(), getColumnsPreferencesKey());
//		}
//	}

	protected void setRedrawTable(boolean redraw) {
		_treeViewer.getControl().setRedraw(redraw);
	}

	protected void showPercentUpdate() {
	}

	protected int updateTableGetColumnNumber() {
		return 0;
	}

	public void update() {
		if (_firstTime)
			firstTimeUpdate();
		// context changes or context handler selection changes
		if (contextChanged() || _page.getTraceViewer().contextHandlerSelectionChanged()) {
			updateContext();
		}
		IContextViewer viewer = getContextViewer();
		if (viewer != null) {
			viewer.getContextSelectionMenuListener().setMOFObject(_page.getMOFObject());
		}
		// set the input of the viewer
		showPercentUpdate();
		setRedrawTable(false);
		getTreeViewer().setInput(_page.getMOFObject());
		// _viewer.refresh();
		expandFirstElement();
		setRedrawTable(true);
		postUpdateEvents();
	}

	protected void firstTimeUpdate() {
		_firstTime = false;
		Tree tree = getTree();
		TreeColumn firstColumn = tree.getColumn(updateTableGetColumnNumber());
		_viewerSorter = getViewerSorterInstance();
		_viewerSorter.setSortedColumn(firstColumn);
		_treeViewer.setSorter(_viewerSorter);
		redrawTable();
	}

	protected void redrawTable() {
		getTree().setRedraw(false);
		_currentColumns = ColumnData.createColumnData(getColumnsPreferencesKey(), getDefaultColumnsTemplate());
		_currentColumns = TIColumnData.convertColumnData(_currentColumns);
		resetColumns(_currentColumns);
		getTree().setRedraw(true);
		//update();
	}

	protected void postUpdateEvents() {
		handleSelectionEvent();
	}

	protected void expandFirstElement() {
	}

	protected String getContext() {
		IContextViewer viewer = getContextViewer();
		return viewer != null ? viewer.getSelectedContext() : null;
	}

	protected boolean contextChanged() {
		IContextViewer viewer = getContextViewer();
		if (viewer != null) {
			if (viewer.contextChanged()) {
				return true;
			} else {
				viewer.setSelectedContext(ContextUpdaterHelper.getCurrentContext(ContextUpdaterHelper.getContexts(_page.getMOFObject()), viewer), false);
				return viewer.contextChanged();
			}
		} else {
			return false;
		}
	}

	protected void updateContext() {
		resetColumns(_currentColumns);
		IContextViewer viewer = getContextViewer();
		if (viewer != null)
			viewer.updateContext();
	}

	private IContextViewer getContextViewer() {
		if (_page != null) {
			TraceViewer traceViewer = _page.getTraceViewer();
			if (traceViewer != null && traceViewer instanceof IContextViewer)
				return (IContextViewer) traceViewer;
		}
		return null;
	}

	/**
	 * This method is called as a result of changing the selected columns. It
	 * takes the new column definition list and adjusts the columns of the table
	 * to match this list.
	 */
	protected void resetColumns(ArrayList list) {
		String sortedColumnKey = ""; // store sorted column
		boolean isDeltaColumn = false;
		int sortSequence = _viewerSorter.getSortSequence(); // store sorted sequence

		Tree tree = getTree();
		setRedrawTable(false);

		// Dispose all the existing columns
		for (int idx = tree.getColumnCount(); idx > 0; idx--) {
			TreeColumn col = tree.getColumn(tree.getColumnCount() - 1);
			StatisticViewColumnInfo staticTblColInfo = (StatisticViewColumnInfo) (col.getData());
			if (staticTblColInfo.isSortColumn()) {
				sortedColumnKey = staticTblColInfo.getColumnData().key();
				isDeltaColumn = staticTblColInfo.isDeltaColumn();
			}
			col.dispose();
		}

		for (int idx = 0; idx < list.size(); idx++) {
			ColumnData data = (ColumnData) list.get(idx);
			data.setContext(getContext());
			if (data.visible()) {
				new StatisticViewColumnInfo(this, data, _deltaColumns);
			}
		}

		tree = getTree();
		for (int idx = 0; idx < tree.getColumnCount(); idx++) {
			StatisticViewColumnInfo treeColumnInfo = getColumnInfo(tree.getColumn(idx));
			if (treeColumnInfo != null
					&& sortedColumnKey.equals(treeColumnInfo.getColumnData().key())
					&& isDeltaColumn == treeColumnInfo.isDeltaColumn()) {
				treeColumnInfo.setSortColumn(true);
				break;
			}
		}

		if (_viewerSorter.resetSortColumn(getTree()) == null) {
			_treeViewer.setSorter(null);
		} else {
			_viewerSorter.setSortSequence(sortSequence * -1);
			_viewerSorter.setSortedColumn(_viewerSorter.resetSortColumn(getTree()));
			_treeViewer.setSorter(_viewerSorter);
		}

		setRedrawTable(true);
	}

	protected Control createDataViewer(Composite parent) {
		_isShowPercent = Utils.loadIntOption(getPreferenceString(TraceConstants.PERCENTAGE)) == 1;
		_deltaColumns = Utils.loadIntOption(getPreferenceString(TraceConstants.DELTA)) == 1;
		_currentColumns = ColumnData.createColumnData(getColumnsPreferencesKey(), getDefaultColumnsTemplate());
		Control table = createTableViewer(parent, _currentColumns);
		return table;
	}

	protected Tree createTree(Composite parent, int options) {
		Tree tree = _toolkit.createTree(parent, options);
		tree.setLinesVisible(true);
		tree.setHeaderVisible(true);
		ColumnLayout layout = new ColumnLayout();
		layout.maxNumColumns = 100;
		tree.setLayout(layout);
		tree.setLayoutData(new GridData(GridData.FILL_BOTH));
		tree.addSelectionListener(this);
		return tree;
	}

	private Control createTableViewer(Composite parent, ArrayList cols) {
		Tree tree = createTree(parent, SWT.SINGLE | SWT.FULL_SELECTION | SWT.FLAT);
		_treeViewer = createTreeViewer(tree);
		for (int idx = 0; idx < cols.size(); idx++) {
			ColumnData data = (ColumnData) cols.get(idx);
			data.setContext(getContext());
			if (data.visible()) {
				new StatisticViewColumnInfo(this, data, _deltaColumns);
			}
		}
		_treeViewer.setContentProvider(getContentProvider());
		_treeViewer.setLabelProvider(getTableLabelProvider());
		UIPlugin.getDefault().addViewSelectionChangedListener(this);
		return _treeViewer.getControl();
	}

	public void dispose() {
		UIPlugin.getDefault().removeViewSelectionChangedListener(this);
		_page = null;
	}

	public void handlePercentChanged() {
		_isShowPercent = !_isShowPercent;
		Utils.saveIntOption(getPreferenceString(TraceConstants.PERCENTAGE), (_isShowPercent ? 1 : 0));
		update();
		updateButtons();
	}

	public void handleDeltaChanged() {
		_deltaColumns = !_deltaColumns;
		Utils.saveIntOption(getPreferenceString(TraceConstants.DELTA), (_deltaColumns ? 1 : 0));
		resetColumns(_currentColumns);
		update();
	}

	protected StatisticViewColumnInfo getColumnInfo(TreeColumn treeColumn) {
		if (treeColumn.getData() != null && treeColumn.getData() instanceof StatisticViewColumnInfo)
			return (StatisticViewColumnInfo) treeColumn.getData();
		return null;
	}

	protected int compareElements(Object obj1, Object obj2, ColumnLabelAdapter col, boolean isDeltaColumn) {
		ColumnDisplayInfo info = getColumnDisplayInfo(col, isDeltaColumn);
		return ColumnLabelComparator.compare(obj1, obj2, col, info);
	}

	protected String getElementColumnText(Object obj, ColumnLabelAdapter col, boolean isDeltaColumn) {
		ColumnDisplayInfo info = getColumnDisplayInfo(col, isDeltaColumn);
		return col.getDisplayString(obj, info);
	}

	protected Image getElementColumnImage(Object obj, ColumnLabelAdapter col, boolean isDeltaColumn) {
		ColumnDisplayInfo info = getColumnDisplayInfo(col, isDeltaColumn);
		return col.getDisplayImage(obj, info);
	}

	protected ColumnDisplayInfo getColumnDisplayInfo(ColumnLabelAdapter col, boolean isDeltaColumn) {
		return col.getColumnDisplayInfo();
	}

	public ISelection getSelection() {
		return _treeViewer.getSelection();
	}
}
