/*****************************************************************************
 * Copyright (c) 2007, Intel Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Intel Corporation - Initial API and implementation
 *    Ruslan A. Scherbakov, Intel - Initial API and implementation
 *
 * $Id: CallStackViewer.java,v 1.5 2007/04/20 13:06:47 ewchan Exp $ 
 *****************************************************************************/

package org.eclipse.tptp.trace.jvmti.internal.client.views;

import java.util.ArrayList;

import org.eclipse.emf.common.util.EList;
import org.eclipse.hyades.models.trace.TRCMethodInvocation;
import org.eclipse.hyades.models.trace.TRCThread;
import org.eclipse.hyades.models.trace.TRCThreadEvent;
import org.eclipse.hyades.trace.ui.IViewSelectionChangedListener;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.ui.ViewSelectionChangedEvent;
import org.eclipse.jface.viewers.DoubleClickEvent;
import org.eclipse.jface.viewers.IDoubleClickListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.TreeSelection;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.tptp.trace.jvmti.internal.client.MethodDetails;
import org.eclipse.tptp.trace.jvmti.internal.client.widgets.Utils;
import org.eclipse.ui.IPageListener;
import org.eclipse.ui.IPartListener2;
import org.eclipse.ui.IViewReference;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchPartReference;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.part.ViewPart;

interface ICallStackProvider {
	Object getCallStack();
}

public class CallStackViewer extends ViewPart implements
		IViewSelectionChangedListener {

	private TreeViewer _calStackCtrl;
	ICallStackProvider _provider;
	private Object _selection;
	private PartListener _partListener;
	private static final Object[] _empty = new Object[0];

	public CallStackViewer() {
	}

	class PartListener implements IPartListener2, IPageListener {

		IWorkbenchWindow _window;
		IWorkbenchPage _page;

		PartListener() {
			IWorkbench wb = UIPlugin.getDefault().getWorkbench();
			_window = wb.getActiveWorkbenchWindow();
			_page = _window.getActivePage();
			_window.addPageListener(this);
			_page.addPartListener(this);
			IViewReference refs[] = _window.getActivePage().getViewReferences();
			for (int i = 0; i < refs.length; i++) {
				Object part = refs[i].getPart(false);
				if (part instanceof ICallStackProvider) {
					_provider = (ICallStackProvider) part;
					update();
					break;
				}
			}
		}

		void dispose() {
			if (_window != null) {
				_window.removePageListener(this);
				if (_window.getActivePage() != null)
					_window.getActivePage().removePartListener(this);
				_window = null;
			}
			_provider = null;
		}

		public void partActivated(IWorkbenchPartReference partRef) {
			Object part = partRef.getPart(false);
			if (part instanceof ICallStackProvider) {
				_provider = (ICallStackProvider) part;
				update();
			}
		}

		public void partBroughtToTop(IWorkbenchPartReference partRef) {
		}

		public void partClosed(IWorkbenchPartReference partRef) {
			if (_provider == partRef.getPart(false)) {
				_provider = null;
				update();
			}
		}

		public void partDeactivated(IWorkbenchPartReference partRef) {
		}

		public void partHidden(IWorkbenchPartReference partRef) {
			if (_provider == partRef.getPart(false)) {
				_provider = null;
				update();
			}
		}

		public void partInputChanged(IWorkbenchPartReference partRef) {
		}

		public void partOpened(IWorkbenchPartReference partRef) {
		}

		public void partVisible(IWorkbenchPartReference partRef) {
		}

		public void pageActivated(IWorkbenchPage page) {
		}

		public void pageClosed(IWorkbenchPage page) {
			if (page == _page) {
				_page.removePartListener(this);
				_page = null;
			}
		}

		public void pageOpened(IWorkbenchPage page) {
		}
	}

	public void createPartControl(Composite parent) {
		_calStackCtrl = new TreeViewer(parent, SWT.H_SCROLL | SWT.V_SCROLL | SWT.FULL_SELECTION | SWT.SINGLE);
		_calStackCtrl.setContentProvider(new CallStackContentProvider());
		_calStackCtrl.setLabelProvider(new CallStackLabelProvider());
		_calStackCtrl.setInput(this);
		_calStackCtrl.addDoubleClickListener(new IDoubleClickListener() {
			public void doubleClick(DoubleClickEvent event) {
				TreeSelection trsel = (TreeSelection) _calStackCtrl.getSelection();
				if (null == trsel)
					return;
				Object element = trsel.getFirstElement();
				if (null == element)
					return;
				if (element instanceof TRCMethodInvocation) {
					Utils.openSource(element);
				} else if (element instanceof MethodDetails) {
					MethodDetails md = (MethodDetails) element;
					String pattern = md.name + Utils.fixMethodSignature(md.signature);
					Utils.openSource(pattern);
				} else if (_calStackCtrl.isExpandable(element)) {
					_calStackCtrl.expandToLevel(element, 1);
				}
			}
		});
		UIPlugin.getDefault().addViewSelectionChangedListener(this);
		handleViewSelectionChangedEvent(UIPlugin.getDefault().getViewSelectionChangedEvent());
		_partListener = new PartListener();
	}

	public void dispose() {
		_partListener.dispose();
		UIPlugin.getDefault().removeViewSelectionChangedListener(this);
		super.dispose();
	}

	public void setFocus() {
		if (null != _calStackCtrl)
			_calStackCtrl.getControl().setFocus();
	}

	void update() {
		if (_provider != null)
			update(_provider.getCallStack());
		else
			update(null);
	}

	/** update view content */
	void update(Object selection) {
		if (null == _calStackCtrl)
			return;
		_selection = selection;
		_calStackCtrl.refresh();
		_calStackCtrl.expandAll();
	}

	/**
	 * Compose String name from object
	 * 
	 * @param obj
	 * @return
	 */
	public String getItemText(Object obj) {
		if (obj instanceof TRCMethodInvocation) {
			return Utils.composeMethodName((TRCMethodInvocation) obj);
		}
		if (obj instanceof TRCThread) {
			return Utils.composeThreadName((TRCThread) obj, true);
		}
		if (obj instanceof TRCThreadEvent) {
			return Utils.composeEventName((TRCThreadEvent) obj);
		}
		if (obj instanceof MethodDetails) {
			return Utils.composeMethodName((MethodDetails) obj);
		}
		return obj.toString();
	}

	class CallStackContentProvider implements ITreeContentProvider {

		public void inputChanged(Viewer v, Object oldInput, Object newInput) {
		}

		public void dispose() {
		}

		public Object[] getElements(Object parent) {
			Object[] ret = _empty;
			if (_selection instanceof IStructuredSelection) {
				ret = ((IStructuredSelection)_selection).toArray();
			}
			else if (null != _selection) {
				ret = new Object[] { _selection };
			}
			return ret;
		}

		public Object[] getChildren(Object parentElement) {
			Object[] ret = _empty;
			if (parentElement instanceof TRCThread) {
				TRCThread thread = (TRCThread) parentElement;
				EList list = thread.getThreadEvents();
				if (null != list && list.size() > 0) {
					TRCThreadEvent event = (TRCThreadEvent) list.get(list.size() - 1);
					ret = Utils.getCallStack(event);
				} else {
					TRCMethodInvocation mi;
					ArrayList arrlist = new ArrayList();
					list = thread.getInitialInvocations();
					while (list != null && !list.isEmpty()) {
						mi = (TRCMethodInvocation) list.get(list.size() - 1);
						arrlist.add(mi);
						list = mi.getInvokes();
					}
					ret = arrlist.toArray();
				}
			} else if (parentElement instanceof TRCThreadEvent) {
				ret = Utils.getCallStack((TRCThreadEvent) parentElement);
			}
			return ret;
		}

		public Object getParent(Object element) {
			return null;
		}

		public boolean hasChildren(Object element) {
			boolean ret = false;
			if (element instanceof TRCThread) {
				ret = true;
			} else if (element instanceof TRCThreadEvent) {
				TRCThreadEvent event = (TRCThreadEvent) element;
				EList list = event.getAnnotations();
				ret = list != null && list.size() > 0;
			}
			return ret;
		}
	}

	class CallStackLabelProvider extends LabelProvider {
		public String getText(Object obj) {
			return getItemText(obj);
		}

		public Image getImage(Object obj) {
			return Utils.getItemImage(obj);
		}
	}

	public void handleViewSelectionChangedEvent(ViewSelectionChangedEvent event) {
		update();
	}
}
