/*****************************************************************************
 * Copyright (c) 2007, 2010 Intel Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Intel Corporation - Initial API and implementation
 *    Ruslan A. Scherbakov, Intel - Initial API and implementation
 *
 * $Id: MemoryAnalysisPage.java,v 1.7 2010/08/27 21:49:40 jcayne Exp $ 
 *****************************************************************************/

package org.eclipse.tptp.trace.jvmti.internal.client.views;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.hyades.models.hierarchy.TRCAgentProxy;
import org.eclipse.hyades.models.hierarchy.TRCCollectionMode;
import org.eclipse.hyades.models.hierarchy.TRCConfiguration;
import org.eclipse.hyades.models.hierarchy.TRCOption;
import org.eclipse.hyades.models.hierarchy.extensions.SimpleSearchQuery;
import org.eclipse.hyades.trace.ui.ITraceSelection;
import org.eclipse.hyades.trace.ui.TraceViewer;
import org.eclipse.hyades.trace.ui.TraceViewerPage;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.ui.internal.util.PerftraceUtil;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.CTabFolder;
import org.eclipse.swt.custom.CTabItem;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;

public class MemoryAnalysisPage extends TraceViewerPage {

	static public final int TAB_MEMORY_STATS = 0;
	static public final int TAB_ALLOC_DETAILS = 1;

	CTabFolder _tabFolder;
	BaseStatisticView _views[];

	public MemoryAnalysisPage(EObject mofObject, TraceViewer viewer) {
		super(mofObject, viewer);
	}

	public void refreshPage() {
		BaseStatisticView view = _views[_tabFolder.getSelectionIndex()];
		view.update();
		updateButtons();
	}

	public void selectionChanged() {
	}

	public void update(boolean newPage) {
		if (newPage) {
			BaseStatisticView view = _views[_tabFolder.getSelectionIndex()];
			view.update();
			updateButtons();
		}
	}

	/** createControl is called every time the tab is opened/reopened-after-closing */
	public void createControl(Composite parent) {
		GridData gd = new GridData();
		gd.horizontalAlignment = GridData.FILL;
		gd.grabExcessHorizontalSpace = true;
		gd.verticalAlignment = GridData.FILL;
		gd.grabExcessVerticalSpace = true;

		_tabFolder = new CTabFolder(parent, SWT.BOTTOM | SWT.FLAT);
		_tabFolder.setLayoutData(gd);
		_tabFolder.addSelectionListener(new SelectionListener() {
			public void widgetDefaultSelected(SelectionEvent e) {
				;
			}
			public void widgetSelected(SelectionEvent e) {
				updateAllocDetails();
				BaseStatisticView view = _views[_tabFolder.getSelectionIndex()];
				view.update();
				view.updateModelSelection();
				updateButtons();
			}
		});
		
		// Enable or disable the object allocation sites tab for bug 219379
		boolean objAllocSitesEnabled = true; // default is show both
		
		if(_mofObject instanceof TRCAgentProxy && _mofObject != null) {
			TRCAgentProxy agProxy = (TRCAgentProxy)_mofObject;
			if(agProxy.getCollectionMode() != null) {
				TRCCollectionMode collectionMode = agProxy.getCollectionMode();
				
				// HEAP_STATISTICS_ONLY_LITERAL indicates that object allocation data will not be collected
				if(collectionMode == TRCCollectionMode.HEAP_STATISTICS_ONLY_LITERAL) {
					objAllocSitesEnabled = false;
				}
				
				/*
				 * Bug 323868
				 * If an extension point contributes to allow the Heap Objects to be stored using HEAP_FULL but does
				 * not enable Object Allocation Sites or it is not previously enabled, then the tab for the Allocation Sites
				 * should not be displayed. The profiler agent has not been started with allocSites=true so the tab is not
				 * displayed.
				 */
				if(objAllocSitesEnabled && collectionMode == TRCCollectionMode.HEAP_FULL_LITERAL) {
					for(TRCConfiguration config : agProxy.getConfigurations()) {
						if(config.getName() != null && config.getName().equals("__JVMPI__")) { //$NON-NLS-1$
							for(TRCOption option : config.getOptions()) {
								/* The allocSite launch constant: ITILaunchConfigurationConstants.ATTR_EXEC_OBJ_ALLOC_SITE */
								if(option.getKey().equals("org.eclipse.tptp.platform.jvmti.client.ATTR_EXEC_OBJ_ALLOC_SITE")) { //$NON-NLS-1$
									objAllocSitesEnabled = Boolean.parseBoolean(option.getValue());
									break;
								}
							}
							break;
						}
					}
				}
			}
		}
		
		// If object allocation is enabled we require two views, if not, only 1
		if(objAllocSitesEnabled) {
			_views = new BaseStatisticView[2];
		} else {
			_views = new BaseStatisticView[1];
		}
		
		CTabItem tabItem;

		tabItem = new CTabItem(_tabFolder, SWT.NONE);
		tabItem.setText(UIMessages.MEM_STATISTICS_TITLE);
		BaseMemoryStatisticView memView = new BaseMemoryStatisticView(_tabFolder, this);
		_views[0] = memView;
		tabItem.setControl(_views[0].getControl());
		
		// Object allocations tab, only show if obj alloc sites is enabled
		if(objAllocSitesEnabled) {
			tabItem = new CTabItem(_tabFolder, SWT.NONE);
			tabItem.setText(UIMessages.MEM_ALLOCATION_DETAILS_TITLE);
			AllocationDetailsView allocView = new AllocationDetailsView(_tabFolder, this);
			_views[1] = allocView;
			tabItem.setControl(_views[1].getControl());
	
			memView._viewDetails = allocView;
		}
		// end obj alloc tab
		_tabFolder.setSelection(0);
	}

	public Control getControl() {
		return _tabFolder;
	}

	public void setFocus() {
		_views[_tabFolder.getSelectionIndex()].getControl().setFocus();
	}

	public SimpleSearchQuery getCurrentFilter() {
		return ((MemoryAnalysisViewer)_viewer).getCurrentFilter();
	}

	public int getViewTab() {
		return _tabFolder.getSelectionIndex();
	}

	public int getViewTabViewMode() {
		int viewMode = -1;
		BaseStatisticView view = _views[_tabFolder.getSelectionIndex()];
		if (view instanceof BaseMemoryStatisticView) {
			BaseMemoryStatisticView view0 = (BaseMemoryStatisticView) view;
			viewMode = view0.getViewMode();
		}
		return viewMode;
	}

	public void setViewMode(int tab, int viewMode) {
		if (TAB_MEMORY_STATS != tab && TAB_ALLOC_DETAILS != tab)
			return;
		_tabFolder.setSelection(tab);
		BaseStatisticView view = _views[_tabFolder.getSelectionIndex()];
		if (view instanceof BaseMemoryStatisticView) {
			BaseMemoryStatisticView view0 = (BaseMemoryStatisticView) view;
			view0.setViewMode(viewMode);
		}
		updateButtons();
	}

	public void updateButtons() {
		if (_viewer instanceof MemoryAnalysisViewer) {
			((MemoryAnalysisViewer)_viewer).updateButtons();
		}
	}

	public void updateFilterAppliedDescription() {
		((MemoryAnalysisViewer) _viewer).updateFilterAppliedDescription();
	}

	public BaseStatisticView getCurrentView() {
		BaseStatisticView view = _views[_tabFolder.getSelectionIndex()];
		return view;
	}

	public boolean isShowDelta() {
		BaseStatisticView view = _views[_tabFolder.getSelectionIndex()];
		return view.showingDeltaColumns();
	}

	public boolean isShowPercent() {
		BaseStatisticView view = _views[_tabFolder.getSelectionIndex()];
		return view.isShowPercent();
	}

	void showAllocDetails() {
		setViewMode(TAB_ALLOC_DETAILS, 0);
		updateAllocDetails();
	}

	void updateAllocDetails() {
		BaseStatisticView view = _views[_tabFolder.getSelectionIndex()];
		if (view instanceof AllocationDetailsView) {
			ITraceSelection model = UIPlugin.getDefault().getSelectionModel(getMOFObject());
			if (model.size() > 0) {
				Object obj = model.getFirstElement();
				((AllocationDetailsView) view).showAllocDetails(obj);
			} else
				((AllocationDetailsView) view).showAllocDetails(null);
		}
	}

	public boolean isEmpty() {
		return PerftraceUtil.getTotalSize(getMOFObject()) <= 0;
	}
}
