/*****************************************************************************
 * Copyright (c) 2008, 2010 Intel Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Intel Corporation - Initial API and implementation
 *    Ruslan A. Scherbakov, Intel - Initial API and implementation
 *
 * $Id: MemoryAnalysisViewer.java,v 1.12 2010/05/12 20:02:37 kchan Exp $ 
 *****************************************************************************/

package org.eclipse.tptp.trace.jvmti.internal.client.views;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.hyades.models.hierarchy.TRCAgentProxy;
import org.eclipse.hyades.trace.internal.ui.PDContentProvider;
import org.eclipse.hyades.trace.internal.ui.PDProjectExplorer;
import org.eclipse.hyades.trace.ui.FilterTraceViewer;
import org.eclipse.hyades.trace.ui.ProfileEvent;
import org.eclipse.hyades.trace.ui.TraceViewerPage;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.views.actions.internal.ContextSelectionMenuListener;
import org.eclipse.hyades.trace.views.adapter.internal.IContextViewer;
import org.eclipse.hyades.trace.views.internal.TracePluginImages;
import org.eclipse.hyades.trace.views.internal.TraceUIMessages;
import org.eclipse.hyades.trace.views.internal.view.columnlabels.ContextUpdaterHelper;
import org.eclipse.hyades.ui.filters.IFilterScopes;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.IToolBarManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.tptp.trace.jvmti.internal.client.AllocationSite;
import org.eclipse.tptp.trace.jvmti.internal.client.context.TIContextAttributes;
import org.eclipse.tptp.trace.jvmti.internal.client.widgets.Utils;
import org.eclipse.tptp.trace.jvmti.provisional.client.extension.CustomAnalysisView;
import org.eclipse.tptp.trace.jvmti.provisional.client.extension.ICustomAnalysisView;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.IMemento;
import org.eclipse.ui.IViewSite;
import org.eclipse.ui.PartInitException;

public class MemoryAnalysisViewer extends FilterTraceViewer implements IContextViewer {
	public static final String VIEW_ID = "org.eclipse.tptp.trace.jvmti.internal.client.views.MemoryAnalysisViewer";
	public static final String RUNTIME_ID = "RUNTIME_ID";

	private static final String GROUP_COLUMNS = "actions.columns";
	
	Action _viewPackage;
	Action _viewClass;
	Action _openSource;   
	Action _chooseColumns;   
	Action _sortByColumn;
	Action _percentMode;
	Action _deltaColumns;
	Menu _contextMenu;
	private String _context;
	private boolean _contextChanged;
	private ContextSelectionMenuListener _contextSelectionMenuListener;

	public TraceViewerPage createPage(EObject mofObject) {
		if(_fi != null)
			_fi.restore();
		return new MemoryAnalysisPage(mofObject, this); 
	}

	public void dispose() {
		if (_contextMenu != null)
			_contextMenu.dispose();
		super.dispose();
	}

	public Object[] getViewerControls() {
		BaseProfilerView view = getCurrentView();
		if (view instanceof BaseStatisticView) {
			return new Object[] { ((BaseStatisticView)view).getTree() };
		}
		return null;
	}

	public String getViewTitle() {
		return UIMessages.MEM_ANALYSIS_TITLE;
	}

	public boolean contextChanged() {
		return _contextChanged;
	}

	public ContextSelectionMenuListener getContextSelectionMenuListener() {
		if (_contextSelectionMenuListener == null)
			_contextSelectionMenuListener = new ContextSelectionMenuListener();
		return _contextSelectionMenuListener;
	}

	public String getSelectedContext() {
		return _context;
	}

	public void setSelectedContext(String context) {
		setSelectedContext(context, true);
	}

	public void setSelectedContext(String context, boolean refresh) {
		if (context == null)
			_contextChanged = _context != null;
		else
			_contextChanged = !context.equals(_context);
		_context = context;
		if (_contextChanged && refresh && getCurrentPage() != null)
			((TraceViewerPage) getCurrentPage()).refreshPage();
	}

	public void updateContext() {
		ContextUpdaterHelper.setCurrentFocusContextLanguage(getSelectedContext());
		_contextChanged = false;
	}

	public void filterUpdated() {
		TraceViewerPage page = (TraceViewerPage) getCurrentPage();
		if (page != null) {
			if (fPartVisible)
				page.refreshPage();
			else
				fForceRefresh = true;
		}
	}

	public String[] getFilterScope() {
		String[] scope = new String[] { IFilterScopes.PROFILING_AGENT_FILTER_SCOPE };
		return scope;
	}

	public void makeActions() {
		if (isInitializedMenu())
			return; //create the toolbar only once
		super.makeActions();
		initializedMenu(true);
		createActions();
		updateButtons();
		IActionBars bars = getViewSite().getActionBars();
		fillLocalPullDown(bars.getMenuManager());
		fillLocalToolBar(bars.getToolBarManager());
		bars.updateActionBars();
	}

	public void updateButtons() {
		MemoryAnalysisPage page = null;
		if (getCurrentPage() instanceof MemoryAnalysisPage)
			page = (MemoryAnalysisPage) getCurrentPage();
		boolean memStatsTab = null != page && page.getViewTab() == MemoryAnalysisPage.TAB_MEMORY_STATS;
		boolean memAllocTab = null != page && page.getViewTab() == MemoryAnalysisPage.TAB_ALLOC_DETAILS;
		boolean classMode = memStatsTab && page.getViewTabViewMode() == BaseMemoryStatisticView.CLASS_MODE;
		boolean packageMode = memStatsTab && page.getViewTabViewMode() == BaseMemoryStatisticView.PACKAGE_MODE;
		boolean isDelta = memStatsTab && page.isShowDelta();
		boolean isPercent = null != page && page.isShowPercent();
		_viewClass.setEnabled(memStatsTab);
		_viewClass.setChecked(classMode);
		_viewPackage.setEnabled(memStatsTab);
		_viewPackage.setChecked(packageMode);
		boolean canSrc = null != page && Utils.canOpenSourceOnThisObject(page.getMOFObject());
		_openSource.setEnabled(canSrc);
		_deltaColumns.setEnabled(memStatsTab);
		_percentMode.setEnabled(memAllocTab || memStatsTab);
		_deltaColumns.setChecked(isDelta);
		_percentMode.setChecked(isPercent);
		setFilterActionEnabled(memStatsTab);
	}

	void createActions() {
		_viewClass = new Action() {
			public void run() {
				MemoryAnalysisPage page = (MemoryAnalysisPage) getCurrentPage();
				page.setViewMode(0, BaseMemoryStatisticView.CLASS_MODE);
			}
		};
		_viewPackage = new Action() {
			public void run() {
				MemoryAnalysisPage page = (MemoryAnalysisPage) getCurrentPage();
				page.setViewMode(0, BaseMemoryStatisticView.PACKAGE_MODE);
			}
		};
		Utils.setActionProperties(_viewPackage, _context, TIContextAttributes.SHOW_PACKAGE_LEVEL);
		Utils.setActionProperties(_viewClass, _context, TIContextAttributes.SHOW_CLASS_LEVEL);

		String openSourceStr = TraceUIMessages._74;
		_openSource = new Action(openSourceStr) {
			public void run() {
				MemoryAnalysisPage page = (MemoryAnalysisPage) getCurrentPage();
				if (null != page) {
					// Bug 302008: Obtain the Element to get the source for.
					Object obj = UIPlugin.getDefault().getSelectionModel(page.getMOFObject()).getFirstElement();
					// For an Allocation Site, get a String representation of the location to open the source.
					if(obj instanceof AllocationSite) {
						Utils.openSource(Utils.composeMethodName((AllocationSite)obj));
					} else {
						Utils.openSource(obj);
					}
				}
			}
		};
		_openSource.setText(openSourceStr);
		TracePluginImages.setImageDescriptors(_openSource,TracePluginImages.T_TOOL,TracePluginImages.IMG_SOURCE);	
		_openSource.setDescription(openSourceStr);   
		_openSource.setToolTipText(openSourceStr);
		_openSource.setEnabled(false);

		String strShowPercent = TraceUIMessages._13;
		_percentMode = new Action(strShowPercent) {
			public void run() {
				MemoryAnalysisPage page = (MemoryAnalysisPage) getCurrentPage();
				if (null != page)
					page.getCurrentView().getShowPercentAction().run();
			}
		};
		_percentMode.setText(strShowPercent);
		TracePluginImages.setImageDescriptors(_percentMode,TracePluginImages.T_LCL,TracePluginImages.IMG_PERCENT);			
		_percentMode.setDescription(strShowPercent);   
		_percentMode.setToolTipText(strShowPercent);
		_percentMode.setEnabled(true);
		
		String strShowDelta = TraceUIMessages._84;
		_deltaColumns = new Action(strShowDelta) {
			public void run() {
				MemoryAnalysisPage page = (MemoryAnalysisPage) getCurrentPage();
				if (null != page)
					page.getCurrentView().getDeltaColumnsAction().run();
			}
		};

		_deltaColumns.setText(strShowDelta);
		TracePluginImages.setImageDescriptors(_deltaColumns,TracePluginImages.T_LCL,TracePluginImages.IMG_DELTACOLUMNS);
		_deltaColumns.setDescription(strShowDelta);   
		_deltaColumns.setToolTipText(strShowDelta);
		_deltaColumns.setEnabled(true);
		
		// Bug 308106 - Add column chooser/resizer
		String strChooseColumns = UIMessages._ChooseColumns;
		_chooseColumns = new Action(strChooseColumns) {
			public void run() {
				BaseProfilerView view = getCurrentView(); 
				if (view instanceof BaseStatisticView) {
					((BaseStatisticView) view).getChooseColumnsAction().run();
				}
			}
		};
		_chooseColumns.setText(strChooseColumns);
		TracePluginImages.setImageDescriptors(_chooseColumns,
				TracePluginImages.T_LCL, TracePluginImages.IMG_CHOOSECOLUMNS);
		_chooseColumns.setDescription(strChooseColumns);
		_chooseColumns.setToolTipText(strChooseColumns);
		
		String strSortByColumn = UIMessages._SortByColumn;
		_sortByColumn = new Action(strSortByColumn) {
			public void run() {
				BaseProfilerView view = getCurrentView(); 
				if (view instanceof ThreadStatisticView) {
					((ThreadStatisticView) view).getSortByColumnAction().run();
				}
			}
		};
		_sortByColumn.setText(strSortByColumn);
		TracePluginImages.setImageDescriptors(_sortByColumn,
				TracePluginImages.T_LCL, TracePluginImages.IMG_SORTCOLUMNS);
		_sortByColumn.setDescription(strSortByColumn);
		_sortByColumn.setToolTipText(strSortByColumn);		
	}

	void fillLocalPullDown(IMenuManager manager) {
		//manager.removeAll();
		manager.add(_openSource);
		manager.add(new Separator(GROUP_COLUMNS));
		manager.add(_chooseColumns);
		manager.add(_sortByColumn);
	}

	void fillLocalToolBar(IToolBarManager manager) {
		//manager.removeAll();
		manager.add(_viewPackage);
		manager.add(_viewClass);
		manager.add(_openSource);
		manager.add(_percentMode);
		manager.add(_deltaColumns);
	}

	public void setFocus() {
		if (getCurrentPage() != null)
			getCurrentPage().setFocus();
	}

	public BaseProfilerView getCurrentView() {
		if (!(getCurrentPage() instanceof MemoryAnalysisPage))
			return null;
		MemoryAnalysisPage page = (MemoryAnalysisPage) getCurrentPage();
		if (page.getControl() == null || page.getControl().isDisposed())
			return null;
		return page.getCurrentView();
	}
	
	public String getViewID() {
		return VIEW_ID;
	}

	public EObject getObjectToView(EObject selObject) {
		return getObjectToView_(selObject);
	}

	public void init(IViewSite site, IMemento memento) throws PartInitException {
		super.init(site, memento);
		if (memento == null) return;

		final String rid = memento.getString(RUNTIME_ID);
		if (rid == null) return;

		loadingTrace();
		
		site.getShell().getDisplay().asyncExec(new Runnable() {
			public void run() {
				PDProjectExplorer.getFromActivePerspective();
				
				EObject obj = PDContentProvider.getAgent(rid);
				if (obj != null) 
					addViewPage(obj, true);
				else
					loadingTraceDone();
			}
		});
	}
	
	public void saveState (IMemento memento) {
		String rid = null;
		if (_mofObject != null && _mofObject instanceof TRCAgentProxy) {
			rid = ((TRCAgentProxy) _mofObject).getRuntimeId();
		}

		if (rid != null) {
			memento.putString(RUNTIME_ID, rid);
			super.saveState(memento);
		}
	}
	
	public void handleProfileEvent(ProfileEvent event) {
		ICustomAnalysisView[] customViewProviders = CustomAnalysisView.getContributors();
		if(customViewProviders != null) {
			BaseProfilerView view = getCurrentView();
			// Check if any overriding of how to handle the element for the view and 
			// update the view if necessary.
			for(int provider = 0; provider < customViewProviders.length; provider++) {
				customViewProviders[provider].handleProfileEvent(event, view);
			}
		}
		
		super.handleProfileEvent(event);
	}
}
