/**********************************************************************
 * Copyright (c) 2008, Intel Corporation.
 * All rights reserved. This content is made available under
 * the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html       
 * $Id: MonitorCallDetails.java,v 1.2 2008/04/14 11:22:28 aalexeev Exp $
 *
 * Contributors:
 *    Intel Corporation - Initial API and implementation
 *    Alexander  N. Alexeev, Intel - Initial API and implementation
 **********************************************************************/  

package org.eclipse.tptp.trace.jvmti.internal.client.views;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;

import org.eclipse.emf.common.util.EList;
import org.eclipse.hyades.models.hierarchy.TRCAnnotation;
import org.eclipse.hyades.models.trace.TRCThreadEvent;
import org.eclipse.hyades.models.trace.TRCThreadSleepingEvent;
import org.eclipse.hyades.models.trace.TRCThreadWaitingForLockEvent;
import org.eclipse.hyades.models.trace.TRCThreadWaitingForObjectEvent;

public class MonitorCallDetails {
	final static String ANNATATION_NAME_LINES = "callStackDumpLineNumbers";
	final static String ANNATATION_NAME_METHODS = "callStackDumpMethods";
	final static String WAIT_METHOD_STR_CONST = "java.lang.Object.wait"; 

	protected ArrayList _waitDeltas = new ArrayList();
	protected ArrayList _blockDeltas = new ArrayList();
	protected HashSet _waitThreadsSet = new HashSet(); 
	protected HashSet _waitObjectsSet = new HashSet();
	protected HashSet _blockThreadsSet = new HashSet(); 
	protected HashSet _blockObjectsSet = new HashSet();
	
	//private ArrayList _methods = new ArrayList(); 
	//private ArrayList _lines = new ArrayList();
	
	private String _idString = "";
	private String _callTitle = "";
	private String _stackString = "";
	private boolean _isObjectWaiting;

	protected double _sleepingTime;
	protected int _sleepingCount;

	protected double _waitingTime;
	protected double _maxWaitingTime = 0;
	protected double _minWaitingTime = Double.MAX_VALUE;
	protected double _avgWaitingTime;
	protected int _waitCount;

	protected double _blockedTime;
	protected double _maxBlockedTime = 0;
	protected double _minBlockedTime = Double.MAX_VALUE;
	protected double _avgBlockedTime;
	protected int _blockedCount;
	
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result
				+ ((_idString == null) ? 0 : _idString.hashCode());
		result = prime * result + (_isObjectWaiting ? 1231 : 1237);
		return result;
	}

	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		MonitorCallDetails other = (MonitorCallDetails) obj;
		if (_idString == null) {
			if (other._idString != null)
				return false;
		} else if (!_idString.equals(other._idString))
			return false;
		if (_isObjectWaiting != other._isObjectWaiting)
			return false;
		return true;
	}

	MonitorCallDetails(TRCThreadEvent event) {
		_isObjectWaiting = false;
		if (event instanceof TRCThreadWaitingForObjectEvent) {
			_isObjectWaiting = true;
		}
		
		TRCAnnotation methodsAnn = null;
		TRCAnnotation linesAnn = null;
	    for (Iterator it=event.getAnnotations().iterator(); it.hasNext(); ) {
	        Object element = it.next();
	        if (element instanceof TRCAnnotation) {
	        	TRCAnnotation ann = (TRCAnnotation)element;
	        	if (ann.getName().equals(ANNATATION_NAME_METHODS)) {
	        		methodsAnn = ann;
	        	} else if (ann.getName().equals(ANNATATION_NAME_LINES)) {
	        		linesAnn = ann;
	        	}
			}
	    }
	    
	    if (null != methodsAnn) {
	    	_idString = "";
	    	EList listMeth = methodsAnn.getValues();
	    	EList listLines = null;
	    	
	    	if (linesAnn != null){
	    		listLines = linesAnn.getValues();
	    	}
	    	
	    	if (!_isObjectWaiting && listMeth.size() > 0) {
				_callTitle = listMeth.get(0).toString();
	    	}
	    	
	    	boolean titleFound = false;
			for (int i = 0; i < listMeth.size(); i++) {
				if (_isObjectWaiting) {
					if (!titleFound && 
						_callTitle.startsWith(WAIT_METHOD_STR_CONST) &&
						!listMeth.get(i).toString().startsWith(WAIT_METHOD_STR_CONST)) {
						_callTitle = listMeth.get(i).toString();
						titleFound = true;
					} else if (!titleFound){
						_callTitle = listMeth.get(i).toString();
					}
				}
				_idString += listMeth.get(i);
				_stackString += listMeth.get(i);
				
				if (null != listLines && i < listLines.size()) {
					_idString += listLines.get(i);
					_stackString += ":" + listLines.get(i);
				}
				_stackString += "\n";
				
			}
		}
    }
	
	public String getCallTitle() {
		return _callTitle;
	}

	public String getStackText() {
		return _stackString;
	}
	
	public int getBlockedCount() {
		return _blockedCount;
	}
	
	public int getWaitingCount() {
		return _waitCount;
	}
	
	public int getSleepingCount() {
		return _sleepingCount;
	}
	
	public double getBlockedTime() {
		return _blockedTime;
	}

	public double getSleepingTime() {
		return _sleepingTime;
	}
	
	public double getWaitingTime() {
		return _waitingTime;
	}
	
	public double getMinBlockedTime() {
		return _minBlockedTime;
	}
	
	public double getAvgBlockedTime() {
		return _avgBlockedTime;
	}

	public double getMaxBlockedTime() {
		return _maxBlockedTime;
	}

	public double getMinWaitingTime() {
		return _minWaitingTime;
	}
	
	public double getAvgWaitingTime() {
		return _avgWaitingTime;
	}

	public double getMaxWaitingTime() {
		return _maxWaitingTime;
	}

	public int getWaitingObjectsCount() {
		return _waitObjectsSet.size();
	}

	public int getBlockedObjectsCount() {
		return _blockObjectsSet.size();
	}

	public int getWaitingThreadsCount() {
		return _waitThreadsSet.size();
	}

	public int getBlockedThreadsCount() {
		return _blockThreadsSet.size();
	}
	
	public void updateCallData(TRCThreadEvent event, double timeSpan) {
		if(event instanceof TRCThreadSleepingEvent) {
			_sleepingTime += timeSpan;
			_sleepingCount++;
		} else if(event instanceof TRCThreadWaitingForObjectEvent) {
			_waitingTime += timeSpan;
			_waitCount++;
			_avgWaitingTime = _waitingTime/_waitCount;
			if (timeSpan > _maxWaitingTime) _maxWaitingTime = timeSpan;
			if (timeSpan < _minWaitingTime) _minWaitingTime = timeSpan;
			_waitThreadsSet.add(event.getThread());
			_waitObjectsSet.add(((TRCThreadWaitingForObjectEvent)event).getObjectWaitingFor());
		} else if(event instanceof TRCThreadWaitingForLockEvent) {
			_blockedTime += timeSpan;
			_blockedCount++;
			_avgBlockedTime = _blockedTime/_blockedCount;
			if (timeSpan > _maxBlockedTime) _maxBlockedTime = timeSpan;
			if (timeSpan < _minBlockedTime) _minBlockedTime = timeSpan;
			_blockThreadsSet.add(event.getThread());
			_blockObjectsSet.add(((TRCThreadWaitingForLockEvent)event).getLockedObject());
		}
	}
}
