/**********************************************************************
 * Copyright (c) 2008, Intel Corporation.
 * All rights reserved. This content is made available under
 * the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html       
 * $Id: MonitorClassStatistics.java,v 1.1 2008/03/12 16:54:03 aalexeev Exp $
 *
 * Contributors:
 *    Intel Corporation - Initial API and implementation
 *    Alexander  N. Alexeev, Intel - Initial API and implementation
 **********************************************************************/  

package org.eclipse.tptp.trace.jvmti.internal.client.views;

import org.eclipse.hyades.ui.util.HyadesFormToolkit;
import org.eclipse.jface.viewers.IBaseLabelProvider;
import org.eclipse.jface.viewers.IContentProvider;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.StructuredViewer;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerSorter;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.ViewForm;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Layout;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.swt.widgets.TreeColumn;
import org.eclipse.swt.widgets.TreeItem;
import org.eclipse.tptp.trace.jvmti.internal.client.widgets.Utils;
import org.eclipse.ui.forms.widgets.ColumnLayout;
import org.eclipse.ui.forms.widgets.Section;

public class MonitorClassStatistics extends LabelProvider implements ITableLabelProvider, SelectionListener{

	// MONSTAT_CLASSES_COL_CLASS_NAME
	private final static int MONSTAT_CLASSES_COL_CLASS_NAME_NUM = 0;
	private final static int MONSTAT_CLASSES_COL_CLASS_NAME_DEFW = 260;
	// MONSTAT_THREADS_COL_BLOCKED_COUNT
	private final static int MONSTAT_CLASSES_COL_BLOCKED_COUNT_NUM = 1;
	private final static int MONSTAT_CLASSES_COL_BLOCKED_COUNT_DEFW = 100;
	// MONSTAT_CLASSES_COL_BLOCKED_TIME
	private final static int MONSTAT_CLASSES_COL_BLOCKED_TIME_NUM = 2;
	private final static int MONSTAT_CLASSES_COL_BLOCKED_TIME_DEFW = 100;
	// MONSTAT_CLASSES_COL_WAITED_COUNT
	private final static int MONSTAT_CLASSES_COL_WAITED_COUNT_NUM = 3;
	private final static int MONSTAT_CLASSES_COL_WAITING_COUNT_DEFW = 100;
	// MONSTAT_CLASSES_COL_WAITED_TIME
	private final static int MONSTAT_CLASSES_COL_WAITED_TIME_NUM = 4;
	private final static int MONSTAT_CLASSES_COL_WAITING_TIME_DEFW = 100;
	
	//overall column number
	private final static int MONSTAT_CLASSES_COL_NUM = 5;
	
	
	protected Composite _viewContainer;
	protected ViewForm _dataPane;
	protected StructuredViewer _viewer;
	protected Layout _layout = null;
	
	protected HyadesFormToolkit _toolkit;
	protected ClassStatisticsSorter _sorter;
	
	MonitorStatisticController monViewCtrl;
	
	public MonitorClassStatistics(Section section, MonitorStatisticController monViewCtrl) {
		this.monViewCtrl = monViewCtrl;
		_toolkit = new HyadesFormToolkit(section.getDisplay());		
		_dataPane = _toolkit.createViewForm(section);
		_viewContainer = _dataPane;
		Control table = createTableViewer(_dataPane);
		_dataPane.setContent(table);	
	}
	
	public Layout getLayout() {
		if (_layout == null)
		{
			_layout = new ColumnLayout();
			((ColumnLayout)_layout).maxNumColumns = 100;
		}
		return _layout;
	}		
	
	private Control createTableViewer(ViewForm parent) {
		Tree tree = _toolkit.createTree(parent, SWT.FULL_SELECTION | SWT.FLAT);
		
		tree.setLinesVisible(true);
		tree.setHeaderVisible(true);
		tree.setLayout(getLayout());
		tree.setLayoutData(new GridData(GridData.FILL_BOTH));
		tree.addSelectionListener(this);
		
		_viewer = new TreeViewer(tree);
		
		_viewer.setContentProvider(getContentProvider());
		_viewer.setLabelProvider(getTableLabelProvider());

		_sorter = new ClassStatisticsSorter();

		_viewer.setSorter(_sorter);
				
		createColumns(tree);		
		_sorter.changeColumnSelection(0);
		
		monViewCtrl.setClassesTimeViewer(_viewer);
		return _viewer.getControl();
	}

	private void createColumns(Tree tree) {

		int colWidth[] = new int[MONSTAT_CLASSES_COL_NUM];
		String colTitle[] = new String[MONSTAT_CLASSES_COL_NUM];
		TreeColumn columns[] = new TreeColumn[MONSTAT_CLASSES_COL_NUM];
		
		colTitle[MONSTAT_CLASSES_COL_CLASS_NAME_NUM] = UIMessages._THREAD_CLASS_NAME;
		colWidth[MONSTAT_CLASSES_COL_CLASS_NAME_NUM] = MONSTAT_CLASSES_COL_CLASS_NAME_DEFW;
		colTitle[MONSTAT_CLASSES_COL_BLOCKED_COUNT_NUM] = UIMessages._BLOCKED_COUNT;
		colWidth[MONSTAT_CLASSES_COL_BLOCKED_COUNT_NUM] = MONSTAT_CLASSES_COL_BLOCKED_COUNT_DEFW;
		colTitle[MONSTAT_CLASSES_COL_BLOCKED_TIME_NUM] = UIMessages._THREAD_BLOCKED_TIME;
		colWidth[MONSTAT_CLASSES_COL_BLOCKED_TIME_NUM] = MONSTAT_CLASSES_COL_BLOCKED_TIME_DEFW;
		colTitle[MONSTAT_CLASSES_COL_WAITED_COUNT_NUM] = UIMessages._WAITING_CONUT;
		colWidth[MONSTAT_CLASSES_COL_WAITED_COUNT_NUM] = MONSTAT_CLASSES_COL_WAITING_COUNT_DEFW;
		colTitle[MONSTAT_CLASSES_COL_WAITED_TIME_NUM] = UIMessages._THREAD_WAITING_TIME;
		colWidth[MONSTAT_CLASSES_COL_WAITED_TIME_NUM] = MONSTAT_CLASSES_COL_WAITING_TIME_DEFW;
		
		for (int i = 0; i < MONSTAT_CLASSES_COL_NUM; i++) {
			columns[i] = new TreeColumn(tree, SWT.NULL | SWT.NO_BACKGROUND | SWT.LEFT);
			columns[i].setResizable(true);
			columns[i].setText(colTitle[i]);
			columns[i].setWidth(colWidth[i]);
			columns[i].addSelectionListener(getColumnSelectionListener(i));
		}
		
		_sorter.setColumns(columns);
	}

	SelectionListener getColumnSelectionListener(final int columnIndex) {
		return new SelectionListener(){
			public void widgetSelected(SelectionEvent e) {
				_sorter.changeColumnSelection(columnIndex);
				_viewer.refresh();
			}
			public void widgetDefaultSelected(SelectionEvent e) {
			}
		};
	}
	
	
	
	private IBaseLabelProvider getTableLabelProvider() {
		return this;
	}

	private IContentProvider getContentProvider() {
		return new MonitorClassesStatisticContentProvider();
	}

	public Composite getViewContainer() {
		return _viewContainer;
	}

	public void widgetDefaultSelected(SelectionEvent e) {
	}
 
	public void widgetSelected(SelectionEvent e) {
		if (!(e.widget instanceof Tree)) {
			// TODO add assert
			return;
		}
		
		Tree tree = (Tree)e.widget;
		TreeItem[] selectedItems = tree.getSelection();
		MonitorClassDetails[] classes = new MonitorClassDetails[selectedItems.length];
		for (int i = 0; i < selectedItems.length; i++) {
			classes[i] = (MonitorClassDetails)selectedItems[i].getData();
		}
		monViewCtrl.setSelectedClass(classes[0]);
	}	
	
	public Image getColumnImage(Object element, int columnIndex) {
		return null;
	}

	public String getColumnText(Object element, int columnIndex) {
		if (!(element instanceof MonitorClassDetails)) {
			return null;
		}
		MonitorClassDetails monClassDetails = (MonitorClassDetails)element;		
		
		switch (columnIndex) {
		case MONSTAT_CLASSES_COL_CLASS_NAME_NUM:
			return monClassDetails.getClassObj().getName();
		case MONSTAT_CLASSES_COL_BLOCKED_COUNT_NUM:
			return new Integer(monClassDetails.getBlockedCount()).toString();
		case MONSTAT_CLASSES_COL_BLOCKED_TIME_NUM:
			return Utils.formatTime(monClassDetails.getBlockedTime());
		case MONSTAT_CLASSES_COL_WAITED_COUNT_NUM:
			return new Integer(monClassDetails.getWaitingCount()).toString();
		case MONSTAT_CLASSES_COL_WAITED_TIME_NUM:
			return Utils.formatTime(monClassDetails.getWaitingTime());
		default:
			break;
		}
		return null;		
		
	}	
		
	class ClassStatisticsSorter extends ViewerSorter {

		private int sortedIdx = -1;
		private boolean ascOrder = true;
		//private TreeColumn columns[] = null;
		
		void changeColumnSelection(int idx) {
			if (sortedIdx == idx) {
				ascOrder = !ascOrder;
			}
			else {
				ascOrder = true;
			} 
			sortedIdx = idx;
		}
		
		void setColumns(TreeColumn columns[]) {
			//this.columns = columns;
		}
		
		public int compare(Viewer viewer, Object e1, Object e2) {
			if (!(e1 instanceof MonitorClassDetails) ||
				!(e2 instanceof MonitorClassDetails)) {
				return 0;
			}
			
			MonitorClassDetails cd1 = (MonitorClassDetails)e1;
			MonitorClassDetails cd2 = (MonitorClassDetails)e2;
			int result = 0; 
			
			switch (sortedIdx) {
			case MONSTAT_CLASSES_COL_CLASS_NAME_NUM:
				result = cd1.getClassObj().getName()
					.compareTo(cd2.getClassObj().getName());
				break;
			case MONSTAT_CLASSES_COL_BLOCKED_COUNT_NUM:
				result = cd1.getBlockedCount() - cd2.getBlockedCount();
				break;
			case MONSTAT_CLASSES_COL_BLOCKED_TIME_NUM:
				result = new Double(cd1.getBlockedTime())
					.compareTo(new Double (cd2.getBlockedTime()));
				break;
			case MONSTAT_CLASSES_COL_WAITED_COUNT_NUM:
				result = cd1.getWaitingCount() - cd2.getWaitingCount();
				break;
			case MONSTAT_CLASSES_COL_WAITED_TIME_NUM:
				result = new Double(cd1.getWaitingTime())
					.compareTo(new Double (cd2.getWaitingTime()));
				break;
			default:
				break;		
			}
			
			return ascOrder ? result : -result;
		}
		
		public boolean isSorterProperty(Object element, String property) {
			 return true;
		}
		
	}	
}

class MonitorClassesStatisticContentProvider implements ITreeContentProvider {
	Object[] classes = null;
	
	public Object[] getChildren(Object parentElement) {
		return null;
	}

	public Object getParent(Object element) {
		return null;
	}

	public boolean hasChildren(Object element) {
		return false;
	}

	public Object[] getElements(Object inputElement) {
		return classes;
	}

	public void dispose() {
	}

	public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {
		if (newInput instanceof Object[]) {
			classes = (Object[])newInput;
		}
	}
	
}
