/**********************************************************************
 * Copyright (c) 2008 Intel Corporation.
 * All rights reserved. This content is made available under
 * the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html       
 * $Id: MonitorThreadDetails.java,v 1.2 2008/05/30 20:04:44 jkubasta Exp $
 *
 * Contributors:
 *    Intel Corporation - Initial API and implementation
 *    Alexander  N. Alexeev, Intel - Initial API and implementation
 **********************************************************************/  

package org.eclipse.tptp.trace.jvmti.internal.client.views;

import java.util.HashMap;
import java.util.Map;

import org.eclipse.emf.common.util.EList;
import org.eclipse.hyades.models.trace.TRCClass;
import org.eclipse.hyades.models.trace.TRCObject;
import org.eclipse.hyades.models.trace.TRCThread;
import org.eclipse.hyades.models.trace.TRCThreadEvent;
import org.eclipse.hyades.models.trace.TRCThreadSleepingEvent;
import org.eclipse.hyades.models.trace.TRCThreadWaitingForLockEvent;
import org.eclipse.hyades.models.trace.TRCThreadWaitingForObjectEvent;
import org.eclipse.hyades.trace.ui.internal.util.PerftraceUtil;


public class MonitorThreadDetails{

	protected HashMap _classMap = new HashMap(); 
	
	protected TRCThread _thread;
	
	protected double _lastUpdateTime = 0;
	
	protected double _sleepTime;
	protected int _sleepCount;

	protected double _waitTime;
	protected int _waitCount;

	protected double _blockTime;
	protected int _blockCount;
	
	public MonitorThreadDetails(TRCThread thread) {
		_thread = thread;
		recalculate();
	}

	public void recalculate() {
		EList events = _thread.getThreadEvents();
		int count = events.size();
		if (count > 0) {
			TRCThreadEvent event = (TRCThreadEvent)events.get(0);
			for (int i = 1; i < count; i++){
				TRCThreadEvent next = (TRCThreadEvent)events.get(i);
				double timeSpan = Math.max(next.getTime() - event.getTime(), 0);
				
				updateMonitorsData(event, timeSpan);
				MonitorClassDetails monObjectClass = getObjectClass(event);
				if (null != monObjectClass) {
					monObjectClass.updateMonitorsData(event, timeSpan);
				}
				
				event = next;
			}
			_lastUpdateTime = ((TRCThreadEvent)events.get(count - 1)).getTime();
		}
	}

	private void updateMonitorsData(TRCThreadEvent event, double timeSpan) {
		if(event instanceof TRCThreadSleepingEvent) {
			_sleepTime += timeSpan;
			_sleepCount++;
		} else if(event instanceof TRCThreadWaitingForObjectEvent) {
			_waitTime += timeSpan;
			_waitCount++;
		} else if(event instanceof TRCThreadWaitingForLockEvent) {
			_blockTime += timeSpan;
			_blockCount++;
		}
	}

	private MonitorClassDetails getObjectClass(TRCThreadEvent event) {
		TRCObject monObject = null;
		if(event instanceof TRCThreadWaitingForObjectEvent) {
			monObject = ((TRCThreadWaitingForObjectEvent)event).getObjectWaitingFor();
		} else if(event instanceof TRCThreadWaitingForLockEvent) {
			monObject = ((TRCThreadWaitingForLockEvent)event).getLockedObject();
		} else {
			return null;
		}
		if (monObject == null) {
			// information about monitor isn't provided in event
			return null;
		}
		TRCClass monObjClass =  PerftraceUtil.getClass(monObject);
		if (!_classMap.containsKey(monObjClass)) {
			MonitorClassDetails classDetails = new MonitorClassDetails(monObjClass);
			_classMap.put(monObjClass, classDetails);
			return classDetails;
		}
		return (MonitorClassDetails)_classMap.get(monObjClass);
	}
	
	public TRCThread getThread() {
		return _thread;
	}
	
	public int getBlockedCount() {
		return _blockCount;
	}
	
	public int getWaitCount() {
		return _waitCount;
	}
	
	public int getSleepCount() {
		return _sleepCount;
	}
	
	public double getBlockTime() {
		return _blockTime;
	}

	public double getSleepTime() {
		return _sleepTime;
	}
	
	public double getWaitingTime() {
		return _waitTime;
	}
	
	public Map getClassMap() {
		return _classMap;
	}
}
