/*****************************************************************************
 * Copyright (c) 2007, 2010 Intel Corporation and others
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Intel Corporation - Initial API and implementation
 *    Ruslan A. Scherbakov, Intel - Initial API and implementation
 *    Alexander N. Alexeev, Intel - Add monitors statistics support
 *
 * $Id: ThreadStatesView.java,v 1.8 2010/05/18 20:41:47 jwest Exp $ 
 *****************************************************************************/

package org.eclipse.tptp.trace.jvmti.internal.client.views;

import org.eclipse.emf.common.util.EList;
import org.eclipse.hyades.models.trace.TRCThread;
import org.eclipse.hyades.models.trace.TRCThreadEvent;
import org.eclipse.hyades.trace.ui.TraceViewerPage;
import org.eclipse.hyades.trace.ui.internal.util.PerftraceUtil;
import org.eclipse.hyades.trace.views.util.internal.ThreadDetails;
import org.eclipse.hyades.ui.util.GridUtil;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ControlAdapter;
import org.eclipse.swt.events.ControlEvent;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.tptp.trace.jvmti.internal.client.widgets.ITimeDataProvider;
import org.eclipse.tptp.trace.jvmti.internal.client.widgets.ThreadStatesCtrl;
import org.eclipse.tptp.trace.jvmti.internal.client.widgets.ThreadsTipHandler;
import org.eclipse.tptp.trace.jvmti.internal.client.widgets.TimeScaleCtrl;
import org.eclipse.tptp.trace.jvmti.internal.client.widgets.TraceColorScheme;
import org.eclipse.tptp.trace.jvmti.internal.client.widgets.Utils;

/** 
 * Data is populated into this view using the 
 * - void setTimeRange(Object threads[]) method.
 */
public class ThreadStatesView extends BaseProfilerView implements
		ITimeDataProvider, SelectionListener {

	/** vars */
	private double _minTimeInterval;
	private double _selectedTime;
	private double _beginTime;
	private double _endTime;

	/** Refers to the start of the viewpoint (what part of the graph the user is currently viewing) in the graph */
	private double _time0;

	/** Refers to the end of the viewpoint (what part of the graph the user is currently viewing) in the graph */
	private double _time1;
	
	/** Refers to the time at which thread data begins in the graph.
	* This variable was previously called '_time0_' */
	private double _timeMinTime;
	
	/** Refers to the time at which thread data (currently) ends in the graph.
 	* This variable was previously called '_time1_' */
	private double _timeMaxTime;
	
	private boolean _timeRangeFixed;
	private int _nameWidth;
	private Composite _dataViewer;
	private ThreadStatesCtrl _stateCtrl;
	private TimeScaleCtrl _timeScaleCtrl;
	private ThreadsTipHandler _threadTip;
	private TraceColorScheme _colors;

	/** ctor */
	protected ThreadStatesView(Composite parent, TraceViewerPage page) {
		super(parent, page);
	}

	public void controlMoved(ControlEvent e) {
	}

	public void controlResized(ControlEvent e) {
		resizeControls();
	}

	public void update() {
		if (null != _stateCtrl) {
			refreshData();
			_stateCtrl.redraw();
			_timeScaleCtrl.redraw();
		}
	}

	public void selectionChanged() {
	}

	protected String getViewTypeStr() {
		return "viewoption.threads";
	}	

	int getMarginWidth(int idx) {
		return 0;
	}

	int getMarginHeight(int idx) {
		return 0;
	}
	
	void loadOptions() {
		_minTimeInterval = .000001;
		_selectedTime = -1;
		_nameWidth = Utils.loadIntOption(getPreferenceString("namewidth"), 96, 6, 1000);
	}

	void saveOptions() {
		Utils.saveIntOption(getPreferenceString("namewidth"), _nameWidth);
	}

	protected Control createDataViewer(Composite parent) {
		loadOptions();
		_colors = new TraceColorScheme();
		_dataViewer = new Composite(parent, SWT.NULL);
		_dataViewer.setLayoutData(GridUtil.createFill());
		_timeScaleCtrl = new TimeScaleCtrl(_dataViewer, _colors);
		_timeScaleCtrl.setTimeProvider(this);
		_timeScaleCtrl.setLayoutData(GridUtil.createFill());
		_stateCtrl = new ThreadStatesCtrl(_dataViewer, _colors);
		_stateCtrl.setTimeProvider(this);
		_stateCtrl.addSelectionListener(this);
		_stateCtrl.setLayoutData(GridUtil.createFill());
		_dataViewer.addControlListener(new ControlAdapter() {
			public void controlResized(ControlEvent event) {
				resizeControls();
			}
		});
		resizeControls();
		_dataViewer.update();
		_threadTip = new ThreadsTipHandler(parent.getShell());
		_threadTip.activateHoverHelp(_stateCtrl);
		return _dataViewer;
	}

	public void dispose() {
		saveOptions();
		_stateCtrl.dispose();
		_dataViewer.dispose();
		_colors.dispose();
		super.dispose();
	}

	public void resizeControls() {
		Rectangle r = _dataViewer.getClientArea();
		if (r.isEmpty())
			return;
		
		int legendHeight = 4;
		int timeScaleHeight = 22;
		_timeScaleCtrl.setBounds(r.x, r.y + legendHeight, r.width, timeScaleHeight);
		_stateCtrl.setBounds(r.x, r.y + legendHeight + timeScaleHeight,
				r.width, r.height - legendHeight - timeScaleHeight);
		int width = r.width;
		if (_nameWidth > width - 6)
			_nameWidth = width - 6;
		if (_nameWidth < 6)
			_nameWidth = 6;
	}
	
	/** Tries to set most convinient time range for display. */
	void setTimeRange(Object threads[]) {
		_endTime = 0;
		_beginTime = -1;
		TRCThreadEvent event;
		for (int i = 0; i < threads.length; i++) {
			TRCThread thread = (TRCThread) threads[i];
			double lastEventTime = thread.getStopTime();
			if (lastEventTime > thread.getStartTime()) {
				if (lastEventTime > _endTime)
					_endTime = lastEventTime;
			}
			EList list = thread.getThreadEvents();
			int len = list.size();
			if (len > 0) {
				event = (TRCThreadEvent) list.get(list.size() - 1);
				lastEventTime = event.getTime();
				if (lastEventTime > _endTime)
					_endTime = lastEventTime;
				event = (TRCThreadEvent) list.get(0);
				if (_beginTime < 0 || _beginTime > event.getTime())
					_beginTime = event.getTime();
			}
		}
		_endTime += 0.1;
		if (_beginTime < 0)
			_beginTime = 0;
	}

	void setTimeBounds() {
		_timeMinTime = _beginTime - (_endTime - _beginTime) * 0.05;
		if (_timeMinTime < 0)
			_timeMinTime = 0;
		//_time1_ = _time0_ + (_endTime - _time0_) * 1.05;
		_timeMaxTime = _endTime;
		_timeMinTime = Math.floor(_timeMinTime);
		_timeMaxTime = Math.ceil(_timeMaxTime);
		if (!_timeRangeFixed) {
			_time0 = _timeMinTime;
			_time1 = _timeMaxTime;
		}
	}

	void refreshData() {
		Object threads[] = PerftraceUtil.getAllThreads(_page.getMOFObject(), false);
		if (null == threads)
			threads = new TRCThread[0];
		setTimeRange(threads);
		setTimeBounds();
		if (_selectedTime < 0 || _selectedTime > _endTime)
			_selectedTime = _endTime;
		_stateCtrl.refreshData(threads);
	}

	public void setFocus() {
		if (null != _stateCtrl)
			_stateCtrl.setFocus();
	}

	public TRCThread getSelectedThread() {
		return _stateCtrl.getSelectedThread();
	}

	public ISelection getSelection() {
		return _stateCtrl.getSelection();
	}

	public double getTime0() {
		return _time0;
	}

	public double getTime1() {
		return _time1;
	}

	public double getMinTimeInterval() {
		return _minTimeInterval;
	}

	public int getNameSpace() {
		return _nameWidth;
	}

	public void setNameSpace(int width) {
		_nameWidth = width;
		width = _stateCtrl.getClientArea().width;
		if (_nameWidth > width - 6)
			_nameWidth = width - 6;
		if (_nameWidth < 6)
			_nameWidth = 6;
		_stateCtrl.adjustScrolls();
		_stateCtrl.redraw();
		_timeScaleCtrl.redraw();
	}
	
	public int getTimeSpace() {
		int w = _stateCtrl.getClientArea().width;
		return w - _nameWidth;
	}

	public double getSelectedTime() {
		return _selectedTime;
	}

	public double getBeginTime() {
		return _beginTime;
	}

	public double getEndTime() {
		return _endTime;
	}

	public double getMaxTime() {
		return _timeMaxTime;
	}

	public double getMinTime() {
		return _timeMinTime;
	}

	public void setStartFinishTime(double time0, double time1) {
		_time0 = time0;
		if (_time0 < _timeMinTime)
			_time0 = _timeMinTime;
		_time1 = time1;
		if (_time1 - _time0 < _minTimeInterval)
			_time1 = _time0 + _minTimeInterval;
		if (_time1 > _timeMaxTime)
			_time1 = _timeMaxTime;
		_timeRangeFixed = true;
		_stateCtrl.adjustScrolls();
		_stateCtrl.redraw();
		_timeScaleCtrl.redraw();
	}

	public void resetStartFinishTime() {
		_timeRangeFixed = false;
		_time0 = _timeMinTime;
		_time1 = _timeMaxTime;
		_stateCtrl.adjustScrolls();
		_stateCtrl.redraw();
		_timeScaleCtrl.redraw();
	}

	public void setSelectedTime(double time, boolean ensureVisible) {
		_selectedTime = time;
		if (_selectedTime > _endTime)
			_selectedTime = _endTime;
		if (_selectedTime < _beginTime)
			_selectedTime = _beginTime;
		if (ensureVisible) {
			double timeSpace = (_time1 - _time0) * .02;
			double timeMid = (_time1 - _time0) * .1;
			if (_selectedTime < _time0 + timeSpace) {
				double dt = _time0 - _selectedTime + timeMid;
				_time0 -= dt;
				_time1 -= dt;
			} else if (_selectedTime > _time1 - timeSpace) {
				double dt = _selectedTime - _time1 + timeMid;
				_time0 += dt;
				_time1 += dt;
			}
			if (_time0 < 0) {
				_time1 -= _time0;
				_time0 = 0;
			} else if (_time1 > _timeMaxTime) {
				_time0 -= _time1 - _timeMaxTime;
				_time1 = _timeMaxTime;
			}
		}
		_stateCtrl.adjustScrolls();
		_stateCtrl.redraw();
		_timeScaleCtrl.redraw();
	}

	public void widgetDefaultSelected(SelectionEvent e) {
		new OpenCallStackViewAction().openView(false);
		updateModelSelection();
	}

	public void widgetSelected(SelectionEvent e) {
		updateModelSelection();
	}

	public void selectNextEvent() {
		_stateCtrl.selectNextEvent();
	}

	public void selectPrevEvent() {
		_stateCtrl.selectPrevEvent();
	}

	public void selectNextThread() {
		_stateCtrl.selectNextThread();
	}

	public void selectPrevThread() {
		_stateCtrl.selectPrevThread();
	}

	public void groupThreads(boolean on) {
		_stateCtrl.groupThreads(on);
	}

	public void filterThreads() {
		if (_dataViewer == null || _dataViewer.isDisposed()) return;
		
		if (ThreadFilterDialog.getThreadFilter(_dataViewer.getShell(), _stateCtrl.getThreads(),
				_stateCtrl.getThreadFilter()))
					_stateCtrl.refreshData();
	}
	
	public void showLegend() {
		if (_dataViewer == null || _dataViewer.isDisposed()) return;
		
		ThreadLegend.open(_dataViewer.getShell());
	}
	
	public void toggleThreadsInteractionDrawing() {
		_stateCtrl.toggleThreadsInteractionDrawing();
	}
	
	public void setThreadJoinDrawing(boolean on) {
		_stateCtrl.setThreadJoinDrawing(on);
	}
	
	public void setThreadWaitDrawing(boolean on) {
		_stateCtrl.setThreadWaitDrawing(on);
	}
	
	public void setThreadReleaseDrawing(boolean on) {
		_stateCtrl.setThreadReleaseDrawing(on);
	}
	
	public boolean getThreadInteractionDrawing() {
		return _stateCtrl.getThreadsInteractionDrawing();
	}
	
	public boolean getThreadJoinDrawing() {
		return _stateCtrl.getThreadJoinDrawing();
	}
	
	public boolean getThreadWaitDrawing() {
		return _stateCtrl.getThreadWaitDrawing();
	}
	
	public boolean getThreadReleaseDrawing() {
		return _stateCtrl.getThreadReleaseDrawing();
	}
	
	protected void select(Object obj) {
		if (obj == null)
			return;
		if (obj instanceof ThreadDetails) {
			obj = ((ThreadDetails) obj).getThread();
		}
		if (obj instanceof TRCThread) {
			//_stateCtrl.selectThread((TRCThread) obj);
		}
	}

	public void zoomIn() {
		_stateCtrl.zoomIn();
	}

	public void zoomOut() {
		_stateCtrl.zoomOut();
	}
}
