/**********************************************************************
 * Copyright (c) 2006, 2008 Intel Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ThreadStatisticView.java,v 1.9 2008/06/16 21:04:49 jkubasta Exp $
 * 
 **********************************************************************/
package org.eclipse.tptp.trace.jvmti.internal.client.views;

import java.util.ArrayList;

import org.eclipse.hyades.models.trace.TRCThread;
import org.eclipse.hyades.models.trace.TRCThreadEvent;
import org.eclipse.hyades.trace.ui.TraceViewerPage;
import org.eclipse.hyades.trace.ui.internal.util.PerftraceUtil;
import org.eclipse.hyades.trace.views.internal.TraceUIPlugin;
import org.eclipse.hyades.trace.views.internal.view.columnlabels.ColumnLabelAdapter;
import org.eclipse.hyades.trace.views.util.internal.ColumnData;
import org.eclipse.hyades.trace.views.util.internal.ThreadDetails;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.graphics.Region;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.swt.widgets.TreeItem;
import org.eclipse.tptp.trace.jvmti.internal.client.context.TIContextAttributes;
import org.eclipse.tptp.trace.jvmti.internal.client.views.columnlabels.ThreadBlockCountColumnLabel;
import org.eclipse.tptp.trace.jvmti.internal.client.views.columnlabels.ThreadBlockedTimeColumnLabel;
import org.eclipse.tptp.trace.jvmti.internal.client.views.columnlabels.ThreadClassNameColumnLabel;
import org.eclipse.tptp.trace.jvmti.internal.client.views.columnlabels.ThreadDeadlockCountColumnLabel;
import org.eclipse.tptp.trace.jvmti.internal.client.views.columnlabels.ThreadDeadlockedTimeColumnLabel;
import org.eclipse.tptp.trace.jvmti.internal.client.views.columnlabels.ThreadNameLabelAdapter;
import org.eclipse.tptp.trace.jvmti.internal.client.views.columnlabels.ThreadRunningTimeColumnLabel;
import org.eclipse.tptp.trace.jvmti.internal.client.views.columnlabels.ThreadStateColumnLabel;
import org.eclipse.tptp.trace.jvmti.internal.client.views.columnlabels.ThreadWaitingTimeColumnLabel;
import org.eclipse.tptp.trace.jvmti.internal.client.widgets.PlainSelection;
import org.eclipse.tptp.trace.jvmti.internal.client.widgets.TraceColorScheme;
import org.eclipse.tptp.trace.jvmti.internal.client.widgets.Utils;

/**
 * This view contains thread statistics data.
 */
public class ThreadStatisticView extends BaseStatisticView {

	static final int COL_THREAD_NAME = 0;
	static final int COL_CLASS_NAME = 1;
	static final int COL_STATE = 2;
	static final int COL_RUN_TIME = 3;
	static final int COL_WAIT_TIME = 4;
	static final int COL_BLOCK_TIME = 5;
	static final int COL_BLOCK_COUNT = 6;
	static final int COL_DEADLOCK_TIME = 7;
	static final int COL_DEADLOCK_COUNT = 8;

	public static String PREFERENCE_KEY_PREFIX = "org.eclipse.hyades.trace.views.statistic.";
	protected ColumnLabelAdapter[] _columns;
	protected TraceColorScheme _colors = new TraceColorScheme();

	public ThreadStatisticView(Composite parent, TraceViewerPage page) {
		super(parent, page);
		createColumnsLabelProviders();
	}

	public void dispose() {
		_colors.dispose();
		super.dispose();
	}
	
	protected String getContextHelpId() {
		return TraceUIPlugin.getPluginId() + ".stvw0001";
	}

	public void createColumnsLabelProviders() {
		_columns = new ColumnLabelAdapter[] {
				new ThreadNameLabelAdapter(),
				new ThreadClassNameColumnLabel(),
				//new ThreadStartTimeColumnLabel(),
				//new ThreadStopTimeColumnLabel(),
				new ThreadStateColumnLabel(),
				new ThreadRunningTimeColumnLabel(),
				//new ThreadSleepingTimeColumnLabel(),
				new ThreadWaitingTimeColumnLabel(),
				new ThreadBlockedTimeColumnLabel(),
				new ThreadBlockCountColumnLabel(),
				new ThreadDeadlockedTimeColumnLabel(),
				new ThreadDeadlockCountColumnLabel() };
	}

	class ThreadStatisticTreeViewer extends TreeViewer {
		public ThreadStatisticTreeViewer(Composite parent) {
			super(parent);
		}

		public ThreadStatisticTreeViewer(Tree tree) {
			super(tree);
		}

		public void expandItem(TreeItem item) {
			item.setExpanded(true);
			createChildren(item);
		}

		/**
		 * Returns the current selection for this provider.
		 * 
		 * @return the current selection
		 */
		public ISelection getSelection() {
			ISelection sel = super.getSelection();
			if (sel != null && !sel.isEmpty()) {
				Object obj = ((IStructuredSelection) sel).getFirstElement();
				if (obj instanceof ThreadDetails)
					return new StructuredSelection(((ThreadDetails) obj).getThread());
			}
			return sel;
		}

	}

	protected Object[] getChildren(Object parentElement) {
		return null;
	}

	protected boolean hasChildren(Object element) {
		return false;
	}

	protected Image getColumnImage(Object element, int columnIndex) {
		StatisticViewColumnInfo info = StatisticViewColumnInfo.getStatisticTableColumnInfo(getTree().getColumn(columnIndex));
		int pos = info.getColumnData().getInitalPos();
		if (info.isDeltaColumn())
			return null;
		if (pos >= 0 && pos < _columns.length) {
			return getElementColumnImage(element, _columns[pos], info.isDeltaColumn());
		} else {
			return null;
		}
	}

	protected String getColumnText(Object element, int columnIndex) {
		StatisticViewColumnInfo info = StatisticViewColumnInfo.getStatisticTableColumnInfo(getTree().getColumn(columnIndex));
		int pos = info.getColumnData().getInitalPos();
		if (pos >= 0 && pos < _columns.length) {
			return getElementColumnText(element, _columns[pos], info.isDeltaColumn());
		} else {
			return "";
		}
	}

	protected Object[] getElements(Object inputElement) {
		Object[] threads = PerftraceUtil.getAllThreads(_page.getMOFObject(), false);
		Object[] elements = new Object[threads.length];
		for (int i = 0; i < threads.length; i++) {
			elements[i] = new ThreadDetails((TRCThread) threads[i]);
		}
		return elements;
	}

	public class ThreadStatisticSorter extends StatisticSorter {
		public int compare(Viewer viewer, Object e1, Object e2) {
			if (_pos >= 0 && _pos < _columns.length) {
				return _sortSequence * compareElements(e1, e2, _columns[_pos], _info.isDeltaColumn());
			} else {
				return 0;
			}
		}
	}

	/*public class ThreadStatisticFilter extends StatisticFilter {

		public ThreadStatisticFilter() {
			super();
		}

		public boolean select(Viewer viewer, Object parent, Object element) {

			boolean flag = true;
			String compareText = (element instanceof TRCThread) ? PerftraceUtil.getThreadName((TRCThread) element) : "";

			if (_noPattern)
				return true;

			if (!_caseSensitive) {
				compareText = compareText.toLowerCase();
			}
			if (_exactMatch) {
				return compareText.compareTo(_prefix) == 0;
			}
			if (_prefix != "") {
				flag = compareText.startsWith(_prefix);
			}
			if (flag && _suffix != "") {
				flag = compareText.endsWith(_suffix);
			}
			if (flag) {
				for (int k = 0; k < _textList.size(); k++) {

					String str1 = (String) _textList.get(k);

					int index = compareText.lastIndexOf(str1);
					if (index == -1) {
						flag = false;
						break;
					}

					compareText = compareText.substring(index + str1.length());
				}
			}
			return flag;
		}
	}*/

	public String getDefaultColumnsTemplate() {
		return ""
				+ TIContextAttributes.THREAD_NAME + ":" + COL_THREAD_NAME + ":" + String.valueOf(ColumnData.NONDELETABLE | ColumnData.IS_VISIBLE | ColumnData.NONMOVABLE) + ":left:120,"
				+ TIContextAttributes.THREAD_CLASS_NAME + ":" + COL_CLASS_NAME + ":" + String.valueOf(ColumnData.NONDELETABLE | ColumnData.IS_VISIBLE | ColumnData.NONMOVABLE) + ":left:150,"
				//+ TIContextAttributes.THREAD_START_TIME + ":2:" + String.valueOf(ColumnData.IS_VISIBLE) + ":right:80,"
				//+ TIContextAttributes.THREAD_STOP_TIME + ":3:" + String.valueOf(ColumnData.IS_VISIBLE) + ":right:80,"
				+ TIContextAttributes.THREAD_STATE + ":" + COL_STATE + ":" + String.valueOf(ColumnData.IS_VISIBLE) + ":right:80,"
				+ TIContextAttributes.THREAD_RUNNING_TIME + ":" + COL_RUN_TIME + ":" + String.valueOf(ColumnData.IS_VISIBLE) + ":right:120," 
				//+ IContextAttributes.THREAD_SLEEPING_TIME + ":6:" + String.valueOf(ColumnData.IS_VISIBLE) + ":right:120,"
				+ TIContextAttributes.THREAD_WAITING_TIME + ":" + COL_WAIT_TIME + ":" + String.valueOf(ColumnData.IS_VISIBLE) + ":right:120,"
				+ TIContextAttributes.THREAD_BLOCKED_TIME + ":" + COL_BLOCK_TIME+ ":" + String.valueOf(ColumnData.IS_VISIBLE) + ":right:120,"
				+ TIContextAttributes.THREAD_BLOCK_COUNT + ":" + COL_BLOCK_COUNT + ":" + String.valueOf(ColumnData.IS_VISIBLE) + ":right:80,"
				+ TIContextAttributes.THREAD_DEADLOCKED_TIME + ":" + COL_DEADLOCK_TIME + ":" + String.valueOf(ColumnData.IS_VISIBLE) + ":right:80,"
				+ TIContextAttributes.THREAD_DEADLOCK_COUNT + ":" + COL_DEADLOCK_COUNT + ":" + String.valueOf(ColumnData.IS_VISIBLE) + ":right:80"
				;
	}

	protected StatisticSorter getViewerSorterInstance() {
		return new ThreadStatisticSorter();
	}

	/*protected StatisticFilter getFilterInstance() {
		return new ThreadStatisticFilter();
	}*/

	public String getColumnsPreferencesKey() {
		return PREFERENCE_KEY_PREFIX + "ThreadStats4";
	}

	public String getViewTypeStr() {
		return org.eclipse.hyades.trace.views.adapter.internal.TraceConstants.THREAD_STATS_VIEW;
	}

	protected void showPercentUpdate() {
	}

	public void update() {
		for (int i = 0; i < _columns.length; i++) {
			_columns[i].resetMap();
		}
		super.update();
	}

	protected Tree createTree(Composite parent, int options) {
		final Tree tree = super.createTree(parent, options);
		Listener listener = new Listener() {
			public void handleEvent(Event event) {
				drawItem(event);
			}
		};
		tree.addListener(SWT.EraseItem, listener);
		return tree;
	}

	protected TreeViewer getTreeViewer(Tree tree) {
		return new ThreadStatisticTreeViewer(tree);
	}

	public ISelection getSelection() {
		TRCThread thread = null;
		ISelection selection = getTreeViewer().getSelection();
		if (selection != null && !selection.isEmpty()) {
			Object sel = ((IStructuredSelection) selection).getFirstElement();
			if (sel instanceof TRCThread)
				thread = (TRCThread) sel;
			if (sel instanceof ThreadDetails)
				thread = ((ThreadDetails) sel).getThread();
		}
		return new PlainSelection(thread);
	}

	public void menuAboutToShow(IMenuManager menu) {
		menu.add(getUpdateAction());
		menu.add(getSeparator());
		resetChooseColumnsAction();
		menu.add(getChooseColumnsAction());
		menu.add(getSortByColumnAction());
	}

	public void widgetSelected(SelectionEvent e) {
		super.widgetSelected(e);
		if (e.widget instanceof TableColumn) {
			expandFirstElement();
			handleSelectionEvent();
		}
	}

	protected void resetColumns(ArrayList list) {
		super.resetColumns(list);
		expandFirstElement();
	}

	protected void select(Object obj) {
		if (obj == null)
			return;
		if (obj instanceof ThreadDetails) {
			obj = ((ThreadDetails) obj).getThread();
		} else if (obj instanceof TRCThreadEvent) {
			obj = ((TRCThreadEvent) obj).getThread();
		}
		Tree tree = getTree();
		TreeItem[] items = tree.getItems();
		TreeItem item = null;
		for (int j = 0; j < items.length; j++) {
			item = items[j];
			if (item != null && !item.isDisposed()) {
				Object data = item.getData();
				if (data instanceof ThreadDetails) {
					data = ((ThreadDetails) data).getThread();
				}
				if (data == obj) {
					tree.setSelection(item);
					return;
				}
			}
		}
	}

	protected int updateTableGetColumnNumber() {
		return 2; // start time
	}

	public void widgetDefaultSelected(SelectionEvent arg0) {
		new OpenCallStackViewAction().openView(false);
	}

	public void drawItem(Event event) {
		Tree tree = getTree();
		Display display = event.display;
		boolean selected = (event.detail & SWT.SELECTED) != 0;
		if (selected) {
			GC gc = event.gc;
			gc.setAdvanced(true);
			if (gc.getAdvanced()) {
				int columnCount = tree.getColumnCount();
				if (event.index == columnCount - 1 || columnCount == 0) {
					Rectangle area = tree.getClientArea();
					int width = area.x + area.width - event.x;
					if (width > 0) {
						Region region = new Region();
						gc.getClipping(region);
						region.add(event.x, event.y, width, event.height);
						gc.setClipping(region);
						region.dispose();
					}
				}
				Color foreground = gc.getForeground();
				Color background = gc.getBackground();
				gc.setForeground(display.getSystemColor(SWT.COLOR_LIST_SELECTION));
				gc.fillRectangle(0, event.y, 100000, event.height);
				gc.setForeground(foreground);
				gc.setBackground(background);
			}
		}

		ThreadDetails thread = (ThreadDetails) event.item.getData();
		int state = ThreadDetails.UNKNOWN;
		if (event.index == COL_STATE) { // thread state
			state = thread.getState();
		} else if (event.index == COL_BLOCK_COUNT // block count
				&& thread.getBlockCount() > 0) {
			state = ThreadDetails.BLOCKED;
		} else if (event.index == COL_DEADLOCK_COUNT // deadlock count
				&& thread.getDeadlockCount() > 0) {
			state = ThreadDetails.DEADLOCK;
		}

		if (state != ThreadDetails.UNKNOWN) {
			if (selected)
				event.detail &= ~SWT.SELECTED;

			GC gc = event.gc;
			Color foreground = gc.getForeground();
			Color background = gc.getBackground();

			int c2 = selected ? SWT.COLOR_GRAY : SWT.COLOR_WHITE;
			Color sysBack = _colors.getColor(state);
			Color sysFore = display.getSystemColor(c2);
			Color fore = Utils.mixColors(display, sysBack, sysFore, 1, 4);
			Color back = Utils.mixColors(display, sysBack, sysFore, 4, 1);

			gc.setAdvanced(true);
			if (gc.getAdvanced()) {
				gc.setAlpha(180);
			}

			gc.setBackground(back);
			gc.setForeground(fore);
			gc.fillGradientRectangle(event.x, event.y, event.width, event.height, false);

			gc.setForeground(foreground);
			gc.setBackground(background);
			back.dispose();
			fore.dispose();
		}
	}
}
