/*****************************************************************************
 * Copyright (c) 2007, 2010 Intel Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Intel Corporation - Initial API and implementation
 *    Ruslan A. Scherbakov, Intel - Initial API and implementation
 *
 * $Id: ThreadStatesCtrl.java,v 1.18 2010/05/18 20:41:46 jwest Exp $ 
 *****************************************************************************/

package org.eclipse.tptp.trace.jvmti.internal.client.widgets;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.eclipse.emf.common.util.EList;
import org.eclipse.hyades.models.trace.TRCThread;
import org.eclipse.hyades.models.trace.TRCThreadEvent;
import org.eclipse.hyades.models.trace.impl.TRCThreadDeadAndNotifyJoinedEventImpl;
import org.eclipse.hyades.models.trace.impl.TRCThreadExecEventImpl;
import org.eclipse.hyades.models.trace.impl.TRCThreadHandoffLockEventImpl;
import org.eclipse.hyades.models.trace.impl.TRCThreadImpl;
import org.eclipse.hyades.models.trace.impl.TRCThreadInterruptThreadEventImpl;
import org.eclipse.hyades.models.trace.impl.TRCThreadNotifyAllEventImpl;
import org.eclipse.hyades.models.trace.impl.TRCThreadNotifyEventImpl;
import org.eclipse.hyades.models.trace.impl.TRCThreadRunningEventImpl;
import org.eclipse.hyades.models.trace.impl.TRCThreadStartThreadEventImpl;
import org.eclipse.hyades.models.trace.impl.TRCThreadWaitTimeoutExceedEventImpl;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ControlEvent;
import org.eclipse.swt.events.ControlListener;
import org.eclipse.swt.events.FocusEvent;
import org.eclipse.swt.events.FocusListener;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.KeyListener;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.events.MouseListener;
import org.eclipse.swt.events.MouseMoveListener;
import org.eclipse.swt.events.MouseTrackListener;
import org.eclipse.swt.events.MouseWheelListener;
import org.eclipse.swt.events.PaintEvent;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.events.TraverseEvent;
import org.eclipse.swt.events.TraverseListener;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Cursor;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.tptp.trace.jvmti.internal.client.views.UIMessages;

/** 
 * Data is populated into the control through 
 * the refreshData(Object[]) method */
public class ThreadStatesCtrl extends TraceCtrl implements FocusListener,
		KeyListener, MouseMoveListener, MouseListener, MouseWheelListener,
		ControlListener, SelectionListener, MouseTrackListener, TraverseListener {

	public static final boolean DEFAULT_DRAW_THREAD_JOIN = true;
	public static final boolean DEFAULT_DRAW_THREAD_WAIT = true;
	public static final boolean DEFAULT_DRAW_THREAD_RELEASE = true;
	
	private final double zoomCoeff = 1.5;
	
	private ITimeDataProvider _timeProvider;
	private boolean _isInFocus = false;
	private boolean _isDragCursor3 = false;
	private boolean _mouseHover = false;
	private int _topItem = 0;
	private int _itemHeight = 18;
	private int _dragState = 0;
	private int _hitIdx = 0;
	private int _dragX0 = 0;
	private int _dragX = 0;
	private int _idealNameWidth = 0;
	private double _timeStep = 0.01;
	private double _time0bak;
	private double _time1bak;
	private ItemData _data = new ItemData();
	private List _selectionListeners;
	private Rectangle _rect0 = new Rectangle(0, 0, 0, 0);
	private Rectangle _rect1 = new Rectangle(0, 0, 0, 0);
	private Cursor _dragCursor3;
	private boolean drawThreadsInteraction = false;
	private boolean drawThreadJoins = DEFAULT_DRAW_THREAD_JOIN;
	private boolean drawThreadWaits = DEFAULT_DRAW_THREAD_WAIT;
	private boolean drawThreadReleases = DEFAULT_DRAW_THREAD_RELEASE;
	
	public ThreadStatesCtrl(Composite parent, TraceColorScheme colors) {
		super(parent, colors, SWT.NO_BACKGROUND | SWT.H_SCROLL | SWT.V_SCROLL | SWT.DOUBLE_BUFFERED);
		addFocusListener(this);
		addMouseListener(this);
		addMouseMoveListener(this);
		addMouseTrackListener(this);
		addMouseWheelListener(this);
		addTraverseListener(this);
		addKeyListener(this);
		addControlListener(this);
		getVerticalBar().addSelectionListener(this);
		getHorizontalBar().addSelectionListener(this);
		_dragCursor3 = new Cursor(parent.getDisplay(), SWT.CURSOR_SIZEWE);
	}

	public void dispose() {
		super.dispose();
		_dragCursor3.dispose();
	}

	public void setTimeProvider(ITimeDataProvider timeProvider) {
		_timeProvider = timeProvider;
		adjustScrolls();
		redraw();
	}

	public void addSelectionListener(SelectionListener listener) {
		if (listener == null)
			SWT.error(SWT.ERROR_NULL_ARGUMENT);
		if (null == _selectionListeners)
			_selectionListeners = new ArrayList();
		_selectionListeners.add(listener);
	}

	public void removeSelectionListener(SelectionListener listener) {
		if (null != _selectionListeners)
			_selectionListeners.remove(listener);
	}

	public void fireSelectionChanged() {
		if (null != _selectionListeners) {
			Iterator it = _selectionListeners.iterator();
			while (it.hasNext()) {
				SelectionListener listener = (SelectionListener) it.next();
				listener.widgetSelected(null);
			}
		}
	}

	public void fireDefaultSelection() {
		if (null != _selectionListeners) {
			Iterator it = _selectionListeners.iterator();
			while (it.hasNext()) {
				SelectionListener listener = (SelectionListener) it.next();
				listener.widgetDefaultSelected(null);
			}
		}
	}

	public Object[] getThreads() {
		return _data.getThreads();
	}
	
	public boolean[] getThreadFilter() {
		return _data.getThreadFilter();
	}
	
	public void refreshData() {
		_data.refreshData();
		adjustScrolls();
		redraw();
	}

	public void refreshData(Object/*<TRCThreadImpl>*/ threads[]) {
		_data.refreshData(threads);
		adjustScrolls();
		redraw();
	}

	public void adjustScrolls() {
		if (null == _timeProvider) {
			getVerticalBar().setValues(0, 1, 1, 1, 1, 1);
			getHorizontalBar().setValues(0, 1, 1, 1, 1, 1);
			return;
		}
		int page = countPerPage();
		if (_topItem + page > _data._items.length)
			_topItem = _data._items.length - page;
		if (_topItem < 0)
			_topItem = 0;
		getVerticalBar().setValues(_topItem, 0, _data._items.length, page, 1,
				page);
		
		double time0 = _timeProvider.getTime0();
		double time1 = _timeProvider.getTime1();
		double timeMin = _timeProvider.getMinTime();
		double timeMax = _timeProvider.getMaxTime();
		
		int timePage = (int) ((time1 - time0) / _timeStep);
		int timePos = (int) (time0 / _timeStep);
		getHorizontalBar().setValues(timePos, (int) (timeMin / _timeStep),
				(int) (timeMax / _timeStep), timePage, 1, timePage);
	}

	boolean ensureVisibleItem(int idx, boolean redraw) {
		boolean changed = false;
		if (idx < 0) {
			for (idx = 0; idx < _data._items.length; idx++) {
				if (((Item)_data._items[idx])._selected)
					break;
			}
		}
		if (idx >= _data._items.length)
			return changed;
		if (idx < _topItem) {
			_topItem = idx;
			getVerticalBar().setSelection(_topItem);
			if (redraw)
				redraw();
			changed = true;
		} else {
			int page = countPerPage();
			if (idx >= _topItem + page) {
				_topItem = idx - page + 1;
				getVerticalBar().setSelection(_topItem);
				if (redraw)
					redraw();
				changed = true;
			}
		}
		return changed;
	}

	public ISelection getSelection() {
		PlainSelection sel = new PlainSelection();
		TRCThread thread = getSelectedThread();
		if (null != thread && null != _timeProvider) {
			double selectedTime = _timeProvider.getSelectedTime();
			TRCThreadEvent event = Utils.findEvent(thread, selectedTime, 0);
			if (event != null)
				sel.add(event);
			else
				sel.add(thread);
		}
		return sel;
	}

	public void selectThread(int n) {
		if (n != 1 && n != -1)
			return;
		boolean changed = false;
		int lastSelection = -1;
		for (int i = 0; i < _data._items.length; i++) {
			Item item = (Item) _data._items[i];
			if (item._selected) {
				lastSelection = i;
				if (1 == n && i < _data._items.length - 1) {
					item._selected = false;
					if (item._hasChildren)
						_data.expandItem(i, true);
					item = (Item) _data._items[i + 1];
					if (item._hasChildren) {
						_data.expandItem(i + 1, true);
						item = (Item) _data._items[i + 2];
					}
					item._selected = true;
					changed = true;
				} else if (-1 == n && i > 0) {
					i--;
					Item prevItem = (Item) _data._items[i];
					if (prevItem._hasChildren) {
						if (prevItem._expanded) {
							if (i > 0) {
								i--;
								prevItem = (Item) _data._items[i];
							}
						}
						if (!prevItem._expanded) {
							int added = _data.expandItem(i, true);
							prevItem = (Item) _data._items[i + added];
							item._selected = false;
							prevItem._selected = true;
							changed = true;
						}
					} else {
						item._selected = false;
						prevItem._selected = true;
						changed = true;
					}
				}
				break;
			}
		}
		if (lastSelection < 0 && _data._items.length > 0) {
			Item item = (Item) _data._items[0];
			if (item._hasChildren) {
				_data.expandItem(0, true);
				item = (Item) _data._items[1];
				item._selected = true;
				changed = true;
			} else {
				item._selected = true;
				changed = true;
			}
		}
		if (changed) {
			ensureVisibleItem(-1, false);
			redraw();
			fireSelectionChanged();
		}
	}

	public void selectEvent(int n) {
		if (null == _timeProvider)
			return;
		TRCThread thread = getSelectedThread();
		if (thread == _timeProvider)
			return;
		double selectedTime = _timeProvider.getSelectedTime();
		double endTime = _timeProvider.getEndTime();
		TRCThreadEvent nextEvent;
		if (-1 == n && selectedTime >= endTime)
			nextEvent = Utils.findEvent(thread, selectedTime, 0);
		else
			nextEvent = Utils.findEvent(thread, selectedTime, n);
		if (null == nextEvent && -1 == n)
			nextEvent = Utils.getFirstEvent(thread);
		if (null != nextEvent) {
			_timeProvider.setSelectedTime(nextEvent.getTime(), true);
			fireSelectionChanged();
		}
		else if (1 == n) {
			_timeProvider.setSelectedTime(endTime, true);
			fireSelectionChanged();
		}
	}

	public void selectNextEvent() {
		selectEvent(1);
	}

	public void selectPrevEvent() {
		selectEvent(-1);
	}

	public void selectNextThread() {
		selectThread(1);
	}

	public void selectPrevThread() {
		selectThread(-1);
	}
	
	public void zoomIn() {
		double _time0 = _timeProvider.getTime0();
		double _time1 = _timeProvider.getTime1();
		double _range = _time1 - _time0;
		double selTime = _timeProvider.getSelectedTime();
		if (selTime <= _time0 || selTime >= _time1) {
			selTime = (_time0 + _time1) / 2;
		}
		double time0 = selTime - (selTime - _time0) / zoomCoeff;
		double time1 = selTime + (_time1 - selTime) / zoomCoeff;

		double inaccuracy = (_timeProvider.getMaxTime() - _timeProvider
				.getMinTime()) - (time1 - time0);
		if (inaccuracy > 0 && inaccuracy < 0.3) {
			_timeProvider.setStartFinishTime(_timeProvider.getMinTime(),
					_timeProvider.getMaxTime());
			return;
		}

		double m = _timeProvider.getMinTimeInterval();
		if ((time1 - time0) < m) {
			time0 = selTime - (selTime - _time0) * m / _range;
			time1 = time0 + m;
		}

		_timeProvider.setStartFinishTime(time0, time1);
	}

	public void zoomOut() {
		double _time0 = _timeProvider.getTime0();
		double _time1 = _timeProvider.getTime1();
		double selTime = _timeProvider.getSelectedTime();
		if (selTime <= _time0 || selTime >= _time1) {
			selTime = (_time0 + _time1) / 2;
		}
		double time0 = selTime - (selTime - _time0) * zoomCoeff;
		double time1 = selTime + (_time1 - selTime) * zoomCoeff;

		double inaccuracy = (_timeProvider.getMaxTime() - _timeProvider
				.getMinTime()) - (time1 - time0);
		if (inaccuracy > 0 && inaccuracy < 0.3) {
			_timeProvider.setStartFinishTime(_timeProvider.getMinTime(),
					_timeProvider.getMaxTime());
			return;
		}

		_timeProvider.setStartFinishTime(time0, time1);
	}

	public void groupThreads(boolean on) {
		_data.groupThreads(on);
		adjustScrolls();
		redraw();
	}
	
	public void toggleThreadsInteractionDrawing() {
		drawThreadsInteraction = !drawThreadsInteraction;
		redraw();
	}
	
	public void setThreadJoinDrawing(boolean on) {
		drawThreadJoins = on;
		drawThreadsInteraction = true;
		redraw();
	}
	
	public void setThreadWaitDrawing(boolean on) {
		drawThreadWaits = on;
		drawThreadsInteraction = true;
		redraw();
	}
	
	public void setThreadReleaseDrawing(boolean on) {
		drawThreadReleases = on;
		drawThreadsInteraction = true;
		redraw();
	}
	
	public boolean getThreadsInteractionDrawing() {
		return drawThreadsInteraction;
	}
	
	public boolean getThreadJoinDrawing() {
		return drawThreadJoins;
	}
	
	public boolean getThreadWaitDrawing() {
		return drawThreadWaits;
	}
	
	public boolean getThreadReleaseDrawing() {
		return drawThreadReleases;
	}
	
	public TRCThread getSelectedThread() {
		TRCThread thread = null;
		int idx = getSelectedIndex();
		if (idx >= 0 && _data._items[idx] instanceof ThreadItem)
			thread = ((ThreadItem) _data._items[idx])._thread;
		return thread;
	}

	public int getSelectedIndex() {
		int idx = -1;
		for (int i = 0; i < _data._items.length; i++) {
			Item item = (Item) _data._items[i];
			if (item._selected) {
				idx = i;
				break;
			}
		}
		return idx;
	}

	boolean toggle(int idx) {
		boolean toggled = false;
		if (idx >= 0 && idx < _data._items.length) {
			Item item = (Item) _data._items[idx];
			if (item._hasChildren) {
				item._expanded = !item._expanded;
				_data.updateItems();
				adjustScrolls();
				redraw();
				toggled = true;
			}
		}
		return toggled;
	}

	int hitTest(int x, int y) {
		if (x < 0 || y < 0)
			return -1;
		int hit = -1;
		int idx = y / _itemHeight;
		idx += _topItem;
		if (idx < _data._items.length)
			hit = idx;
		return hit;
	}

	int hitSplitTest(int x, int y) {
		if (x < 0 || y < 0 || null == _timeProvider)
			return -1;
		int w = 4;
		int hit = -1;
		int nameWidth = _timeProvider.getNameSpace();
		if (x > nameWidth - w && x < nameWidth + w)
			hit = 1;
		return hit;
	}

	public Item getItem(Point pt) {
	    int idx = hitTest(pt.x, pt.y);
	    return idx >= 0 ? (Item)_data._items[idx] : null;
	}

	double hitTimeTest(int x, int y) {
		if (null == _timeProvider)
			return -1;
		double hitTime = -1;
		Point size = getCtrlSize();
		double time0 = _timeProvider.getTime0();
		double time1 = _timeProvider.getTime1();
		int nameWidth = _timeProvider.getNameSpace();
		x -= nameWidth;
		if (x >= 0 && size.x >= nameWidth) {
			hitTime = time0 + (time1 - time0) * x / (size.x - nameWidth);
		}
		return hitTime;
	}

	void selectItem(int idx, boolean addSelection) {
		if (addSelection) {
			if (idx >= 0 && idx < _data._items.length) {
				Item item = (Item) _data._items[idx];
				item._selected = true;
			}
		} else {
			for (int i = 0; i < _data._items.length; i++) {
				Item item = (Item) _data._items[i];
				item._selected = i == idx;
			}
		}
		boolean changed = ensureVisibleItem(idx, true);
		if (!changed)
			redraw();
	}

	public int countPerPage() {
		int height = getCtrlSize().y;
		int count = 0;
		if (height > 0)
			count = height / _itemHeight;
		return count;
	}

	public int getTopIndex() {
		int idx = -1;
		if (_data._items.length > 0)
			idx = 0;
		return idx;
	}

	public int getBottomIndex() {
		int idx = _data._items.length - 1;
		return idx;
	}

	Point getCtrlSize() {
		Point size = getSize();
		size.x -= getVerticalBar().getSize().x;
		size.y -= getHorizontalBar().getSize().y;
		return size;
	}

	void getNameRect(Rectangle rect, Rectangle bound, int idx, int nameWidth) {
		idx -= _topItem;
		rect.x = bound.x;
		rect.y = bound.y + idx * _itemHeight;
		rect.width = nameWidth;
		rect.height = _itemHeight;
	}

	void getStatesRect(Rectangle rect, Rectangle bound, int idx, int nameWidth) {
		idx -= _topItem;
		rect.x = bound.x + nameWidth;
		rect.y = bound.y + idx * _itemHeight;
		rect.width = bound.width - rect.x;
		rect.height = _itemHeight;
	}

	private int getThreadNumber(int tid) {
		int num = -1;
		
		Object[] items = _data._items;
		for (int i=_topItem; i<items.length; i++) {
			Item item = (Item) items[i];
			if ((item instanceof ThreadItem)) {
				TRCThread thread =  ((ThreadItem) item)._thread;
				if (thread != null &&thread.getId() == tid) {
					num = i;
					break;
				}
			}
		}
		
		return num;
	}

	private void drawArrow(GC gc, int x0, int y0, int x1, int y1, Color c) {
		gc.setForeground(c);
		gc.drawLine(x0, y0, x1, y1);
		
		if (y1 > y0) {
			gc.drawLine(x1-3, y1-3, x1, y1);
			gc.drawLine(x1+3, y1-3, x1, y1);
		} else {
			gc.drawLine(x1-3, y1+3, x1, y1);
			gc.drawLine(x1+3, y1+3, x1, y1);
		}
	}

	private void drawThreadThreadEvent(Rectangle bound, TRCThreadEvent e, TRCThread thread, int nItem, int color, GC gc) {
		if (thread == null) return;
		
		int tid = thread.getId();
		if (tid < 0 || getThreadNumber(tid) == -1) return;

		int nameWidth = _timeProvider.getNameSpace();

		double time0 = _timeProvider.getTime0();
		double time1 = _timeProvider.getTime1();
		if (time0 == time1) return;
		
		int xr = bound.x + nameWidth;
		double K = (double) (bound.width - xr) / (time1 - time0);
		
		int x0 = xr + (int) ((e.getTime() - time0) * K);
		if (x0 < xr) x0 = xr;
		
		int x1 = xr + (int) ((thread.getStartTime() - time0) * K);
		if (x1 < xr) return;

		int y0 = bound.y + (nItem-_topItem)*_itemHeight+3+(_itemHeight-6)/2;
		int y1 = bound.y + (getThreadNumber(tid)-_topItem)*_itemHeight+3+(_itemHeight-6)/2;
		
		drawArrow(gc, x0, y0, x1, y1, _colors.getColor(color));
	}
	
	private void drawThreadEvent(Rectangle bound, TRCThreadEvent e, int nItem, int color, GC gc) {
		int nameWidth = _timeProvider.getNameSpace();

		double time0 = _timeProvider.getTime0();
		double time1 = _timeProvider.getTime1();
		if (time0 == time1) return;
		
		int xr = bound.x + nameWidth;
		double K = (double) (bound.width - xr) / (time1 - time0);
		
		int x0 = xr + (int) ((e.getTime() - time0) * K);
		if (x0 < xr) return;
		
		int y0 = bound.y + (nItem-_topItem)*_itemHeight + 3;
		
		gc.setBackground(_colors.getColor(color));
		int c[] = { x0-3, y0-3, x0, y0, x0+3, y0-3 };
		gc.fillPolygon(c);
	}
	
	private void drawExecEvent(Rectangle bound, TRCThreadExecEventImpl e, int nitem, int color, GC gc) {
		List runnings = e.getRunningEvents(); 
		if (runnings == null) return;

		int nameWidth = _timeProvider.getNameSpace();

		double time0 = _timeProvider.getTime0();
		double time1 = _timeProvider.getTime1();
		if (time0 == time1) return;
		
		int xr = bound.x + nameWidth;
		double K = (double) (bound.width - xr) / (time1 - time0);
		
		int x0 = xr + (int) ((e.getTime() - time0) * K);
		if (x0 < xr) x0 = xr;
		
		Iterator it = runnings.iterator();
		while (it.hasNext()) {
			TRCThreadRunningEventImpl re = (TRCThreadRunningEventImpl) it.next();

			//bug 295395
			TRCThread reThread = re.getThread();
			if (reThread == null) continue;
			int tid = reThread.getId();

			if (tid < 0 || getThreadNumber(tid) == -1) continue;
			
			int x1 = xr + (int) ((re.getTime() - time0) * K);
			if (x1 < xr) continue;
			
			int y0 = bound.y + (nitem -_topItem)*_itemHeight+3+(_itemHeight-6)/2;
			int y1 = bound.y + (getThreadNumber(tid)-_topItem)*_itemHeight+3+(_itemHeight-6)/2;
			
			drawArrow(gc, x0, y0, x1, y1, _colors.getColor(color));
		}
	}

	private void drawThreadInteractions(Rectangle bound, GC gc) {
		int nameWidth = _timeProvider.getNameSpace();
		Object[] items = _data._items;

		double time0 = _timeProvider.getTime0();
		double time1 = _timeProvider.getTime1();

		if (time0 == time1) return;
		
		int xr = bound.x + nameWidth;
		double K = (double) (bound.width - xr) / (time1 - time0);
		
		for (int i=0; i<items.length; i++) {
			Item item = (Item) items[i];
			if (!(item instanceof ThreadItem)) continue;
			
			TRCThread thread = ((ThreadItem) item)._thread;
			if (thread == null) continue;
			
			EList list = thread.getThreadEvents();
			Iterator it = list.iterator();
			while (it.hasNext()) {
				TRCThreadEvent te = (TRCThreadEvent) it.next();
				if (te instanceof TRCThreadStartThreadEventImpl) {
					TRCThread child = ((TRCThreadStartThreadEventImpl) te).getStartedThread();
					drawThreadThreadEvent(bound, te, child, i, TraceColorScheme.TI_START_THREAD, gc);
				}
				else if (te instanceof TRCThreadHandoffLockEventImpl) {
					if (drawThreadReleases)
						drawExecEvent(bound, (TRCThreadExecEventImpl) te, i, TraceColorScheme.TI_HANDOFF_LOCK, gc);
				}
				else if (te instanceof TRCThreadNotifyAllEventImpl) {
					if (drawThreadWaits)
						drawExecEvent(bound, (TRCThreadExecEventImpl) te, i, TraceColorScheme.TI_NOTIFY_ALL, gc);
				}
				else if (te instanceof TRCThreadNotifyEventImpl) {
					if (drawThreadWaits)
						drawExecEvent(bound, (TRCThreadExecEventImpl) te, i, TraceColorScheme.TI_NOTIFY, gc);
				}
				else if (te instanceof TRCThreadDeadAndNotifyJoinedEventImpl) {
					if (drawThreadJoins)
						drawExecEvent(bound, (TRCThreadExecEventImpl) te, i, TraceColorScheme.TI_NOTIFY_JOINED, gc);
				}
				else if (te instanceof TRCThreadInterruptThreadEventImpl) {
					if (drawThreadWaits)
						drawExecEvent(bound, (TRCThreadExecEventImpl) te, i, TraceColorScheme.TI_INTERRUPT, gc);
				}
				else if (te instanceof TRCThreadWaitTimeoutExceedEventImpl) {
					drawThreadEvent(bound, te, i, TraceColorScheme.TI_WAIT_EXCEEDED, gc);
				}
			}
		}
	}
	
	void paint(Rectangle bound, PaintEvent e) {
		_itemHeight = getFontHeight() + 6;
		if (bound.width < 2 || bound.height < 2 || null == _timeProvider) return;

		_idealNameWidth = 0;
		GC gc = e.gc;
		int nameWidth = _timeProvider.getNameSpace();
		double time0 = _timeProvider.getTime0();
		double time1 = _timeProvider.getTime1();
		double endTime = _timeProvider.getEndTime();
		double selectedTime = _timeProvider.getSelectedTime();
		// draw thread states
		Object[] items = _data._items;
		for (int i = _topItem; i < items.length; i++) {
			Item item = (Item) items[i];
			getNameRect(_rect0, bound, i, nameWidth);
			if (_rect0.y >= bound.y + bound.height)
				break;
			
			if (item instanceof GroupItem) {
				getStatesRect(_rect1, bound, i, nameWidth);
				_rect0.width += _rect1.width;
				drawName(item, _rect0, gc);
			} else {
				drawName(item, _rect0, gc);
			}
			getStatesRect(_rect0, bound, i, nameWidth);
			drawItemData(item, _rect0, time0, time1, endTime, selectedTime, gc);
		}

		if (drawThreadsInteraction)
			drawThreadInteractions(bound, e.gc);
		
		// fill free canvas area
		_rect0.x = bound.x;
		_rect0.y += _rect0.height;
		_rect0.width = bound.width;
		_rect0.height = bound.y + bound.height - _rect0.y;
		if (_rect0.y < bound.y + bound.height) {
			gc.setBackground(_colors.getColor(TraceColorScheme.BACKGROUND));
			gc.fillRectangle(_rect0);
		}
		// draw drag line
		if (3 == _dragState) {
			gc.setForeground(_colors.getColor(TraceColorScheme.BLACK));
			gc.drawLine(bound.x + nameWidth, bound.y, bound.x + nameWidth, bound.y + bound.height - 1);
		} else if (0 == _dragState && _mouseHover) {
			gc.setForeground(_colors.getColor(TraceColorScheme.RED));
			gc.drawLine(bound.x + nameWidth, bound.y, bound.x + nameWidth, bound.y + bound.height - 1);
		}
	}

	void drawName(Item item, Rectangle rect, GC gc) {
		boolean group = item instanceof GroupItem;
		int elemHeight = rect.height / 2;
		int elemWidth = elemHeight;
		String name = item._name;
		if (group) {
			
			gc.setBackground(_colors.getBkColorGroup(item._selected, _isInFocus));
			gc.fillRectangle(rect);
			if (item._selected && _isInFocus) {
				gc.setForeground(_colors.getBkColor(item._selected, _isInFocus, false));
				gc.drawRectangle(rect.x, rect.y, rect.width - 2, rect.height - 2);
			}
			gc.setForeground(_colors.getBkColor(false, false, false));
			gc.drawLine(rect.x, rect.y + rect.height - 1, rect.width - 1, rect.y + rect.height - 1);
			gc.setForeground(_colors.getFgColorGroup(false, false));
			gc.setBackground(_colors.getBkColor(false, false, false));
			Utils.init(_rect1, rect);
			_rect1.x += MARGIN;
			_rect1.y += (rect.height - elemHeight) / 2;
			_rect1.width = elemWidth;
			_rect1.height = elemHeight;
			gc.fillRectangle(_rect1);
			gc.drawRectangle(_rect1.x, _rect1.y, _rect1.width - 1, _rect1.height - 1);
			int p = _rect1.y + _rect1.height / 2;
			gc.drawLine(_rect1.x + 2, p, _rect1.x + _rect1.width - 3, p);
			if (!item._expanded) {
				p = _rect1.x + _rect1.width / 2;
				gc.drawLine(p, _rect1.y + 2, p, _rect1.y + _rect1.height - 3);
			}
			gc.setForeground(_colors.getFgColorGroup(item._selected, _isInFocus));
			elemWidth+= MARGIN;
		} else {
			gc.setBackground(_colors.getBkColor(item._selected, _isInFocus, true));
			gc.setForeground(_colors.getFgColor(item._selected, _isInFocus));
			gc.fillRectangle(rect);
			Utils.init(_rect1, rect);
			_rect1.x += MARGIN;
			// draw icon
			TRCThread thread = ((ThreadItem)item)._thread;
			Image img = Utils.getItemImage(thread);
			if (null != img) {
				_rect1.y += (rect.height - img.getImageData().height) / 2;
				gc.drawImage(img, _rect1.x, _rect1.y);
			}
			elemWidth = SMALL_ICON_SIZE;
			// cut long string with "..."
			Point size = gc.stringExtent(name);
			if (_idealNameWidth < size.x)
				_idealNameWidth = size.x;
			int width = rect.width - MARGIN - MARGIN - elemWidth;
			int cuts = 0;
			while (size.x > width && name.length() > 1) {
				cuts++;
				name = name.substring(0, name.length() - 1);
				size = gc.stringExtent(name + "...");
			}
			if (cuts > 0)
				name += "...";
			elemWidth+= MARGIN;
		}
		Utils.init(_rect1, rect);
		int leftMargin = MARGIN + elemWidth;
		_rect1.x += leftMargin;
		_rect1.width -= leftMargin;
		int textWidth = 0;
		// draw text
		if (_rect1.width > 0) {
			_rect1.y += 2;
			textWidth = Utils.drawText(gc, name, _rect1, true) + 8;
			_rect1.y -= 2;
		}
		// draw middle line
		if (_rect1.width > 0 && !group) {
			Utils.init(_rect1, rect);
			_rect1.x += leftMargin + textWidth;
			_rect1.width -= textWidth;
			gc.setForeground(_colors.getColor(TraceColorScheme.MID_LINE));
			int midy = _rect1.y + _rect1.height / 2;
			gc.drawLine(_rect1.x, midy, _rect1.x + _rect1.width, midy);
		}
		// gc.drawLine(_rect1.x + _rect1.width - 1, _rect1.y, _rect1.x +
		// _rect1.width - 1, _rect1.y + _rect1.height);
	}

	void drawItemData(Item item, Rectangle rect, double time0, double time1,
			double endTime, double selectedTime, GC gc) {
		if (rect.isEmpty())
			return;
		if (time1 <= time0) {
			gc.setBackground(_colors.getBkColor(false, false, false));
			gc.fillRectangle(rect);
			return;
		}

		Utils.init(_rect1, rect);
		boolean selected = item._selected;
		double K = (double) rect.width / (time1 - time0);
		boolean group = item instanceof GroupItem;

		if (group) {
			//gc.setBackground(_colors.getBkColorGroup(selected, _isInFocus));
			//gc.fillRectangle(rect);
		} else if (item instanceof ThreadItem) {
			TRCThread thread = ((ThreadItem) item)._thread;
			
			int x0 = rect.x;
			
			EList list = thread.getThreadEvents();
			
			int count = list.size();
			TRCThreadEvent lastEvent = null;
			if (count > 0) {
				
				// Pull the first event
				TRCThreadEvent currEvent = (TRCThreadEvent) list.get(0);
				TRCThreadEvent nextEvent = null;
				double currEventTime = currEvent.getTime();
				double nextEventTime = currEventTime;
				x0 = rect.x + (int) ((currEventTime - time0) * K);
				int xEnd = rect.x + (int) ((time1 - time0) * K);
				int x1 = -1;
				int idx = 1;

				// Reduce rect
				_rect1.y += 3;
				_rect1.height -= 6;
				// Fill space before first event
				if (x0 > rect.x) {
					_rect1.width = (x0 <= xEnd ? x0 : xEnd) - _rect1.x;
					gc.setBackground(_colors.getBkColor(selected, _isInFocus, false));
					gc.fillRectangle(_rect1);
					// draw middle line
					gc.setForeground(_colors.getColor(TraceColorScheme.MID_LINE));
					int midy = _rect1.y + _rect1.height / 2;
					gc.drawLine(_rect1.x, midy, _rect1.x + _rect1.width, midy);
				}

				// Draw event states
				while (x0 <= xEnd && null != currEvent) {
					// For each event....
					boolean stopped = false;//currEvent instanceof TRCThreadDeadEvent;
					if (idx < count) {
						nextEvent = (TRCThreadEvent) list.get(idx);
						nextEventTime = nextEvent.getTime();
						idx++;
					} else if (stopped) {
						nextEvent = null;
						nextEventTime = time1;
					} else {
						nextEvent = null;
						nextEventTime = endTime;
					}
					x1 = rect.x + (int) ((nextEventTime - time0) * K);
					if (x1 >= rect.x) {
						_rect1.x = x0 >= rect.x ? x0 : rect.x;
						_rect1.width = (x1 <= xEnd ? x1 : xEnd) - _rect1.x;
						boolean timeSelected = currEventTime <= selectedTime && selectedTime < nextEventTime;
						Utils.drawState(_colors, currEvent, _rect1, gc, selected, false, timeSelected);
					}
					lastEvent = currEvent;
					currEvent = nextEvent;
					currEventTime = nextEventTime;
					x0 = x1;
				}
			}

			// Fill space after last event
			int xEnd = rect.x + rect.width;
			if (x0 < xEnd) {
				_rect1.x = x0 >= rect.x ? x0 : rect.x;
				_rect1.width = xEnd - _rect1.x;
				gc.setBackground(_colors.getBkColor(selected, _isInFocus, false));
				gc.fillRectangle(_rect1);
				// draw middle line
				gc.setForeground(_colors.getColor(Utils.getEventColor(lastEvent)));
				int midy = _rect1.y + _rect1.height / 2;
				int lw = gc.getLineWidth();
				gc.setLineWidth(2);
				gc.drawLine(_rect1.x, midy, _rect1.x + _rect1.width, midy);
				gc.setLineWidth(lw);
			}

			// Draw focus ares
			Utils.init(_rect1, rect);
			gc.setForeground(_colors.getBkColor(selected, _isInFocus, false));
			int y = _rect1.y;
			gc.drawLine(_rect1.x, y, _rect1.x + _rect1.width, y);
			y++;
			gc.drawLine(_rect1.x, y, _rect1.x + _rect1.width, y);
			y++;
			gc.drawLine(_rect1.x, y, _rect1.x + _rect1.width, y);
			y = _rect1.y + _rect1.height - 1;
			gc.drawLine(_rect1.x, y, _rect1.x + _rect1.width, y);
			y--;
			gc.drawLine(_rect1.x, y, _rect1.x + _rect1.width, y);
			y--;
			gc.drawLine(_rect1.x, y, _rect1.x + _rect1.width, y);
		}

		// draw selected time
		int x = rect.x + (int) ((selectedTime - time0) * K);
		if (x >= rect.x && x < rect.x + rect.width) {
			gc.setForeground(_colors.getColor(TraceColorScheme.SELECTED_TIME));
			if (group)
				gc.drawLine(x, rect.y + rect.height - 1, x, rect.y + rect.height);
			else
				gc.drawLine(x, rect.y, x, rect.y + rect.height);
		}
	}

	public void keyTraversed(TraverseEvent e) {
		if ((e.detail==SWT.TRAVERSE_TAB_NEXT)||
		(e.detail==SWT.TRAVERSE_TAB_PREVIOUS))
			e.doit=true;
	}
	
	public void keyPressed(KeyEvent e) {
		int idx = -1;
		if (SWT.HOME == e.keyCode) {
			idx = getTopIndex();
		} else if (SWT.END == e.keyCode) {
			idx = getBottomIndex();
		} else if (SWT.ARROW_DOWN == e.keyCode) {
			idx = getSelectedIndex();
			if (idx < 0)
				idx = 0;
			else if (idx < _data._items.length - 1)
				idx++;
		} else if (SWT.ARROW_UP == e.keyCode) {
			idx = getSelectedIndex();
			if (idx < 0)
				idx = 0;
			else if (idx > 0)
				idx--;
		} else if (SWT.ARROW_LEFT == e.keyCode) {
			selectPrevEvent();
		} else if (SWT.ARROW_RIGHT == e.keyCode) {
			selectNextEvent();
		} else if (SWT.PAGE_DOWN == e.keyCode) {
			int page = countPerPage();
			idx = getSelectedIndex();
			if (idx < 0)
				idx = 0;
			idx += page;
			if (idx >= _data._items.length)
				idx = _data._items.length - 1;
		} else if (SWT.PAGE_UP == e.keyCode) {
			int page = countPerPage();
			idx = getSelectedIndex();
			if (idx < 0)
				idx = 0;
			idx -= page;
			if (idx < 0)
				idx = 0;
		} else if (SWT.CR == e.keyCode) {
			idx = getSelectedIndex();
			if (idx >= 0) {
				if (_data._items[idx] instanceof ThreadItem)
					fireDefaultSelection();
				else if (_data._items[idx] instanceof GroupItem)
					toggle(idx);
			}
			idx = -1;
		}
		if (idx >= 0) {
			selectItem(idx, false);
			fireSelectionChanged();
		}
	}

	public void keyReleased(KeyEvent e) {
	}

	public void focusGained(FocusEvent e) {
		_isInFocus = true;
		redraw();
	}

	public void focusLost(FocusEvent e) {
		_isInFocus = false;
		if (0 != _dragState) {
			setCapture(false);
			_dragState = 0;
		}
		redraw();
	}

	public void mouseMove(MouseEvent e) {
		if (null == _timeProvider)
			return;
		Point size = getCtrlSize();
		if (1 == _dragState) {
			int nameWidth = _timeProvider.getNameSpace();
			int x = e.x - nameWidth;
			if (x > 0 && size.x > nameWidth && _dragX != x) {
				_dragX = x;
				double K = (double) (size.x - nameWidth) / (_time1bak - _time0bak);
				double timeDelta = (_dragX - _dragX0) / K;
				double time1 = _time1bak - timeDelta;
				double maxTime = _timeProvider.getMaxTime();
				if (time1 > maxTime)
					time1 = maxTime;
				double time0 = time1 - (_time1bak - _time0bak);
				if (time0 < _timeProvider.getMinTime()) {
					time0 = _timeProvider.getMinTime();
					time1 = time0 + (_time1bak - _time0bak);
				}
				_timeProvider.setStartFinishTime(time0, time1);
			}
		} else if (3 == _dragState) {
			_dragX = e.x;
			_timeProvider.setNameSpace(_hitIdx + _dragX - _dragX0);
		} else if (0 == _dragState) {
			boolean mouseHover = hitSplitTest(e.x, e.y) > 0;
			if (_mouseHover != mouseHover)
				redraw();
			_mouseHover = mouseHover;
		}
		updateCursor(e.x, e.y);
	}

	public void mouseDoubleClick(MouseEvent e) {
		if (null == _timeProvider)
			return;
		if (1 == e.button) {
			int idx = hitSplitTest(e.x, e.y);
			if (idx >= 0) {
				_timeProvider.setNameSpace(_idealNameWidth + 3 * MARGIN + SMALL_ICON_SIZE);
				return;
			}
			idx = hitTest(e.x, e.y);
			if (idx >= 0) {
				selectItem(idx, false);
				if (_data._items[idx] instanceof ThreadItem) {
					fireDefaultSelection();
				}
			}
		}
	}

	void updateCursor(int x, int y) {
		int idx = hitSplitTest(x, y);
		if (idx > 0 && !_isDragCursor3) {
			setCursor(_dragCursor3);
			_isDragCursor3 = true;
		} else if (idx <= 0 && _isDragCursor3) {
			setCursor(null);
			_isDragCursor3 = false;
		}
	}

	public void mouseDown(MouseEvent e) {
		if (null == _timeProvider)
			return;
		if (1 == e.button) {
			int idx = hitSplitTest(e.x, e.y);
			if (idx > 0) {
				_dragState = 3;
				_dragX = _dragX0 = e.x;
				_hitIdx = _timeProvider.getNameSpace();;
				_time0bak = _timeProvider.getTime0();
				_time1bak = _timeProvider.getTime1();
				redraw();
				return;
			}
			idx = hitTest(e.x, e.y);
			if (idx >= 0) {
				if (_data._items[idx] instanceof ThreadItem) {
					double hitTime = hitTimeTest(e.x, e.y);
					if (hitTime >= 0) {
						_timeProvider.setSelectedTime(hitTime, false);
						setCapture(true);
						_dragState = 1;
						_dragX = _dragX0 = e.x - _timeProvider.getNameSpace();
						_time0bak = _timeProvider.getTime0();
						_time1bak = _timeProvider.getTime1();
					}
				} else if (_data._items[idx] instanceof GroupItem) {
					_hitIdx = idx;
					_dragState = 2;
				}
				selectItem(idx, false);
				fireSelectionChanged();
			}
		}
	}

	public void mouseUp(MouseEvent e) {
		if (0 != _dragState) {
			setCapture(false);
			if (2 == _dragState) {
				if (hitTest(e.x, e.y) == _hitIdx)
					toggle(_hitIdx);
			} else if (3 == _dragState) {
				redraw();
			}
			_dragState = 0;
		}
	}

	public void controlMoved(ControlEvent e) {
	}

	public void controlResized(ControlEvent e) {
		adjustScrolls();
	}

	public void widgetDefaultSelected(SelectionEvent e) {
	}

	public void widgetSelected(SelectionEvent e) {
		if (e.widget == getVerticalBar()) {
			_topItem = getVerticalBar().getSelection();
			if (_topItem < 0)
				_topItem = 0;
			redraw();
		} else if (e.widget == getHorizontalBar() && null != _timeProvider) {
			int startTime = getHorizontalBar().getSelection();
			double time0 = _timeProvider.getTime0();
			double time1 = _timeProvider.getTime1();
			double range = time1 - time0;
			// _timeRangeFixed = true;
			time0 = _timeStep * startTime;
			time1 = time0 + range;
			_timeProvider.setStartFinishTime(time0, time1);
		}
	}

	public void mouseEnter(MouseEvent e) {
	}

	public void mouseExit(MouseEvent e) {
		if (_mouseHover) {
			_mouseHover = false;
			redraw();
		}
	}

	public void mouseHover(MouseEvent e) {
	}

	public void mouseScrolled(MouseEvent e) {
		// Removing link to zoom in/out from mouse wheel events for bug 295656
//		if (e.count > 0) {
//			zoomIn();
//		}
//		else if (e.count < 0) {
//			zoomOut();
//		}
	}
}

class Item {
	public boolean _expanded;
	public boolean _selected;
	public boolean _hasChildren;
	public String _name;

	Item(String name) {
		_name = name;
	}

	public String toString() {
		return _name;
	}
}

class ThreadItem extends Item {
	public TRCThread _thread;

	ThreadItem(TRCThread thread) {
		super(Utils.composeThreadName(thread, false));
		_thread = thread;
	}
}

class GroupItem extends Item {
	public List _threads;

	GroupItem(String name) {
		super(name);
		_threads = new ArrayList();
		_hasChildren = true;
	}

	void add(Object thread) {
		_threads.add(thread);
	}
}

/** Data is populated into ItemData by the 
 * refreshData(Object[]) method, which updates _threads and causes the other
 * items to be updated*/
class ItemData {
	public Object[] _items = new Object[0];
	private Object _threads[] = new Object[0];
	private boolean threadFilter[] = new boolean[0];
	private Map _groupTable = new HashMap();
	private boolean _flatList = false;
	
	protected void groupThreads(boolean on) {
		if (_flatList == on) {
			_flatList = !on;
			updateItems();
		}
	}

	void clearGroups() {
		Iterator it = _groupTable.values().iterator();
		while (it.hasNext()) {
			GroupItem group = (GroupItem) it.next();
			group._threads.clear();
		}
	}

	void deleteEmptyGroups() {
		Iterator it = _groupTable.values().iterator();
		while (it.hasNext()) {
			GroupItem group = (GroupItem) it.next();
			if (group._threads.size() == 0) it.remove();
		}
	}

	ThreadItem findThreadItem(TRCThread thread) {
		if (thread == null) return null;
		
		int threadId = thread.getId();
		ThreadItem threadItem = null;
		
		for (int i=0; i<_items.length; i++) {
			Object item = _items[i];
			if (item instanceof ThreadItem) {
				ThreadItem ti = (ThreadItem) item;
				if (ti._thread.getId() == threadId) {
					threadItem = ti;
					break;
				}
			}
		}

		return threadItem;
	}

	public void updateItems() {
		List itemList = new ArrayList();
		
		Iterator it = _groupTable.values().iterator();
		while (it.hasNext()) {
			GroupItem group = (GroupItem) it.next();
			if (!_flatList)
				itemList.add(group);
			
			if (_flatList || group._expanded) {
				Iterator it2 = group._threads.iterator();
				while (it2.hasNext()) {
					TRCThread thread = (TRCThread) it2.next();
					ThreadItem threadItem = findThreadItem(thread);
					if (threadItem == null)
						threadItem = new ThreadItem(thread);
					itemList.add(threadItem);
				}
			}
		}
		_items = itemList.toArray();
	}

	public int expandItem(int idx, boolean expand) {
		if (idx < 0 || idx >= _items.length)
			return 0;
		int ret = 0;
		Item item = (Item) _items[idx];
		if (item._hasChildren && !item._expanded) {
			item._expanded = expand;
			ret = _items.length;
			updateItems();
			ret = _items.length - ret;
		}
		return ret;
	}

	public void refreshData(Object threads[]) {
		if (threads == null || threads.length == 0)
			threadFilter = null;
		else if (threadFilter == null || threads.length != threadFilter.length) {
			threadFilter = new boolean[threads.length];
			java.util.Arrays.fill(threadFilter, true);
		}
		
		_threads = threads;

		refreshData();
	}

	public void refreshData() {
		clearGroups();
		String undef = UIMessages._UNDEFINED_GROUP;
		List groupList = new ArrayList();
		for (int i = 0; i < _threads.length; i++) {
			TRCThread thread = (TRCThread) _threads[i];
			if (!threadFilter[i]) continue;
			
			String groupName = thread.getGroupName();
			if (null == groupName) groupName = undef;
			
			GroupItem group = (GroupItem) _groupTable.get(groupName);
			if (null == group) {
				group = new GroupItem(NLS.bind(UIMessages._THREAD_GROUP_LABEL, groupName));
				group._expanded = !groupName.equalsIgnoreCase("system")
						&& !groupName.equalsIgnoreCase(undef);
				_groupTable.put(groupName, group);
				groupList.add(group);
			}
			group.add(thread);
		}
		
		deleteEmptyGroups();
		updateItems();
	}
	
	public Object[] getThreads() {
		return _threads;
	}
	
	public boolean[] getThreadFilter() {
		return threadFilter;
	}
}
