/*****************************************************************************
 * Copyright (c) 2007, 2009 Intel Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Intel Corporation - Initial API and implementation
 *    Ruslan A. Scherbakov, Intel - Initial API and implementation
 *
 * $Id: TimeScaleCtrl.java,v 1.6 2009/08/11 20:54:55 ewchan Exp $ 
 *****************************************************************************/

package org.eclipse.tptp.trace.jvmti.internal.client.widgets;

import org.eclipse.swt.SWT;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.events.MouseListener;
import org.eclipse.swt.events.MouseMoveListener;
import org.eclipse.swt.events.PaintEvent;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.tptp.trace.jvmti.internal.client.views.UIMessages;

public class TimeScaleCtrl extends TraceCtrl implements MouseListener, MouseMoveListener {

	public TimeScaleCtrl(Composite parent, TraceColorScheme colors) {
		super(parent, colors, SWT.NO_BACKGROUND | SWT.NO_FOCUS | SWT.DOUBLE_BUFFERED);
		addMouseListener(this);
		addMouseMoveListener(this);
	}

	private ITimeDataProvider _timeProvider;
	private int _dragState = 0;
	private int _dragX0 = 0;
	private int _dragX = 0;
	private double _time0bak;
	private double _time1bak;
	private boolean _isInUpdate;
	private Rectangle _rect0 = new Rectangle(0, 0, 0, 0);

	public void setTimeProvider(ITimeDataProvider timeProvider) {
		_timeProvider = timeProvider;
	}

	private double _timeDeltaD;
	private long _timeDelta;
	private void calcTimeDelta(int width, double K) {
		long D[] = { 1, 2, 5, };
		long pow = 1;
		double powD = 0.000000001;
		long td = pow;
		double tdD = powD;
		double dx = tdD * K;
		int i = 0;
		while (dx < width) {
			td = D[i] * pow;
			tdD = D[i] * powD;
			dx = tdD * K;
			i++;
			if (i == 3) {
				i = 0;
				pow *= 10;
				powD *= 10;
			}
		}
		_timeDeltaD = tdD;
		_timeDelta = td;
	}

	static private TimeDraw _tds[] = new TimeDraw[] {
		new TimeDrawSec(),
		new TimeDrawMillisec(),
		new TimeDrawMicrosec(),
		new TimeDrawNanosec(),
	};
	
	static TimeDraw getTimeDraw(long timeDelta) {
		TimeDraw timeDraw;
		if (timeDelta >= 1000000000)
			timeDraw = _tds[0];
		else if (timeDelta >= 1000000)
			timeDraw = _tds[1];
		else if (timeDelta >= 1000) 
			timeDraw = _tds[2];
		else
			timeDraw = _tds[3];
		return timeDraw;
	}

	void paint(Rectangle rect, PaintEvent e) {

		if (_isInUpdate || null == _timeProvider)
			return;

		GC gc = e.gc;
		if (null == _timeProvider) {
			gc.fillRectangle(rect);
			return;
		}

		gc.setBackground(_colors.getColor(TraceColorScheme.TOOL_BACKGROUND));
		gc.setForeground(_colors.getColor(TraceColorScheme.TOOL_FOREGROUND));
		double time0 = _timeProvider.getTime0();
		double time1 = _timeProvider.getTime1();
		double selectedTime = _timeProvider.getSelectedTime();
		int leftSpace = _timeProvider.getNameSpace();
		int timeSpace = _timeProvider.getTimeSpace();

		if (time1 <= time0 || timeSpace < 2) {
			gc.fillRectangle(rect);
			return;
		}

		double timeRange = time1 - time0;
		int numDigits = 8; // 11:222
		if (timeRange < .00001)
			numDigits = 16; // 11:222:333:444__
		else if (timeRange < .01)
			numDigits = 12; // 11:222:333__

		Utils.init(_rect0, rect);
		int labelWidth = gc.getCharWidth('0') * numDigits;
		double K = 1;
		if (rect.width - leftSpace > 0) {
			K = (double) timeSpace / (time1 - time0);
			calcTimeDelta(labelWidth, K);
		}
		TimeDraw timeDraw = getTimeDraw(_timeDelta);

		// draw top left area
		_rect0.width = leftSpace;
		gc.fillRectangle(_rect0);
		_rect0.x += 4;
		_rect0.width -= 4;
		if (_rect0.width > 0) {
			if (false && rect.width - leftSpace > 0)
				Utils.drawText(gc, UIMessages._Timescale + " : " + timeDraw.hint(), _rect0, true);
			else
				Utils.drawText(gc, UIMessages._Timescale + " : ", _rect0, true);
		}
		_rect0.x -= 4;
		_rect0.width += 4;

		// prepare and draw right rect of the timescale
		_rect0.x += leftSpace;
		_rect0.width = rect.width - leftSpace;

		// draw bottom border and erase all other area
		gc.drawLine(rect.x, rect.y + rect.height - 1, rect.x + rect.width - 1,
				rect.y + rect.height - 1);
		_rect0.height--;
		gc.fillRectangle(_rect0);

		if (_rect0.isEmpty())
			return;

		// draw selected time
		int x = _rect0.x + (int) ((selectedTime - time0) * K);
		if (x >= _rect0.x && x < _rect0.x + _rect0.width) {
			gc.setForeground(_colors.getColor(TraceColorScheme.SELECTED_TIME));
			gc.drawLine(x, _rect0.y + _rect0.height - 6, x, _rect0.y + _rect0.height);
			gc.setForeground(_colors.getColor(TraceColorScheme.TOOL_FOREGROUND));
		}

		// draw time scale ticks
		_rect0.y = rect.y;
		_rect0.height = rect.height - 4;
		_rect0.width = labelWidth;
		double time = Math.floor(time0 / _timeDeltaD) * _timeDeltaD;
		long t = (long)(time * 1000000000);
		int y = _rect0.y + _rect0.height;
		while (true) {
			x = rect.x + leftSpace + (int) ((time - time0) * K);
			if (x >= rect.x + leftSpace + rect.width - _rect0.width) {
				break;
			}
			if (x >= rect.x + leftSpace) {
				gc.drawLine(x, y, x, y + 4);
				_rect0.x = x;
				if (x + _rect0.width <= rect.x + rect.width)
					timeDraw.draw(gc, t, _rect0);
			}
			time += _timeDeltaD;
			t += _timeDelta;
		}
	}

	public void mouseDown(MouseEvent e) {
		if (1 == e.button && null != _timeProvider) {
			setCapture(true);
			_dragState = 1;
			_dragX = _dragX0 = e.x - _timeProvider.getNameSpace();
			_time0bak = _timeProvider.getTime0();
			_time1bak = _timeProvider.getTime1();
		}
	}

	public void mouseUp(MouseEvent e) {
		if (1 == _dragState) {
			setCapture(false);
			_dragState = 0;
		}
	}

	public void mouseMove(MouseEvent e) {
		if(_dragX0 < 0){
			return;
		}
		Point size = getSize();
		if (1 == _dragState && null != _timeProvider) {
			int leftSpace = _timeProvider.getNameSpace();
			int x = e.x - leftSpace;
			if (x > 0 && size.x > leftSpace && _dragX != x) {
				_dragX = x;
				double time1 = _time0bak + (_time1bak - _time0bak) * _dragX0 / _dragX;
				_timeProvider.setStartFinishTime(_time0bak, time1);
			}
		}
	}

	public void mouseDoubleClick(MouseEvent e) {
		if (null != _timeProvider) {
			_timeProvider.resetStartFinishTime();
		}
	}
}

abstract class TimeDraw {
	static String S = ":";
	static String S0 = ":0";
	static String S00 = ":00";

	static String pad(long n) {
		String s = S;
		if (n < 10)
			s = S00;
		else if (n < 100)
			s = S0;
		return s + n;
	}

	public abstract void draw(GC gc, long time, Rectangle rect);
	public abstract String hint();
}

class TimeDrawSec extends TimeDraw {
	static String _hint = "sec";

	public void draw(GC gc, long time, Rectangle rect) {
		time /= 1000000000;
		Utils.drawText(gc, time + "", rect, true);
	}

	public String hint() {
		return _hint;
	}
}

class TimeDrawMillisec extends TimeDraw {
	static String _hint = "s:ms";

	public void draw(GC gc, long time, Rectangle rect) {
		time /= 1000000;
		long ms = time % 1000;
		time /= 1000;
		Utils.drawText(gc, time + pad(ms), rect, true);
	}

	public String hint() {
		return _hint;
	}
}

class TimeDrawMicrosec extends TimeDraw {
	static String _hint = "s:ms:mcs";

	public void draw(GC gc, long time, Rectangle rect) {
		time /= 1000;
		long mcs = time % 1000;
		time /= 1000;
		long ms = time % 1000;
		time /= 1000;
		Utils.drawText(gc, time + pad(ms) + pad(mcs), rect, true);
	}

	public String hint() {
		return _hint;
	}
}

class TimeDrawNanosec extends TimeDraw {
	static String _hint = "s:ms:mcs:ns";

	public void draw(GC gc, long time, Rectangle rect) {
		long ns = time % 1000;
		time /= 1000;
		long mcs = time % 1000;
		time /= 1000;
		long ms = time % 1000;
		time /= 1000;
		Utils.drawText(gc, time + pad(ms) + pad(mcs) + pad(ns), rect, true);
	}

	public String hint() {
		return _hint;
	}
}
