/*****************************************************************************
 * Copyright (c) 2008 Intel Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Intel Corporation - Initial API and implementation
 *    Ruslan A. Scherbakov, Intel - Initial API and implementation
 *
 * $Id: TraceColorScheme.java,v 1.3 2008/05/09 16:11:24 jkubasta Exp $ 
 *****************************************************************************/

package org.eclipse.tptp.trace.jvmti.internal.client.widgets;

import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Color;

public class TraceColorScheme {

	// state elements color indices
	static public final int UNKNOWN = 0;
	static public final int RUNNING = 1;
	static public final int SLEEPING = 2;
	static public final int WAITING = 3;
	static public final int BLOCKED = 4;
	static public final int DEADLOCK = 5;
	static public final int STOPPED = 6;

	static public final int STATES0 = 0;
	static public final int STATES1 = 7;

	// selected state elements color indices
	static public final int UNKNOWN_SEL = 7;
	static public final int RUNNING_SEL = 8;
	static public final int SLEEPING_SEL = 9;
	static public final int WAITING_SEL = 10;
	static public final int BLOCKED_SEL = 11;
	static public final int DEADLOCK_SEL = 12;
	static public final int STOPPED_SEL = 13;

	static public final int STATES_SEL0 = 7;
	static public final int STATES_SEL1 = 13;

	// colors indices for viewer controls
	static public final int BACKGROUND = 14;
	static public final int FOREGROUND = 15;
	static public final int BACKGROUND_SEL = 16;
	static public final int FOREGROUND_SEL = 17;
	static public final int BACKGROUND_SEL_NOFOCUS = 18;
	static public final int FOREGROUND_SEL_NOFOCUS = 19;
	static public final int TOOL_BACKGROUND = 20;
	static public final int TOOL_FOREGROUND = 21;

	// misc colors
	static public final int FIX_COLOR = 22;
	static public final int WHITE = 23;
	static public final int GRAY = 24;
	static public final int BLACK = 25;
	static public final int DARK_GRAY = 26;

	// selected state elements border color indices
	static public final int UNKNOWN_BORDER = 27;
	static public final int RUNNING_BORDER = 28;
	static public final int SLEEPING_BORDER = 29;
	static public final int WAITING_BORDER = 30;
	static public final int BLOCKED_BORDER = 31;
	static public final int DEADLOCK_BORDER = 32;
	static public final int STOPPED_BORDER = 33;

	static public final int STATES_BORDER0 = 27;
	static public final int STATES_BORDER1 = 33;

	static public final int MID_LINE = 34;
	static public final int RED = 35;
	static public final int GREEN = 36;
	static public final int BLUE = 37;
	static public final int YELLOW = 38;
	static public final int CYAN = 39;
	static public final int MAGENTA = 40;

	static public final int SELECTED_TIME = 41;
	static public final int LEGEND_BACKGROUND = 42;
	static public final int LEGEND_FOREGROUND = 43;

	// group items' colors
	static public final int GR_BACKGROUND = 44;
	static public final int GR_FOREGROUND = 45;
	static public final int GR_BACKGROUND_SEL = 46;
	static public final int GR_FOREGROUND_SEL = 47;
	static public final int GR_BACKGROUND_SEL_NOFOCUS = 48;
	static public final int GR_FOREGROUND_SEL_NOFOCUS = 49;
	
	static public final int LIGHT_LINE = 50;
	static public final int BACKGROUND_NAME = 51;

	// Threads interraction's colors
	static public final int TI_START_THREAD = BLACK;
	static public final int TI_HANDOFF_LOCK = BLUE;
	static public final int TI_NOTIFY_ALL = GREEN;
	static public final int TI_NOTIFY = GREEN;
	static public final int TI_NOTIFY_JOINED = DARK_GRAY;
	static public final int TI_INTERRUPT = RED;
	static public final int TI_WAIT_EXCEEDED = BLUE;
	
	static interface IColorProvider {
		public Color get();
	}

	static class SysCol implements IColorProvider {
		int syscol;

		SysCol(int syscol) {
			this.syscol = syscol;
		}

		public Color get() {
			return Utils.getSysColor(syscol);
		}
	}

	static class RGB implements IColorProvider {
		int r;
		int g;
		int b;

		RGB(int r, int g, int b) {
			this.r = r;
			this.g = g;
			this.b = b;
		}

		public Color get() {
			return new Color(null, r, g, b);
		}
	}

	static class Mix implements IColorProvider {
		IColorProvider cp1;
		IColorProvider cp2;
		int w1;
		int w2;

		Mix(IColorProvider cp1, IColorProvider cp2, int w1, int w2) {
			this.cp1 = cp1;
			this.cp2 = cp2;
			this.w1 = w1;
			this.w2 = w2;
		}

		Mix(IColorProvider cp1, IColorProvider cp2) {
			this.cp1 = cp1;
			this.cp2 = cp2;
			this.w1 = 1;
			this.w2 = 1;
		}

		public Color get() {
			Color col1 = cp1.get();
			Color col2 = cp2.get();
			Color col = Utils.mixColors(col1, col2, w1, w2);
			return col;
		}
	}

	static private final IColorProvider _providersMap[] = {
			//
			new RGB(100, 100, 100), // UNKNOWN
			new RGB(174, 200, 124), // RUNNING
			new Mix(new SysCol(SWT.COLOR_BLUE), new SysCol(SWT.COLOR_GRAY), 1, 3), // SLEEPING
			new RGB(210, 150, 60), // WAITING
			new RGB(242, 225, 168), // BLOCKED
			new Mix(new SysCol(SWT.COLOR_RED), new SysCol(SWT.COLOR_GRAY), 1, 3), // DEADLOCK
			new RGB(200, 200, 200), // STOPPED

			new SysCol(SWT.COLOR_WHITE), // UNKNOWN_SEL
			new SysCol(SWT.COLOR_GREEN), // RUNNING_SEL
			new SysCol(SWT.COLOR_BLUE), // SLEEPING_SEL
			new SysCol(SWT.COLOR_CYAN), // WAITING_SEL
			new SysCol(SWT.COLOR_YELLOW), // BLOCKED_SEL
			new SysCol(SWT.COLOR_RED), // DEADLOCK_SEL
			new SysCol(SWT.COLOR_DARK_GRAY), // STOPPED_SEL

			new SysCol(SWT.COLOR_LIST_BACKGROUND), // BACKGROUND
			new SysCol(SWT.COLOR_LIST_FOREGROUND), // FOREGROUND
			new SysCol(SWT.COLOR_LIST_SELECTION), // BACKGROUND_SEL
			new SysCol(SWT.COLOR_LIST_SELECTION_TEXT), // FOREGROUND_SEL
			new SysCol(SWT.COLOR_WIDGET_BACKGROUND), // BACKGROUND_SEL_NOFOCUS
			new SysCol(SWT.COLOR_WIDGET_FOREGROUND), // FOREGROUND_SEL_NOFOCUS
			new SysCol(SWT.COLOR_WIDGET_BACKGROUND), // TOOL_BACKGROUND
			new SysCol(SWT.COLOR_WIDGET_DARK_SHADOW), // TOOL_FOREGROUND

			new SysCol(SWT.COLOR_GRAY), // FIX_COLOR
			new SysCol(SWT.COLOR_WHITE), // WHITE
			new SysCol(SWT.COLOR_GRAY), // GRAY
			new SysCol(SWT.COLOR_BLACK), // BLACK
			new SysCol(SWT.COLOR_DARK_GRAY), // DARK_GRAY

			new SysCol(SWT.COLOR_DARK_GRAY), // UNKNOWN_BORDER
			new RGB(75, 115, 120), // RUNNING_BORDER
			new SysCol(SWT.COLOR_DARK_BLUE), // SLEEPING_BORDER
			new RGB(242, 225, 168), // WAITING_BORDER
			new RGB(210, 150, 60), // BLOCKED_BORDER
			new SysCol(SWT.COLOR_DARK_RED), // DEADLOCK_BORDER
			new SysCol(SWT.COLOR_BLACK), // STOPPED_BORDER

			new SysCol(SWT.COLOR_GRAY), // MID_LINE
			new SysCol(SWT.COLOR_RED), // RED
			new SysCol(SWT.COLOR_GREEN), // GREEN
			new SysCol(SWT.COLOR_BLUE), // BLUE
			new SysCol(SWT.COLOR_YELLOW), // YELLOW
			new SysCol(SWT.COLOR_CYAN), // CYAN
			new SysCol(SWT.COLOR_MAGENTA), // MAGENTA

			new SysCol(SWT.COLOR_BLUE), // SELECTED_TIME
			new SysCol(SWT.COLOR_WIDGET_BACKGROUND), // LEGEND_BACKGROUND
			new SysCol(SWT.COLOR_WIDGET_DARK_SHADOW), // LEGEND_FOREGROUND

			new Mix(new RGB(150, 200, 240), new SysCol(SWT.COLOR_LIST_BACKGROUND)),// GR_BACKGROUND
			new RGB(0, 0, 50), // GR_FOREGROUND
			new Mix(new RGB(200, 200, 100), new SysCol(SWT.COLOR_LIST_SELECTION)),  // GR_BACKGROUND_SEL
			new Mix(new RGB(150, 200, 240), new SysCol(SWT.COLOR_LIST_SELECTION_TEXT)), // GR_FOREGROUND_SEL
			new Mix(new RGB(222, 222, 155), new SysCol(SWT.COLOR_WIDGET_BACKGROUND)), // GR_BACKGROUND_SEL_NOFOCUS
			new RGB(0, 0, 50), // GR_FOREGROUND_SEL_NOFOCUS

			new Mix(new SysCol(SWT.COLOR_GRAY), new SysCol(SWT.COLOR_LIST_BACKGROUND), 1, 3), // LIGHT_LINE
			new Mix(new SysCol(SWT.COLOR_GRAY), new SysCol(SWT.COLOR_LIST_BACKGROUND), 1, 6), // BACKGROUND_NAME
	};

	private Color _colors[];

	public TraceColorScheme() {
		_colors = new Color[_providersMap.length];
	}

	public void dispose() {
		for (int i = 0; i < _colors.length; i++) {
			Utils.dispose(_colors[i]);
			_colors[i] = null;
		}
	}

	public Color getColor(int idx) {
		if (null == _colors[idx]) {
			if (idx >= STATES_SEL0 && idx <= STATES_SEL1) {
				Color col1 = getColor(idx - STATES_SEL0);
				Color col2 = getColor(BACKGROUND_SEL);
				_colors[idx] = Utils.mixColors(col1, col2, 3, 1);
			} else {
				_colors[idx] = _providersMap[idx].get();
			}
		}
		return _colors[idx];
	}

	public Color getBkColor(boolean selected, boolean focused, boolean name) {
		if (selected && focused)
			return getColor(BACKGROUND_SEL);
		if (selected)
			return getColor(BACKGROUND_SEL_NOFOCUS);
		if (name)
			return getColor(BACKGROUND_NAME);
		return getColor(BACKGROUND);
	}

	public Color getFgColor(boolean selected, boolean focused) {
		if (selected && focused)
			return getColor(FOREGROUND_SEL);
		if (selected)
			return getColor(FOREGROUND_SEL_NOFOCUS);
		return getColor(FOREGROUND);
	}

	public Color getBkColorGroup(boolean selected, boolean focused) {
		if (selected && focused)
			return getColor(GR_BACKGROUND_SEL);
		if (selected)
			return getColor(GR_BACKGROUND_SEL_NOFOCUS);
		return getColor(GR_BACKGROUND);
	}

	public Color getFgColorGroup(boolean selected, boolean focused) {
		if (selected && focused)
			return getColor(GR_FOREGROUND_SEL);
		if (selected)
			return getColor(GR_FOREGROUND_SEL_NOFOCUS);
		return getColor(GR_FOREGROUND);
	}
}
