/*****************************************************************************
 * Copyright (c) 2007, 2010 Intel Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Intel Corporation - Initial API and implementation
 *    Ruslan A. Scherbakov, Intel - Initial API and implementation
 *
 * $Id: Utils.java,v 1.12 2010/11/23 13:59:52 mreid Exp $ 
 *****************************************************************************/

package org.eclipse.tptp.trace.jvmti.internal.client.widgets;

import java.util.Iterator;

import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.hyades.models.hierarchy.TRCAgentProxy;
import org.eclipse.hyades.models.hierarchy.TRCAnnotation;
import org.eclipse.hyades.models.hierarchy.TRCNode;
import org.eclipse.hyades.models.hierarchy.TRCProcessProxy;
import org.eclipse.hyades.models.trace.TRCClass;
import org.eclipse.hyades.models.trace.TRCFullTraceObject;
import org.eclipse.hyades.models.trace.TRCGCEvent;
import org.eclipse.hyades.models.trace.TRCMethod;
import org.eclipse.hyades.models.trace.TRCMethodInvocation;
import org.eclipse.hyades.models.trace.TRCObjectAllocationAnnotation;
import org.eclipse.hyades.models.trace.TRCPackage;
import org.eclipse.hyades.models.trace.TRCThread;
import org.eclipse.hyades.models.trace.TRCThreadDeadEvent;
import org.eclipse.hyades.models.trace.TRCThreadDeadLockEvent;
import org.eclipse.hyades.models.trace.TRCThreadEvent;
import org.eclipse.hyades.models.trace.TRCThreadRunningEvent;
import org.eclipse.hyades.models.trace.TRCThreadSleepingEvent;
import org.eclipse.hyades.models.trace.TRCThreadWaitingForLockEvent;
import org.eclipse.hyades.models.trace.TRCThreadWaitingForObjectEvent;
import org.eclipse.hyades.trace.ui.ITraceSelection;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.ui.internal.util.TString;
import org.eclipse.hyades.trace.ui.internal.util.TraceMessages;
import org.eclipse.hyades.trace.views.internal.TraceUIMessages;
import org.eclipse.hyades.trace.views.internal.TraceUIPlugin;
import org.eclipse.hyades.trace.views.util.internal.OpenSource;
import org.eclipse.hyades.ui.internal.util.OpenJavaSource;
import org.eclipse.hyades.ui.provisional.context.ContextManager;
import org.eclipse.hyades.ui.provisional.context.IContextLabelFormatProvider;
import org.eclipse.jdt.core.search.IJavaSearchConstants;
import org.eclipse.jdt.core.search.SearchEngine;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.BusyIndicator;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Device;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.widgets.Display;
import org.eclipse.tptp.trace.jvmti.internal.client.AllocationSite;
import org.eclipse.tptp.trace.jvmti.internal.client.MethodDetails;
import org.eclipse.tptp.trace.jvmti.internal.client.TITracePlugin;
import org.eclipse.tptp.trace.jvmti.internal.client.views.UIMessages;
import org.eclipse.ui.plugin.AbstractUIPlugin;

public class Utils {

	static public final int IMG_THREAD_RUNNING = 0;
	static public final int IMG_THREAD_SUSPENDED = 1;
	static public final int IMG_THREAD_STOPPED = 2;
	static public final int IMG_METHOD_RUNNING = 3;
	static public final int IMG_METHOD = 4;
	static public final int IMG_NUM = 5;

	static public final Object[] _empty = new Object[0];

	static private String _externalPlugin[] = {
			"org.eclipse.debug.ui",
			"org.eclipse.debug.ui",
			"org.eclipse.debug.ui",
			"org.eclipse.debug.ui",
			"org.eclipse.debug.ui",
			};

	static private String _externalPath[] = {
			"icons/full/obj16/thread_obj.gif", // running thread
			"icons/full/obj16/threads_obj.gif", // suspended
			"icons/full/obj16/threadt_obj.gif", // stopped
			"icons/full/obj16/stckframe_running_obj.gif", // running stack frame
			"icons/full/obj16/stckframe_obj.gif", // stack frame
			};

	static public Image getImage(int idx) {
		if (idx < 0 || idx >= IMG_NUM)
			SWT.error(SWT.ERROR_INVALID_ARGUMENT);
		String key = "trace.img." + idx;
		Image img = TITracePlugin.getDefault().getImageRegistry().get(key);
		if (null == img) {
			ImageDescriptor desc = AbstractUIPlugin.imageDescriptorFromPlugin(
					_externalPlugin[idx], _externalPath[idx]);
			TITracePlugin.getDefault().getImageRegistry().put(key, desc);
			img = TITracePlugin.getDefault().getImageRegistry().get(key);
		}
		return img;
	}

	static public void init(Rectangle rect) {
		rect.x = 0;
		rect.y = 0;
		rect.width = 0;
		rect.height = 0;
	}

	static public void init(Rectangle rect, int x, int y, int width, int height) {
		rect.x = x;
		rect.y = y;
		rect.width = width;
		rect.height = height;
	}

	static public void init(Rectangle rect, Rectangle source) {
		rect.x = source.x;
		rect.y = source.y;
		rect.width = source.width;
		rect.height = source.height;
	}

	static public void deflate(Rectangle rect, int x, int y) {
		rect.x += x;
		rect.y += y;
		rect.width -= x + x;
		rect.height -= y + y;
	}

	static public void inflate(Rectangle rect, int x, int y) {
		rect.x -= x;
		rect.y -= y;
		rect.width += x + x;
		rect.height += y + y;
	}

	static void dispose(Color col) {
		if (null != col)
			col.dispose();
	}

	static public Color mixColors(Device display, Color c1, Color c2, int w1,
			int w2) {
		return new Color(display, (w1 * c1.getRed() + w2 * c2.getRed())
				/ (w1 + w2), (w1 * c1.getGreen() + w2 * c2.getGreen())
				/ (w1 + w2), (w1 * c1.getBlue() + w2 * c2.getBlue())
				/ (w1 + w2));
	}

	static public Color getSysColor(int id) {
		Color col = Display.getCurrent().getSystemColor(id);
		return new Color(col.getDevice(), col.getRGB());
	}

	static public Color mixColors(Color col1, Color col2, int w1, int w2) {
		return mixColors(Display.getCurrent(), col1, col2, w1, w2);
	}

	static public void drawState(TraceColorScheme colors, TRCThreadEvent event,
			Rectangle rect, GC gc, boolean selected, boolean rectBound, boolean timeSelected) {
		int colorIdx = getEventColor(event);
		drawState(colors, colorIdx, rect, gc, selected, rectBound, timeSelected);
	}

	static public void drawState(TraceColorScheme colors, int colorIdx,
			Rectangle rect, GC gc, boolean selected, boolean rectBound, boolean timeSelected) {
		if (rect.isEmpty())
			return;
		// fill all rect area
		int colorIdx1 = colorIdx;
		timeSelected = timeSelected && selected;
		if (timeSelected)
			colorIdx1 = colorIdx + TraceColorScheme.STATES_SEL0 - TraceColorScheme.STATES0;
		gc.setBackground(colors.getColor(colorIdx1));
		gc.fillRectangle(rect);
		colorIdx1 = colorIdx + TraceColorScheme.STATES_BORDER0 - TraceColorScheme.STATES0;
		gc.setForeground(colors.getColor(colorIdx1));
		// draw bounds
		if (!timeSelected) {
			if (rectBound && rect.width >= 3) {
				gc.drawRectangle(rect.x, rect.y, rect.width - 1, rect.height - 1);
			} else {
				gc.drawLine(rect.x, rect.y, rect.x + rect.width - 1, rect.y);
				gc.drawLine(rect.x, rect.y + rect.height - 1, rect.x + rect.width - 1, rect.y + rect.height - 1);
			}
		}
		// draw decoration middle line
		int mindy = rect.y + rect.height / 2;
		if (TraceColorScheme.BLOCKED == colorIdx || TraceColorScheme.WAITING == colorIdx) {
			int s = gc.getLineStyle();
			int w = gc.getLineWidth();
			gc.setLineStyle(SWT.LINE_DOT);
			gc.setLineWidth(2);
			gc.drawLine(rect.x, mindy, rect.x + rect.width, mindy);
			gc.setLineStyle(s);
			gc.setLineWidth(w);
		} else if (TraceColorScheme.DEADLOCK == colorIdx || TraceColorScheme.STOPPED == colorIdx) {
			int w = gc.getLineWidth();
			gc.setLineWidth(2);
			gc.drawLine(rect.x, mindy, rect.x + rect.width, mindy);
			gc.setLineWidth(w);
		}
		// draw selection bounds
		if (timeSelected) {
			gc.setForeground(colors.getColor(TraceColorScheme.SELECTED_TIME));
			if (rect.width >= 3) {
				gc.drawRectangle(rect.x, rect.y, rect.width - 1, rect.height - 1);
				//gc.drawRectangle(rect.x + 1, rect.y + 1, rect.width - 3, rect.height - 3);
			} else {
				gc.drawLine(rect.x, rect.y, rect.x + rect.width - 1, rect.y);
				gc.drawLine(rect.x, rect.y + rect.height - 1, rect.x + rect.width - 1, rect.y + rect.height - 1);				
			}
			gc.drawLine(rect.x, rect.y + 1, rect.x + rect.width - 1, rect.y + 1);
			gc.drawLine(rect.x, rect.y + rect.height - 2, rect.x + rect.width - 1, rect.y + rect.height - 2);
		}
	}

	static public int getEventColor(TRCThreadEvent event) {
		if (event instanceof TRCThreadRunningEvent)
			return TraceColorScheme.RUNNING;
		else if (event instanceof TRCThreadSleepingEvent)
			return TraceColorScheme.SLEEPING;
		else if (event instanceof TRCThreadDeadLockEvent)
			return TraceColorScheme.DEADLOCK;
		else if (event instanceof TRCThreadWaitingForLockEvent)
			return TraceColorScheme.BLOCKED;
		else if (event instanceof TRCThreadWaitingForObjectEvent)
			return TraceColorScheme.WAITING;
		else if (event instanceof TRCThreadDeadEvent)
			return TraceColorScheme.STOPPED;
		return TraceColorScheme.UNKNOWN;
	}

	static public int drawText(GC gc, String text, Rectangle rect, boolean transp) {
		Point size = gc.stringExtent(text);
		gc.drawText(text, rect.x, rect.y, transp);
		return size.x;
	}

	static public int drawText(GC gc, String text, int x, int y, boolean transp) {
		Point size = gc.stringExtent(text);
		gc.drawText(text, x, y, transp);
		return size.x;
	}

	static public TRCClass getThreadClass(TRCThread thread) {
		TRCClass cls = null;
		if (thread.getThreadObject() != null) {
			cls = thread.getThreadObject().getIsA();
		} else if (thread.getInitialInvocations().size() > 0) {
			cls = ((TRCMethodInvocation) thread.getInitialInvocations().get(0)).getMethod().getDefiningClass();
		}
		return cls;
	}

	static public String getThreadClassName(TRCThread thread) {
		String name = "";
		TRCClass cls = getThreadClass(thread);
		if (cls != null)
			name = getPackageClassName(cls);
		return name;
	}

	static public String getPackageClassName(TRCClass cls) {
		String name = "";
		if (cls != null) {
			TRCPackage pkg = cls.getPackage();
			if (pkg != null && pkg.getName().length() > 0)
				name = pkg.getName() + "." + cls.getName();
			else
				name = cls.getName();
		}
		return name;
	}

	static public String getFullMethodName(TRCMethod m, boolean sig) {
		String name = "";
		name += getPackageClassName(m.getDefiningClass());
		name += ".";
		name += m.getName();
		if (sig)
			name += m.getSignature();
		return name;
	}

	static public String getEventName(TRCThreadEvent event) {
		return getEventName(event, true, false);
	}

	static public String getEventName(TRCThreadEvent event, boolean upper, boolean extInfo) {
		String name = upper ? UIMessages._Unknown : UIMessages._unknown;
		if (event instanceof TRCThreadRunningEvent) {
			name = upper ? UIMessages._Running : UIMessages._running;
		} else if (event instanceof TRCThreadSleepingEvent) {
			name = upper ? UIMessages._Sleeping : UIMessages._sleeping;
		} else if (event instanceof TRCThreadDeadLockEvent) {
			name = upper ? UIMessages._Deadlocked : UIMessages._deadlocked;
		} else if (event instanceof TRCThreadWaitingForLockEvent) {
			name = upper ? UIMessages._Blocked : UIMessages._blocked;
			if (extInfo) {
				TRCThread thread = ((TRCThreadWaitingForLockEvent)event).getLockingThread();
				if (null != thread)
					name += " by thread=" + thread.getName();
			}
		} else if (event instanceof TRCThreadWaitingForObjectEvent) {
			name = upper ? UIMessages._Waiting : UIMessages._waiting;
			if (extInfo)
				name += " timeout=" + ((TRCThreadWaitingForObjectEvent)event).getTimeout();
		} else if (event instanceof TRCThreadDeadEvent) {
			name = upper ? UIMessages._Stopped : UIMessages._stopped;
		}
		return name;
	}

	static public String composeThreadName(TRCThread thread, boolean inclState) {
		String name = thread.getName();
		String threadClass = getThreadClassName(thread);
		if (threadClass != null && threadClass.length() > 0) {
			name += " [" + threadClass + "]";
		}
		if (inclState) {
			EList list = thread.getThreadEvents();
			if (null != list && list.size() > 0) {
				TRCThreadEvent event = (TRCThreadEvent)list.get(list.size() - 1);
				name+= " (" + getEventName(event, false, true) + ")";
			}
		}
		return name;
	}

	static public String composeMethodName(TRCMethodInvocation method) {
		return getFullMethodName(method.getMethod(), true);
	}

	static public String composeEventName(TRCThreadEvent event) {
		String name = event.getThread().getName();
		String threadClass = getThreadClassName(event.getThread());
		if (threadClass != null && threadClass.length() > 0) {
			name += " [" + threadClass + "]";
		}
		name += " (" + getEventName(event, false, true) + ")";
		return name;
	}

	static public String composeMethodName(MethodDetails method) {
		String name = "";
		if (method.name != null)
			name += method.name;
		if (method.signature != null) {
			// Bug 283637 - The method signature is wrapped in parentheses when it comes from the agent
			String unwrapped = method.signature.substring(1, method.signature.length()-1);
			int pos = unwrapped.indexOf('(');
			if (pos >= 0)
				name += unwrapped.substring(pos);
		}
		if (method.line >= 0)
			name += " line: " + method.line;
		return name;
	}

	static public String composeMethodName(AllocationSite allocSite) {
		String name = "";
		if (allocSite._method != null)
			name += getFullMethodName(allocSite._method, true);
		if (allocSite._lineNo >= 0)
			name += " line:" + allocSite._lineNo;
		return name;
	}

	static public Image getItemImage(Object obj) {
		if (obj instanceof MethodDetails) {
			if (((MethodDetails)obj).running)
				return getImage(IMG_METHOD_RUNNING);
			else
				return getImage(IMG_METHOD);
		}
		if (obj instanceof TRCThread) {
			EList list = ((TRCThread)obj).getThreadEvents();
			if (null != list && list.size() > 0)
				obj = list.get(list.size() - 1);
			else if (((TRCThread)obj).getStopTime() > 0)
				return getImage(IMG_THREAD_STOPPED);
			else
				return getImage(IMG_THREAD_RUNNING);
		}
		if (obj instanceof TRCThreadEvent) {
			if (obj instanceof TRCThreadDeadEvent)
				return getImage(IMG_THREAD_STOPPED);
			else if (obj instanceof TRCThreadRunningEvent)
				return getImage(IMG_THREAD_RUNNING);
			else
				return getImage(IMG_THREAD_SUSPENDED);
		}
		return null;
	}

	static public void setActionProperties(Action action, String context,
			String attributeId) {
		String strLevel = null;
		Image icon = null;
		IContextLabelFormatProvider clfp = ContextManager.getContextLabelFormatProvider(context
				, attributeId, IContextLabelFormatProvider.MODE_VIEW_ACTION);
		if (clfp != null) {
			strLevel = clfp.getDisplayStringFromElement(attributeId, null, IContextLabelFormatProvider.MODE_VIEW_ACTION);
			icon = clfp.getDisplayImageByElement(attributeId, null, IContextLabelFormatProvider.MODE_VIEW_ACTION);
		} else {
			strLevel = attributeId;
		}
		action.setText(strLevel);
		action.setImageDescriptor(ImageDescriptor.createFromImage(icon));
		action.setDescription(strLevel);
		action.setToolTipText(strLevel);
	}

	static public Object[] getCallStack(TRCThreadEvent event) {
		Object[] ret = _empty;
		EList list = event.getAnnotations();
		TRCAnnotation callStackDumpMethods = null;
		TRCAnnotation callStackDumpLineNumbers = null;
		if (null != list) {
			for (int i = 0; i < list.size(); i++) {
				TRCAnnotation annotation = (TRCAnnotation) list.get(i);
				if (annotation.getName().equals("callStackDumpMethods"))
					callStackDumpMethods = annotation;
				else if (annotation.getName().equals("callStackDumpLineNumbers"))
					callStackDumpLineNumbers = annotation;
			}
		}
		if (null != callStackDumpMethods) {
			list = callStackDumpMethods.getValues();
			ret = new Object[list.size()];
			for (int i = 0; i < list.size(); i++) {
				String name = list.get(i).toString();
				String sig = null;
				int pos = name.indexOf(' ');
				if (pos >= 0) {
					sig = name.substring(pos + 1);
					name = name.substring(0, pos);
				}
				MethodDetails method = new MethodDetails(name, sig);
				method.running = event instanceof TRCThreadRunningEvent;
				ret[i] = method;
			}
			if (null != callStackDumpLineNumbers) {
				list = callStackDumpLineNumbers.getValues();
				for (int i = 0; i < ret.length && i < list.size(); i++) {
					((MethodDetails)ret[i]).line = Integer.parseInt(list.get(i).toString());
				}
			}
		}
		return ret;
	}

	static public boolean canOpenSourceOnThisObject(EObject mofObj) {
		ITraceSelection _model = UIPlugin.getDefault().getSelectionModel(mofObj);
		if (_model.size() == 0 || _model.getFirstElement() == null) {
			return false;
		}
		Object firstElement = _model.getFirstElement();
		if (firstElement instanceof TRCClass &&
			((TRCClass)firstElement).getSourceInfo() != null &&
			((TRCClass)firstElement).getSourceInfo().getLocation().length() > 0) {
			return true;
		} else if (firstElement instanceof TRCMethod) {
			return true;
		}
		return false;
	}

	/**
	 * Formats time in format: MM:SS:NNN
	 * @param v
	 * @return
	 */
	static public String formatTime(double v) {
		StringBuffer str = new StringBuffer();
		boolean neg = v < 0;
		if (neg) {
			v = -v;
			str.append('-');
		}
		long sec = (long) v;
		if (sec / 60 < 10)
			str.append('0');
		str.append(sec / 60);
		str.append(':');
		sec %= 60;
		if (sec < 10)
			str.append('0');
		str.append(sec);
		str.append(':');
		long ms = (long) (v * 1000);
		ms %= 1000;
		if (ms < 10)
			str.append("00");
		else if (ms < 100)
			str.append('0');
		str.append(ms);
		return str.toString();
	}

	static public int loadIntOption(String opt, int def, int min, int max) {
		int val = TraceUIPlugin.getDefault().getPreferenceStore().getInt(opt);
		if (0 == val)
			val = def;
		if (val < min)
			val = min;
		if (val > max)
			val = max;
		return val;
	}

	static public int loadIntOption(String opt) {
		int val = TraceUIPlugin.getDefault().getPreferenceStore().getInt(opt);
		return val;
	}

	static public void saveIntOption(String opt, int val) {
		TraceUIPlugin.getDefault().getPreferenceStore().setValue(opt, val);
	}

	static TRCThreadEvent getFirstEvent(TRCThread thread) {
		if (null == thread)
			return null;
		EList list = thread.getThreadEvents();
		TRCThreadEvent event = null;
		if (!list.isEmpty())
			event = (TRCThreadEvent) list.get(0);
		return event;
	}

	static TRCThreadEvent findEvent(TRCThread thread, double time, int n) {
		if (null == thread)
			return null;
		EList list = thread.getThreadEvents();
		Iterator it = list.iterator();
		TRCThreadEvent event = null;
		TRCThreadEvent prevEvent = null;
		TRCThreadEvent nextEvent = null;
		if (it.hasNext()) {
			event = (TRCThreadEvent) it.next();
			double currTime = event.getTime();
			if (time < currTime) {
				if (1 != n)
					event = null;
				return event;
			}
			while (it.hasNext()) {
				nextEvent = (TRCThreadEvent) it.next();
				double nextTime = nextEvent.getTime();
				if (currTime <= time && time < nextTime) {
					if (1 == n)
						event = nextEvent;
					else if (-1 == n)
						event = prevEvent;
					return event;
				}
				prevEvent = event;
				event = nextEvent;
			}
		}
		if (1 == n)
			event = null;
		else if (-1 == n)
			event = prevEvent;
		return event;
	}

	static public TRCPackage getPackage(Object element) {
		if (element instanceof TRCPackage)
			return (TRCPackage) element;
		if (element instanceof TRCClass)
			return ((TRCClass) element).getPackage();
		return null;
	}

	static public TRCObjectAllocationAnnotation getAllocationAnnotation(TRCClass cls) {
		TRCObjectAllocationAnnotation aa = null;
		EList list = cls.getAnnotations();
		int len = list.size();
		for (int i = 0; i < len; i++) {
			TRCAnnotation annotation = (TRCAnnotation) list.get(i);
			if (annotation instanceof TRCObjectAllocationAnnotation)
				aa = (TRCObjectAllocationAnnotation) annotation;
		}
		return aa;
	}

	/**
	 * "Fixes" the given method signature as returned by the agent and returns it in a format
	 * that can be used with the Java search API.
	 * 
	 * @param sig - The method signature, as formatted from the agent
	 * @return The modified method signature suitable for use with the JDT SearchEngine API
	 */
	static public String fixMethodSignature(String sig) {
		sig = sig.substring(1,sig.length()-1); // Bug 283637 - strip off the wrapped parentheses
		int pos = sig.indexOf('(');
		if (pos >= 0) {
			String ret = sig.substring(0, pos);
			sig = sig.substring(pos);
			sig = sig + " " + ret;
		}
		return sig;
	}

	static public String restoreMethodSignature(String sig) {
		String ret = "";
		int pos = sig.indexOf('(');
		if (pos >= 0) {
			ret = sig.substring(0, pos);
			sig = sig.substring(pos + 1);
		}
		pos = sig.indexOf(')');
		if (pos >= 0) {
			sig = sig.substring(0, pos);
		}
		String args[] = sig.split(",");
		sig = "(";
		for (int i = 0; i < args.length; i++) {
			String arg = args[i].trim();
			if (arg.length() == 0 && args.length == 1)
				break;
			sig += getTypeSignature(arg);
		}
		sig += ")" + getTypeSignature(ret);
		return sig;
	}

	static public String getTypeSignature(String type) {
		int dim = 0;
		for (int j = 0; j < type.length(); j++) {
			if (type.charAt(j) == '[')
				dim++;
		}
		int pos = type.indexOf('[');
		if (pos >= 0)
			type = type.substring(0, pos);
		String sig = "";
		for (int j = 0; j < dim; j++)
			sig += "[";
		if (type.equals("boolean"))
			sig+= "Z";
		else if (type.equals("byte"))
			sig+= "B";
		else if (type.equals("char"))
			sig+= "C";
		else if (type.equals("short"))
			sig+= "S";
		else if (type.equals("int"))
			sig+= "I";
		else if (type.equals("long"))
			sig+= "J";
		else if (type.equals("float"))
			sig+= "F";
		else if (type.equals("double"))
			sig+= "D";
		else if (type.equals("void"))
			sig+= "V";
		else
			sig+= "L" + type.replace('.', '/') + ";";
		return sig;
	}

	static public boolean openSource(Object element) {
		if (element instanceof String) {
			System.out.println("openSource: " + element);
			final String pattern = (String) element;
			final int javaType = IJavaSearchConstants.METHOD;
			BusyIndicator.showWhile(Display.getDefault(), new Runnable() {
				public void run() {
					if (!OpenJavaSource.openSource(pattern, javaType,
							SearchEngine.createWorkspaceScope(), true)) {
						MessageDialog.openInformation(UIPlugin.getDefault()
								.getWorkbench().getActiveWorkbenchWindow()
								.getShell(), TraceMessages.TRC_MSGT, NLS.bind(
								TraceUIMessages._68, pattern));
					}
				}
			});
		}
		OpenSource.openSource(element);
		return true;
	}

	static public int getObjAge(TRCFullTraceObject obj, EList listGC) {
		int age = 0;
		double t0 = obj.getCreateTime();
		double t1 = obj.getCollectTime();
		int len = listGC.size();
		for (int j = 0; j < len; j++) {
			TRCGCEvent gcEvent = (TRCGCEvent) listGC.get(j);
			if (gcEvent.getType().equals("finish")) {
				double time = gcEvent.getTime();
				if (time <= t0)
					continue;
				if (t1 > 0 && time >= t1)
					break;
				age++;
			}
		}
		return age;
	}

	static public int compare(double d1, double d2) {
		if (d1 > d2)
			return 1;
		if (d1 < d2)
			return 1;
		return 0;
	}

	static public int compare(String s1, String s2) {
		if (s1 != null && s2 != null)
			return s1.compareToIgnoreCase(s2);
		if (s1 != null)
			return 1;
		if (s2 != null)
			return -1;
		return 0;
	}

	static public String getAgentName(EObject mofObject) {
		String name = "";
		EList list = null;
		if (mofObject instanceof TRCAgentProxy) {
			name = ((TRCAgentProxy) mofObject).getName();
		} else if (mofObject instanceof TRCProcessProxy) {
			list = ((TRCProcessProxy) mofObject).getAgentProxies();
		} else if (mofObject instanceof TRCNode) {
			list = ((TRCNode) mofObject).getProcessProxies();
		}
		if (list != null && list.size() > 0)
			name = getAgentName((EObject) list.get(0));
		return name;
	}

	static public String formatPercent(int val, int max) {
		String s = max > 0 && max >= val ?
			TString.formatAsPercentage((double)val / (double)max) : "";
		return s;
	}
}
