/***********************************************************************
 Copyright (c) 2007, 2010 IBM Corporation and others.
 All rights reserved. This program and the accompanying materials
 are made available under the terms of the Eclipse Public License v1.0
 which accompanies this distribution, and is available at
 http://www.eclipse.org/legal/epl-v10.html
 $Id: ConfigGenerator.java,
 
 Contributors:
     IBM - initial implementation
 **********************************************************************/

package org.eclipse.tptp.platform.iac.administrator.internal.common;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;

import org.eclipse.core.resources.IResourceStatus;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.FileLocator;
import org.eclipse.core.runtime.IBundleGroup;
import org.eclipse.core.runtime.IBundleGroupProvider;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Status;
import org.eclipse.jdt.launching.IVMInstall;
import org.eclipse.jdt.launching.JavaRuntime;
import org.eclipse.jdt.launching.VMStandin;
import org.eclipse.jdt.launching.environments.ExecutionEnvironmentDescription;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.tptp.platform.agentcontroller.config.ConfigFile;
import org.eclipse.tptp.platform.agentcontroller.config.Constants;
import org.eclipse.tptp.platform.iac.administrator.AdminPlugin;
import org.eclipse.tptp.platform.iac.administrator.internal.config.ConfigGenerator;
import org.eclipse.tptp.platform.iac.administrator.internal.preference.PreferenceMessages;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.PlatformUI;
import org.osgi.framework.Bundle;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;

/**
 * Contains common util functions used in this plugin
 * 
 * @author Navid Mehregani
 *
 */
public class AdminUtil {
	
	static final String JVM_EE_TYPE = "EEVMType";
	
	/**
	 * True if stringToCheck is not NULL or empty; false otherwise
	 */
	public static boolean validString(String stringToCheck)
	{
		return ((stringToCheck != null) && (stringToCheck.trim().length() > 0));
	}
	
	/**
	 * Used to parse the given XML file.  This is only used in the JUnits to parse and verify the config file
	 * @param fileName The absolute path of the XML file to parse
	 * @param validation DTD validation. set to true if DTD validation is required; false otherwise
	 * @return The parsed Document object
	 */
	public static Document parseFile(String fileName, boolean validation)
	{
		Document document = null;
		
		DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
        factory.setValidating(validation);   
        factory.setNamespaceAware(validation);

        /* parse files */
        try {
        	DocumentBuilder builder = factory.newDocumentBuilder();
        	document = builder.parse(new File(fileName));           
        }
        catch (SAXParseException e) {
        	Status status = new Status(Status.ERROR, AdminPlugin.PLUGIN_ID, 0, e.getMessage(), e);
        	AdminPlugin.getDefault().getLog().log(status);
        }
        catch (Exception e) {
        	Status status = new Status(Status.ERROR, AdminPlugin.PLUGIN_ID, 0, e.getMessage(), e);
        	AdminPlugin.getDefault().getLog().log(status);        
            if (e instanceof SAXException) {
                Exception nested = ((SAXException)e).getException();
                if (nested != null) {
                    e = nested;
                }
            }
            else if (e instanceof FileNotFoundException)
            {
            	Status status1 = new Status(Status.ERROR, AdminPlugin.PLUGIN_ID, IACPluginMessages.FILE_NOT_FOUND);
            	AdminPlugin.getDefault().getLog().log(status1);
            }
        }         
        
        return document;

	}
	
	/**
	 * Provided for backward compatibility
	 * 
	 * @param fileName
	 * @return
	 */
	public static Document parseFile(String fileName)
	{
		return parseFile(fileName, true);
	}
	

	/**
	 * Returns the value of the attribute specified in 'attributeToGet' that meets the
	 * following criterias:
	 * It's specified in the element called 'elementName' that contains the attribute
	 * 'attributeToCheck' with value 'attributeValueToCheck' 
	 * In other words, it returns what's specified with the question mark below:
	 * &lt;elementName attributeToCheck=attributeValueToCheck attributeToGet=?&gt;
	 * 
	 * @return  The value of the question mark above;  empty string if it can't be found
	 */
	public static String getAttributeWithMoreDetails(Document document, String elementName, String attributeToCheck, String attributeValueToCheck, String attributeToGet)
	{
		NodeList elementList = document.getElementsByTagName(elementName);
		
		for (int i=0; i < elementList.getLength(); i++)
		{
			Element element = (Element)elementList.item(i);
			
			if (element.getAttribute(attributeToCheck).equals(attributeValueToCheck))
				return element.getAttribute(attributeToGet);
		}
		 
		return "";
	}
	
	/**
	 * Used to get the value of an attribute of an element in the config file.
	 * If the attribute does not exist, an empty string is returned.  
	 * 
	 * @param document  Document to look in 
	 * @param elementName  The name of the element the attribute belongs to
	 * @param attributeName  The attribute we're interested in  
	 * @param index  Indicates which element we should look into, incase there are multiple elements with the name specified in elementName
	 * 
	 * @return  The value of the requested attribute, empty string if it doesn't exist
	 */
	public static String getAttributeOfElement(Document document, String elementName, String attributeName, int index)
	{		
		String attributeValue = "";
		NodeList elementList = document.getElementsByTagName(elementName);
		
		if (index < elementList.getLength())
		{
			Element element = (Element)elementList.item(index);
			attributeValue = element.getAttribute(attributeName);
		}
		
		return attributeValue;
	}
	
	/**
	 * Return value of the  first child of the element with the given specifications
	 * 
	 * @param document  Document to look in 
	 * @param elementName  Name of the element
	 * @param index  The index of the element. In case there are multiple elements with the given name in the document
	 * @return  Element with the given specifications
	 */
	public static String getElementValue(Document document, String elementName, int index)
	{
		Element element = (Element) getElement(document, elementName, index);
		return element.getFirstChild().getNodeValue();
	}
	
	/**
	 * Returns the element with the given specifications
	 * 
	 * @param document  Document to look in 
	 * @param elementName  Name of the element 
	 * @param index  The index of the element. In case there are multiple elements with the given name in the document
	 * @return  Element with the given specifications
	 */
	public static Element getElement(Document document, String elementName, int index)
	{
		NodeList elementList = document.getElementsByTagName(elementName);
		Element element = null;
		
		if (index < elementList.getLength())
		{
			element = (Element)elementList.item(index);
		}
		
		return element;			
	}
	
	/**
	 * Creates a grid data object that occupies vertical and horizontal
	 * space.
	 */
	public static GridData createFill() 
	{
		return new GridData(SWT.FILL, SWT.FILL, true, true);
	}
	/**
	 * Creates a grid data object that occupies horizontal space.
	 */
	public static GridData createHorizontalFill() 
	{		
		return new GridData(SWT.FILL, SWT.DEFAULT, true, false);		
	}
	/**
	 * Creates a grid data object that occupies vertical space.
	 */
	public static GridData createVerticalFill() 
	{
		return new GridData(SWT.DEFAULT, SWT.FILL, false, true);
	}
	
	/**
	 * @return an array of Files, one for each non-jarred plugin that contains
	 * a config.jar file in its root
	 * @throws IOException
	 */
	public static File[] getIACPlugins() {
		List configJars = new ArrayList();
		IBundleGroupProvider providers[] = Platform.getBundleGroupProviders();
		if (providers.length == 0){
			Bundle b = Platform.getBundle("org.eclipse.update.configurator");
			try{
			if (b != null)
				  b.start(Bundle.START_TRANSIENT);
			}
			catch (Exception e){
				// if this fails, we revert to IAC won't start behavior
			}
			providers = Platform.getBundleGroupProviders();
		}
		for (int i = 0; i < providers.length; i++) {
			IBundleGroupProvider provider = providers[i];
			IBundleGroup groups[] = provider.getBundleGroups();
			for (int j = 0; j < groups.length; j++) {
				IBundleGroup group = groups[j];
				Bundle bundles[] = group.getBundles();
				for (int k = 0; k < bundles.length; k++) {
					Bundle bundle = bundles[k];	
					File configJar = getConfigJar(bundle);
					if (configJar != null)
						configJars.add(configJar);
				}
			}
		}
		return (File[]) configJars.toArray(new File[0]);
	}

	private static File getConfigJar(Bundle bundle) {
		File configJar = null;
		try
		{
			URL rootEntry = bundle.getEntry("/");
			rootEntry = FileLocator.resolve(rootEntry);
			if ("file".equals(rootEntry.getProtocol())) {
				configJar = new File(rootEntry.getPath(), "config.jar");
				if (!configJar.exists())
					configJar = null;
				else
					configJar = configJar.getParentFile();
			}
		} catch (IOException e)
		{
			Status status = new Status(Status.ERROR, AdminPlugin.PLUGIN_ID, 0, e.getMessage(), e);
			AdminPlugin.getDefault().getLog().log(status);
		}
		return configJar;
	}
	
	/**
	 * Used to generate IAC's configuration file based on the options selected in the preference page
	 */
	public static void generateConfigFile(boolean invokedFromWorkbench)
	{
		//bug 234486
		if(!isIACConfigFileWritable(invokedFromWorkbench)){
			Status status = new Status(Status.ERROR, AdminPlugin.PLUGIN_ID, IACPluginMessages.CONFIG_FILE_ACCESS_DENIED);
			AdminPlugin.getDefault().getLog().log(status);	
			return;
		}
		//end of sbug 234486
		ConfigGenerator configGenerator = new ConfigGenerator();
		String iacHome = getIACHome();
		if (iacHome == null)
		{
			Status status = new Status(Status.ERROR, AdminPlugin.PLUGIN_ID, IACPluginMessages.AC_HOME_NULL);
			AdminPlugin.getDefault().getLog().log(status);
			return;
		}
		AdminPlugin adminPlugin = AdminPlugin.getDefault();
		
		String javaExecutable = adminPlugin.getString(PreferenceMessages.JAVA_EXECUTABLE);
		String allowedHosts = adminPlugin.getString(PreferenceMessages.ALLOWED_HOSTS);
		String hostList = adminPlugin.getString(PreferenceMessages.HOST_LIST_LABEL);
		boolean security = 	adminPlugin.getBoolean(PreferenceMessages.AC_SECURITY);
		String usertype = adminPlugin.getString(PreferenceMessages.ALLOWED_USERS);
		String userList = adminPlugin.getString(PreferenceMessages.USER_LIST_LABEL);
		
		configGenerator.generateConfiguration(iacHome, javaExecutable, getPluginsFolder(), getIACPlugins(), security, usertype, userList, allowedHosts, hostList);
	}

	/**
	 * @return The absolute folder of IAC.  This method IS platform dependent.  Returns null if the absolute
	 * folder of IAC cannot be resolved.  Callers should always check for the null value and handle it gracefully.
	 */
	public static String getIACHome()
	{
		String iacPluginId = CommonConstants.IAC_PLUGIN_ID + ".";
		String iacHome = null;
		
		String platformSymbolicName = getPlatformSymbolicName();
		
		if (platformSymbolicName == null)
		{
			Status status = new Status(Status.ERROR, AdminPlugin.PLUGIN_ID, IACPluginMessages.CANT_SOLVE_PLATFORM + CommonConstants.OPERATING_SYSTEM +":"+ CommonConstants.SYSTEM_ARCH);
			AdminPlugin.getDefault().getLog().log(status);
			return null;
		}
		
		iacPluginId = iacPluginId + platformSymbolicName;
		
		try {
			Bundle iacBundle = Platform.getBundle(iacPluginId);
			
			/* Was the bundle resolved? */
			if (iacBundle == null)
			{
				Status status = new Status(Status.ERROR, AdminPlugin.PLUGIN_ID, IACPluginMessages.CANT_SOLVE_BUNDLE + iacPluginId);
				AdminPlugin.getDefault().getLog().log(status);
				return null;
			}
			
			iacHome = FileLocator.resolve(iacBundle.getEntry("/")).getPath();
			
			/* Windows */
			if ((CommonConstants.OPERATING_SYSTEM.indexOf("windows") != -1) && (iacHome.startsWith("/")))
			{
				iacHome = iacHome.substring(1);
			}
			
			iacHome = iacHome + "agent_controller";
			
			if (CommonConstants.FILE_SEPARATOR.equals("\\"))
				iacHome = iacHome.replaceAll("/", "\\\\");
			
			/* Make sure the director does in fact exist */
			if (!(new File(iacHome)).exists())
				return null;
			
		} catch (IOException e) {
			Status status = new Status(Status.ERROR, AdminPlugin.PLUGIN_ID, 0, e.getMessage(), e);
			AdminPlugin.getDefault().getLog().log(status);
			return null;
		}
		
		return iacHome;
	}
		
	/**
	 * @return The absolute location of the plugins folder that should be used by IAC
	 * This should be the same "plugins" folder as the workbench
	 */
	public static String getPluginsFolder()
	{
		/* By default, we'll take the location that the IAC plugin is in */
		String iacHome = getIACHome();
		String iacPluginsFolder = null;
		
		if (iacHome != null)
		{
			iacPluginsFolder = iacHome.substring(0, iacHome.indexOf(CommonConstants.IAC_PLUGIN_ID) - 1);
		}
		else
		{
			/* If IAC home can't be resolved, return the 'plugins' folder of the workbench */
			String workbenchPluginsFolder = Platform.getInstallLocation().getURL().getPath() + "plugins";
			if ((workbenchPluginsFolder.charAt(0) == '/') || (workbenchPluginsFolder.charAt(0) == '\\'))
				workbenchPluginsFolder = workbenchPluginsFolder.substring(1);
			
			iacPluginsFolder = workbenchPluginsFolder;
		}
		
		if (CommonConstants.FILE_SEPARATOR.equals("\\"))
			iacPluginsFolder = iacPluginsFolder.replaceAll("/", "\\\\");
		
		return iacPluginsFolder;
	}
	
	/**
	 * @return true if IAC's config file needs to be generated (i.e. either IAC's config file
	 * doesn't exist or doesn't have anything in it). 
	 */
	public static boolean isConfigEmpty(boolean invokedFromWorkbench)
	{
		String iacConfigFileLocation = getIACConfigFile(invokedFromWorkbench);
		if (iacConfigFileLocation == null)
			return true;
		
		File configFile = new File(iacConfigFileLocation);
		return !checkFileValidity(configFile);		
	}
	
	/**
	 * Helper function to check validity of IAC config files
	 * @param configFile  The configuration file to check
	 * @return  True if given file exists and it's NOT empty; false otherwise
	 */
	private static boolean checkFileValidity(File configFile)
	{
		return (configFile.exists() && (configFile.length() > 0));
	}
	
	/**
	 * @return The absolute location of the JVM executable for the workbench
	 */
	public static String getJVMExecutable()
	{
		String jvmExecutable = null;
		
		String jvmLocation = JavaRuntime.getDefaultVMInstall().getInstallLocation().getPath();
	
		//240389, 259570
		if(isJVMEEType(JavaRuntime.getDefaultVMInstall())){
			VMStandin vms = new VMStandin(JavaRuntime.getDefaultVMInstall());
			if(vms != null) {
				// Bug 308169: Use EXECUTABLE not EXECUTABLE_CONSOLE because we want javaw not java
				jvmExecutable = vms.getAttribute(ExecutionEnvironmentDescription.EXECUTABLE);
			}
		}
		//end of 240389, 259570
		
		
		if (jvmExecutable == null)
			return getJVMExecutable(jvmLocation);
		else
			return jvmExecutable;
	}
	
	private static boolean isJVMEEType(IVMInstall vm){
		String vmType = vm.getVMInstallType().toString();
		if(vmType.indexOf(JVM_EE_TYPE) != -1)
			return true;
		else
			return false;
	}
	
	/**
	 * @return The absolute location of the JVM executable for the given JVM
	 */
	public static String getJVMExecutable(String jvmLocation)
	{
		String jvmExecutable = jvmLocation + CommonConstants.FILE_SEPARATOR + "jre" + CommonConstants.FILE_SEPARATOR + "bin" + CommonConstants.FILE_SEPARATOR + CommonConstants.JAVA_EXECUTABLE_NAME + CommonConstants.EXECUTABLE_EXTENSION;
		
		/* Make sure the java executable exists */
		if (!(new File(jvmExecutable).exists()))
		{
			jvmExecutable = jvmLocation + CommonConstants.FILE_SEPARATOR + "bin" + CommonConstants.FILE_SEPARATOR + CommonConstants.JAVA_EXECUTABLE_NAME + CommonConstants.EXECUTABLE_EXTENSION;
			
			/* Check for the java executable now */
			if (!(new File(jvmExecutable).exists()))
			{
				/* This should never happen. */
				Status status = new Status(Status.ERROR, AdminPlugin.PLUGIN_ID, IACPluginMessages.CANT_FIND_JAVA + jvmLocation);
				AdminPlugin.getDefault().getLog().log(status);
			}
		}		
		
		return jvmExecutable;
	}
	
	/**
	 * @return Absolute location of IAC's executable (e.g. RAServer.exe).  Null if IAC executable can't be resolved
	 */
	public static String getIACExecutable() {
		String[] iacExecutableArray = getIACExecutableArray();
		String iacExecutable = null;
		
		if(iacExecutableArray == null) {
			return null;
		}
		
		for(int i = 0; i < iacExecutableArray.length; i++) {
			iacExecutable = iacExecutable + " " + iacExecutableArray[i];
		}
		return iacExecutable;
	}
	
	/**
	 * @return Absolute location of IAC's executable (e.g. RAServer.exe).  Null if IAC executable can't be resolved
	 */
	public static String[] getIACExecutableArray()
	{
		ArrayList iacExecutableList = new ArrayList();
		iacExecutableList.add(getIACBinFolder());
		String iacExecutable = (String)iacExecutableList.get(0);
		if (iacExecutable == null)
			return null;
		
		if (CommonConstants.OPERATING_SYSTEM.indexOf("linux") != -1)
			iacExecutableList.set(0, iacExecutable + CommonConstants.FILE_SEPARATOR + "ACStart.sh");
		else if (CommonConstants.OPERATING_SYSTEM.indexOf("windows") != -1)
			iacExecutableList.set(0, iacExecutable + CommonConstants.FILE_SEPARATOR + "ACServer");
		else
		{
			Status status = new Status(Status.ERROR, AdminPlugin.PLUGIN_ID, IACPluginMessages.CANT_SOLVE_PLATFORM + CommonConstants.OPERATING_SYSTEM);
			AdminPlugin.getDefault().getLog().log(status);
			return null;
		}
		
		//242291 
		if(iacExecutable != null){
			String configDir = (new IACConfigUtil()).getACConfigDirectory(getIACHome(), true, true).getAbsolutePath();
			iacExecutableList.add("-d");
			iacExecutableList.add(configDir);
		} 
		
		return (String[])iacExecutableList.toArray(new String[iacExecutableList.size()]);
	}
	
	/**
	 * @return  The shutdown command that should be used to terminate IAC only
	 */
	public static String getShutdownCommand()
	{
		String[] commandArray = getShutdownCommandArray();
		String command = null;
		
		if(commandArray == null) {
			return null;
		}
		
		for(int i = 0; i < commandArray.length; i++) {
			command = command + " " + commandArray[i];
		}
		
		return command;
	}
	
	/**
	 * @return  The shutdown command that should be used to terminate IAC only
	 */
	public static String[] getShutdownCommandArray()
	{
		ArrayList commandList = new ArrayList();
		commandList.add(getIACBinFolder());
		String command = (String)commandList.get(0);
		if (command == null)
			return null;
		
		if (CommonConstants.OPERATING_SYSTEM.indexOf("linux") != -1)
			commandList.set(0, command + CommonConstants.FILE_SEPARATOR + "ACStop.sh");
		else if (CommonConstants.OPERATING_SYSTEM.indexOf("windows") != -1) {
			commandList.set(0, command + CommonConstants.FILE_SEPARATOR + "ACServer");
			commandList.add("-shutdown");
		}
		else
		{
			Status status = new Status(Status.ERROR, AdminPlugin.PLUGIN_ID, IACPluginMessages.CANT_SOLVE_PLATFORM + CommonConstants.OPERATING_SYSTEM);
			AdminPlugin.getDefault().getLog().log(status);
			command = null;
		}
		//243214
		if(command != null){
			String configDir = (new IACConfigUtil()).getACConfigDirectory(getIACHome(), true, true).getAbsolutePath();
			commandList.add("-d");
			commandList.add(configDir);
		}
		return (String[])commandList.toArray(new String[commandList.size()]);
		
	}
	
	/**
	 * @return Absolute location of IAC's config file
	 */
	public static String getIACConfigFile(boolean invokedFromWorkbench)
	{
		String iacHome = getIACHome();
		if (iacHome == null)
			return null;
		
		String iacConfigFile;
		
		if(!invokedFromWorkbench)
			iacConfigFile = iacHome + CommonConstants.FILE_SEPARATOR + "config" + CommonConstants.FILE_SEPARATOR + "serviceconfig.xml";
		else
			iacConfigFile = (new IACConfigUtil()).getACConfigDirectory(iacHome, true, true) + CommonConstants.FILE_SEPARATOR + "serviceconfig.xml";
		return iacConfigFile;
	}
	
	//bug 234486
	/**
	 * @return true if IAC config file is readable and writable; if not exisit then try to create one.
	 */
	public static boolean isIACConfigFileWritable(boolean invokedFromWorkbench)
	{
		String configFileName = getIACConfigFile(invokedFromWorkbench);
		if (configFileName == null)
			return false;
		
		File configFile = new File(configFileName);
		
		if(configFile.exists()){
			return (configFile.canWrite() && configFile.canRead());
		}else{	
			try{
				return configFile.createNewFile();		
			}catch(IOException ioe){
				//Access is denied
				return false;
			}
		}
		
	}
	
	/**
	 * @return  Absolute location of IAC's bin folder.  Returns null if IAC's bin folder can't be
	 * found.  Callers should always check for the null value and handle it gracefully.
	 */
	public static String getIACBinFolder()
	{
		String iacHome = getIACHome();
		if (iacHome == null)
			return iacHome;
		
		
		return iacHome + CommonConstants.FILE_SEPARATOR + "bin";
	}
	
	/**
	 * Returns a confirmation dialog with the given parameters
	 */
	public static MessageDialog getConfirmationDialog(Shell parentShell, String title, String message)
	{
		return new MessageDialog(parentShell, title, null, message, MessageDialog.QUESTION, new String[] {IDialogConstants.YES_LABEL, IDialogConstants.NO_LABEL}, 0);
	}
	
	
	
	/**
	 * @return  The absolute path of the Probekit native files.  Returns null if location of native files can't be resolved.
	 * Callers should always check for null. 
	 */
	public static String getProbekitNativePath()
	{
		String probekitNativePath = null;
//		String platformSymbolicName = getPlatformSymbolicName();
		Bundle probekitBundle = Platform.getBundle(CommonConstants.PROBEKIT_PLUGIN_ID);
		
		if ((probekitBundle == null)) //|| (platformSymbolicName == null))
			return probekitNativePath;
		
		try
		{			
			probekitNativePath = FileLocator.resolve(probekitBundle.getEntry("/")).getPath();
				
		} catch (IOException e) {
			Status status = new Status(Status.ERROR, AdminPlugin.PLUGIN_ID, 0, e.getMessage(), e);
			AdminPlugin.getDefault().getLog().log(status);
			return null;
		}
		
		/* Windows */
		if ((CommonConstants.OPERATING_SYSTEM.indexOf("windows") != -1) && (probekitNativePath.startsWith("/")))
		{
			probekitNativePath = probekitNativePath+File.separatorChar+"os"+File.separatorChar+"win32"+File.separatorChar+"x86";
		}
		/* Linux */
		else if ((CommonConstants.OPERATING_SYSTEM.indexOf("linux") != -1) && (probekitNativePath.startsWith("/")))
		{
			probekitNativePath = probekitNativePath+File.separatorChar+"os"+File.separatorChar+"linux"+File.separatorChar+"x86";
		}
		
		if (CommonConstants.FILE_SEPARATOR.equals("\\"))
			probekitNativePath = probekitNativePath.replaceAll("/", "\\\\");
		
		return probekitNativePath;
	}
	
	/**
	 * @return  The absolute path of the JVMTI native files.  Returns null if location of native files can't be resolved.
	 * Callers should always check for null. 
	 */
	public static String getJvmtiNativePath()
	{
		String jvmtiNativePath = null;
		String platformSymbolicName = getPlatformSymbolicName();
		Bundle jvmtiBundle = Platform.getBundle(CommonConstants.JVMTI_PLUGIN_ID);
		
		if ((jvmtiBundle == null) || (platformSymbolicName == null))
			return jvmtiNativePath;
		
		try
		{			
			jvmtiNativePath = FileLocator.resolve(jvmtiBundle.getEntry("/")).getPath() + "agent_files" + CommonConstants.FILE_SEPARATOR + platformSymbolicName;
				
		} catch (IOException e) {
			Status status = new Status(Status.ERROR, AdminPlugin.PLUGIN_ID, 0, e.getMessage(), e);
			AdminPlugin.getDefault().getLog().log(status);
			return null;
		}
		
		/* Windows */
		if ((CommonConstants.OPERATING_SYSTEM.indexOf("windows") != -1) && (jvmtiNativePath.startsWith("/")))
		{
			jvmtiNativePath = jvmtiNativePath.substring(1);
		}
		
		if (CommonConstants.FILE_SEPARATOR.equals("\\"))
			jvmtiNativePath = jvmtiNativePath.replaceAll("/", "\\\\");
		
		return jvmtiNativePath;
	}
	
	/**
	 * Helper function that returns the corresponding platform symbolic name.  Null if symbolic name
	 * can't be resolved.  Callers should always check for null.
	 */
	private static String getPlatformSymbolicName()
	{
		String platformSymbolicName = null;
		
		/* Windows */
		if (CommonConstants.OPERATING_SYSTEM.indexOf("windows") != -1)
		{	
			/* Now check the system architecture */		
						
			/* IA32 */
			if (CommonConstants.SYSTEM_ARCH.indexOf("x86") != -1)
				platformSymbolicName = CommonConstants.WIN_IA32;
			/* EM64T */
			else if (CommonConstants.SYSTEM_ARCH.indexOf("x86_64") != -1)
				platformSymbolicName = CommonConstants.WIN_EM64T;
			/* AMD64 */
			else if (CommonConstants.SYSTEM_ARCH.indexOf("amd64") != -1)
				platformSymbolicName = CommonConstants.WIN_EM64T;
			/* IPF */
			else if (CommonConstants.SYSTEM_ARCH.indexOf("ia64") != -1)
				platformSymbolicName = CommonConstants.WIN_IPF;
			
		}
		/* Linux */
		else if (CommonConstants.OPERATING_SYSTEM.indexOf("linux") != -1)
		{
			/* Now check the system architecture */		
			
			/* IA32 */
			if ((CommonConstants.SYSTEM_ARCH.indexOf("x86") != -1) || (CommonConstants.SYSTEM_ARCH.indexOf("i386") != -1))
				platformSymbolicName = CommonConstants.LINUX_IA32;
			/* EM64T */
			else if (CommonConstants.SYSTEM_ARCH.indexOf("x86_64") != -1)
				platformSymbolicName = CommonConstants.LINUX_EM64T;
			/* AMD64 */
			else if (CommonConstants.SYSTEM_ARCH.indexOf("amd64") != -1)
				platformSymbolicName = CommonConstants.LINUX_EM64T;
			/* IPF */
			else if (CommonConstants.SYSTEM_ARCH.indexOf("ia64") != -1)
				platformSymbolicName = CommonConstants.LINUX_IPF;
		}
		
		return platformSymbolicName;
	}
	
	/**
	 * This function checks to see if IAC's configuration file needs to be regenerated.
	 * 
	 * The configuration file needs to be generated when 
	 * 
	 * 1. The current config file is not valid or
	 * 2. Workbench's IAC's home is different from the one specified in the exiting configuration file.
	 * 
	 * @param invokedFromWorkbench
	 * @return
	 */
	public static boolean isNeedToGenerateConfigFile(boolean invokedFromWorkbench){
		
		if(!isConfigurationValid(invokedFromWorkbench)){
			return true;
		}
		String runtimeIACHome = getIACHome();		
		if(runtimeIACHome == null)
			return true;
		
		ConfigFile configFile = new ConfigFile(AdminUtil.getIACConfigFile(true));
		String configFileIACHome = configFile.getValue(Constants.RASERVER_HOME);
		
		return (!runtimeIACHome.equalsIgnoreCase(configFileIACHome));
	}
	
	/**
	 * This function checks to see whether the main config file and the config file for the execution framework are valid.
	 * It also checks to see whether the Workbench JVM matches the JVM specified in IAC's config file.  This is done to make
	 * sure that IAC's config file is never in an invalid state.
	 * 
	 * @return  True iff the main config file and the config file for the execution framework are valid AND 
	 *          workbench JVM matches the JVM specified in IAC's config file; false otherwise
	 */
	public static boolean isConfigurationValid(boolean invokedFromWorkbench)
	{
		/* Check main config file */
		if (isConfigEmpty(invokedFromWorkbench))
			return false;
		
		/* Check config file of execution framework */
		boolean valid = false;
				
		File[] iacPlugins = getIACPlugins();
		for (int i=0; i < iacPlugins.length; i++)
		{
			if (iacPlugins[i].toString().indexOf(CommonConstants.EXECUTION_FRAMEWORK + "_") != -1)
			{
				valid = checkFileValidity(new File(iacPlugins[i].toString() + CommonConstants.FILE_SEPARATOR + "config" + CommonConstants.FILE_SEPARATOR + "pluginconfig.xml"));
				break;
			}
		}
		
		if (!valid)
		{
			Status status = new Status(Status.ERROR, AdminPlugin.PLUGIN_ID, IACPluginMessages.EXECUTION_FRAMEWORK_CONFIG_FILE_INVALID);
        	AdminPlugin.getDefault().getLog().log(status);
        	return false;
		}
		
		/* Now see if workbench JVM matches the JVM in IAC's config file */
		Document document = AdminUtil.parseFile(getIACConfigFile(invokedFromWorkbench), false);
		if (document == null) {
			Status status = new Status(Status.WARNING, AdminPlugin.PLUGIN_ID, IACPluginMessages.EXECUTION_FRAMEWORK_CONFIG_FILE_INVALID);
        	AdminPlugin.getDefault().getLog().log(status);
        	return false;
		}
		
		String iacJVM = AdminUtil.getAttributeWithMoreDetails(document, "Variable", "name", "JAVA_PATH", "value");
		String workbenchJVM = getJVMExecutable();
		
		if (!iacJVM.equals(workbenchJVM))
		{
			AdminPlugin.getDefault().setString(PreferenceMessages.JAVA_EXECUTABLE, workbenchJVM);
			Status status = new Status(Status.WARNING, AdminPlugin.PLUGIN_ID, "Execution framework config file is invalid.");
        	AdminPlugin.getDefault().getLog().log(status);
        	return false;
			
		}
		
		return valid;
	}
	
	
	//bug 208831
	public static boolean isSupportedPlatform(){
		
		return getPlatformSymbolicName() != null;
	}
	
	
	/*
	 * Copy from ConnectUtilUI.java because can't import that class because of circular reference.
	 * 
	 **/
	private static Shell getValidShell() {
		Shell shell = null;
		IWorkbench workbench = PlatformUI.getWorkbench();
		if (workbench != null) {
			if (workbench.getActiveWorkbenchWindow() != null) {
				shell = workbench.getActiveWorkbenchWindow().getShell();
				if ((shell != null) && (!shell.isDisposed()))
					return shell;
			}

			if (workbench.getWorkbenchWindows().length > 0) {
				shell = workbench.getWorkbenchWindows()[0].getShell();
				if ((shell != null) && (!shell.isDisposed()))
					return shell;
			}
		}

		return null;
	}
	
	
	/*
	 * Copy from ConnectUtilUI.java because can't import that class because of circular reference.
	 * 
	 **/
	public static void openErrorDialog(final String title, final String message){
		Display.getDefault().syncExec(new Runnable() {
			public void run() {
				
				Status err = new Status(Status.ERROR, ResourcesPlugin.PI_RESOURCES, IResourceStatus.ERROR, message,new Exception(message));
				ErrorDialog.openError(getValidShell(), title,PreferenceMessages.ERROR_GENERAL, err);
			}
		});
		
	}
}
