/**********************************************************************
 * Copyright (c) 2007, 2009 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: IACPreferencePage.java,v 1.16 2009/08/10 20:27:52 ewchan Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.tptp.platform.iac.administrator.internal.preference;

import java.util.StringTokenizer;

import org.eclipse.core.runtime.Status;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.preference.PreferencePage;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.List;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;
import org.eclipse.tptp.platform.agentcontroller.config.ConfigFile;
import org.eclipse.tptp.platform.agentcontroller.config.Constants;
import org.eclipse.tptp.platform.iac.administrator.AdminPlugin;
import org.eclipse.tptp.platform.iac.administrator.internal.common.AdminUtil;
import org.eclipse.tptp.platform.iac.administrator.internal.common.CommonConstants;
import org.eclipse.tptp.platform.iac.administrator.internal.common.IACPluginMessages;
import org.eclipse.tptp.platform.iac.administrator.internal.startstop.AutoStartStop;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchPreferencePage;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.dialogs.PreferenceLinkArea;
import org.eclipse.ui.preferences.IWorkbenchPreferenceContainer;

/**
 * IAC preference page
 * 
 * @author Navid Mehregani
 *
 */
public class IACPreferencePage extends PreferencePage	implements IWorkbenchPreferencePage {

	/* This is the ID of the 'Installed JREs' preference page */
	private static final String JRE_PAGE_ID = "org.eclipse.jdt.debug.ui.preferences.VMPreferencePage";//$NON-NLS-1$
	
	/* These are the variables that will store the user's selection in the preference page */
	private boolean iacEnabled = true;
	private String allowedHosts = "";
	private String listOfCustomHosts = "";
	private String javaExecutable = "";
	private boolean security = false;
	private String allowedUsers = "";
	private String listOfCustomUsers = "";
	
	/* SWT widgets that make up the preference page */
	private Composite composite = null;
	private Label label = null;
	private Group allowedHostsGroup = null;
	private Composite composite5 = null;
	private List listOfHosts = null;
	private Composite composite6 = null;
	private Label listOfHostLabel = null;
	private Button addHostButton = null;
	private Button removeHostButton = null;
	private Button editHostButton = null;
	private Group acSecurityGroup = null;
	private Group listOfUsersGroup = null;
	private Label listOfUsersLabel = null;
	private Composite composite1 = null;
	private List listOfUsers = null;
	private Composite composite2 = null;
	private Button addUserButton = null;
	private Button removeUserButton = null;
	private Button editUserButton = null;
	private Composite composite3 = null;
	private Button securityOnButton = null;
	private Button securityOffButton = null;
	private Composite composite4 = null;
	private Button customHostButton = null;
	private Button localHostButton = null;
	private Button allHostButton = null;
	private Composite composite8 = null;
	private Text javaPathText = null;
	private PreferenceLinkArea modifyLink = null;
	private Composite composite7 = null;
	private Button customUserButton = null;
	private Button anyUserButton = null;
	
	/* Enable/disable composite */
	private Composite enableDisableComposite = null;
	private Label enableDisableLabel = null;
	private Button enableIACButton = null;
	private Button disableIACButton = null;
	
	/* Instance of the plugin activator class */
	private static AdminPlugin adminPlugin = AdminPlugin.getDefault();
	
	/* Stores the preference composite. Used for displaying warning when config file has to be regenerated */
	private Composite preferenceComposite = null;
	
	//bug 208831
	private Composite notSupportPlatformComposite = null;
	private Label notSupportPlatformMsg = null;
	//end bug 208831

	//bug 234486
	private Composite accessIACConfigDeniedComposite = null;
	private Label accessIACConfigDeniedMsg = null;
	//end of bug 234486
	protected Control createContents(Composite parent) {
		
		//bug 208831, bug 234486
		if(!AdminUtil.isSupportedPlatform()){
			createNotSupportPlatformErrorComposite(parent);
			AdminPlugin.getDefault().getLog().log(new Status(Status.INFO, AdminPlugin.PLUGIN_ID,IACPluginMessages.NOT_SUPPORT_PLATFORM ));		
		}else if(!AdminUtil.isIACConfigFileWritable(true)){//end bug 208831
			createCannotAccessIACConfigFile(parent);
			AdminPlugin.getDefault().getLog().log(new Status(Status.ERROR, AdminPlugin.PLUGIN_ID, IACPluginMessages.CONFIG_FILE_ACCESS_DENIED));		
		}
 		//end of bug 208831, bug 234486
		
		preferenceComposite = parent;
		createEnableDisableComposite(parent);
		GridLayout gridLayout = new GridLayout();
		gridLayout.horizontalSpacing = 5;
		gridLayout.numColumns = 1;
		GridData gridData1 = new GridData();
		gridData1.grabExcessHorizontalSpace = false;
		gridData1.horizontalAlignment = org.eclipse.swt.layout.GridData.BEGINNING;
		gridData1.verticalAlignment = GridData.CENTER;
		gridData1.horizontalIndent = 0;
		gridData1.grabExcessVerticalSpace = false;
		GridData gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.grabExcessVerticalSpace = true;
		gridData.verticalAlignment = GridData.FILL;
		composite = new Composite(parent, SWT.NONE);
		composite.setLayoutData(gridData);
		composite.setLayout(gridLayout);
		label = new Label(composite, SWT.NONE);
		label.setText(PreferenceMessages.JAVA_EXECUTABLE);
		label.setLayoutData(gridData1);
		createJavaExecutableText();
		createAllowedHostsGroup();
		createAcSecurityGroup();
		
		/* Get the values from the config file is a valid one exist; otherwise get it from preference store */
		if(AdminUtil.isConfigurationValid(true))
			refreshValuesFromConfigFile();
		else
			refreshValuesFromPreference();
		
		
		//bug 184625, add CSH
		PlatformUI.getWorkbench().getHelpSystem().setHelp(enableIACButton, AdminPlugin.PLUGIN_ID+".iacpp0001");
		PlatformUI.getWorkbench().getHelpSystem().setHelp(disableIACButton, AdminPlugin.PLUGIN_ID+".iacpp0002");
		PlatformUI.getWorkbench().getHelpSystem().setHelp(modifyLink.getControl(), AdminPlugin.PLUGIN_ID+".iacpp0003");
		PlatformUI.getWorkbench().getHelpSystem().setHelp(localHostButton, AdminPlugin.PLUGIN_ID+".iacpp0004");
		PlatformUI.getWorkbench().getHelpSystem().setHelp(customHostButton, AdminPlugin.PLUGIN_ID+".iacpp0005");
		PlatformUI.getWorkbench().getHelpSystem().setHelp(allHostButton, AdminPlugin.PLUGIN_ID+".iacpp0006");
		PlatformUI.getWorkbench().getHelpSystem().setHelp(addHostButton, AdminPlugin.PLUGIN_ID+".iacpp0007");
		PlatformUI.getWorkbench().getHelpSystem().setHelp(removeHostButton, AdminPlugin.PLUGIN_ID+".iacpp0008");
		PlatformUI.getWorkbench().getHelpSystem().setHelp(editHostButton, AdminPlugin.PLUGIN_ID+".iacpp0009");
		PlatformUI.getWorkbench().getHelpSystem().setHelp(securityOnButton, AdminPlugin.PLUGIN_ID+".iacpp0010");
		PlatformUI.getWorkbench().getHelpSystem().setHelp(securityOffButton, AdminPlugin.PLUGIN_ID+".iacpp0011");
		PlatformUI.getWorkbench().getHelpSystem().setHelp(customUserButton, AdminPlugin.PLUGIN_ID+".iacpp0012");
		PlatformUI.getWorkbench().getHelpSystem().setHelp(anyUserButton, AdminPlugin.PLUGIN_ID+".iacpp0013");
		PlatformUI.getWorkbench().getHelpSystem().setHelp(addUserButton, AdminPlugin.PLUGIN_ID+".iacpp0014");
		PlatformUI.getWorkbench().getHelpSystem().setHelp(removeUserButton, AdminPlugin.PLUGIN_ID+".iacpp0015");
		PlatformUI.getWorkbench().getHelpSystem().setHelp(editUserButton, AdminPlugin.PLUGIN_ID+".iacpp0016");
		
		//208831
		if(!AdminUtil.isSupportedPlatform()){
			
			enableIACButton.setSelection(false);
			disableIACButton.setSelection(false);
			enableIACButton.setEnabled(false);
			disableIACButton.setEnabled(false);
			
			setIACEnable(false);
		}
		//end bug 208831
		
		//bug 234486
		if(!AdminUtil.isIACConfigFileWritable(true)){
			disableIACButton.setEnabled(false);
			enableIACButton.setEnabled(false);
			if(!AdminUtil.isConfigurationValid(true)){
				enableIACButton.setSelection(false);
				disableIACButton.setSelection(true);		
			}else{
				enableIACButton.setSelection(true);
				disableIACButton.setSelection(false);			
			}
			setIACEnable(false);
		}
		//end of bug 234486
		return composite;
	}
	
	/**
	 * This helper method is used to extract the preference page values from the preference store. 
	 * It's called when the preference page is created under 'createContents'
	 */
	private void refreshValuesFromPreference() {
		iacEnabled = adminPlugin.getBoolean(PreferenceMessages.ENABLE_DISABLE_LABEL);
		allowedHosts = adminPlugin.getString(PreferenceMessages.ALLOWED_HOSTS);
		javaExecutable = adminPlugin.getString(PreferenceMessages.JAVA_EXECUTABLE);
		listOfCustomHosts = adminPlugin.getString(PreferenceMessages.HOST_LIST_LABEL);
		security = adminPlugin.getBoolean(PreferenceMessages.AC_SECURITY);
		allowedUsers = adminPlugin.getString(PreferenceMessages.ALLOWED_USERS);
		listOfCustomUsers = adminPlugin.getString(PreferenceMessages.USER_LIST_LABEL);
		
		/* Update the widgets */
		updateWidgets(true);
	}
	
	
	/**
	 * This helper method is used to extract the preference page values from the iac config file. 
     * Perference store will be updated too.
	 */
	private void refreshValuesFromConfigFile() {
		
		
		ConfigFile configFile = new ConfigFile(AdminUtil.getIACConfigFile(true));
		
		
		iacEnabled = adminPlugin.getBoolean(PreferenceMessages.ENABLE_DISABLE_LABEL);
		
		javaExecutable = configFile.getValue(Constants.JAVA_PATH);
		adminPlugin.setString(PreferenceMessages.JAVA_EXECUTABLE, javaExecutable);
		
		String s = configFile.getValue(Constants.SECURITY);
		if(Constants.SECURITY_ON.equalsIgnoreCase(s))
			security = true;
		else
			security = false;
		adminPlugin.setBoolean(PreferenceMessages.AC_SECURITY, security);
		
		allowedHosts = configFile.getValue(Constants.ALLOW_HOST_TYPE);
		if(allowedHosts.equalsIgnoreCase(Constants.ALLOW_ALL))
			allowedHosts = PreferenceMessages.ALL_HOSTS;
		else if(allowedHosts.equalsIgnoreCase(Constants.ALLOW_LOCAL))
			allowedHosts = PreferenceMessages.LOCALHOST;
		else{
			allowedHosts = PreferenceMessages.CUSTOM_HOSTS;
			listOfCustomHosts = configFile.getValue(Constants.HOSTS);
		}
		adminPlugin.setString(PreferenceMessages.ALLOWED_HOSTS, allowedHosts);
		adminPlugin.setString(PreferenceMessages.HOST_LIST_LABEL, listOfCustomHosts);
		
		
		if(security){
			allowedUsers = configFile.getValue(Constants.USERTYPE);
			if(allowedUsers.equalsIgnoreCase(Constants.USER_ALL))
				allowedUsers = PreferenceMessages.ANY_USER;
			else{
				allowedUsers = PreferenceMessages.CUSTOM_USER;
				listOfCustomUsers = configFile.getValue(Constants.USERLIST);
			}			
		}else{
			allowedUsers = PreferenceMessages.ANY_USER;
		}
		adminPlugin.setString(PreferenceMessages.ALLOWED_USERS, allowedUsers);
		adminPlugin.setString(PreferenceMessages.USER_LIST_LABEL, listOfCustomUsers);
		/* Update the widgets */
		updateWidgets(true);
	}
	/**
	 * This helper method is used to restore the default values on AC's preference page
	 */
	private void restoreValuesFromPreferenceDefault() {
		
		iacEnabled = adminPlugin.getDefaultBoolean(PreferenceMessages.ENABLE_DISABLE_LABEL);
		allowedHosts = adminPlugin.getDefaultString(PreferenceMessages.ALLOWED_HOSTS);
		javaExecutable = adminPlugin.getDefaultString(PreferenceMessages.JAVA_EXECUTABLE);
		listOfCustomHosts = adminPlugin.getDefaultString(PreferenceMessages.HOST_LIST_LABEL);
		security = adminPlugin.getDefaultBoolean(PreferenceMessages.AC_SECURITY);
		allowedUsers = adminPlugin.getDefaultString(PreferenceMessages.ALLOWED_USERS);
		listOfCustomUsers = adminPlugin.getDefaultString(PreferenceMessages.USER_LIST_LABEL);

		/* Update the widgets */
		updateWidgets(false);
	}
	
	/**
	 * Used to update the widgets after the values have been retrieved from the preference store
	 * @param restoreLists - True if the lists should also be updated; false otherwise
	 */
	private void updateWidgets(boolean restoreLists)
	{
		/* IAC enabled/disabled */
		enableIACButton.setSelection(iacEnabled);
		disableIACButton.setSelection(!iacEnabled);
		setIACEnable(iacEnabled);
		
		/* JVM Executable */
		javaPathText.setText(javaExecutable);
		
		/* Allowed hosts */
		localHostButton.setSelection(allowedHosts.equals(PreferenceMessages.LOCALHOST));
		allHostButton.setSelection(allowedHosts.equals(PreferenceMessages.ALL_HOSTS));
		customHostButton.setSelection(allowedHosts.equals(PreferenceMessages.CUSTOM_HOSTS));
		
		/* Custom host list */
		if (restoreLists)
		{
			StringTokenizer stListOfHosts = new StringTokenizer(listOfCustomHosts, ",");
			while (stListOfHosts.hasMoreElements())
			{
				listOfHosts.add(stListOfHosts.nextToken());
			}
		}
		
		setEnableListOfHosts(iacEnabled && allowedHosts.equals(PreferenceMessages.CUSTOM_HOSTS));
		
		/* AC Security */
		securityOnButton.setSelection(security);
		securityOffButton.setSelection(!security);
		setEnableSecurityButtons(iacEnabled && security);
		anyUserButton.setSelection(allowedUsers.equals(PreferenceMessages.ANY_USER));
		customUserButton.setSelection(allowedUsers.equals(PreferenceMessages.CUSTOM_USER));
		/* Custom user list */
		if (restoreLists)
		{
			StringTokenizer stListOfUsers = new StringTokenizer(listOfCustomUsers, ",");
			while (stListOfUsers.hasMoreElements())
			{
				listOfUsers.add(stListOfUsers.nextToken());
			}
		}
		
		setEnableListOfUsers(iacEnabled && security && allowedUsers.equals(PreferenceMessages.CUSTOM_USER));
	}
	
	/**
	 * This method initializes allowedHostsGroup	
	 *
	 */
	private void createAllowedHostsGroup() {
		GridLayout gridLayout2 = new GridLayout();
		gridLayout2.numColumns = 1;
		GridData gridData18 = new GridData();
		gridData18.horizontalIndent = 20;
		GridData gridData3 = new GridData();
		gridData3.horizontalIndent = 0;
		gridData3.horizontalAlignment = org.eclipse.swt.layout.GridData.FILL;
		gridData3.grabExcessHorizontalSpace = true;
		allowedHostsGroup = new Group(composite, SWT.NONE);
		allowedHostsGroup.setText(PreferenceMessages.ALLOWED_HOSTS);
		allowedHostsGroup.setLayout(gridLayout2);
		allowedHostsGroup.setLayoutData(gridData3);
		createAllowedHostButtons();
		listOfHostLabel = new Label(allowedHostsGroup, SWT.NONE);
		listOfHostLabel.setText(PreferenceMessages.HOST_LIST_LABEL);
		listOfHostLabel.setLayoutData(gridData18);		
		createListOfHosts();
	}
	/**
	 * This method creates the list of hosts	
	 *
	 */
	private void createListOfHosts() {
		GridLayout gridLayout3 = new GridLayout();
		gridLayout3.numColumns = 2;
		GridData gridData19 = new GridData();
		gridData19.heightHint = 75;
		gridData19.horizontalIndent = 5;
		gridData19.grabExcessHorizontalSpace = true;
		gridData19.horizontalAlignment = org.eclipse.swt.layout.GridData.FILL;
		gridData19.widthHint = 80;
		GridData gridData11 = new GridData();
		gridData11.grabExcessHorizontalSpace = true;
		gridData11.horizontalIndent = 10;
		gridData11.horizontalAlignment = org.eclipse.swt.layout.GridData.FILL;
		composite5 = new Composite(allowedHostsGroup, SWT.NONE);
		composite5.setLayoutData(gridData11);
		composite5.setLayout(gridLayout3);
		listOfHosts = new List(composite5, SWT.V_SCROLL | SWT.BORDER | SWT.MULTI);		
		listOfHosts.setLayoutData(gridData19); 
		listOfHosts.addSelectionListener(new org.eclipse.swt.events.SelectionAdapter() {
			public void widgetSelected(org.eclipse.swt.events.SelectionEvent e) {
				if (listOfHosts.getSelectionCount() > 0)
				{
					removeHostButton.setEnabled(true);
					
					/* Enable the 'edit' button only if ONE item has been selected in the list */
					if (listOfHosts.getSelectionCount() == 1)
					{
						editHostButton.setEnabled(true);
					}
					else
					{
						editHostButton.setEnabled(false);
					}
				}
				else
				{
					removeHostButton.setEnabled(false);
					editHostButton.setEnabled(false);
				}
			}
		});
		listOfHosts.setData(CommonConstants.HOST_LIST); /* Used to indicate which list we're dealing with */
		createHostListButtons();
	}
	/**
	 * This method creates the host list buttons	
	 *
	 */
	private void createHostListButtons() {
		GridData gridData20 = new GridData();
		gridData20.horizontalAlignment = org.eclipse.swt.layout.GridData.FILL;
		gridData20.grabExcessHorizontalSpace = false;
		composite6 = new Composite(composite5, SWT.NONE);
		composite6.setLayout(new GridLayout());
		composite6.setLayoutData(gridData20);
		addHostButton = new Button(composite6, SWT.NONE);
		addHostButton.setText(PreferenceMessages.ADD_BUTTON);		
		addHostButton.setLayoutData(new GridData(SWT.FILL, SWT.DEFAULT, true, false));
		
		
		addHostButton.addSelectionListener(new org.eclipse.swt.events.SelectionAdapter() {
			public void widgetSelected(org.eclipse.swt.events.SelectionEvent e) {
				addToList(listOfHosts, PreferenceMessages.ADD_HOST, PreferenceMessages.HOST_NAME);
			}
		});
		removeHostButton = new Button(composite6, SWT.NONE);
		removeHostButton.setText(PreferenceMessages.REMOVE_BUTTON);
		removeHostButton.setLayoutData(new GridData(SWT.FILL, SWT.DEFAULT, true, false));
		
		removeHostButton.addSelectionListener(new org.eclipse.swt.events.SelectionAdapter() {
			public void widgetSelected(org.eclipse.swt.events.SelectionEvent e) {
				removeFromList(listOfHosts);
			}
		});		
		editHostButton = new Button(composite6, SWT.NONE);
		editHostButton.setText(PreferenceMessages.EDIT_BUTTON);		
		editHostButton.setLayoutData(new GridData(SWT.FILL, SWT.DEFAULT, true, false));
		
		editHostButton.addSelectionListener(new org.eclipse.swt.events.SelectionAdapter() {
			public void widgetSelected(org.eclipse.swt.events.SelectionEvent e) {			
				editList(listOfHosts, PreferenceMessages.EDIT_HOST, PreferenceMessages.HOST_NAME);
			}
		});
	}
	
	/**
	 * This method initializes acSecurityGroup	
	 *
	 */
	private void createAcSecurityGroup() {
		GridData gridData4 = new GridData();
		gridData4.widthHint = 290;
		gridData4.grabExcessHorizontalSpace = true;
		gridData4.horizontalAlignment = org.eclipse.swt.layout.GridData.FILL;
		GridLayout gridLayout4 = new GridLayout();
		gridLayout4.numColumns = 1;
		acSecurityGroup = new Group(composite, SWT.NONE);
		acSecurityGroup.setText(PreferenceMessages.AC_SECURITY);
		createSecurityButtons();
		acSecurityGroup.setLayout(gridLayout4);
		acSecurityGroup.setLayoutData(gridData4);
		createListOfUsersGroup();
	}
	/**
	 * This method initializes listOfUsersGroup	
	 *
	 */
	private void createListOfUsersGroup() {
		GridLayout gridLayout7 = new GridLayout();
		gridLayout7.numColumns = 1;
		GridData gridData12 = new GridData();
		gridData12.horizontalIndent = 5;
		GridData gridData7 = new GridData();
		gridData7.horizontalIndent = 10;
		gridData7.horizontalAlignment = org.eclipse.swt.layout.GridData.FILL;
		gridData7.grabExcessHorizontalSpace = true;
		listOfUsersGroup = new Group(acSecurityGroup, SWT.NONE);
		listOfUsersGroup.setText(PreferenceMessages.ALLOWED_USERS);		
		listOfUsersGroup.setLayout(gridLayout7);
		listOfUsersGroup.setLayoutData(gridData7);
		createCustomAnyButtons();
		listOfUsersLabel = new Label(listOfUsersGroup, SWT.NONE);
		listOfUsersLabel.setText(PreferenceMessages.USER_LIST_LABEL);		
		listOfUsersLabel.setLayoutData(gridData12);
		createListOfUsers();
	}
	/**
	 * This method creates the list of users	
	 *
	 */
	private void createListOfUsers() {
		GridLayout gridLayout1 = new GridLayout();
		gridLayout1.numColumns = 2;
		GridData gridData8 = new GridData();
		gridData8.widthHint = 80;
		gridData8.horizontalIndent = 5;
		gridData8.grabExcessHorizontalSpace = true;
		gridData8.horizontalAlignment = org.eclipse.swt.layout.GridData.FILL;
		gridData8.heightHint = 75;
		GridData gridData13 = new GridData();
		gridData13.horizontalAlignment = org.eclipse.swt.layout.GridData.FILL;
		gridData13.grabExcessHorizontalSpace = true;
		composite1 = new Composite(listOfUsersGroup, SWT.NONE);
		composite1.setLayoutData(gridData13);
		composite1.setLayout(gridLayout1);
		listOfUsers = new List(composite1, SWT.V_SCROLL | SWT.BORDER | SWT.MULTI);		
		listOfUsers.setLayoutData(gridData8);
		listOfUsers.addSelectionListener(new org.eclipse.swt.events.SelectionAdapter() {
			public void widgetSelected(org.eclipse.swt.events.SelectionEvent e) {
				if (listOfUsers.getSelectionCount() > 0)
				{	
					removeUserButton.setEnabled(true);
					
					/* Enable the 'edit' button only if ONE item has been selected in the list */
					if (listOfUsers.getSelectionCount() == 1)
					{
						editUserButton.setEnabled(true);
					}
					else
					{
						editUserButton.setEnabled(false);
					}
				}
				else
				{
					removeUserButton.setEnabled(false);
					editUserButton.setEnabled(false);
				}
			}
		});
		listOfUsers.setData(CommonConstants.USER_LIST);
		createUserListButtons();
	}
	/**
	 * This method creates the buttons for the list of users	
	 *
	 */
	private void createUserListButtons() {
		
		
		GridData gridData14 = new GridData();
		gridData14.grabExcessHorizontalSpace = false;
		gridData14.horizontalAlignment = org.eclipse.swt.layout.GridData.BEGINNING;		
		composite2 = new Composite(composite1, SWT.NONE);
		composite2.setLayoutData(gridData14);
		composite2.setLayout(new GridLayout());
		addUserButton = new Button(composite2, SWT.NONE);
		addUserButton.setText(PreferenceMessages.ADD_BUTTON2);		
		addUserButton.setLayoutData(new GridData(SWT.FILL, SWT.DEFAULT, true, false));
	
		addUserButton
				.addSelectionListener(new org.eclipse.swt.events.SelectionAdapter() {
					public void widgetSelected(org.eclipse.swt.events.SelectionEvent e) {
						addToList(listOfUsers, PreferenceMessages.ADD_USER, PreferenceMessages.USER_NAME);
					}
				});
		
		removeUserButton = new Button(composite2, SWT.NONE);
		removeUserButton.setText(PreferenceMessages.REMOVE_BUTTON2);		
		removeUserButton.setLayoutData(new GridData(SWT.FILL, SWT.DEFAULT, true, false));
		removeUserButton
				.addSelectionListener(new org.eclipse.swt.events.SelectionAdapter() {
					public void widgetSelected(org.eclipse.swt.events.SelectionEvent e) {
						removeFromList(listOfUsers);
					}
				});
		
		editUserButton = new Button(composite2, SWT.NONE);
		editUserButton.setText(PreferenceMessages.EDIT_BUTTON2);
		editUserButton.setLayoutData(new GridData(SWT.FILL, SWT.DEFAULT, true, false));
		editUserButton
				.addSelectionListener(new org.eclipse.swt.events.SelectionAdapter() {
					public void widgetSelected(org.eclipse.swt.events.SelectionEvent e) {
						editList(listOfUsers, PreferenceMessages.ADD_USER, PreferenceMessages.USER_NAME);
					}
				});
	}
	/**
	 * This method creates the security ON/OFF buttons
	 *
	 */
	private void createSecurityButtons() {
		GridData gridData5 = new GridData();
		gridData5.horizontalIndent = 15;
		GridLayout gridLayout5 = new GridLayout();
		gridLayout5.numColumns = 2;
		GridData gridData15 = new GridData();
		gridData15.horizontalIndent = 10;
		GridData gridData6 = new GridData();
		gridData6.grabExcessHorizontalSpace = true;
		gridData6.horizontalIndent = 5;
		gridData6.horizontalAlignment = org.eclipse.swt.layout.GridData.FILL;
		composite3 = new Composite(acSecurityGroup, SWT.NONE);
		composite3.setLayoutData(gridData6);
		composite3.setLayout(gridLayout5);
		securityOnButton = new Button(composite3, SWT.RADIO);
		securityOnButton.setText(PreferenceMessages.SECURITY_ON);
		securityOnButton.setSelection(false);
		securityOnButton.setLayoutData(gridData15);
		securityOnButton
				.addSelectionListener(new org.eclipse.swt.events.SelectionAdapter() {
					public void widgetSelected(org.eclipse.swt.events.SelectionEvent e) {
						security = true;
						
						/* Enable list of users if need be */
						if (customUserButton.getSelection())
							setEnableListOfUsers(true);				
						
						/* Enable 'custom' and 'any user' radio buttons */
						setEnableSecurityButtons(true);
					}
				});
		securityOffButton = new Button(composite3, SWT.RADIO);
		securityOffButton.setText(PreferenceMessages.SECURITY_OFF);
		securityOffButton.setLayoutData(gridData5);
		securityOffButton.addSelectionListener(new org.eclipse.swt.events.SelectionAdapter() {
			public void widgetSelected(org.eclipse.swt.events.SelectionEvent e) {
				security = false;
				
				/* Disable list of users if need be */
				if (customUserButton.getSelection())
					setEnableListOfUsers(false);				
				
				/* Disable 'custom' and 'any user' radio buttons */
				setEnableSecurityButtons(false);
			}
		});
	}
	/**
	 * This method creates the allowed hosts buttons (i.e. all, local, and custom)	
	 *
	 */
	private void createAllowedHostButtons() {
		GridData gridData24 = new GridData();
		gridData24.horizontalIndent = 5;
		GridData gridData22 = new GridData();
		gridData22.horizontalIndent = 5;
		GridLayout gridLayout6 = new GridLayout();
		gridLayout6.numColumns = 3;
		GridData gridData17 = new GridData();
		gridData17.horizontalIndent = 15;
		GridData gridData21 = new GridData();
		gridData21.grabExcessHorizontalSpace = true;
		gridData21.horizontalIndent = 0;
		gridData21.horizontalAlignment = org.eclipse.swt.layout.GridData.FILL;
		composite4 = new Composite(allowedHostsGroup, SWT.NONE);
		composite4.setLayoutData(gridData21);
		composite4.setLayout(gridLayout6);
		customHostButton = new Button(composite4, SWT.RADIO);
		customHostButton.setText(PreferenceMessages.CUSTOM_HOSTS);
		customHostButton.setLayoutData(gridData17);
		customHostButton.addSelectionListener(new org.eclipse.swt.events.SelectionAdapter() {
			public void widgetSelected(org.eclipse.swt.events.SelectionEvent e) {
				allowedHosts = PreferenceMessages.CUSTOM_HOSTS;
				setEnableListOfHosts(true);						
			}
		});
		localHostButton = new Button(composite4, SWT.RADIO);
		localHostButton.setText(PreferenceMessages.LOCALHOST);
		localHostButton.setLayoutData(gridData22);		
		localHostButton.addSelectionListener(new org.eclipse.swt.events.SelectionAdapter() {
			public void widgetSelected(org.eclipse.swt.events.SelectionEvent e) {
				allowedHosts = PreferenceMessages.LOCALHOST;
				setEnableListOfHosts(false);
			}
		});
		allHostButton = new Button(composite4, SWT.RADIO);
		allHostButton.setText(PreferenceMessages.ALL_HOSTS);
		allHostButton.setLayoutData(gridData24);
		allHostButton.addSelectionListener(new org.eclipse.swt.events.SelectionAdapter() {
			public void widgetSelected(org.eclipse.swt.events.SelectionEvent e) {
				allowedHosts = PreferenceMessages.ALL_HOSTS;
				setEnableListOfHosts(false);
			}
		});
	}
	/**
	 * This method creates the text box and the modify link for the Java executable
	 *
	 */
	private void createJavaExecutableText() {
		GridData gridData16 = new GridData();
		gridData16.horizontalIndent = 0;
		GridData gridData2 = new GridData();
		gridData2.widthHint = 240;
		gridData2.horizontalAlignment = org.eclipse.swt.layout.GridData.FILL;
		gridData2.grabExcessHorizontalSpace = true;
		GridLayout gridLayout8 = new GridLayout();
		gridLayout8.numColumns = 2;
		GridData gridData23 = new GridData();
		gridData23.grabExcessHorizontalSpace = true;
		gridData23.horizontalAlignment = org.eclipse.swt.layout.GridData.FILL;
		composite8 = new Composite(composite, SWT.NONE);
		composite8.setLayoutData(gridData23);
		composite8.setLayout(gridLayout8);
		javaPathText = new Text(composite8, SWT.BORDER);
		javaPathText.setEditable(false);
		javaPathText.setLayoutData(gridData2);
		modifyLink = new PreferenceLinkArea(composite8,	SWT.WRAP, JRE_PAGE_ID, PreferenceMessages.MODIFY_LINK, (IWorkbenchPreferenceContainer) getContainer(), null);
		modifyLink.getControl().setLayoutData(gridData16);
	}
	/**
	 * This method creates the 'custom' and 'any user' buttons for security 
	 *
	 */
	private void createCustomAnyButtons() {
		GridData gridData10 = new GridData();
		gridData10.horizontalIndent = 10;
		GridData gridData9 = new GridData();
		gridData9.horizontalIndent = 0;
		GridData gridData25 = new GridData();
		gridData25.grabExcessHorizontalSpace = true;
		gridData25.horizontalAlignment = org.eclipse.swt.layout.GridData.FILL;
		GridLayout gridLayout9 = new GridLayout();
		gridLayout9.numColumns = 2;
		composite7 = new Composite(listOfUsersGroup, SWT.NONE);
		composite7.setLayout(gridLayout9);
		composite7.setLayoutData(gridData25);
		customUserButton = new Button(composite7, SWT.RADIO);
		customUserButton.setText(PreferenceMessages.CUSTOM_USER);		
		customUserButton.setLayoutData(gridData9);
		customUserButton
				.addSelectionListener(new org.eclipse.swt.events.SelectionAdapter() {
					public void widgetSelected(org.eclipse.swt.events.SelectionEvent e) {		
						allowedUsers = PreferenceMessages.CUSTOM_USER;
						setEnableListOfUsers(true);
					}
				});
		anyUserButton = new Button(composite7, SWT.RADIO);
		anyUserButton.setText(PreferenceMessages.ANY_USER);
		anyUserButton.setLayoutData(gridData10);
		anyUserButton.addSelectionListener(new org.eclipse.swt.events.SelectionAdapter() {
			public void widgetSelected(org.eclipse.swt.events.SelectionEvent e) {				
				allowedUsers = PreferenceMessages.ANY_USER;
				setEnableListOfUsers(false);
			}
		});
	}
	//bug 208831
	private void createNotSupportPlatformErrorComposite(Composite parent) {
		
		GridData gridData1 = new GridData();
		gridData1.grabExcessHorizontalSpace = true;
		gridData1.horizontalAlignment = org.eclipse.swt.layout.GridData.FILL;
		
		GridLayout gridLayout2 = new GridLayout();
		gridLayout2.numColumns = 17;
		notSupportPlatformComposite = new Composite(parent, SWT.NONE);
		notSupportPlatformComposite.setLayout(gridLayout2);
		notSupportPlatformComposite.setLayoutData(gridData1);

		
		GridData gridData3 = new GridData();			
		gridData3.grabExcessHorizontalSpace = false;
		gridData3.horizontalAlignment = org.eclipse.swt.layout.GridData.BEGINNING;
		notSupportPlatformMsg = new Label(notSupportPlatformComposite, SWT.NONE);
		notSupportPlatformMsg.setText(PreferenceMessages.NOT_SUPPORT_PLATFORM);
		notSupportPlatformMsg.setSize(150, 20);
		notSupportPlatformMsg.setForeground(new Color(null, 255, 0, 0));
		
		notSupportPlatformMsg.setLayoutData(gridData3);
	}
	//end bug 208831
	
	//bug 234486		
	private void createCannotAccessIACConfigFile(Composite parent) {
		
		GridData gridData1 = new GridData();
		gridData1.grabExcessHorizontalSpace = true;
		gridData1.horizontalAlignment = org.eclipse.swt.layout.GridData.FILL;
		
		GridLayout gridLayout2 = new GridLayout();
		gridLayout2.numColumns = 17;
		accessIACConfigDeniedComposite = new Composite(parent, SWT.NONE);
		accessIACConfigDeniedComposite.setLayout(gridLayout2);
		accessIACConfigDeniedComposite.setLayoutData(gridData1);

		
		GridData gridData3 = new GridData();			
		gridData3.grabExcessHorizontalSpace = false;
		gridData3.horizontalAlignment = org.eclipse.swt.layout.GridData.BEGINNING;
		accessIACConfigDeniedMsg = new Label(accessIACConfigDeniedComposite, SWT.NONE);
		if(AdminUtil.isConfigurationValid(true))
			accessIACConfigDeniedMsg.setText(PreferenceMessages.ACCESS_TO_IAC_CONFIG_DENIED_WITH_VALID_CONFIG);
		else
			accessIACConfigDeniedMsg.setText(PreferenceMessages.ACCESS_TO_IAC_CONFIG_DENIED_WITH_INVALID_CONFIG);

		accessIACConfigDeniedMsg.setSize(150, 20);
		accessIACConfigDeniedMsg.setForeground(new Color(null, 255, 0, 0));
		
		accessIACConfigDeniedMsg.setLayoutData(gridData3);
	}
	
	
	
	/**
	 * This method creates the enable disable buttons	
	 *
	 */
	private void createEnableDisableComposite(Composite parent) {
		GridData gridData31 = new GridData();
		gridData31.grabExcessHorizontalSpace = false;
		gridData31.horizontalAlignment = org.eclipse.swt.layout.GridData.BEGINNING;
		GridData gridData30 = new GridData();
		gridData30.horizontalAlignment = org.eclipse.swt.layout.GridData.END;
		gridData30.verticalAlignment = org.eclipse.swt.layout.GridData.CENTER;
		gridData30.horizontalSpan = 15;
		gridData30.grabExcessHorizontalSpace = false;
		GridData gridData29 = new GridData();
		gridData29.horizontalAlignment = org.eclipse.swt.layout.GridData.BEGINNING;
		gridData29.grabExcessHorizontalSpace = true;
		GridData gridData28 = new GridData();
		gridData28.grabExcessHorizontalSpace = true;
		gridData28.horizontalAlignment = org.eclipse.swt.layout.GridData.FILL;
		GridLayout gridLayout10 = new GridLayout();
		gridLayout10.numColumns = 17;
		enableDisableComposite = new Composite(parent, SWT.NONE);
		enableDisableComposite.setLayout(gridLayout10);
		enableDisableComposite.setLayoutData(gridData28);
		enableDisableLabel = new Label(enableDisableComposite, SWT.NONE);
		enableDisableLabel.setText(PreferenceMessages.ENABLE_DISABLE_LABEL);
		enableDisableLabel.setLayoutData(gridData31);
		enableIACButton = new Button(enableDisableComposite, SWT.RADIO);
		enableIACButton.setText(PreferenceMessages.ENABLE_IAC);
		enableIACButton.setLayoutData(gridData30);		
		enableIACButton.addSelectionListener(new org.eclipse.swt.events.SelectionAdapter() {
			public void widgetSelected(org.eclipse.swt.events.SelectionEvent e) {
				iacEnabled = true;
				setIACEnable(true);
			}
		});
		disableIACButton = new Button(enableDisableComposite, SWT.RADIO);
		disableIACButton.setText(PreferenceMessages.DISABLE_IAC);
		disableIACButton.setLayoutData(gridData29);
		disableIACButton.addSelectionListener(new org.eclipse.swt.events.SelectionAdapter() {
			public void widgetSelected(org.eclipse.swt.events.SelectionEvent e) {
				iacEnabled = false;
				setIACEnable(false);
			}
		});
	}
	
	/**
	 * Helper function used to enable/disable IAC.
	 */
	private void setIACEnable(boolean enabled)
	{
		label.setEnabled(enabled);
		javaPathText.setEnabled(enabled);
		modifyLink.getControl().setEnabled(enabled);
		customHostButton.setEnabled(enabled);
		localHostButton.setEnabled(enabled);
		allHostButton.setEnabled(enabled);
		
		if (((enabled) && (customHostButton.getSelection())) || (!enabled))
		{
			/* Enable these only when custom host button is selected */
			listOfHosts.setEnabled(enabled);
			listOfHostLabel.setEnabled(enabled);
			addHostButton.setEnabled(enabled);
			
			if ((listOfHosts.getSelectionCount() > 0) || (!enabled))
			{
				/* Enable these only if something is selected in the list of hosts */
				removeHostButton.setEnabled(enabled);
				editHostButton.setEnabled(enabled);
			}			
		}
		
		securityOnButton.setEnabled(enabled);
		securityOffButton.setEnabled(enabled);
		
		if (((enabled) && (securityOnButton.getSelection())) || (!enabled))
		{
			/* Enable these only if security is on */
			customUserButton.setEnabled(enabled);
			anyUserButton.setEnabled(enabled);
			
			if ((customUserButton.getSelection()) || (!enabled))
			{
				/* Enable these only if custom user button is selected */
				listOfUsersLabel.setEnabled(enabled);
				listOfUsers.setEnabled(enabled);
				addUserButton.setEnabled(enabled);
				
				if ((listOfUsers.getSelectionCount() > 0) || (!enabled))
				{
					/* Enable these only if something is selected in the list of users */
					removeUserButton.setEnabled(enabled);
					editUserButton.setEnabled(enabled);
				}
			}
		}
	}
	
	/**
	 * Helper function to add to the host or user list. Opens a dialog box to collect content to be added to the list.
	 * @param list The list to use
	 * @param title  Title of the dialog box
	 * @param entryLabel  Label of the entry in the dialog box
	 */
	private void addToList(List list, String title, String entryLabel)
	{
		HostUserDialog dialog = new HostUserDialog(title, entryLabel, null);
		dialog.open();
		
		if (dialog.getReturnCode() == Window.OK)
		{
			String entry = dialog.getEntry();
			list.add(entry);
			
			if (((String)list.getData()).equals(CommonConstants.HOST_LIST))
			{
				/* Append the newly added host to listOfCustomHosts */
				if (!listOfCustomHosts.trim().equals(""))
					listOfCustomHosts = listOfCustomHosts + ",";
				
				listOfCustomHosts = listOfCustomHosts + entry;
			}
			else
			{
				/* Append the newly added user to listOfCustomUsers */
				if (!listOfCustomUsers.trim().equals(""))
					listOfCustomUsers = listOfCustomUsers + ",";
				
				listOfCustomUsers = listOfCustomUsers + entry;
			}
			
		}
	}
	
	/**
	 * Helper function used to remove a selected item from the given list
	 *  
	 * @param list - List to remove the item from
	 */
	private void removeFromList(List list)
	{
		list.remove(list.getSelectionIndices());		
		
		/* If all the items have been deleted, disable the list */
		if (list.getSelectionCount() <= 0)
		{
			String listDataItem = (String)list.getData();
			
			if (listDataItem.equals(CommonConstants.HOST_LIST))
			{
				removeHostButton.setEnabled(false);
				editHostButton.setEnabled(false);
			}
			else
			{
				removeUserButton.setEnabled(false);
				editUserButton.setEnabled(false);
			}
		}
		
		String listData = (String)list.getData();
		
		/* Update string list */
		if (listData.equals(CommonConstants.HOST_LIST))
			updateHostStringList();
		else
			updateUserStringList();
	}
	
	/**
	 * Helper function to edit the host or user list. Opens a dialog box to edit the selected content in the list.
	 * @param list The list to use
	 * @param title  Title of the dialog box
	 * @param entryLabel  Label of the entry in the dialog box
	 */
	private void editList(List list, String title, String entryLabel)
	{
		HostUserDialog dialog = new HostUserDialog(title, entryLabel, list.getSelection()[0]);
		dialog.open();
		
		if (dialog.getReturnCode() == Window.OK)
		{
			list.setItem(list.getSelectionIndex(), dialog.getEntry());
		}
		
		String listData = (String)list.getData();
		
		/* Update string list */
		if (listData.equals(CommonConstants.HOST_LIST))
			updateHostStringList();
		else
			updateUserStringList();
	}
	
	/**
	 * Used to build up the list of custom host string.  Not the most optimum way to doing this, but
	 * it's not a big deal, since the lists are typically small.
	 */
	private void updateHostStringList()
	{
		if (listOfHosts.getItemCount() > 0)
		{
			listOfCustomHosts = listOfHosts.getItem(0);
			for (int i=1; i < listOfHosts.getItemCount(); i++)
			{
				listOfCustomHosts = listOfCustomHosts + "," + listOfHosts.getItem(i);
			}
		}
		else
		{
			listOfCustomHosts = "";
		}
	}
	
	/**
	 * Used to build up the list of custom user string.  Not the most optimum way to doing this, but
	 * it's not a big deal, since the lists are typically small.
	 */	
	private void updateUserStringList()
	{
		if (listOfUsers.getItemCount() > 0)
		{
			listOfCustomUsers = listOfUsers.getItem(0);
			for (int i=1; i < listOfUsers.getItemCount(); i++)
			{
				listOfCustomUsers = listOfCustomUsers + "," + listOfUsers.getItem(i);
			}
		}		
		else
		{
			listOfCustomUsers = "";
		}
	}
	
	
	/**
	 * Check this list for the given item
	 * @param list  The list to check
	 * @param itemToCheck  The item to check for
	 * @return  true if item is in list; false otherwise
	 */
	private boolean isItemInList(List list, String itemToCheck)
	{
		/* Avoid duplicate entries in the list */
		String[] items = list.getItems();		
		for (int i=0; i < items.length; i++)
		{
			if (itemToCheck.equals(items[i]))
				return true;
		}
		return false;
	}
	
	/**
	 * Helper function used to enable/disable list of hosts and its corresponding buttons
	 * @param enable true if it should be enahbled; false otherwise
	 */
	private void setEnableListOfUsers(boolean enable)
	{
		listOfUsersLabel.setEnabled(enable);
		listOfUsers.setEnabled(enable);
		addUserButton.setEnabled(enable);		
		if (listOfUsers.getSelectionCount() > 0)
		{
			removeUserButton.setEnabled(true);
			
			if (listOfUsers.getSelectionCount() == 1)
			{
				editUserButton.setEnabled(true);
			}
		}
		else
		{
			removeUserButton.setEnabled(false);
			editUserButton.setEnabled(false);
		}
		
		if (!enable)
		{
			editUserButton.setEnabled(false);
			removeUserButton.setEnabled(false);
		}
	}
	
	/**
	 * Helper function used to enable/disable list of hosts and its corresponding buttons
	 * @param enable true if it should be enahbled; false otherwise
	 */
	private void setEnableListOfHosts(boolean enable)
	{
		listOfHostLabel.setEnabled(enable);
		listOfHosts.setEnabled(enable);
		addHostButton.setEnabled(enable);
		if (listOfHosts.getSelectionCount() > 0)
		{
			removeHostButton.setEnabled(true);
			
			if (listOfHosts.getSelectionCount() == 1)
			{
				editHostButton.setEnabled(true);
			}
		}
		else
		{
			removeHostButton.setEnabled(false);
			editHostButton.setEnabled(false);
		}
		
		if (!enable)
		{
			editHostButton.setEnabled(false);
			removeHostButton.setEnabled(false);
		}
	}
	
	/**
	 * Helper function used to enable/disable radio buttons for security
	 * @param enable true if it should be enahbled; false otherwise
	 */
	private void setEnableSecurityButtons(boolean enable)
	{
		customUserButton.setEnabled(enable);
		anyUserButton.setEnabled(enable);
	}
	
	
	/**
	 * @see PreferencePage#computeSize()
	 */
	public Point computeSize() {
		Point p = super.computeSize();

		return p;
	}
	/**
	 * @see IWorkbenchPreferencePage#init(IWorkbench)
	 */
	public void init(IWorkbench workbench) {
	}
	
	/**
	 * @see PreferencePage#performDefaults()
	 */
	protected void performDefaults() {
		restoreValuesFromPreferenceDefault();
		super.performDefaults();

	}
	
	protected void performApply() {
		
		/* Don't do anything if preference page hasn't changed */
		if (!hasPrefernceChanged())
			return;
				
		adminPlugin.setBoolean(PreferenceMessages.ENABLE_DISABLE_LABEL, iacEnabled);
		adminPlugin.setString(PreferenceMessages.ALLOWED_HOSTS, allowedHosts);
		adminPlugin.setString(PreferenceMessages.HOST_LIST_LABEL, listOfCustomHosts);
		adminPlugin.setBoolean(PreferenceMessages.AC_SECURITY, security);
		adminPlugin.setString(PreferenceMessages.ALLOWED_USERS, allowedUsers);
		adminPlugin.setString(PreferenceMessages.USER_LIST_LABEL, listOfCustomUsers);
				
		//bug 235323
		boolean stopIAC = false;	
		if (AutoStartStop.getIACStarted() ){
			if(iacEnabled){
				/* Confirm whether the user wants to restart IAC */
				MessageDialog confirmationDialog = AdminUtil.getConfirmationDialog(preferenceComposite.getShell(), PreferenceMessages.CONFIRMATION_TITLE, PreferenceMessages.CONFIRMATION_MESSAGE);
				stopIAC = (confirmationDialog.open() == 0);
			}else{
				MessageDialog confirmationDialog = AdminUtil.getConfirmationDialog(preferenceComposite.getShell(), PreferenceMessages.CONFIRMATION_TITLE, PreferenceMessages.CONFIRMATION_STOP_MESSAGE);
				stopIAC = (confirmationDialog.open() == 0);
			}
		}
						
		if (stopIAC)
			AutoStartStop.stopIAC();
		if (iacEnabled){					
			AdminUtil.generateConfigFile(true);
			/* IAC will automatically start next time we attempt to connect to it */		
		}
		//end of 235323
	}
	
	private boolean hasPrefernceChanged()
	{
		if (adminPlugin.getBoolean(PreferenceMessages.ENABLE_DISABLE_LABEL) != iacEnabled)
			return true;
		else if (!adminPlugin.getString(PreferenceMessages.ALLOWED_HOSTS).equals(allowedHosts))
			return true;
		else if	(!adminPlugin.getString(PreferenceMessages.HOST_LIST_LABEL).equals(listOfCustomHosts))
			return true;
		else if (adminPlugin.getBoolean(PreferenceMessages.AC_SECURITY) != security)
			return true;
		else if (!adminPlugin.getString(PreferenceMessages.ALLOWED_USERS).equals(allowedUsers))
			return true;
		else if (!adminPlugin.getString(PreferenceMessages.USER_LIST_LABEL).equals(listOfCustomUsers))
			return true;
		
		return false;
	}
	
	public boolean performOk() {
		performApply();

		return true;
	}
	
	/**
	 * Dialog that's used to add/edit host and user names
	 * Part of bugzilla 173330
	 * 
	 * @author Navid Mehregani
	 *
	 */
	class HostUserDialog extends Dialog implements ModifyListener {
		private Text entryText;
		private String entryString;
        private String title;
        private String entryLabel;
        private Label errorLabel;
		
        /**
         * @param title Title of the dialog box
         * @param entryString  Set when editing; null when adding
         * @param entryLabel Label of the entry
         */
		public HostUserDialog(String title, String entryLabel, String entryString) {
			super(Display.getCurrent().getActiveShell());
			this.title = title;
			this.entryString = entryString;
			this.entryLabel = entryLabel; 
		}
		
		protected void configureShell(Shell shell) {
			super.configureShell(shell);
			shell.setText(title);
		}

		protected Control createDialogArea(Composite parent) {
			Composite result = (Composite) super.createDialogArea(parent);

			GridLayout layout;
			GridData data;
			Label label;

			layout = new GridLayout();
			layout.numColumns = 2;
			result.setLayout(layout);
			data = AdminUtil.createFill();
			data.widthHint = 300;
			result.setLayoutData(data);			
			
			Composite errorComposite = new Composite(result, SWT.NONE);
			GridLayout layoutForErrorComposite = new GridLayout();
			GridData dataForErrorComposite = new GridData();
			layoutForErrorComposite.numColumns = 2;
			dataForErrorComposite.horizontalAlignment = SWT.FILL;
			dataForErrorComposite.grabExcessHorizontalSpace = true;
			dataForErrorComposite.horizontalSpan = 2;
			errorComposite.setLayout(layoutForErrorComposite);
			errorComposite.setLayoutData(dataForErrorComposite);
			errorLabel = new Label (errorComposite, SWT.NONE);
			errorLabel.setLayoutData(AdminUtil.createHorizontalFill());
			errorLabel.setText("");
			
			label = new Label(result, SWT.NONE);
			label.setText(entryLabel);  
			entryText = new Text(result, SWT.BORDER);
			entryText.setLayoutData(AdminUtil.createHorizontalFill());
			entryText.addModifyListener(this);
			
			if (entryString != null)
				entryText.setText(entryString);		
			
			entryText.setFocus();
			entryText.selectAll();			
			
			return result;
		}
	
		protected void okPressed() {
			entryString = entryText.getText().trim();
			super.okPressed();
		}
		
		/*
		 * Set the initial Ok button enablement.
		 */
		protected Control createContents(Composite parent) {
			Control result = super.createContents(parent);
			getButton(IDialogConstants.OK_ID).setEnabled(entryText.getText().trim().length() > 0);
			return result;
		}

		/**
		 * @return Returns the entry string.
		 */
		public String getEntry() {
			return entryString;
		}
		
		public void modifyText(ModifyEvent e) {
			if (getButton(IDialogConstants.OK_ID) != null)
			{
				List list = null;
				if (entryLabel.equalsIgnoreCase(PreferenceMessages.HOST_NAME))
					list = listOfHosts;
				else
					list = listOfUsers;
				
				String entryString = entryText.getText().trim();
				
				boolean itemAlreadyInList = isItemInList(list, entryString);
				
				if (itemAlreadyInList)
					errorLabel.setText(PreferenceMessages.ITEM_ALREADY_IS_IN_LIST);
				else
					errorLabel.setText("");
				
				getButton(IDialogConstants.OK_ID).setEnabled((entryString.length() > 0) && (!itemAlreadyInList));	
			}
			
		}
	}

}
