/**********************************************************************
 Copyright (c) 2007, 2009 IBM Corporation and others.
 All rights reserved. This program and the accompanying materials
 are made available under the terms of the Eclipse Public License v1.0
 which accompanies this distribution, and is available at
 http://www.eclipse.org/legal/epl-v10.html
 $Id: AutoStartStop.java,v 1.20 2009/05/08 16:58:13 jcayne Exp $
 
 Contributors:
     IBM - initial implementation
 **********************************************************************/

package org.eclipse.tptp.platform.iac.administrator.internal.startstop;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.Socket;
import java.net.UnknownHostException;

import org.eclipse.core.runtime.Status;
import org.eclipse.tptp.platform.iac.administrator.AdminPlugin;
import org.eclipse.tptp.platform.iac.administrator.internal.common.AdminUtil;
import org.eclipse.tptp.platform.iac.administrator.internal.common.CommonConstants;
import org.eclipse.tptp.platform.iac.administrator.internal.common.IACPluginMessages;
import org.eclipse.tptp.platform.iac.administrator.internal.config.ConfigGenerator;
import org.eclipse.tptp.platform.iac.administrator.internal.preference.PreferenceMessages;


/**
 * This class is responsible for the auto start/stop of IAC.  Note that in the context of this
 * class, IAC simply refers to the Agent Controller that's shipped with the workbench.
 * 
 * @author Navid Mehregani
 *
 */
public class AutoStartStop {
	
	/* Set to true when WE start IAC; false otherwise.  
	 * This boolean is unaffected if the user has manually started an instance the stand-alone AC */
	private static boolean iacStarted = false;
	private static boolean shownNotSupportPlatformError = false;
		
	/* Instance of the plugin activator class */
	private static AdminPlugin adminPlugin = AdminPlugin.getDefault();
	
	/**
	 * Used to start the Integrated Agent Controller
	 * 
	 * @return  Indicates the status of starting IAC.  0 if successful, nonzero if not.         
	 */
	public static synchronized int startIAC()
	{
		/* Do we have an instance of our plugin activator */
		if (adminPlugin == null)
		{
			Status status = new Status(Status.ERROR, AdminPlugin.PLUGIN_ID, IACPluginMessages.IAC_PLUGIN_NULL);
			AdminPlugin.getDefault().getLog().log(status);
			return 1;
		}
		
		if(!AdminUtil.isSupportedPlatform() && !shownNotSupportPlatformError){
			AdminUtil.openErrorDialog(PreferenceMessages.ERROR_DIALOG_TITLE, PreferenceMessages.NOT_SUPPORT_PLATFORM);
			shownNotSupportPlatformError = true;
		}
		
		
		
		/* Do nothing if IAC is disabled */
		if (!adminPlugin.getBoolean(PreferenceMessages.ENABLE_DISABLE_LABEL))
			return 0;
			
		java.lang.Process process = null;
		int result = 0;
				
		/* Start IAC only if it hasn't been started before */
		if (!isIACRunning())
		{
			/* Does IAC exist? */
			if (AdminUtil.getIACHome() == null)
			{
				Status status = new Status(Status.ERROR, AdminPlugin.PLUGIN_ID, IACPluginMessages.IAC_NOT_FOUND);
	        	AdminPlugin.getDefault().getLog().log(status);
	        	return 1;
			}
			
			/* Do we need to generate IAC's configuration file? */
			if (AdminUtil.isNeedToGenerateConfigFile(true))
			{
				AdminUtil.generateConfigFile(true);
			
				/* If configuration is still invalid, return an error  */
				if (!AdminUtil.isConfigurationValid(true))
				{
					Status status = new Status(Status.ERROR, AdminPlugin.PLUGIN_ID, IACPluginMessages.CANT_START_IAC_CONFIG_FILE_INVALID);
		        	AdminPlugin.getDefault().getLog().log(status);
		        	return 1;
				}
			}						
			
			Runtime rt = Runtime.getRuntime();
			String[] acExecutable = AdminUtil.getIACExecutableArray();
							
			try
			{				
				/* Run the 'RAServer' command.  We can't use the CommandExecutor class here because that class
				 * waits until its error/output threads are terminated */				
				process = rt.exec(acExecutable, null, new File(AdminUtil.getIACBinFolder()));
				
				new AutoStartStop().new ProcessOutputReader(process.getErrorStream(), "Error Stream").start();
				new AutoStartStop().new ProcessOutputReader(process.getInputStream(), "Output Stream").start();									
							
			} catch (Exception e)
			{				
				Status status = new Status(Status.ERROR, AdminPlugin.PLUGIN_ID, 0, e.getMessage(), e);
				AdminPlugin.getDefault().getLog().log(status);
				result = 1;
			}	
			
			if (hasProcessTerminated(process))
			{
				/* Oh oh, RAServer has terminated.  Something went wrong.  Try to recover from this state: We'll 
				 * regenerate IAC's configuration from its "default default" value and start it again */
				ConfigGenerator configGenerator = new ConfigGenerator();
				String iacHome = AdminUtil.getIACHome();
				if (iacHome == null)
					return 2;
				
				configGenerator.setUpLocallyWithNoSecurity(iacHome, AdminUtil.getJVMExecutable(), AdminUtil.getPluginsFolder(), AdminUtil.getIACPlugins());
				
				try
				{
					process = rt.exec(acExecutable);
					
					new AutoStartStop().new ProcessOutputReader(process.getErrorStream(), "Error Stream").start();
					new AutoStartStop().new ProcessOutputReader(process.getInputStream(), "Output Stream").start();
				} catch (Exception e)
				{				
					Status status = new Status(Status.ERROR, AdminPlugin.PLUGIN_ID, 0, e.getMessage(), e);
					AdminPlugin.getDefault().getLog().log(status);
					result = 1;
				}	
				
				/* Let's try this again */
				if (hasProcessTerminated(process))
				{
					/* Our attempt to recover has failed.  Just return an error code */
					result = 2;
				}
			}
			
			/* Wait until IAC is started */
			if (result == 0)
			{
				while (!isIACRunning()&& !hasProcessTerminated(process))
				{
					try {
						Thread.sleep(1000);
					} catch (InterruptedException e) {
						Status status = new Status(Status.ERROR, AdminPlugin.PLUGIN_ID, 0, e.getMessage(), e);
						AdminPlugin.getDefault().getLog().log(status);
						result = 1;
					}
				}
			}
			
			if (result == 0)
				iacStarted = true; /* Indicate that we started IAC */

		}		
		
		return result;
		
	}

	/**
	 * Used to stop IAC
	 * 
	 * @return  Indicates the status of stopping IAC.  0 if successful, nonzero if not.
	 */
	public static synchronized int stopIAC()
	{
		/* Do nothing if IAC is disabled */
		/**
		 bug 235323
		if (!adminPlugin.getBoolean(PreferenceMessages.ENABLE_DISABLE_LABEL))
			return 0;
		*/
		
		int result = 0;
		
		if (!iacStarted) 
			return result;
		
		/* Use the 'RAServer -shutdown' command to forcibly kill IAC */
		CommandExecutor commandExecutor = new CommandExecutor();
		String[] command = AdminUtil.getShutdownCommandArray();
		StringBuffer stdOut = new StringBuffer();
		StringBuffer stdErr = new StringBuffer();
		
		commandExecutor.executeCommand(command, null, AdminUtil.getIACBinFolder(), stdOut, stdErr);
		
		if (stdErr.length() != 0)
		{
			Status status = new Status(Status.ERROR, AdminPlugin.PLUGIN_ID,IACPluginMessages.IAC_SHUTDOWN_ERROR + stdErr);
			AdminPlugin.getDefault().getLog().log(status);
			result = 1;
		}
		
		/* Wait until IAC is terminated */
		if (result == 0)
		{
			while (isIACRunning())
			{
				try {
					Thread.sleep(1000);
				} catch (InterruptedException e) {
					Status status = new Status(Status.ERROR, AdminPlugin.PLUGIN_ID, 0, IACPluginMessages.IAC_SHUTDOWN_ERROR + e.getMessage(), e);
					AdminPlugin.getDefault().getLog().log(status);
					result = 1;
				}
			}
			
			iacStarted = false;  /* Indicated that we terminated IAC */
		}		
						
		return result;
	}
	
	/**
	 * Indicates whether IAC is running or not.  This method simply checks to see whether or not IAC's socket is up
	 * and running.  This method can easily get confused by another program listening on this socket.  There is no
	 * easier way of checking whether IAC is running.  Another way would be to check the named pipes, but I didn't
	 * want to introduce native code and create more complexity.
	 * 
	 * @return true if IAC is running; false otherwise
	 */
	public static boolean isIACRunning()
	{
		boolean result = true;
		
		/* Check to see whether IAC's socket is up and running */
		Socket socket = null;
		try {
			socket = new Socket("localhost", 10002);  // TODO: Extract the port number from the config file
			socket.close();
		} catch (UnknownHostException e) {
			Status status = new Status(Status.ERROR, AdminPlugin.PLUGIN_ID, 0, e.getMessage(), e);
			AdminPlugin.getDefault().getLog().log(status);
			result = false;
		} catch (IOException e) {
			/* Socket isn't up and running, return false */
			result = false;
		} 
		
		return result;
		
	}
	
	/**
	 * Used to restart IAC.  Equivalent to calling stopIAC() and then startIAC()
	 * 
	 * @return  0 if restarting is successful. Non-zero value if not.
	 */
	public static int restartIAC()
	{
		int result = stopIAC();
		
		if (result != 0)
			return result;
		
		result = startIAC();
		
		return result;
	}
	
	/**
	 * @param process  The process to check for
	 * @return  True if the given process has terminated; false otherwise.
	 */
	private static boolean hasProcessTerminated(Process process)
	{
		try
		{
			//220197.   Process is null if it failed to start
			if(process == null)
				return true;
			
			process.exitValue();
			return true;
			
		} catch (IllegalThreadStateException e)
		{
			return false;
		}
	}
		
	class ProcessOutputReader extends Thread {

        private BufferedReader reader;
        private String line = "";
        private String streamName = "";

        public ProcessOutputReader(InputStream inputStream, String stream) {
        	setName(stream);
            reader = new BufferedReader(new InputStreamReader(inputStream));
            streamName = stream;
        }

        public void run() {

            try {
                
                while ((line = reader.readLine()) != null){
                	if ((line.indexOf("started successfully") == -1) && (line.indexOf("Starting Agent Controller") == -1) && (line.indexOf("ACServer stopped") == -1) && (line.indexOf("Shutting down the Agent Controller") == -1))
                	{
	                 	line = line.trim() + CommonConstants.LINE_SEPARATOR;
	                	Status status = new Status(Status.ERROR, AdminPlugin.PLUGIN_ID, IACPluginMessages.IAC_PROCESS_ERROR + " ["+streamName+"]:" + line);
						AdminPlugin.getDefault().getLog().log(status);     
                	}
                }
            }
            catch (IOException e) {
            	Status status = new Status(Status.ERROR, AdminPlugin.PLUGIN_ID, e.getMessage());
				AdminPlugin.getDefault().getLog().log(status);            	        	
            }
        }
    }

	/**
	 * @return The iacStarted boolean that's set by startIAC and stopIAC.
	 */
	public static boolean getIACStarted() {
		return iacStarted;
	}

}
