/**********************************************************************
 Copyright (c) 2007, 2009 IBM Corporation and others.
 All rights reserved. This program and the accompanying materials
 are made available under the terms of the Eclipse Public License v1.0
 which accompanies this distribution, and is available at
 http://www.eclipse.org/legal/epl-v10.html
 $Id: ConfigGenerator.java,
 
 Contributors:
     IBM - initial implementation
 **********************************************************************/


package org.eclipse.tptp.platform.iac.administrator.internal.startstop;
import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.UnsupportedEncodingException;

import org.eclipse.core.runtime.Status;
import org.eclipse.tptp.platform.iac.administrator.AdminPlugin;
import org.eclipse.tptp.platform.iac.administrator.internal.common.IACPluginMessages;

/**
 * Used to execute system commands.  Stores the error/output stream of the process executed
 * 
 * @author Navid Mehregani
 *
 */
public final class CommandExecutor {
	
	private Object currentThreadLock = new Object();
	private int childThreadsDone = 0;

	/**
	 * Main method used to test this class.  This should be replaced by JUnits
	 */
	public static void main(String[] args) {
		
		StringBuffer stdout = new StringBuffer();
		StringBuffer stderr = new StringBuffer();
		CommandExecutor ce = new CommandExecutor();
		ce.executeCommand("dir ../..", null, null, stdout, stderr);
		
		if (stderr.length() != 0)
		{
			System.err.println("Command generated error(s): " + stderr.toString());
		}
		
		System.out.println("Ouptut: " + stdout.toString());
	}
	
	/**
	 * Used to execute the specified command with the given settings
	 * 
	 * @param command  Command to execute
	 * @param environmentVars  The environment variables to set prior to executing the command.  Format: <variable_name>=<variable_value>. Can be null.
	 * @param workingDirectory  The working directory that the command should be executed from. Can be null.
	 * @param stdout  Output stream of the process executed is stored in this StringBuffer
	 * @param stderr  Error stream of the process executed is stored in this StringBuffer
	 */
	public void executeCommand(String[] command, String[] environmentVars, String workingDirectory, StringBuffer stdout, StringBuffer stderr)
	{				
        try {
        	
        	/* Is a working directory specified? */
        	File workingDirectoryFile = null;
        	if (workingDirectory != null)
        	{
        		workingDirectoryFile = new File(workingDirectory);
	        	
	        	/* Does the specified working directory exist? */ 
	        	if (!workingDirectoryFile.exists())	    
	        	{
	        		Status status = new Status(Status.ERROR, AdminPlugin.PLUGIN_ID, IACPluginMessages.DIRECTORY_NOT_EXIST + workingDirectory);
	        		AdminPlugin.getDefault().getLog().log(status);
	        	}
        	}
        	
        	/* Execute the command */
            java.lang.Process	process = Runtime.getRuntime().exec(command, environmentVars, workingDirectoryFile);
            
            /* Capture any stdout and stderr output from the execution of the command */
            new ProcessOutputReader(process.getErrorStream(), stderr, "Error Stream").start();
            new ProcessOutputReader(process.getInputStream(), stdout, "Output Stream").start();

            /* Wait for both of the ProcessOutputReader threads spinned off for output/error streams to be completed */ 
            synchronized(currentThreadLock)
            {
            	currentThreadLock.wait();
            }
            
            /* Get the process' exit value */
            int exitValue =  process.waitFor();
            
            if (exitValue != 0)
            {
            	Status status = new Status(Status.ERROR, AdminPlugin.PLUGIN_ID, IACPluginMessages.PROCESS_ERROR);
    			AdminPlugin.getDefault().getLog().log(status);    		
            }
            
        }
        catch (Exception  e) {
        	Status status = new Status(Status.ERROR, AdminPlugin.PLUGIN_ID, 0, e.getMessage(), e);
        	AdminPlugin.getDefault().getLog().log(status);        
        }
    }
	
	/**
	 * Used to execute the specified command with the given settings
	 * 
	 * @param command  Command to execute
	 * @param environmentVars  The environment variables to set prior to executing the command.  Format: <variable_name>=<variable_value>. Can be null.
	 * @param workingDirectory  The working directory that the command should be executed from. Can be null.
	 * @param stdout  Output stream of the process executed is stored in this StringBuffer
	 * @param stderr  Error stream of the process executed is stored in this StringBuffer
	 */
	public void executeCommand(String command, String[] environmentVars, String workingDirectory, StringBuffer stdout, StringBuffer stderr)
	{				
		String[] commandArray = {command};
		executeCommand(commandArray, environmentVars, workingDirectory, stdout, stderr);
    }
	
	/**
	 * This inner class is used to capture the error/output streams of the command executed
	 * 
	 * @author Navid Mehregani
	 *
	 */
    class ProcessOutputReader extends Thread {

        private BufferedReader reader;
        private StringBuffer output;

        /**
         * Class' Constructor
         * 
         * @param inputStream  Either the error or output stream of the command that has been executed
         * @param output  StringBuffer used to store the content of the stream
         * @param streamName  Name of the stream.  Used to set the name of the thread.
         */
        public ProcessOutputReader(InputStream inputStream, StringBuffer output, String streamName) {
        	
        	String platform = System.getProperty("os.name").toLowerCase();
        	
        	/* Need to use the right encoding for AS400, since it uses EBCDIC */
        	if (platform.indexOf("os/400") != -1)
        	{
    			try {
					reader = new BufferedReader(new InputStreamReader(inputStream,"Cp037"));
				} catch (UnsupportedEncodingException e) {
					Status status = new Status(Status.ERROR, AdminPlugin.PLUGIN_ID, 0, e.getMessage(), e);
					AdminPlugin.getDefault().getLog().log(status);        
				}
        	}        		
        	else
        	{
        		reader = new BufferedReader(new InputStreamReader(inputStream));
        	}
        		
        	
            this.output = output;
            setName(streamName);
        }

        /**
         * Monitors the specified stream and copies it to the given StringBuffer
         */
        public void run() {

            try {
            	
            	String newline = System.getProperty("line.separator");
                
            	String line = "";
            	
                /* Read the input stream until the stream is closed (i.e. null termination) and append the read string to the storage buffer */
                while ((line = reader.readLine()) != null){                
                    output.append(line + newline);                    
                }
            }
            catch (IOException e) {
            	Status status = new Status(Status.ERROR, AdminPlugin.PLUGIN_ID, 0, e.getMessage(), e);
            	AdminPlugin.getDefault().getLog().log(status);         
            }

            /* If both input streams (i.e. stdout and stderr) are closed, notify the waiting parent thread */
            synchronized (currentThreadLock) {
                if (++childThreadsDone == 2)
                    currentThreadLock.notify();
            	
            }
        }
    }

}
