package org.eclipse.buckminster.subversion.utilities;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;
import java.util.Set;

/**
 * This class is used to autogenerate the "svn_exception_messages.properties" file. This file is located in
 * org.eclipse.buckminster.subversion package and used by SvnExceptionHandler class to analyze the exceptions thrown by
 * subversion clients. Check SvnExceptionHandler class for more informations.
 * 
 * @author Guillaume Chatelet
 * 
 */
public class ExceptionExtractor
{
	private static final Properties MESSAGES = new Properties();

	private static final Properties TRANSLATIONS = new Properties();

	private static final String COMMENT = "This file is autogenerated from " + ExceptionExtractor.class.getName()
			+ "\n#Please consult it for more informations.\n";

	/**
	 * Statically reads the messages that are to be translated
	 */
	static
	{
		try
		{
			MESSAGES.load(ExceptionExtractor.class.getResourceAsStream("exception_messages.properties"));
		}
		catch(IOException e)
		{
			e.printStackTrace();
			throw new RuntimeException(e.getMessage());
		}
	}

	/**
	 * Each argument points to root of subversion sources folder. Version is extracted from the forlder's name (
	 * expecting "subversion-version" )
	 * 
	 * @param args
	 * @throws IOException
	 */
	public static void main(String[] args) throws IOException
	{
		final List<String> versions = new ArrayList<String>();
		for(String folder : args)
		{
			final String folderName = new File(folder).getName();
			String version = folderName.split("-")[1];
			versions.add(version);
			// building the default language
			new ExceptionExtractor(version);
			// building variants
			final File[] files = gatherPoFiles(new File(folder + "/subversion/po"));
			for(File file : files)
				new ExceptionExtractor(version, file);
		}
		final StringBuilder builder = new StringBuilder(COMMENT);
		builder.append('\n');
		builder.append("# Message used\n");
		builder.append("# ------------\n");
		for(Object value : MESSAGES.values())
			builder.append("# " + value).append('\n');
		builder.append('\n');
		builder.append("# Version used\n");
		builder.append("# ------------\n");
		for(String version : versions)
			builder.append("# " + version).append('\n');
		builder.append('\n');
		TRANSLATIONS.store(new FileOutputStream(
				"src/org/eclipse/buckminster/subversion/svn_exception_messages.properties"), builder.toString());
	}

	private static File[] gatherPoFiles(File file)
	{
		if(file.isDirectory() == false)
			return new File[0];
		return file.listFiles(new FilenameFilter()
		{

			public boolean accept(File dir, String name)
			{
				return name.endsWith(".po");
			}

		});
	}

	private static String getLangage(File file)
	{
		final String name = file.getName();
		final int indexOf = name.indexOf(".po");
		return name.substring(0, indexOf);
	}

	final String ID;

	protected ExceptionExtractor(String version) throws IOException
	{
		ID = version;

		for(Object key : MESSAGES.keySet())
		{
			final String exceptionKey = (String)key;
			final String message = (String)MESSAGES.get(key);
			TRANSLATIONS.put(getExceptionName(exceptionKey), getLongestTextPart(cleanLine(message)));
		}
	}

	protected ExceptionExtractor(String version, File file) throws IOException
	{
		ID = version + '|' + getLangage(file);
		final InputStreamReader reader = new InputStreamReader(new FileInputStream(file), "UTF-8");
		System.out.println(">> reading " + file.getAbsolutePath());
		try
		{
			processStream(new BufferedReader(reader));
		}
		finally
		{
			reader.close();
		}
	}

	/**
	 * returns the line without quoting and msgstr
	 */
	private String cleanLine(String line)
	{
		int startIndex = line.indexOf('\"');
		int lastIndex = line.lastIndexOf('\"');
		return line.substring(startIndex + 1, lastIndex);
	}

	private String getExceptionName(String exception)
	{
		return exception + '|' + ID;
	}

	private String getLongestTextPart(String translation)
	{
		final String[] split = translation.split("%ld|%s");
		int maxLenght = 0;
		int maxIndex = 0;
		for(int i = 0; i < split.length; ++i)
			if(split[i].length() > maxLenght)
			{
				maxIndex = i;
				maxLenght = split[i].length();
			}
		return split[maxIndex];
	}

	private String getTranslation(BufferedReader reader) throws IOException
	{
		StringBuilder builder = new StringBuilder();
		String line = null;

		// going to TRANSLATIONS line start
		while((line = reader.readLine()).indexOf("msgstr") == -1)
			;
		// gather message over lines
		do
		{
			builder.append(cleanLine(line));
		} while((line = reader.readLine()).indexOf('\"') != -1);
		return builder.toString();
	}

	/**
	 * returns the identified exception message if any
	 */
	private String isExceptionMessage(String line)
	{

		final Set<Object> keySet = MESSAGES.keySet();
		for(Object object : keySet)
		{
			final String exception = (String)object;
			final String message = MESSAGES.getProperty(exception);
			if(line.indexOf(message) != -1)
				return exception;
		}
		return null;
	}

	private void processStream(BufferedReader reader) throws IOException
	{
		String line = null;
		while((line = reader.readLine()) != null)
		{
			String exception;
			if((exception = isExceptionMessage(line)) != null)
				TRANSLATIONS.put(getExceptionName(exception), getLongestTextPart(getTranslation(reader)));
		}
	}
}
