/**
 * Copyright (c) 2004 - 2010 Eike Stepper (Berlin, Germany) and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Eike Stepper - initial API and implementation
 *    Stefan Winkler - 271444: [DB] Multiple refactorings bug 271444
 */
package org.eclipse.emf.cdo.server.internal.db.mapping.horizontal;

import org.eclipse.emf.cdo.common.id.CDOID;
import org.eclipse.emf.cdo.common.id.CDOIDUtil;
import org.eclipse.emf.cdo.common.model.CDOClassifierRef;
import org.eclipse.emf.cdo.common.revision.CDORevision;
import org.eclipse.emf.cdo.eresource.CDOResourceNode;
import org.eclipse.emf.cdo.eresource.EresourcePackage;
import org.eclipse.emf.cdo.server.IStoreAccessor.QueryResourcesContext;
import org.eclipse.emf.cdo.server.db.IDBStoreAccessor;
import org.eclipse.emf.cdo.server.db.IObjectTypeCache;
import org.eclipse.emf.cdo.server.db.mapping.IClassMapping;
import org.eclipse.emf.cdo.server.internal.db.CDODBSchema;
import org.eclipse.emf.cdo.server.internal.db.bundle.OM;
import org.eclipse.emf.cdo.server.internal.db.mapping.AbstractMappingStrategy;

import org.eclipse.net4j.db.DBException;
import org.eclipse.net4j.db.DBUtil;
import org.eclipse.net4j.db.IDBAdapter;
import org.eclipse.net4j.db.ddl.IDBField;
import org.eclipse.net4j.db.ddl.IDBTable;
import org.eclipse.net4j.util.lifecycle.LifecycleUtil;
import org.eclipse.net4j.util.om.trace.ContextTracer;

import org.eclipse.emf.ecore.EClass;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Collection;

/**
 * * This abstract base class refines {@link AbstractMappingStrategy} by implementing aspects common to horizontal
 * mapping strategies -- namely:
 * <ul>
 * <li>object type cache (table cdo_objects)
 * <li>resource query handling
 * </ul>
 * 
 * @author Eike Stepper
 * @since 2.0
 */
public abstract class AbstractHorizontalMappingStrategy extends AbstractMappingStrategy
{
  private static final ContextTracer TRACER = new ContextTracer(OM.DEBUG, AbstractHorizontalMappingStrategy.class);

  /**
   * The associated object type cache.
   */
  private IObjectTypeCache objectTypeCache;

  public CDOClassifierRef readObjectType(IDBStoreAccessor accessor, CDOID id)
  {
    return objectTypeCache.getObjectType(accessor, id);
  }

  public void putObjectType(IDBStoreAccessor accessor, CDOID id, EClass type)
  {
    objectTypeCache.putObjectType(accessor, id, type);
  }

  public long[] repairAfterCrash(IDBAdapter dbAdapter, Connection connection)
  {
    long minLocalID = getMinLocalID(connection);
    long maxID = objectTypeCache.getMaxID(connection);
    long maxTime = getMaxTime(connection);

    long[] result = { minLocalID, maxID, maxTime };
    return result;
  }

  public void queryResources(IDBStoreAccessor accessor, QueryResourcesContext context)
  {
    // only support timestamp in audit mode
    if (context.getTimeStamp() != CDORevision.UNSPECIFIED_DATE && !hasAuditSupport())
    {
      throw new UnsupportedOperationException("Mapping Strategy does not support audits."); //$NON-NLS-1$
    }

    EresourcePackage resourcesPackage = EresourcePackage.eINSTANCE;

    // first query folders
    boolean shallContinue = queryResources(accessor, getClassMapping(resourcesPackage.getCDOResourceFolder()), context);

    // not enough results? -> query resources
    if (shallContinue)
    {
      queryResources(accessor, getClassMapping(resourcesPackage.getCDOResource()), context);
    }
  }

  @Override
  protected Collection<EClass> getClassesWithObjectInfo()
  {
    return getClassMappings().keySet();
  }

  @Override
  protected void doActivate() throws Exception
  {
    super.doActivate();
    if (objectTypeCache == null)
    {
      objectTypeCache = createObjectTypeCache();
      LifecycleUtil.activate(objectTypeCache);
    }
  }

  @Override
  protected void doDeactivate() throws Exception
  {
    LifecycleUtil.deactivate(objectTypeCache);
    objectTypeCache = null;
    super.doDeactivate();
  }

  private IObjectTypeCache createObjectTypeCache()
  {
    ObjectTypeCache cache = new ObjectTypeCache();
    cache.setMappingStrategy(this);
    return cache;
  }

  /**
   * This is an intermediate implementation. It should be changed after classmappings support a general way to implement
   * queries ...
   * 
   * @param accessor
   *          the accessor to use.
   * @param classMapping
   *          the class mapping of a class instanceof {@link CDOResourceNode} which should be queried.
   * @param context
   *          the query context containing the parameters and the result.
   * @return <code>true</code> if result context is not yet full and query should continue false, if result context is
   *         full and query should stop.
   */
  private boolean queryResources(IDBStoreAccessor accessor, IClassMapping classMapping, QueryResourcesContext context)
  {
    PreparedStatement stmt = null;
    ResultSet rset = null;

    CDOID folderID = context.getFolderID();
    String name = context.getName();
    boolean exactMatch = context.exactMatch();

    try
    {
      stmt = classMapping.createResourceQueryStatement(accessor, folderID, name, exactMatch, context);
      rset = stmt.executeQuery();

      while (rset.next())
      {
        long longID = rset.getLong(1);
        if (TRACER.isEnabled())
        {
          TRACER.trace("Resource query returned ID " + longID); //$NON-NLS-1$
        }

        CDOID id = CDOIDUtil.createLong(longID);
        if (!context.addResource(id))
        {
          // No more results allowed
          return false; // don't continue
        }
      }

      return true; // continue with other results
    }
    catch (SQLException ex)
    {
      throw new DBException(ex);
    }
    finally
    {
      DBUtil.close(rset);
      accessor.getStatementCache().releasePreparedStatement(stmt);
    }
  }

  private long getMinLocalID(Connection connection)
  {
    long min = Long.MAX_VALUE;
    if (getStore().getRepository().isSupportingBranches())
    {
      for (IClassMapping classMapping : getClassMappings().values())
      {
        IDBTable table = classMapping.getDBTables().get(0);
        IDBField field = table.getField(CDODBSchema.ATTRIBUTES_ID);
        long id = DBUtil.selectMinimumLong(connection, field, "0>" + CDODBSchema.ATTRIBUTES_BRANCH); //$NON-NLS-1$
        if (id < min)
        {
          min = id;
        }
      }
    }

    return min;
  }

  private long getMaxTime(Connection connection)
  {
    long max = CDORevision.UNSPECIFIED_DATE;
    for (IClassMapping classMapping : getClassMappings().values())
    {
      IDBTable table = classMapping.getDBTables().get(0);
      IDBField field = table.getField(CDODBSchema.ATTRIBUTES_CREATED);
      long timeStamp = DBUtil.selectMaximumLong(connection, field);
      if (timeStamp > max)
      {
        max = timeStamp;
      }
    }

    return max;
  }
}
