/*******************************************************************************
 * Copyright (c) 2009, 2010 Oracle. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0, which accompanies this distribution
 * and is available at http://www.eclipse.org/legal/epl-v10.html.
 * 
 * Contributors:
 *     Oracle - initial API and implementation
 ******************************************************************************/
package org.eclipse.jpt.core.internal.context.java;

import java.util.Iterator;
import java.util.List;
import org.eclipse.jdt.core.dom.CompilationUnit;
import org.eclipse.jpt.core.context.Entity;
import org.eclipse.jpt.core.context.JoiningStrategy;
import org.eclipse.jpt.core.context.RelationshipMapping;
import org.eclipse.jpt.core.context.TypeMapping;
import org.eclipse.jpt.core.context.java.JavaAssociationOverride;
import org.eclipse.jpt.core.context.java.JavaAssociationOverrideRelationshipReference;
import org.eclipse.jpt.core.context.java.JavaJoinColumnInAssociationOverrideJoiningStrategy;
import org.eclipse.jpt.core.resource.java.AssociationOverrideAnnotation;
import org.eclipse.jpt.core.utility.TextRange;
import org.eclipse.jpt.utility.Filter;
import org.eclipse.wst.validation.internal.provisional.core.IMessage;
import org.eclipse.wst.validation.internal.provisional.core.IReporter;

public abstract class AbstractJavaAssociationOverrideRelationshipReference extends AbstractJavaJpaContextNode
	implements JavaAssociationOverrideRelationshipReference
{

	// cache the strategy for property change notification
	protected JoiningStrategy cachedPredominantJoiningStrategy;

	protected final JavaJoinColumnInAssociationOverrideJoiningStrategy joinColumnJoiningStrategy;

	protected AbstractJavaAssociationOverrideRelationshipReference(JavaAssociationOverride parent) {
		super(parent);
		this.joinColumnJoiningStrategy = buildJoinColumnJoiningStrategy();
	}
	
	protected JavaJoinColumnInAssociationOverrideJoiningStrategy buildJoinColumnJoiningStrategy() {
		return new GenericJavaJoinColumnInAssociationOverrideJoiningStrategy(this);
	}
	
	@Override
	public JavaAssociationOverride getParent() {
		return (JavaAssociationOverride) super.getParent();
	}
	
	public JavaAssociationOverride getAssociationOverride() {
		return getParent();
	}
	
	public boolean isOverridableAssociation() {
		return false;
	}
	
	public TypeMapping getTypeMapping() {
		return getAssociationOverride().getOwner().getTypeMapping();
	}

	public Entity getEntity() {
		TypeMapping typeMapping = getTypeMapping();
		return (typeMapping instanceof Entity) ? (Entity) typeMapping : null;
	}

	public boolean isTargetForeignKeyRelationship() {
		return false;
	}

	// **************** RelationshipMapping implementation *******************************
	
	public boolean isParentVirtual() {
		return getAssociationOverride().isVirtual();
	}
	
	// **************** predominant joining strategy ***************************
	
	public JoiningStrategy getPredominantJoiningStrategy() {
		return this.cachedPredominantJoiningStrategy;
	}
	
	protected void setPredominantJoiningStrategy(JoiningStrategy newJoiningStrategy) {
		JoiningStrategy oldJoiningStrategy = this.cachedPredominantJoiningStrategy;
		this.cachedPredominantJoiningStrategy = newJoiningStrategy;
		firePropertyChanged(PREDOMINANT_JOINING_STRATEGY_PROPERTY, oldJoiningStrategy, newJoiningStrategy);
	}

	public void initialize(AssociationOverrideAnnotation associationOverride) {
		initializeJoiningStrategies(associationOverride);
		this.cachedPredominantJoiningStrategy = calculatePredominantJoiningStrategy();
	}		
	
	protected void initializeJoiningStrategies(AssociationOverrideAnnotation associationOverride) {
		this.joinColumnJoiningStrategy.initialize(associationOverride);		
	}

	public void update(AssociationOverrideAnnotation associationOverride) {
		updateJoiningStrategies(associationOverride);
		setPredominantJoiningStrategy(calculatePredominantJoiningStrategy());
	}	
		
	protected void updateJoiningStrategies(AssociationOverrideAnnotation associationOverride) {
		this.joinColumnJoiningStrategy.update(associationOverride);
	}
	
	protected abstract JoiningStrategy calculatePredominantJoiningStrategy();

	@Override
	public Iterator<String> javaCompletionProposals(int pos, Filter<String> filter, CompilationUnit astRoot) {
		Iterator<String> result = super.javaCompletionProposals(pos, filter, astRoot);
		if (result != null) {
			return result;
		}
		result = this.joinColumnJoiningStrategy.javaCompletionProposals(pos, filter, astRoot);
		if (result != null) {
			return result;
		}
		return null;
	}

	
	// ********** validation **********

	@Override
	public void validate(List<IMessage> messages, IReporter reporter, CompilationUnit astRoot) {
		super.validate(messages, reporter, astRoot);
		this.joinColumnJoiningStrategy.validate(messages, reporter, astRoot);
	}

	public TextRange getValidationTextRange(CompilationUnit astRoot) {
		return getAssociationOverride().getValidationTextRange(astRoot);
	}

	// **************** join columns *******************************************
	

	public JavaJoinColumnInAssociationOverrideJoiningStrategy getJoinColumnJoiningStrategy() {
		return this.joinColumnJoiningStrategy;
	}
	
	public boolean usesJoinColumnJoiningStrategy() {
		return getPredominantJoiningStrategy() == this.joinColumnJoiningStrategy;
	}
	
	public void setJoinColumnJoiningStrategy() {
		this.joinColumnJoiningStrategy.addStrategy();
	}
	
	public void unsetJoinColumnJoiningStrategy() {
		this.joinColumnJoiningStrategy.removeStrategy();
	}
	
	public boolean mayHaveDefaultJoinColumn() {
		return false;
	}

	public RelationshipMapping getRelationshipMapping() {
		return getAssociationOverride().getOwner().getRelationshipMapping(getAssociationOverride().getName());
	}

	public boolean isOwnedBy(RelationshipMapping mapping) {
		return getRelationshipMapping().isOwnedBy(mapping);
	}

	public boolean isRelationshipOwner() {
		return getRelationshipMapping().isRelationshipOwner();
	}

}
