/*******************************************************************************
 * Copyright (c) 2007 Actuate Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *  Actuate Corporation  - initial API and implementation
 *******************************************************************************/

package org.eclipse.birt.chart.reportitem.ui;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.eclipse.birt.chart.exception.ChartException;
import org.eclipse.birt.chart.model.Chart;
import org.eclipse.birt.chart.model.ChartWithAxes;
import org.eclipse.birt.chart.model.DialChart;
import org.eclipse.birt.chart.model.component.Series;
import org.eclipse.birt.chart.model.data.Query;
import org.eclipse.birt.chart.model.data.SeriesDefinition;
import org.eclipse.birt.chart.model.data.impl.QueryImpl;
import org.eclipse.birt.chart.model.type.BubbleSeries;
import org.eclipse.birt.chart.model.type.DifferenceSeries;
import org.eclipse.birt.chart.model.type.GanttSeries;
import org.eclipse.birt.chart.model.type.StockSeries;
import org.eclipse.birt.chart.plugin.ChartEnginePlugin;
import org.eclipse.birt.chart.reportitem.ChartReportItemUtil;
import org.eclipse.birt.chart.reportitem.ChartXTabUtil;
import org.eclipse.birt.chart.reportitem.ui.dialogs.ChartColumnBindingDialog;
import org.eclipse.birt.chart.reportitem.ui.dialogs.ExtendedItemFilterDialog;
import org.eclipse.birt.chart.reportitem.ui.dialogs.ReportItemParametersDialog;
import org.eclipse.birt.chart.reportitem.ui.i18n.Messages;
import org.eclipse.birt.chart.reportitem.ui.views.attributes.provider.ChartCubeFilterHandleProvider;
import org.eclipse.birt.chart.reportitem.ui.views.attributes.provider.ChartFilterProviderDelegate;
import org.eclipse.birt.chart.ui.swt.ColorPalette;
import org.eclipse.birt.chart.ui.swt.ColumnBindingInfo;
import org.eclipse.birt.chart.ui.swt.CustomPreviewTable;
import org.eclipse.birt.chart.ui.swt.DataDefinitionTextManager;
import org.eclipse.birt.chart.ui.swt.DefaultChartDataSheet;
import org.eclipse.birt.chart.ui.swt.SimpleTextTransfer;
import org.eclipse.birt.chart.ui.swt.interfaces.IDataServiceProvider;
import org.eclipse.birt.chart.ui.swt.wizard.ChartAdapter;
import org.eclipse.birt.chart.ui.util.ChartHelpContextIds;
import org.eclipse.birt.chart.ui.util.ChartUIConstants;
import org.eclipse.birt.chart.ui.util.ChartUIUtil;
import org.eclipse.birt.chart.util.ChartUtil;
import org.eclipse.birt.core.data.ExpressionUtil;
import org.eclipse.birt.core.exception.BirtException;
import org.eclipse.birt.core.ui.frameworks.taskwizard.WizardBase;
import org.eclipse.birt.report.designer.internal.ui.dialogs.ExpressionFilter;
import org.eclipse.birt.report.designer.internal.ui.views.ViewsTreeProvider;
import org.eclipse.birt.report.designer.internal.ui.views.attributes.provider.AbstractFilterHandleProvider;
import org.eclipse.birt.report.designer.ui.actions.NewDataSetAction;
import org.eclipse.birt.report.designer.ui.cubebuilder.action.NewCubeAction;
import org.eclipse.birt.report.designer.ui.dialogs.ColumnBindingDialog;
import org.eclipse.birt.report.designer.ui.dialogs.ExpressionProvider;
import org.eclipse.birt.report.designer.util.DEUtil;
import org.eclipse.birt.report.item.crosstab.core.de.CrosstabReportItemHandle;
import org.eclipse.birt.report.model.api.ComputedColumnHandle;
import org.eclipse.birt.report.model.api.DataSetHandle;
import org.eclipse.birt.report.model.api.ExtendedItemHandle;
import org.eclipse.birt.report.model.api.ReportItemHandle;
import org.eclipse.birt.report.model.api.metadata.IClassInfo;
import org.eclipse.birt.report.model.api.olap.CubeHandle;
import org.eclipse.birt.report.model.api.olap.LevelHandle;
import org.eclipse.birt.report.model.api.olap.MeasureHandle;
import org.eclipse.emf.common.util.EList;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.action.IContributionItem;
import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.resource.JFaceResources;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.CCombo;
import org.eclipse.swt.custom.StackLayout;
import org.eclipse.swt.dnd.DND;
import org.eclipse.swt.dnd.DragSource;
import org.eclipse.swt.dnd.DragSourceEvent;
import org.eclipse.swt.dnd.DragSourceListener;
import org.eclipse.swt.dnd.Transfer;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.MenuItem;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.swt.widgets.TreeItem;
import org.eclipse.swt.widgets.Widget;
import org.eclipse.ui.PlatformUI;

/**
 * Data sheet implementation for Standard Chart
 */

public class StandardChartDataSheet extends DefaultChartDataSheet implements
		Listener
{

	final private ExtendedItemHandle itemHandle;
	final private ReportDataServiceProvider dataProvider;

	private Button btnInherit = null;
	private Button btnUseData = null;
	private boolean bIsInheritSelected = true;

	private CCombo cmbDataItems = null;

	private StackLayout stackLayout = null;
	private Composite cmpStack = null;
	private Composite cmpCubeTree = null;
	private Composite cmpDataPreview = null;

	private CustomPreviewTable tablePreview = null;
	private TreeViewer cubeTreeViewer = null;

	private Button btnFilters = null;
	private Button btnParameters = null;
	private Button btnBinding = null;
	private String currentData = null;

	public static final int SELECT_NONE = 1;
	public static final int SELECT_NEXT = 2;
	public static final int SELECT_DATA_SET = 4;
	public static final int SELECT_DATA_CUBE = 8;
	public static final int SELECT_REPORT_ITEM = 16;
	public static final int SELECT_NEW_DATASET = 32;
	public static final int SELECT_NEW_DATACUBE = 64;
	
	private final int iSupportedDataItems;

	private List<Integer> selectDataTypes = new ArrayList<Integer>( );

	public StandardChartDataSheet( ExtendedItemHandle itemHandle,
			ReportDataServiceProvider dataProvider, int iSupportedDataItems )
	{
		this.itemHandle = itemHandle;
		this.dataProvider = dataProvider;
		this.iSupportedDataItems = iSupportedDataItems;
	}

	public StandardChartDataSheet( ExtendedItemHandle itemHandle,
			ReportDataServiceProvider dataProvider )
	{
		this( itemHandle, dataProvider, 0 );
	}

	public Composite createActionButtons( Composite parent )
	{
		Composite composite = ChartUIUtil.createCompositeWrapper( parent );
		{
			composite.setLayoutData( new GridData( GridData.FILL_HORIZONTAL
					| GridData.VERTICAL_ALIGN_END ) );
		}

		btnFilters = new Button( composite, SWT.NONE );
		{
			btnFilters.setAlignment( SWT.CENTER );
			GridData gridData = new GridData( GridData.FILL_HORIZONTAL );
			btnFilters.setLayoutData( gridData );
			btnFilters.setText( Messages.getString( "StandardChartDataSheet.Label.Filters" ) ); //$NON-NLS-1$
			btnFilters.addListener( SWT.Selection, this );
		}

		btnParameters = new Button( composite, SWT.NONE );
		{
			btnParameters.setAlignment( SWT.CENTER );
			GridData gridData = new GridData( GridData.FILL_HORIZONTAL );
			btnParameters.setLayoutData( gridData );
			btnParameters.setText( Messages.getString( "StandardChartDataSheet.Label.Parameters" ) ); //$NON-NLS-1$
			btnParameters.addListener( SWT.Selection, this );
		}

		btnBinding = new Button( composite, SWT.NONE );
		{
			btnBinding.setAlignment( SWT.CENTER );
			GridData gridData = new GridData( GridData.FILL_HORIZONTAL );
			btnBinding.setLayoutData( gridData );
			btnBinding.setText( Messages.getString( "StandardChartDataSheet.Label.DataBinding" ) ); //$NON-NLS-1$
			btnBinding.addListener( SWT.Selection, this );
		}

		setEnabledForButtons( );
		return composite;
	}

	private void setEnabledForButtons( )
	{
		if ( isCubeMode( ) )
		{
			// getDataServiceProvider( ).checkState(
			// IDataServiceProvider.SHARE_QUERY )
			boolean disabled = getDataServiceProvider( ).isInXTabAggrCell( )
					|| getDataServiceProvider( ).isInXTabMeasureCell( );
			btnFilters.setEnabled( !disabled );
			btnBinding.setEnabled( getDataServiceProvider( ).isInvokingSupported( )
					|| getDataServiceProvider( ).isSharedBinding( ) );
			btnParameters.setEnabled( false );
		}
		else
		{
			 btnFilters.setEnabled( hasDataSet( ) );
	
			// Bugzilla#177704 Chart inheriting data from container doesn't
			// support parameters due to limitation in DtE
			btnParameters.setEnabled( getDataServiceProvider( ).getBoundDataSet( ) != null
					&& getDataServiceProvider( ).isInvokingSupported( ) );
			btnBinding.setEnabled( hasDataSet( )
					&& ( getDataServiceProvider( ).isInvokingSupported( ) || getDataServiceProvider( ).isSharedBinding( ) ) );
		}
	}

	private boolean hasDataSet( )
	{
		return getDataServiceProvider( ).getReportDataSet( ) != null
				|| getDataServiceProvider( ).getBoundDataSet( ) != null;
	}

	void fireEvent( Widget widget, int eventType )
	{
		Event event = new Event( );
		event.data = this;
		event.widget = widget;
		event.type = eventType;
		notifyListeners( event );
	}

	public Composite createDataDragSource( Composite parent )
	{
		cmpStack = new Composite( parent, SWT.NONE );
		cmpStack.setLayoutData( new GridData( GridData.FILL_BOTH ) );
		stackLayout = new StackLayout( );
		stackLayout.marginHeight = 0;
		stackLayout.marginWidth = 0;
		cmpStack.setLayout( stackLayout );

		cmpCubeTree = ChartUIUtil.createCompositeWrapper( cmpStack );
		cmpDataPreview = ChartUIUtil.createCompositeWrapper( cmpStack );

		Label label = new Label( cmpCubeTree, SWT.NONE );
		{
			label.setText( Messages.getString( "StandardChartDataSheet.Label.CubeTree" ) ); //$NON-NLS-1$
			label.setFont( JFaceResources.getBannerFont( ) );
		}

		if ( !dataProvider.isInXTabMeasureCell( )
				&& !dataProvider.isInMultiView( ) )
		{
			// No description if dnd is disabled
			Label description = new Label( cmpCubeTree, SWT.WRAP );
			{
				GridData gd = new GridData( GridData.FILL_HORIZONTAL );
				description.setLayoutData( gd );
				description.setText( Messages.getString( "StandardChartDataSheet.Label.DragCube" ) ); //$NON-NLS-1$
			}
		}

		cubeTreeViewer = new TreeViewer( cmpCubeTree, SWT.SINGLE
				| SWT.H_SCROLL
				| SWT.V_SCROLL
				| SWT.BORDER );
		cubeTreeViewer.getTree( )
				.setLayoutData( new GridData( GridData.FILL_BOTH ) );
		( (GridData) cubeTreeViewer.getTree( ).getLayoutData( ) ).heightHint = 120;
		ViewsTreeProvider provider = new ViewsTreeProvider( );
		cubeTreeViewer.setLabelProvider( provider );
		cubeTreeViewer.setContentProvider( provider );
		cubeTreeViewer.setInput( getCube( ) );

		final DragSource dragSource = new DragSource( cubeTreeViewer.getTree( ),
				DND.DROP_COPY );
		dragSource.setTransfer( new Transfer[]{
			SimpleTextTransfer.getInstance( )
		} );
		dragSource.addDragListener( new DragSourceListener( ) {

			private String text = null;

			public void dragFinished( DragSourceEvent event )
			{
				// TODO Auto-generated method stub

			}

			public void dragSetData( DragSourceEvent event )
			{
				event.data = text;
			}

			public void dragStart( DragSourceEvent event )
			{
				text = createCubeExpression( );
				if ( text == null )
				{
					event.doit = false;
				}
			}
		} );

		cubeTreeViewer.getTree( ).addListener( SWT.MouseDown, new Listener( ) {

			public void handleEvent( Event event )
			{
				if ( event.button == 3 && event.widget instanceof Tree )
				{
					Tree tree = (Tree) event.widget;
					TreeItem treeItem = tree.getSelection( )[0];
					if ( treeItem.getData( ) instanceof LevelHandle
							|| treeItem.getData( ) instanceof MeasureHandle )
					{
						tree.setMenu( createMenuManager( treeItem.getData( ) ).createContextMenu( tree ) );
						// tree.getMenu( ).setVisible( true );
					}
					else
					{
						tree.setMenu( null );
					}
				}
			}
		} );

		label = new Label( cmpDataPreview, SWT.NONE );
		{
			label.setText( Messages.getString( "StandardChartDataSheet.Label.DataPreview" ) ); //$NON-NLS-1$
			label.setFont( JFaceResources.getBannerFont( ) );
		}

		if ( !dataProvider.isInXTabMeasureCell( )
				&& !dataProvider.isInMultiView( ) )
		{
			// No description if dnd is disabled
			Label description = new Label( cmpDataPreview, SWT.WRAP );
			{
				GridData gd = new GridData( GridData.FILL_HORIZONTAL );
				description.setLayoutData( gd );
				description.setText( Messages.getString( "StandardChartDataSheet.Label.ToBindADataColumn" ) ); //$NON-NLS-1$
			}
		}

		tablePreview = new CustomPreviewTable( cmpDataPreview, SWT.SINGLE
				| SWT.H_SCROLL
				| SWT.V_SCROLL
				| SWT.FULL_SELECTION );
		{
			GridData gridData = new GridData( GridData.FILL_BOTH );
			gridData.widthHint = 400;
			gridData.heightHint = 120;
			tablePreview.setLayoutData( gridData );
			tablePreview.setHeaderAlignment( SWT.LEFT );
			tablePreview.addListener( CustomPreviewTable.MOUSE_RIGHT_CLICK_TYPE,
					this );
		}

		updateDragDataSource( );
		return cmpStack;
	}

	private void updateDragDataSource( )
	{
		if ( isCubeMode( ) )
		{
			stackLayout.topControl = cmpCubeTree;
			cubeTreeViewer.setInput( getCube( ) );
		}
		else
		{
			stackLayout.topControl = cmpDataPreview;
			refreshTablePreview( );
		}
		cmpStack.layout( );
	}

	public Composite createDataSelector( Composite parent )
	{
		Composite cmpDataSet = ChartUIUtil.createCompositeWrapper( parent );
		{
			cmpDataSet.setLayoutData( new GridData( GridData.FILL_HORIZONTAL ) );
		}

		Label label = new Label( cmpDataSet, SWT.NONE );
		{
			label.setText( Messages.getString( "StandardChartDataSheet.Label.SelectDataSet" ) ); //$NON-NLS-1$
			label.setFont( JFaceResources.getBannerFont( ) );
		}

		Composite cmpDetail = new Composite( cmpDataSet, SWT.NONE );
		{
			GridLayout gridLayout = new GridLayout( 3, false );
			gridLayout.marginWidth = 10;
			gridLayout.marginHeight = 0;
			cmpDetail.setLayout( gridLayout );
			cmpDetail.setLayoutData( new GridData( GridData.FILL_HORIZONTAL ) );
		}

		Composite compRadios = ChartUIUtil.createCompositeWrapper( cmpDetail );
		{
			GridData gd = new GridData( );
			gd.verticalSpan = 2;
			compRadios.setLayoutData( gd );
		}

		btnInherit = new Button( compRadios, SWT.RADIO );
		btnInherit.setText( Messages.getString( "StandardChartDataSheet.Label.UseReportData" ) ); //$NON-NLS-1$
		btnInherit.addListener( SWT.Selection, this );

		btnUseData = new Button( compRadios, SWT.RADIO );
		btnUseData.setText( Messages.getString( "StandardChartDataSheet.Label.UseDataSet" ) ); //$NON-NLS-1$
		btnUseData.addListener( SWT.Selection, this );

		new Label( cmpDetail, SWT.NONE );
		new Label( cmpDetail, SWT.NONE );

		cmbDataItems = new CCombo( cmpDetail, SWT.DROP_DOWN
				| SWT.READ_ONLY
				| SWT.BORDER );
		cmbDataItems.setLayoutData( new GridData( GridData.FILL_HORIZONTAL ) );
		cmbDataItems.addListener( SWT.Selection, this );

		initDataSelector( );
		updatePredefinedQueries( );
		return cmpDataSet;
	}

	int invokeNewDataSet( )
	{
		IAction action = new NewDataSetAction( );
		PlatformUI.getWorkbench( ).getHelpSystem( ).setHelp( action,
				ChartHelpContextIds.DIALOG_NEW_DATA_SET );
		action.run( );
		// Due to the limitation of the action execution, always return ok
		return Window.OK;
	}

	int invokeEditFilter( )
	{
		ExtendedItemHandle handle = getItemHandle( );
		handle.getModuleHandle( ).getCommandStack( ).startTrans( null );
		ExtendedItemFilterDialog page = new ExtendedItemFilterDialog( handle );

		AbstractFilterHandleProvider provider = ChartFilterProviderDelegate.createFilterProvider( handle,
				handle );
		if ( provider instanceof ChartCubeFilterHandleProvider )
		{
			( (ChartCubeFilterHandleProvider) provider ).setContext( getContext( ) );
		}
		page.setFilterHandleProvider( provider );

		int openStatus = page.open( );
		if ( openStatus == Window.OK )
		{
			handle.getModuleHandle( ).getCommandStack( ).commit( );
		}
		else
		{
			handle.getModuleHandle( ).getCommandStack( ).rollback( );
		}

		return openStatus;
	}

	int invokeEditParameter( )
	{
		ReportItemParametersDialog page = new ReportItemParametersDialog( getItemHandle( ) );
		return page.open( );
	}

	int invokeDataBinding( )
	{
		Shell shell = new Shell( Display.getDefault( ), SWT.DIALOG_TRIM
				| SWT.RESIZE
				| SWT.APPLICATION_MODAL );
		// #194163: Do not register CS help in chart since it's registered in
		// super column binding dialog.
		// ChartUIUtil.bindHelp( shell,
		// ChartHelpContextIds.DIALOG_DATA_SET_COLUMN_BINDING );
		ExtendedItemHandle handle = getItemHandle( );

		handle.getModuleHandle( ).getCommandStack( ).startTrans( null );
		ColumnBindingDialog page = new ChartColumnBindingDialog( handle,
				shell,
				getContext( ) );

		ExpressionProvider ep = new ExpressionProvider( getItemHandle( ) );
		ep.addFilter( new ExpressionFilter( ) {

			public boolean select( Object parentElement, Object element )
			{
				// Remove unsupported expression. See bugzilla#132768
				return !( parentElement.equals( ExpressionProvider.BIRT_OBJECTS )
						&& element instanceof IClassInfo && ( (IClassInfo) element ).getName( )
						.equals( "Total" ) ); //$NON-NLS-1$
			}
		} );
		page.setExpressionProvider( ep );

		// Make all bindings under share binding case read-only.
		( (ChartColumnBindingDialog) page ).setReadOnly( getDataServiceProvider( ).isSharedBinding( )
				|| getDataServiceProvider( ).isInheritanceOnly( ) );

		int openStatus = page.open( );
		if ( openStatus == Window.OK )
		{
			handle.getModuleHandle( ).getCommandStack( ).commit( );
			updatePredefinedQueries( );
		}
		else
		{
			handle.getModuleHandle( ).getCommandStack( ).rollback( );
		}

		return openStatus;
	}

	private void initDataSelector( )
	{
		// create Combo items
		cmbDataItems.setItems( createDataComboItems( ) );
		cmbDataItems.setVisibleItemCount( cmbDataItems.getItemCount( ) );

		// Select report item reference
		// Since handle may have data set or data cube besides reference, always
		// check reference first
		String sItemRef = getDataServiceProvider( ).getReportItemReference( );
		if ( sItemRef != null )
		{
			btnUseData.setSelection( true );
			bIsInheritSelected = false;
			cmbDataItems.setText( sItemRef );
			currentData = sItemRef;
			return;
		}

		// Select data set
		String sDataSet = getDataServiceProvider( ).getBoundDataSet( );
		if ( sDataSet != null && !getDataServiceProvider( ).isInheritanceOnly( ) )
		{
			btnUseData.setSelection( true );
			bIsInheritSelected = false;
			cmbDataItems.setText( sDataSet );
			currentData = sDataSet;
			if ( sDataSet != null )
			{
				switchDataTable( );
			}
			return;
		}

		// Select data cube
		String sDataCube = getDataServiceProvider( ).getDataCube( );
		if ( sDataCube != null
				&& !getDataServiceProvider( ).isInheritanceOnly( ) )
		{
			btnUseData.setSelection( true );
			bIsInheritSelected = false;
			cmbDataItems.setText( sDataCube );
			currentData = sDataCube;
			return;
		}

		btnInherit.setSelection( true );
		bIsInheritSelected = true;
		if ( getDataServiceProvider( ).isInheritanceOnly( ) )
		{
			btnUseData.setSelection( false );
			btnUseData.setEnabled( false );
		}
		cmbDataItems.select( 0 );
		currentData = null;
		cmbDataItems.setEnabled( false );
		// Initializes column bindings from container
		getDataServiceProvider( ).setDataSet( null );
		String reportDataSet = getDataServiceProvider( ).getReportDataSet( );
		if ( reportDataSet != null )
		{
			switchDataTable( );
		}

		// select reference item
		// selectItemRef( );
		// if ( cmbReferences.getSelectionIndex( ) > 0 )
		// {
		// cmbDataSet.setEnabled( false );
		// btnUseReference.setSelection( true );
		// btnUseReportData.setSelection( false );
		// btnUseDataSet.setSelection( false );
		// }
		// else
		// {
		// cmbReferences.setEnabled( false );
		// }
		//
		// String dataCube = getDataServiceProvider( ).getDataCube( );
		// if ( dataCube != null )
		// {
		// cmbCubes.setText( dataCube );
		// btnUseReference.setSelection( false );
		// btnUseReportData.setSelection( false );
		// btnUseDataSet.setSelection( false );
		// btnUseCubes.setSelection( true );
		// }
		// else
		// {
		// cmbCubes.select( 0 );
		// }
	}

	public void handleEvent( Event event )
	{
		// Right click to display the menu. Menu display by clicking
		// application key is triggered by os, so do nothing.
		if ( event.type == CustomPreviewTable.MOUSE_RIGHT_CLICK_TYPE )
		{
			if ( getDataServiceProvider( ).getBoundDataSet( ) != null
					|| getDataServiceProvider( ).getReportDataSet( ) != null )
			{
				if ( event.widget instanceof Button )
				{
					Button header = (Button) event.widget;

					// Bind context menu to each header button
					if ( header.getMenu( ) == null )
					{
						header.setMenu( createMenuManager( event.data ).createContextMenu( tablePreview ) );
					}

					header.getMenu( ).setVisible( true );
				}
			}

		}
		else if ( event.type == SWT.Selection )
		{
			if ( event.widget instanceof MenuItem )
			{
				MenuItem item = (MenuItem) event.widget;
				IAction action = (IAction) item.getData( );
				action.setChecked( !action.isChecked( ) );
				action.run( );
			}
			else if ( event.widget == btnFilters )
			{
				if ( invokeEditFilter( ) == Window.OK )
				{
					refreshTablePreview( );
					// Update preview via event
					fireEvent( btnFilters, EVENT_PREVIEW );
				}
			}
			else if ( event.widget == btnParameters )
			{
				if ( invokeEditParameter( ) == Window.OK )
				{
					refreshTablePreview( );
					// Update preview via event
					fireEvent( btnParameters, EVENT_PREVIEW );
				}
			}
			else if ( event.widget == btnBinding )
			{
				if ( invokeDataBinding( ) == Window.OK )
				{
					refreshTablePreview( );
					// Update preview via event
					fireEvent( btnBinding, EVENT_PREVIEW );
				}
			}

			try
			{
				if ( event.widget == btnInherit )
				{
					ColorPalette.getInstance( ).restore( );

					// Skip when selection is false
					if ( !btnInherit.getSelection( ) )
					{
						return;
					}

					// Avoid duplicate loading data set.
					if ( bIsInheritSelected )
					{
						return;
					}

					bIsInheritSelected = true;

					getDataServiceProvider( ).setReportItemReference( null );
					getDataServiceProvider( ).setDataSet( null );
					switchDataSet( null );

					cmbDataItems.select( 0 );
					currentData = null;
					cmbDataItems.setEnabled( false );
					setEnabledForButtons( );
					updateDragDataSource( );
					updatePredefinedQueries( );
				}
				else if ( event.widget == btnUseData )
				{
					// Skip when selection is false
					if ( !btnUseData.getSelection( ) )
					{
						return;
					}

					// Avoid duplicate loading data set.
					if ( !bIsInheritSelected )
					{
						return;
					}

					bIsInheritSelected = false;

					getDataServiceProvider( ).setReportItemReference( null );
					getDataServiceProvider( ).setDataSet( null );
					selectDataSet( );
					cmbDataItems.setEnabled( true );
					setEnabledForButtons( );
					updateDragDataSource( );
					updatePredefinedQueries( );
				}
				else if ( event.widget == cmbDataItems )
				{
					ColorPalette.getInstance( ).restore( );
					int selectedIndex = cmbDataItems.getSelectionIndex( );
					Integer selectState = selectDataTypes.get( selectedIndex );
					switch ( selectState.intValue( ) )
					{
						case SELECT_NONE :
							// Inherit data from container
							btnInherit.setSelection( true );
							btnUseData.setSelection( false );
							btnInherit.notifyListeners( SWT.Selection,
									new Event( ) );
							break;
						case SELECT_NEXT :
							selectedIndex++;
							selectState = selectDataTypes.get( selectedIndex );
							cmbDataItems.select( selectedIndex );
							break;
					}
					switch ( selectState.intValue( ) )
					{
						case SELECT_DATA_SET :
							if ( getDataServiceProvider( ).getReportItemReference( ) == null
									&& getDataServiceProvider( ).getBoundDataSet( ) != null
									&& getDataServiceProvider( ).getBoundDataSet( )
											.equals( cmbDataItems.getText( ) ) )
							{
								return;
							}
							getDataServiceProvider( ).setDataSet( cmbDataItems.getText( ) );
							currentData = cmbDataItems.getText( );
							switchDataSet( cmbDataItems.getText( ) );
							setEnabledForButtons( );
							updateDragDataSource( );
							break;
						case SELECT_DATA_CUBE :
							getDataServiceProvider( ).setDataCube( cmbDataItems.getText( ) );
							currentData = cmbDataItems.getText( );
							updateDragDataSource( );
							setEnabledForButtons( );
							// Update preview via event
							DataDefinitionTextManager.getInstance( )
									.refreshAll( );
							fireEvent( tablePreview, EVENT_PREVIEW );
							break;
						case SELECT_REPORT_ITEM :
							if ( cmbDataItems.getText( )
									.equals( getDataServiceProvider( ).getReportItemReference( ) ) )
							{
								return;
							}
							getDataServiceProvider( ).setReportItemReference( cmbDataItems.getText( ) );
							
							// TED 10163
							// Following calls will revise chart model for
							// report item sharing case, in older version of
							// chart, it is allowed to set grouping on category
							// series when sharing report item, but now it isn't
							// allowed, so this calls will revise chart model to
							// remove category series grouping flag for the
							// case.
							ChartReportItemUtil.reviseChartModel( ChartReportItemUtil.REVISE_REFERENCE_REPORT_ITEM,
									this.getContext( ).getModel( ),
									itemHandle );
							
							currentData = cmbDataItems.getText( );
							// selectDataSet( );
							// switchDataSet( cmbDataItems.getText( ) );

							// Update preview via event
							DataDefinitionTextManager.getInstance( )
									.refreshAll( );
							fireEvent( tablePreview, EVENT_PREVIEW );

							setEnabledForButtons( );
							updateDragDataSource( );
							break;
						case SELECT_NEW_DATASET :
							// Bring up the dialog to create a dataset
							int result = invokeNewDataSet( );
							if ( result == Window.CANCEL )
							{
								return;
							}

							cmbDataItems.removeAll( );
							cmbDataItems.setItems( createDataComboItems( ) );
							cmbDataItems.setVisibleItemCount( cmbDataItems.getItemCount( ) );
							if ( currentData == null )
							{
								cmbDataItems.select( 0 );
							}
							else
							{
								cmbDataItems.setText( currentData );
							}
							break;
						case SELECT_NEW_DATACUBE :
							if ( getDataServiceProvider( ).getAllDataSets( ).length == 0 )
							{
								invokeNewDataSet( );
							}
							if ( getDataServiceProvider( ).getAllDataSets( ).length != 0 )
							{
								new NewCubeAction( ).run( );
							}

							cmbDataItems.removeAll( );
							cmbDataItems.setItems( createDataComboItems( ) );
							cmbDataItems.setVisibleItemCount( cmbDataItems.getItemCount( ) );
							if ( currentData == null )
							{
								cmbDataItems.select( 0 );
							}
							else
							{
								cmbDataItems.setText( currentData );
							}
							break;
					}
					updatePredefinedQueries( );
				}
				// else if ( event.widget == btnUseReference )
				// {
				// // Skip when selection is false
				// if ( !btnUseReference.getSelection( ) )
				// {
				// return;
				// }
				// cmbDataSet.setEnabled( false );
				// cmbReferences.setEnabled( true );
				// selectItemRef( );
				// setEnabledForButtons( );
				// }
				// else if ( event.widget == cmbReferences )
				// {
				// if ( cmbReferences.getSelectionIndex( ) == 0 )
				// {
				// if ( getDataServiceProvider( ).getReportItemReference( ) ==
				// null )
				// {
				// return;
				// }
				// getDataServiceProvider( ).setReportItemReference( null );
				//
				// // Auto select the data set
				// selectDataSet( );
				// cmbReferences.setEnabled( false );
				// cmbDataSet.setEnabled( true );
				// btnUseReference.setSelection( false );
				// btnUseDataSet.setSelection( true );
				// }
				// else
				// {
				// if ( cmbReferences.getText( )
				// .equals( getDataServiceProvider( ).getReportItemReference( )
				// ) )
				// {
				// return;
				// }
				// getDataServiceProvider( ).setReportItemReference(
				// cmbReferences.getText( ) );
				// selectDataSet( );
				// }
				// switchDataSet( cmbDataSet.getText( ) );
				// setEnabledForButtons( );
				// }
			}
			catch ( ChartException e1 )
			{
				WizardBase.showException( e1.getLocalizedMessage( ) );
			}
		}
	}

	private void selectDataSet( )
	{
		String currentDS = getDataServiceProvider( ).getBoundDataSet( );
		if ( currentDS == null )
		{
			cmbDataItems.select( 0 );
			currentData = null;
		}
		else
		{
			cmbDataItems.setText( currentDS );
			currentData = currentDS;
		}
	}

	private void refreshTablePreview( )
	{
		if ( dataProvider.getDataSetFromHandle( ) == null )
		{
			return;
		}
		tablePreview.clearContents( );
		switchDataTable( );
		tablePreview.layout( );
	}

	private void switchDataSet( String datasetName ) throws ChartException
	{
		if ( isCubeMode( ) )
		{
			return;
		}
		try
		{
			// Clear old dataset and preview data
			tablePreview.clearContents( );

			// Try to get report data set
			if ( datasetName == null )
			{
				datasetName = getDataServiceProvider( ).getReportDataSet( );
			}

			if ( datasetName != null )
			{
				switchDataTable( );
			}
			else
			{
				tablePreview.createDummyTable( );
			}
			tablePreview.layout( );
		}
		catch ( Throwable t )
		{
			throw new ChartException( ChartEnginePlugin.ID,
					ChartException.DATA_BINDING,
					t );
		}

		DataDefinitionTextManager.getInstance( ).refreshAll( );
		// Update preview via event
		fireEvent( tablePreview, EVENT_PREVIEW );
	}

	/**
	 * Update column headers and data to table.
	 * 
	 * @param headers
	 * @param dataList
	 */
	private void updateTablePreview( final ColumnBindingInfo[] headers,
			final List dataList )
	{
		fireEvent( tablePreview, EVENT_QUERY );

		if ( tablePreview.isDisposed( ) )
		{
			return;
		}

		if ( headers == null )
		{
			tablePreview.setEnabled( false );
			tablePreview.createDummyTable( );
		}
		else
		{
			tablePreview.setEnabled( true );
			tablePreview.setColumns( headers );

			refreshTableColor( );

			// Add data value
			if ( dataList != null )
			{
				for ( Iterator iterator = dataList.iterator( ); iterator.hasNext( ); )
				{
					String[] dataRow = (String[]) iterator.next( );
					for ( int i = 0; i < dataRow.length; i++ )
					{
						tablePreview.addEntry( dataRow[i], i );
					}
				}
			}
		}
		tablePreview.layout( );
	}

	private void switchDataTable( )
	{
		if ( isCubeMode( ) )
		{
			return;
		}
		// 1. Create a runnable.
		Runnable runnable = new Runnable( ) {

			/*
			 * (non-Javadoc)
			 * 
			 * @see org.eclipse.jface.operation.IRunnableWithProgress#run(org.eclipse.core.runtime.IProgressMonitor)
			 */
			public void run( )
			{
				ColumnBindingInfo[] headers = null;
				List dataList = null;
				try
				{
					// Get header and data in other thread.
					headers = getDataServiceProvider( ).getPreviewHeadersInfo( );
					dataList = getDataServiceProvider( ).getPreviewData( );
					getDataServiceProvider( ).setPredefinedExpressions( headers );

					final ColumnBindingInfo[] headerInfo = headers;
					final List data = dataList;
					// Execute UI operation in UI thread.
					Display.getDefault( ).syncExec( new Runnable( ) {

						public void run( )
						{
							updateTablePreview( headerInfo, data );
						}
					} );
				}
				catch ( Exception e )
				{
					final ColumnBindingInfo[] headerInfo = headers;
					final List data = dataList;

					// Catch any exception.
					final String msg = e.getMessage( );
					Display.getDefault( ).syncExec( new Runnable( ) {

						/*
						 * (non-Javadoc)
						 * 
						 * @see java.lang.Runnable#run()
						 */
						public void run( )
						{
							// Still update table preview in here to ensure the
							// column headers of table preview can be updated
							// and user can select expression from table preview
							// even if there is no preview data.
							updateTablePreview( headerInfo, data );

							WizardBase.showException( msg );
						}
					} );
				}
			}
		};

		// 2. Run it.
		new Thread( runnable ).start( );
	}

	private void refreshTableColor( )
	{
		if ( isCubeMode( ) )
		{
			return;
		}
		// Reset column color
		for ( int i = 0; i < tablePreview.getColumnNumber( ); i++ )
		{
			tablePreview.setColumnColor( i,
					ColorPalette.getInstance( )
							.getColor( ExpressionUtil.createJSRowExpression( tablePreview.getColumnHeading( i ) ) ) );
		}
	}

	protected void manageColorAndQuery( Query query, String expr )
	{
		// If it's not used any more, remove color binding
		if ( DataDefinitionTextManager.getInstance( )
				.getNumberOfSameDataDefinition( query.getDefinition( ) ) == 0 )
		{
			ColorPalette.getInstance( ).retrieveColor( query.getDefinition( ) );
		}

		// Update query, if it is sharing binding case, the specified expression
		// will be converted and set to query, else directly set specified
		// expression to query.
		// DataDefinitionTextManager.getInstance( ).updateQuery( query, expr );
		query.setDefinition( getActualExpression( expr ) );

		DataDefinitionTextManager.getInstance( ).updateText( query );
		// Reset table column color
		refreshTableColor( );
		// Refresh all data definition text
		DataDefinitionTextManager.getInstance( ).refreshAll( );
	}

	/**
	 * Returns actual expression for common and sharing query case.
	 * 
	 * @param query
	 * @param expr
	 * @return
	 */
	private String getActualExpression( String expr )
	{
		if ( !dataProvider.checkState( IDataServiceProvider.SHARE_QUERY ) )
		{
			return expr;
		}

		// Convert to actual expression.
		Object obj = tablePreview.getCurrentColumnHeadObject( );
		if ( obj instanceof ColumnBindingInfo )
		{
			ColumnBindingInfo cbi = (ColumnBindingInfo) obj;
			int type = cbi.getColumnType( );
			if ( type == ColumnBindingInfo.GROUP_COLUMN
					|| type == ColumnBindingInfo.AGGREGATE_COLUMN )
			{
				return cbi.getExpression( );
			}
		}

		return expr;
	}

	class CategoryXAxisAction extends Action
	{

		Query query;
		String expr;

		CategoryXAxisAction( String expr )
		{
			super( getBaseSeriesTitle( getChartModel( ) ) );
			this.query = ( (Query) ( (SeriesDefinition) ChartUIUtil.getBaseSeriesDefinitions( getChartModel( ) )
					.get( 0 ) ).getDesignTimeSeries( )
					.getDataDefinition( )
					.get( 0 ) );
			this.expr = expr;

			setEnabled( DataDefinitionTextManager.getInstance( )
					.isAcceptableExpression( query,
							expr,
							dataProvider.isSharedBinding( ) ) );
		}

		public void run( )
		{
			manageColorAndQuery( query, expr );
		}
	}

	class GroupYSeriesAction extends Action
	{

		Query query;
		String expr;

		GroupYSeriesAction( Query query, String expr )
		{
			super( getGroupSeriesTitle( getChartModel( ) ) );
			this.query = query;
			this.expr = expr;

			setEnabled( DataDefinitionTextManager.getInstance( )
					.isAcceptableExpression( query,
							expr,
							dataProvider.isSharedBinding( ) ) );
		}

		public void run( )
		{
			// Use the first group, and copy to the all groups
			ChartAdapter.beginIgnoreNotifications( );
			ChartUIUtil.setAllGroupingQueryExceptFirst( getChartModel( ), expr );
			ChartAdapter.endIgnoreNotifications( );

			manageColorAndQuery( query, expr );
		}
	}

	class ValueYSeriesAction extends Action
	{

		Query query;
		String expr;

		ValueYSeriesAction( Query query, String expr )
		{
			super( getOrthogonalSeriesTitle( getChartModel( ) ) );
			this.query = query;
			this.expr = expr;

			// Grouping expressions can't be set on value series.
			boolean enabled = true;
			if ( dataProvider.checkState( IDataServiceProvider.SHARE_QUERY ) )
			{
				Object obj = tablePreview.getCurrentColumnHeadObject( );
				if ( obj instanceof ColumnBindingInfo
						&& ( (ColumnBindingInfo) obj ).getColumnType( ) == ColumnBindingInfo.GROUP_COLUMN )
				{
					enabled = false;
				}
			}

			setEnabled( enabled );
		}

		public void run( )
		{
			manageColorAndQuery( query, expr );
		}
	}

	class HeaderShowAction extends Action
	{

		HeaderShowAction( String header )
		{
			super( header );
			setEnabled( false );
		}
	}

	ExtendedItemHandle getItemHandle( )
	{
		return this.itemHandle;
	}

	ReportDataServiceProvider getDataServiceProvider( )
	{
		return this.dataProvider;
	}

	protected List<Object> getActionsForTableHead( String expr )
	{
		List<Object> actions = new ArrayList<Object>( 3 );
		actions.add( getBaseSeriesMenu( getChartModel( ), expr ) );
		actions.add( getOrthogonalSeriesMenu( getChartModel( ), expr ) );
		actions.add( getGroupSeriesMenu( getChartModel( ), expr ) );
		return actions;
	}

	private MenuManager createMenuManager( final Object data )
	{
		MenuManager menuManager = new MenuManager( );
		menuManager.setRemoveAllWhenShown( true );
		menuManager.addMenuListener( new IMenuListener( ) {

			public void menuAboutToShow( IMenuManager manager )
			{
				if ( data instanceof Integer )
				{
					// Menu for table
					addMenu( manager,
							new HeaderShowAction( tablePreview.getCurrentColumnHeading( ) ) );
					String expr = ExpressionUtil.createJSRowExpression( tablePreview.getCurrentColumnHeading( ) );
					List<Object> actions = getActionsForTableHead( expr );
					for ( Object act : actions )
					{
						addMenu( manager, act );
					}
				}
				else if ( data instanceof MeasureHandle )
				{
					// Menu for Measure
					String expr = createCubeExpression( );
					if ( expr != null )
					{
						addMenu( manager,
								getOrthogonalSeriesMenu( getChartModel( ), expr ) );
					}
				}
				else if ( data instanceof LevelHandle )
				{
					// Menu for Level
					String expr = createCubeExpression( );
					if ( expr != null )
					{
						// bug#220724
						if ( ( (Boolean) dataProvider.checkData( ChartUIConstants.QUERY_CATEGORY,
								expr ) ).booleanValue( ) )
						{
							addMenu( manager,
									getBaseSeriesMenu( getChartModel( ),
								expr ) );
						}
						
						if ( dataProvider.checkState( IDataServiceProvider.MULTI_CUBE_DIMENSIONS )
								&& ( (Boolean) dataProvider.checkData( ChartUIConstants.QUERY_OPTIONAL,
										expr ) ).booleanValue( ) )
						{
							addMenu( manager,
									getGroupSeriesMenu( getChartModel( ),
								expr ) );
						}
						
					}
				}
			}

			private void addMenu( IMenuManager manager, Object item )
			{
				if ( item instanceof IAction )
				{
					manager.add( (IAction) item );
				}
				else if ( item instanceof IContributionItem )
				{
					manager.add( (IContributionItem) item );
				}

				// Do not allow customized query in xtab
				if ( getDataServiceProvider( ).isPartChart( ) )
				{
					if ( item instanceof IAction )
					{
						( (IAction) item ).setEnabled( false );
					}
				}
			}
		} );
		return menuManager;
	}

	private Object getBaseSeriesMenu( Chart chart, String expr )
	{
		EList sds = ChartUIUtil.getBaseSeriesDefinitions( chart );
		if ( sds.size( ) == 1 )
		{
			return new CategoryXAxisAction( expr );
		}
		return null;
	}

	private Object getGroupSeriesMenu( Chart chart, String expr )
	{
		IMenuManager topManager = new MenuManager( getGroupSeriesTitle( getChartModel( ) ) );
		int axisNum = ChartUIUtil.getOrthogonalAxisNumber( chart );
		for ( int axisIndex = 0; axisIndex < axisNum; axisIndex++ )
		{
			List sds = ChartUIUtil.getOrthogonalSeriesDefinitions( chart,
					axisIndex );
			for ( int i = 0; i < sds.size( ); i++ )
			{
				SeriesDefinition sd = (SeriesDefinition) sds.get( i );
				IAction action = new GroupYSeriesAction( sd.getQuery( ), expr );
				// ONLY USE FIRST GROUPING SERIES FOR CHART ENGINE SUPPORT
				// if ( axisNum == 1 && sds.size( ) == 1 )
				{
					// Simply cascade menu
					return action;
				}
				// action.setText( getSecondMenuText( axisIndex,
				// i,
				// sd.getDesignTimeSeries( ) ) );
				// topManager.add( action );
			}
		}
		return topManager;
	}

	private Object getOrthogonalSeriesMenu( Chart chart, String expr )
	{
		IMenuManager topManager = new MenuManager( getOrthogonalSeriesTitle( getChartModel( ) ) );
		int axisNum = ChartUIUtil.getOrthogonalAxisNumber( chart );
		for ( int axisIndex = 0; axisIndex < axisNum; axisIndex++ )
		{
			List sds = ChartUIUtil.getOrthogonalSeriesDefinitions( chart,
					axisIndex );
			for ( int i = 0; i < sds.size( ); i++ )
			{
				Series series = ( (SeriesDefinition) sds.get( i ) ).getDesignTimeSeries( );
				EList dataDefns = series.getDataDefinition( );

				if ( series instanceof StockSeries )
				{
					IMenuManager secondManager = new MenuManager( getSecondMenuText( axisIndex,
							i,
							series ) );
					topManager.add( secondManager );
					for ( int j = 0; j < dataDefns.size( ); j++ )
					{
						IAction action = new ValueYSeriesAction( (Query) dataDefns.get( j ),
								expr );
						action.setText( ChartUIUtil.getStockTitle( j )
								+ Messages.getString( "StandardChartDataSheet.Label.Component" ) ); //$NON-NLS-1$
						secondManager.add( action );
					}
				}
				else if ( series instanceof BubbleSeries )
				{
					IMenuManager secondManager = new MenuManager( getSecondMenuText( axisIndex,
							i,
							series ) );
					topManager.add( secondManager );
					for ( int j = 0; j < dataDefns.size( ); j++ )
					{
						IAction action = new ValueYSeriesAction( (Query) dataDefns.get( j ),
								expr );
						action.setText( ChartUIUtil.getBubbleTitle( j )
								+ Messages.getString( "StandardChartDataSheet.Label.Component" ) ); //$NON-NLS-1$
						secondManager.add( action );
					}
				}
				else if ( series instanceof DifferenceSeries )
				{
					IMenuManager secondManager = new MenuManager( getSecondMenuText( axisIndex,
							i,
							series ) );
					topManager.add( secondManager );
					for ( int j = 0; j < dataDefns.size( ); j++ )
					{
						IAction action = new ValueYSeriesAction( (Query) dataDefns.get( j ),
								expr );
						action.setText( ChartUIUtil.getDifferenceTitle( j )
								+ Messages.getString( "StandardChartDataSheet.Label.Component" ) ); //$NON-NLS-1$
						secondManager.add( action );
					}
				}
				else if ( series instanceof GanttSeries )
				{
					IMenuManager secondManager = new MenuManager( getSecondMenuText( axisIndex,
							i,
							series ) );
					topManager.add( secondManager );
					for ( int j = 0; j < dataDefns.size( ); j++ )
					{
						IAction action = new ValueYSeriesAction( (Query) dataDefns.get( j ),
								expr );
						action.setText( ChartUIUtil.getGanttTitle( j )
								+ Messages.getString( "StandardChartDataSheet.Label.Component" ) ); //$NON-NLS-1$
						secondManager.add( action );
					}
				}
				else
				{
					IAction action = new ValueYSeriesAction( (Query) dataDefns.get( 0 ),
							expr );
					if ( axisNum == 1 && sds.size( ) == 1 )
					{
						// Simplify cascade menu
						return action;
					}
					action.setText( getSecondMenuText( axisIndex, i, series ) );
					topManager.add( action );
				}
			}
		}
		return topManager;
	}

	private String getSecondMenuText( int axisIndex, int seriesIndex,
			Series series )
	{
		StringBuffer sb = new StringBuffer( );
		if ( ChartUIUtil.getOrthogonalAxisNumber( getChartModel( ) ) > 1 )
		{
			sb.append( Messages.getString( "StandardChartDataSheet.Label.Axis" ) ); //$NON-NLS-1$
			sb.append( axisIndex + 1 );
			sb.append( " - " ); //$NON-NLS-1$
		}
		sb.append( Messages.getString( "StandardChartDataSheet.Label.Series" ) //$NON-NLS-1$
				+ ( seriesIndex + 1 )
				+ " (" + series.getDisplayName( ) + ")" ); //$NON-NLS-1$ //$NON-NLS-2$
		return sb.toString( );
	}

	private String getBaseSeriesTitle( Chart chart )
	{
		if ( chart instanceof ChartWithAxes )
		{
			return Messages.getString( "StandardChartDataSheet.Label.UseAsCategoryXAxis" ); //$NON-NLS-1$
		}
		return Messages.getString( "StandardChartDataSheet.Label.UseAsCategorySeries" ); //$NON-NLS-1$
	}

	private String getOrthogonalSeriesTitle( Chart chart )
	{
		if ( chart instanceof ChartWithAxes )
		{
			return Messages.getString( "StandardChartDataSheet.Label.PlotAsValueYSeries" ); //$NON-NLS-1$
		}
		else if ( chart instanceof DialChart )
		{
			return Messages.getString( "StandardChartDataSheet.Label.PlotAsGaugeValue" ); //$NON-NLS-1$
		}
		return Messages.getString( "StandardChartDataSheet.Label.PlotAsValueSeries" ); //$NON-NLS-1$
	}

	private String getGroupSeriesTitle( Chart chart )
	{
		if ( chart instanceof ChartWithAxes )
		{
			return Messages.getString( "StandardChartDataSheet.Label.UseToGroupYSeries" ); //$NON-NLS-1$
		}
		return Messages.getString( "StandardChartDataSheet.Label.UseToGroupValueSeries" ); //$NON-NLS-1$
	}

	private boolean isCubeMode( )
	{
		return ChartXTabUtil.getBindingCube( itemHandle ) != null;
	}

	private CubeHandle getCube( )
	{
		return ChartXTabUtil.getBindingCube( itemHandle );
	}

	/**
	 * Creates the cube expression
	 * 
	 * @return expression
	 */
	private String createCubeExpression( )
	{
		if ( cubeTreeViewer == null )
		{
			return null;
		}
		TreeItem[] selection = cubeTreeViewer.getTree( ).getSelection( );
		String expr = null;
		if ( selection.length > 0
				&& !dataProvider.isSharedBinding( )
				&& !dataProvider.isPartChart( ) )
		{
			TreeItem treeItem = selection[0];
			ComputedColumnHandle binding = null;
			if ( treeItem.getData( ) instanceof LevelHandle )
			{
				binding = ChartXTabUtil.findBinding( itemHandle,
						ChartXTabUtil.createDimensionExpression( (LevelHandle) treeItem.getData( ) ) );
			}
			else if ( treeItem.getData( ) instanceof MeasureHandle )
			{
				binding = ChartXTabUtil.findBinding( itemHandle,
						ChartXTabUtil.createMeasureExpression( (MeasureHandle) treeItem.getData( ) ) );
			}
			if ( binding != null )
			{
				expr = ExpressionUtil.createJSDataExpression( binding.getName( ) );
			}
		}
		return expr;
	}
	
	private boolean isDataItemSupported( int type )
	{
		return iSupportedDataItems == 0
				|| ( iSupportedDataItems & type ) == type;
	}

	private String[] createDataComboItems( )
	{
		List<String> items = new ArrayList<String>( );
		selectDataTypes.clear( );

		if ( isDataItemSupported( SELECT_NONE ) )
		{
			if ( DEUtil.getDataSetList( itemHandle.getContainer( ) )
							.size( ) > 0 )
			{
				items.add( Messages.getString( "ReportDataServiceProvider.Option.Inherits", //$NON-NLS-1$
						( (DataSetHandle) DEUtil.getDataSetList( itemHandle.getContainer( ) )
								.get( 0 ) ).getName( ) ) );
			}
			else
			{
				items.add( ReportDataServiceProvider.OPTION_NONE );
			}
			selectDataTypes.add( new Integer( SELECT_NONE ) );
		}

		if ( isDataItemSupported( SELECT_DATA_SET ) )
		{
			String[] dataSets = getDataServiceProvider( ).getAllDataSets( );
			if ( dataSets.length > 0 )
			{
				if ( isDataItemSupported( SELECT_NEXT ) )
				{
					items.add( Messages.getString( "StandardChartDataSheet.Combo.DataSets" ) ); //$NON-NLS-1$
					selectDataTypes.add( new Integer( SELECT_NEXT ) );
				}

				for ( int i = 0; i < dataSets.length; i++ )
				{
					items.add( dataSets[i] );
					selectDataTypes.add( new Integer( SELECT_DATA_SET ) );
				}
			}
			if ( isDataItemSupported( SELECT_NEW_DATASET ) )
			{
				items.add( Messages.getString( "StandardChartDataSheet.NewDataSet" ) ); //$NON-NLS-1$
				selectDataTypes.add( new Integer( SELECT_NEW_DATASET ) );
			}
		}

		if ( isDataItemSupported( SELECT_DATA_CUBE ) )
		{
			String[] dataCubes = getDataServiceProvider( ).getAllDataCubes( );
			if ( dataCubes.length > 0 )
			{
				if ( isDataItemSupported( SELECT_NEXT ) )
				{
					items.add( Messages.getString( "StandardChartDataSheet.Combo.DataCubes" ) ); //$NON-NLS-1$
					selectDataTypes.add( new Integer( SELECT_NEXT ) );
				}
				for ( int i = 0; i < dataCubes.length; i++ )
				{
					items.add( dataCubes[i] );
					selectDataTypes.add( new Integer( SELECT_DATA_CUBE ) );
				}
			}
			if ( isDataItemSupported( SELECT_NEW_DATACUBE ) )
			{
				items.add( Messages.getString( "StandardChartDataSheet.NewDataCube" ) ); //$NON-NLS-1$
				selectDataTypes.add( new Integer( SELECT_NEW_DATACUBE ) );
			}
		}

		if ( isDataItemSupported( SELECT_REPORT_ITEM ) )
		{
			String[] dataRefs = getDataServiceProvider( ).getAllReportItemReferences( );
			if ( dataRefs.length > 0 )
			{
				if ( isDataItemSupported( SELECT_NEXT ) )
				{
					items.add( Messages.getString( "StandardChartDataSheet.Combo.ReportItems" ) ); //$NON-NLS-1$
					selectDataTypes.add( new Integer( SELECT_NEXT ) );
				}
				for ( int i = 0; i < dataRefs.length; i++ )
				{
					items.add( dataRefs[i] );
					selectDataTypes.add( new Integer( SELECT_REPORT_ITEM ) );
				}
			}
		}
		return items.toArray( new String[items.size( )] );
	}

	private void updatePredefinedQueries( )
	{
		if ( dataProvider.isInXTabMeasureCell( ) )
		{
			try
			{
				CrosstabReportItemHandle xtab = ChartXTabUtil.getXtabContainerCell( itemHandle )
						.getCrosstab( );

				if ( dataProvider.isPartChart( ) )
				{
					List<String> levels = ChartXTabUtil.getAllLevelsBindingExpression( xtab );
					String[] exprs = levels.toArray( new String[levels.size( )] );
					if ( exprs.length == 2 && dataProvider.isInXTabAggrCell( ) )
					{
						// Only one direction is valid for chart in total cell
						if ( ( (ChartWithAxes) getChartModel( ) ).isTransposed( ) )
						{
							exprs = new String[]{
								exprs[1]
							};
						}
						else
						{
							exprs = new String[]{
								exprs[0]
							};
						}
					}
					getContext( ).addPredefinedQuery( ChartUIConstants.QUERY_CATEGORY,
							exprs );
				}
				else
				{
					Iterator columnBindings = ChartXTabUtil.getAllColumnBindingsIterator( itemHandle );
					List<String> levels = ChartXTabUtil.getAllLevelsBindingExpression( columnBindings );
					String[] exprs = levels.toArray( new String[levels.size( )] );
					getContext( ).addPredefinedQuery( ChartUIConstants.QUERY_CATEGORY,
							exprs );
					getContext( ).addPredefinedQuery( ChartUIConstants.QUERY_OPTIONAL,
							exprs );

					columnBindings = ChartXTabUtil.getAllColumnBindingsIterator( itemHandle );
					List<String> measures = ChartXTabUtil.getAllMeasuresBindingExpression( columnBindings );
					exprs = measures.toArray( new String[measures.size( )] );
					getContext( ).addPredefinedQuery( ChartUIConstants.QUERY_VALUE,
							exprs );
				}

			}
			catch ( BirtException e )
			{
				WizardBase.displayException( e );
			}
		}
		else
		{
			CubeHandle cube = getCube( );
			if ( cube == null )
			{
				getContext( ).addPredefinedQuery( ChartUIConstants.QUERY_CATEGORY,
						null );
				getContext( ).addPredefinedQuery( ChartUIConstants.QUERY_VALUE,
						null );
				getContext( ).addPredefinedQuery( ChartUIConstants.QUERY_OPTIONAL,
						null );
			}
			else
			{
				if ( dataProvider.isInheritanceOnly( ) // Is in multiple view
						|| dataProvider.isSharedBinding( ) ) // Is sharing
				// query case.
				{
					// Get all column bindings.
					List<String> dimensionExprs = new ArrayList<String>( );
					List<String> measureExprs = new ArrayList<String>( );
					ReportItemHandle reportItemHandle = dataProvider.getReportItemHandle( );
					for ( Iterator iter = reportItemHandle.getColumnBindings( )
							.iterator( ); iter.hasNext( ); )
					{
						ComputedColumnHandle cch = (ComputedColumnHandle) iter.next( );
						String dataExpr = ExpressionUtil.createJSDataExpression( cch.getName( ) );
						if ( ChartXTabUtil.isDimensionExpresion( cch.getExpression( ) ) )
						{
							dimensionExprs.add( dataExpr );
						}
						else if ( ChartXTabUtil.isMeasureExpresion( cch.getExpression( ) ) )
						{
							// Fixed issue ED 28.
							// Underlying code was reverted to the earlier than
							// bugzilla 246683, since we have enhanced it to
							// support all available measures defined in shared
							// item.

							// Bugzilla 246683.
							// Here if it is sharing with crosstab or
							// multi-view, we just put the measure expression
							// whose aggregate-ons is most into prepared
							// expression query. It will keep correct value to
							// shared crosstab or multi-view.
							measureExprs.add( dataExpr );

						}
					}
					String[] categoryExprs = dimensionExprs.toArray( new String[dimensionExprs.size( )] );
					String[] yOptionalExprs = categoryExprs;
					String[] valueExprs = measureExprs.toArray( new String[measureExprs.size( )] );

					ReportItemHandle referenceHandle = ChartReportItemUtil.getReportItemReference( itemHandle );
					if ( referenceHandle instanceof ExtendedItemHandle
							&& ChartReportItemUtil.isChartReportItemHandle( referenceHandle ) )
					{
						// If the final reference handle is cube with other
						// chart, the valid category and Y optional expressions
						// only allow those expressions defined in shared chart.
						Chart referenceCM = ChartReportItemUtil.getChartFromHandle( (ExtendedItemHandle) referenceHandle );
						categoryExprs = ChartUtil.getCategoryExpressions( referenceCM );
						yOptionalExprs = ChartUtil.getYOptoinalExpressions( referenceCM );
						valueExprs = ChartUtil.getValueSeriesExpressions( referenceCM );

						Chart cm = getChartModel( );
						if ( categoryExprs.length > 0 )
						{
							updateCategoryExpression( cm, categoryExprs[0] );
						}
						if ( yOptionalExprs.length > 0 )
						{
							updateYOptionalExpressions( cm, yOptionalExprs[0] );
						}
					}
					else if ( dataProvider.checkState( IDataServiceProvider.SHARE_CROSSTAB_QUERY ) )
					{
						// In sharing query with crosstab, the category
						// expression and Y optional expression is decided by
						// value series expression, so here set them to null.
						// And in UI, when the value series expression is
						// selected, it will trigger to set correct category and
						// Y optional expressions.
						categoryExprs = null;
						yOptionalExprs = null;
					}

					getContext( ).addPredefinedQuery( ChartUIConstants.QUERY_CATEGORY,
							categoryExprs );
					getContext( ).addPredefinedQuery( ChartUIConstants.QUERY_OPTIONAL,
							yOptionalExprs );
					getContext( ).addPredefinedQuery( ChartUIConstants.QUERY_VALUE,
							valueExprs );
				}
				else
				{
					Iterator columnBindings = ChartXTabUtil.getAllColumnBindingsIterator( itemHandle );
					List<String> levels = ChartXTabUtil.getAllLevelsBindingExpression( columnBindings );
					String[] exprs = levels.toArray( new String[levels.size( )] );
					getContext( ).addPredefinedQuery( ChartUIConstants.QUERY_CATEGORY,
							exprs );
					getContext( ).addPredefinedQuery( ChartUIConstants.QUERY_OPTIONAL,
							exprs );

					columnBindings = ChartXTabUtil.getAllColumnBindingsIterator( itemHandle );
					List<String> measures = ChartXTabUtil.getAllMeasuresBindingExpression( columnBindings );
					exprs = measures.toArray( new String[measures.size( )] );
					getContext( ).addPredefinedQuery( ChartUIConstants.QUERY_VALUE,
							exprs );
				}
			}
		}

		// Fire event to update predefined queries in outside UI
		fireEvent( btnBinding, EVENT_QUERY );
	}

	/**
	 * Update Y Optional expression with specified expression if current Y
	 * optional expression is null or empty.
	 * 
	 * @param cm
	 *            chart model.
	 * @param expr
	 *            specified expression.
	 */
	private void updateYOptionalExpressions( Chart cm, String expr )
	{
		List<SeriesDefinition> orthSDs = ChartUtil.getAllOrthogonalSeriesDefinitions( cm );
		for ( SeriesDefinition sd : orthSDs )
		{
			Query q = sd.getQuery( );

			if ( q == null )
			{
				sd.setQuery( QueryImpl.create( expr ) );
				continue;
			}

			if ( q.getDefinition( ) == null
					|| "".equals( q.getDefinition( ).trim( ) ) ) //$NON-NLS-1$
			{
				q.setDefinition( expr );
			}
		}
	}

	/**
	 * Update category expression with specified expression if current category
	 * expression is null or empty.
	 * 
	 * @param cm
	 *            chart model.
	 * @param expr
	 *            specified expression.
	 */
	private void updateCategoryExpression( Chart cm, String expr )
	{
		EList<SeriesDefinition> baseSDs = ChartUtil.getBaseSeriesDefinitions( cm );
		for ( SeriesDefinition sd : baseSDs )
		{
			EList<Query> dds = sd.getDesignTimeSeries( ).getDataDefinition( );
			Query q = dds.get( 0 );
			if ( q.getDefinition( ) == null
					|| "".equals( q.getDefinition( ).trim( ) ) ) //$NON-NLS-1$
			{
				q.setDefinition( expr );
			}
		}
	}
}
