/*******************************************************************************
 * Copyright (c) 2006 Actuate Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *  Actuate Corporation  - initial API and implementation
 *******************************************************************************/

package org.eclipse.birt.chart.internal.datafeed;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;

import org.eclipse.birt.chart.engine.i18n.Messages;
import org.eclipse.birt.chart.exception.ChartException;
import org.eclipse.birt.chart.factory.IActionEvaluator;
import org.eclipse.birt.chart.log.ILogger;
import org.eclipse.birt.chart.log.Logger;
import org.eclipse.birt.chart.model.Chart;
import org.eclipse.birt.chart.model.ChartWithAxes;
import org.eclipse.birt.chart.model.ChartWithoutAxes;
import org.eclipse.birt.chart.model.component.Axis;
import org.eclipse.birt.chart.model.component.Series;
import org.eclipse.birt.chart.model.data.DataPackage;
import org.eclipse.birt.chart.model.data.Query;
import org.eclipse.birt.chart.model.data.SeriesDefinition;
import org.eclipse.birt.chart.model.data.SeriesGrouping;
import org.eclipse.birt.chart.plugin.ChartEnginePlugin;
import org.eclipse.birt.chart.util.ChartUtil;
import org.eclipse.emf.common.util.EList;

import com.ibm.icu.util.ULocale;

/**
 * Helper to lookup the index of each data definition in the evaluator data.
 */

public class GroupingLookupHelper
{

	private static ILogger logger = Logger.getLogger( "org.eclipse.birt.chart.engine/trace" ); //$NON-NLS-1$

	private LinkedHashMap lhmAggExp = new LinkedHashMap( 8 );

	private List lstAll = new ArrayList( 8 );
	
	private List lstTransformedExprs = new ArrayList( 8 );
	
	private LinkedHashMap lhmTransformedExprs = new LinkedHashMap( 8 );
	
	private String strBaseAggExp = null;

	private int iLookup = 0;

	private int iLookupTransformedExprs = 0;
	
	private ULocale locale;

	/** The expression index of sort expression on base series. */
	private int fBaseSortExprIndex = -1;
	
	/**
	 * The expression index of sort key on Y grouping.
	 * @since BIRT 2.3
	 **/
	private int fYSortExprIndex = -1;
	
	/**
	 * Constructor. Finds all data expressions and aggregation expressions in
	 * the chart model in the order and restore them in the lookup list
	 * 
	 * @param cm
	 *            chart model
	 * @param iae
	 *            IActionEvaluator to get the expressions in triggers
	 * @param locale 
	 * @throws ChartException
	 */
	public GroupingLookupHelper( Chart cm, IActionEvaluator iae, ULocale locale )
			throws ChartException
	{
		this.locale = locale;
		if ( cm instanceof ChartWithAxes )
		{
			initRowExpressions( (ChartWithAxes) cm, iae );
		}
		else if ( cm instanceof ChartWithoutAxes )
		{
			initRowExpressions( (ChartWithoutAxes) cm, iae );
		}
		
	}

	/**
	 * Constructor. Restore all expressions in the lookup list.
	 * 
	 * @param dataExps
	 *            data expressions collection
	 * @param aggExps
	 *            aggregation expressions collection
	 * 
	 */
	public GroupingLookupHelper( Collection dataExps, Collection aggExps )
	{
		Iterator dataIterator = dataExps.iterator( );
		Iterator aggIterator = aggExps.iterator( );
		while ( dataIterator.hasNext( ) && aggIterator.hasNext( ) )
		{
			String dataExp = (String) dataIterator.next( );
			String aggExp = (String) aggIterator.next( );
			
			lstAll.add( dataExp );
			lhmAggExp.put( generateKey( dataExp, aggExp ),
					new Integer( iLookup++ ) );
		}
	}

	/**
	 * Gets the list for all data expressions. Only for lookup, and can't be
	 * changed directly.
	 * 
	 * @return the list for all data expressions
	 */
	public List getExpressions( )
	{
		return lstAll;
	}

	/**
	 * Returns transformed expressions for aggregate case.
	 * @return
	 * @since 2.3
	 */
	public List getExpressionsForAggregate( )
	{
		return lstTransformedExprs;
	}
	
	private String generateKey( String dataExp, String aggExp )
	{
		if ( aggExp == null || "".equals( aggExp ) ) //$NON-NLS-1$
		{
			return dataExp;
		}
		return dataExp + "_" + aggExp; //$NON-NLS-1$
	}
	
	/**
	 * Finds the index of base series according to the data expression .
	 * 
	 * @param dataExp
	 *            the data expression to lookup
	 * @return the index of the data expression in the evaluator data
	 */
	public int findIndexOfBaseSeries( String dataExp )
	{
		return findIndex( dataExp, "" ); //$NON-NLS-1$
	}

	/**
	 * Finds the index of orthogonal series according to the combination of data
	 * expression and aggregation expression.
	 * 
	 * @param dataExp
	 *            data expression
	 * @param aggExp
	 *            aggregation expression. If it's null, will use aggregation
	 *            expression of base series instead
	 * @return the index in the evaluator data
	 */
	public int findIndex( String dataExp, String aggExp )
	{
		Object value = null;
		if ( aggExp == null )
		{
			value = lhmAggExp.get( generateKey( dataExp, this.strBaseAggExp ) );
		}
		else
		{
			value = lhmAggExp.get( generateKey( dataExp, aggExp ) );

		}
		return value instanceof Integer ? ( (Integer) value ).intValue( ) : -1;
	}

	/**
	 * Finds the index according to the combination of data expression and
	 * aggregation expression in a batch. Note that all data expressions must
	 * match the same aggregation expression.
	 * 
	 * @param dataExpArray
	 *            data expression array
	 * @param aggExp
	 *            aggregation expression
	 * @return the index array in the evaluator data
	 */
	public int[] findBatchIndex( String[] dataExpArray, String aggExp )
	{
		int[] indexArray = new int[dataExpArray.length];
		for ( int i = 0; i < indexArray.length; i++ )
		{
			indexArray[i] = this.findIndex( dataExpArray[i], aggExp );
		}
		return indexArray;
	}
	
	private boolean addDataExpOfBaseSeries( String dataExp )
	{
		boolean result = addDataExp( dataExp, "" ); //$NON-NLS-1$
		if ( result )
		{
			addDataExpForAggregate( dataExp, "" );
		}
		return result;
	}

	private boolean addDataExp( String dataExp, String aggExp )
	{
		if ( dataExp != null && dataExp.trim( ).length( ) > 0 )
		{
			String key = generateKey( dataExp, aggExp );
			if ( !lhmAggExp.containsKey( key ) )
			{
				lhmAggExp.put( key, new Integer( iLookup++ ) );
				lstAll.add( dataExp );
			}
			return true;
		}
		return false;
	}
	
	private void addDataExpForAggregate(String expr, String aggExp )
	{
		String key = generateKey( expr, aggExp );
		if ( !lhmTransformedExprs.containsKey( key ) )
		{
			lhmTransformedExprs.put( key, new Integer( iLookupTransformedExprs++ ) );
			lstTransformedExprs.add( expr );
		}
	}
	
	private void addLookupForBaseSeries( SeriesDefinition baseSD )
			throws ChartException
	{
		final Query qBaseSeriesDefinition = baseSD.getQuery( );
		String sExpression = qBaseSeriesDefinition.getDefinition( );
		if ( sExpression != null && sExpression.trim( ).length( ) > 0 )
		{
			// Ignore expression for base series definition
			logger.log( ILogger.WARNING,
					Messages.getString( "dataprocessor.log.baseSeriesDefn3", sExpression, ULocale.getDefault( ) ) ); //$NON-NLS-1$
		}

		// PROJECT THE EXPRESSION ASSOCIATED WITH THE BASE SERIES EXPRESSION
		final Series seBase = baseSD.getDesignTimeSeries( );
		EList elBaseSeries = seBase.getDataDefinition( );
		if ( elBaseSeries.size( ) != 1 )
		{
			throw new ChartException( ChartEnginePlugin.ID,
					ChartException.DATA_BINDING,
					"dataprocessor.exception.FoundDefnAssociatedWithX", //$NON-NLS-1$
					new Object[]{
						String.valueOf( elBaseSeries.size( ) )
					},
					Messages.getResourceBundle( this.locale ) );
		}

		String baseSeriesExpression =  ( (Query) elBaseSeries.get( 0 ) ).getDefinition( );
		if ( !addDataExpOfBaseSeries( baseSeriesExpression ) )
		{
			throw new ChartException( ChartEnginePlugin.ID,
					ChartException.DATA_BINDING,
					"dataprocessor.exception.DefinitionUnspecified", //$NON-NLS-1$
					Messages.getResourceBundle( this.locale) );
		}
		
		// Set base sort key index if it equals base series expression.
		String sortKey = getSortKey( baseSD );
		if ( baseSD.isSetSorting( ) && sortKey == null ) {
			sortKey = baseSeriesExpression;
		}
		if ( sortKey != null && sortKey.equals( baseSeriesExpression )
				&& baseSD.eIsSet( DataPackage.eINSTANCE.getSeriesDefinition_Sorting( ) ) )
		{
			fBaseSortExprIndex = findIndexOfBaseSeries( baseSeriesExpression );
		}
	}

	private void addLookupForOrthogonalSeries( SeriesDefinition baseSD, EList lstOrthogonalSDs,
			IActionEvaluator iae ) throws ChartException
	{
		for ( int k = 0; k < lstOrthogonalSDs.size( ); k++ )
		{
			SeriesDefinition orthoSD = (SeriesDefinition) lstOrthogonalSDs.get( k );
			Query qOrthogonalSeriesDefinition = orthoSD.getQuery( );
			if ( qOrthogonalSeriesDefinition == null )
			{
				return;
			}

			String strOrthoAgg = getOrthogonalAggregationExpression( orthoSD );
			
			if ( addDataExp( qOrthogonalSeriesDefinition.getDefinition( ),
					strOrthoAgg ) )
			{
				addDataExpForAggregate( qOrthogonalSeriesDefinition.getDefinition( ), strOrthoAgg );	
			}
			
			// Get sort key of Y grouping.
			String ySortKey = getSortKey( orthoSD );

			Series seOrthogonal = orthoSD.getDesignTimeSeries( );
			EList elOrthogonalSeries = seOrthogonal.getDataDefinition( );
			if ( elOrthogonalSeries.isEmpty( ) )
			{
				throw new ChartException( ChartEnginePlugin.ID,
						ChartException.DATA_BINDING,
						"dataprocessor.exception.DefnExpMustAssociateY", //$NON-NLS-1$
						new Object[]{
								String.valueOf( k ), seOrthogonal
						},
						Messages.getResourceBundle( this.locale ) );
			}

			boolean bAnyQueries = false;
			for ( int i = 0; i < elOrthogonalSeries.size( ); i++ )
			{
				Query qOrthogonalSeries = (Query) elOrthogonalSeries.get( i );
				if ( qOrthogonalSeries == null ) // NPE PROTECTION
				{
					continue;
				}

				if ( addDataExp( qOrthogonalSeries.getDefinition( ),
						strOrthoAgg ) )
				{
					addDataExpForAggregate( ChartUtil.createValueSeriesRowFullExpression( qOrthogonalSeries.getDefinition( ),
							orthoSD,
							baseSD ), strOrthoAgg );
					
					bAnyQueries = true;
					
					// Set base sort index if it equals the value series
					// expression.
					if ( fBaseSortExprIndex < 0 )
					{
						String sortExpr = getSortKey( baseSD );
						if ( sortExpr != null &&
								sortExpr.equals( qOrthogonalSeries.getDefinition( ) )
								&& baseSD.eIsSet( DataPackage.eINSTANCE.getSeriesDefinition_Sorting( ) ) )
						{
							fBaseSortExprIndex = findIndex( qOrthogonalSeries.getDefinition( ),
									strOrthoAgg );
						}
					}
					
					// Get Y sort expression index.
					if ( fYSortExprIndex < 0 )
					{
						if ( ySortKey != null &&
								ySortKey.equals( qOrthogonalSeries.getDefinition( ) )
								&& orthoSD.eIsSet( DataPackage.eINSTANCE.getSeriesDefinition_Sorting( ) ) )
						{
							fYSortExprIndex = findIndex( qOrthogonalSeries.getDefinition( ),
									strOrthoAgg );
						}
					}
				}
			}

			if ( !bAnyQueries )
			{
				throw new ChartException( ChartEnginePlugin.ID,
						ChartException.DATA_BINDING,
						"dataprocessor.exception.AtLeastOneDefnExpMustAssociateY", //$NON-NLS-1$
						new Object[]{
								String.valueOf( k ), seOrthogonal
						},
						Messages.getResourceBundle( this.locale ) );
			}

			// Add orthogonal series trigger expressions.
			String[] triggerExprs = DataProcessor.getSeriesTriggerExpressions( seOrthogonal,
					iae );
			if ( triggerExprs != null )
			{
				for ( int t = 0; t < triggerExprs.length; t++ )
				{
					if ( addDataExp( triggerExprs[t], strOrthoAgg ) )
					{
						addDataExpForAggregate( triggerExprs[t], strOrthoAgg );
					}
				}
			}
		}
	}

	private void initRowExpressions( ChartWithoutAxes cwoa, IActionEvaluator iae )
			throws ChartException
	{
		EList elSD = cwoa.getSeriesDefinitions( );
		if ( elSD.size( ) != 1 )
		{
			throw new ChartException( ChartEnginePlugin.ID,
					ChartException.DATA_BINDING,
					"dataprocessor.exception.CannotDecipher", //$NON-NLS-1$
					Messages.getResourceBundle( this.locale ) );
		}

		// PROJECT THE EXPRESSION ASSOCIATED WITH THE BASE SERIES DEFINITION
		SeriesDefinition baseSD = (SeriesDefinition) elSD.get( 0 );		
		this.strBaseAggExp = getBaseAggregationExpression( baseSD );
		addLookupForBaseSeries( baseSD );

		// PROJECT ALL DATA DEFINITIONS ASSOCIATED WITH THE ORTHOGONAL SERIES
		addLookupForOrthogonalSeries( baseSD, baseSD.getSeriesDefinitions( ), iae );

		addCommonSortKey( baseSD );
	}

	private void initRowExpressions( ChartWithAxes cwa, IActionEvaluator iae )
			throws ChartException
	{
		final Axis axPrimaryBase = cwa.getPrimaryBaseAxes( )[0];
		EList elSD = axPrimaryBase.getSeriesDefinitions( );
		if ( elSD.size( ) != 1 )
		{
			throw new ChartException( ChartEnginePlugin.ID,
					ChartException.GENERATION,
					"dataprocessor.exception.CannotDecipher2", //$NON-NLS-1$
					Messages.getResourceBundle( this.locale ) );
		}

		// PROJECT THE EXPRESSION ASSOCIATED WITH THE BASE SERIES DEFINITION
		SeriesDefinition baseSD = (SeriesDefinition) elSD.get( 0 );
		this.strBaseAggExp = getBaseAggregationExpression( baseSD );
		addLookupForBaseSeries( baseSD );

		// PROJECT ALL DATA DEFINITIONS ASSOCIATED WITH THE ORTHOGONAL SERIES
		final Axis[] axaOrthogonal = cwa.getOrthogonalAxes( axPrimaryBase, true );
		for ( int j = 0; j < axaOrthogonal.length; j++ )
		{
			addLookupForOrthogonalSeries( baseSD, axaOrthogonal[j].getSeriesDefinitions( ),
					iae );
		}

		// If base sort expression is not base series or value series, directly
		// add the expression.
		addCommonSortKey( baseSD );
	}

	/**
	 * Add common sort expression by the specified expression.
	 * 
	 * @param baseSD
	 */
	private void addCommonSortKey( SeriesDefinition baseSD )
	{
		if ( fBaseSortExprIndex < 0
				&& baseSD.eIsSet( DataPackage.eINSTANCE.getSeriesDefinition_Sorting( ) ) )
		{
			String sortExpr = getSortKey( baseSD );
			if ( sortExpr != null )
			{
				if ( addDataExp( sortExpr, "" ) ) //$NON-NLS-1$
				{
					addDataExpForAggregate( sortExpr, "" );
				}
				fBaseSortExprIndex = findIndexOfBaseSeries( sortExpr );
			}
		}
	}

	/**
	 * Returns sort key of series definition.
	 * 
	 * @param sd
	 * @return
	 */
	private String getSortKey( SeriesDefinition sd )
	{
		if ( sd == null || sd.getSortKey( ) == null )
		{
			return null;
		}
		
		return sd.getSortKey( ).getDefinition( );
	}
	
//	static String getAggregationExp( SeriesDefinition baseSD,
//			SeriesDefinition orthoSD )
//	{
//		boolean bBaseGroupEnabled = baseSD.getGrouping( ).isEnabled( );
//		String strOrthoAgg = null;
//		if ( bBaseGroupEnabled )
//		{
//			String strBaseAgg = baseSD.getGrouping( ).getAggregateExpression( );
//			boolean bOrthoGroupEnabled = orthoSD.getGrouping( ).isEnabled( );
//			strOrthoAgg = bOrthoGroupEnabled ? orthoSD.getGrouping( )
//					.getAggregateExpression( ) : strBaseAgg;
//		}
//		return strOrthoAgg;
//	}

	/**
	 * Simply gets aggregation expressions for the series definitions. If
	 * grouping is not enabled, return null
	 * 
	 * @param sd
	 *            series definition
	 * @return aggregation expressions for the series definitions, or null if
	 *         grouping is disabled.
	 */
	static String getBaseAggregationExpression( SeriesDefinition sd )
	{
		SeriesGrouping grouping = sd.getGrouping( );
		if ( grouping != null &&
				grouping.isSetEnabled( ) &&
				grouping.isEnabled( ) )
		{
			return grouping.getAggregateExpression( );
		}
		return null;
	}

	/**
	 * Gets aggregation expressions of orthogonal series definition. If base
	 * series doesn't enable grouping, return null. If its own grouping is null,
	 * return the one of base grouping, otherwise, return its own.
	 * 
	 * @param orthoSD
	 *            orthogonal series definition
	 * @return If base series doesn't enable grouping, return null. If its own
	 *         grouping is null, return the one of base grouping, otherwise,
	 *         return its own.
	 */
	public String getOrthogonalAggregationExpression( SeriesDefinition orthoSD )
	{
		 String strOrthoAgg = null;
		 SeriesGrouping grouping = orthoSD.getGrouping( );
		 if ( grouping != null
				&& grouping.isSetEnabled( )
				&& grouping.isEnabled( ) )
		{
			// Set own group
			strOrthoAgg = grouping.getAggregateExpression( );
		}

		if ( ( strOrthoAgg == null || strOrthoAgg.length( ) == 0 )
				&& this.strBaseAggExp != null )
		{
			strOrthoAgg = strBaseAggExp;
		}
		return strOrthoAgg;
	}
	
	/**
	 * Returns sort expression of base series, <code>-1</code> means no sort
	 * expression is set for base series.
	 * 
	 * @return
	 * @since 2.3
	 */
	int getBaseSortExprIndex( )
	{
		return fBaseSortExprIndex;
	}
	
	/**
	 * Returns sort expression of Y grouping, <code>-1</code> means no sort
	 * expression is set for Y grouping.
	 * 
	 * @return
	 * @since BIRT 2.3
	 */
	int getYSortExprIndex( )
	{
		return fYSortExprIndex;
	}
}
