package org.eclipse.birt.report.tests.model.acceptance;

import java.util.Iterator;

import junit.framework.Test;
import junit.framework.TestSuite;

import org.eclipse.birt.report.model.api.DesignEngine;
import org.eclipse.birt.report.model.api.ElementFactory;
import org.eclipse.birt.report.model.api.EmbeddedImageHandle;
import org.eclipse.birt.report.model.api.ImageHandle;
import org.eclipse.birt.report.model.api.LabelHandle;
import org.eclipse.birt.report.model.api.PropertyHandle;
import org.eclipse.birt.report.model.api.SharedStyleHandle;
import org.eclipse.birt.report.model.api.StructureFactory;
import org.eclipse.birt.report.model.api.StructureHandle;
import org.eclipse.birt.report.model.api.TextItemHandle;
import org.eclipse.birt.report.model.api.command.NameException;
import org.eclipse.birt.report.model.api.core.IDesignElement;
import org.eclipse.birt.report.model.api.elements.DesignChoiceConstants;
import org.eclipse.birt.report.model.api.elements.structures.Action;
import org.eclipse.birt.report.model.api.elements.structures.EmbeddedImage;
import org.eclipse.birt.report.model.elements.ReportDesign;
import org.eclipse.birt.report.model.elements.TextItem;
import org.eclipse.birt.report.model.elements.interfaces.IReportItemModel;
import org.eclipse.birt.report.model.elements.interfaces.IImageItemModel;
import org.eclipse.birt.report.model.elements.interfaces.IStyleModel;
import org.eclipse.birt.report.tests.model.BaseTestCase;

import com.ibm.icu.util.ULocale;

public class ImageHandleTest extends BaseTestCase
{
	
	private ElementFactory factory = null;
    private ImageHandle image = null;
    private EmbeddedImage embeddedimage = null;
    private EmbeddedImage embeddedimage1 = null;

    
	public ImageHandleTest(String name) {
		super(name);
		// TODO Auto-generated constructor stub
	}
	public static Test suite()
	{
		return new TestSuite(ImageHandleTest.class);
	}
	public void setUp( ) throws Exception
	{
		super.setUp();
		sessionHandle = DesignEngine.newSession(ULocale.ENGLISH);
		designHandle = sessionHandle.createDesign();
		design = (ReportDesign)designHandle.getModule();
		factory = new ElementFactory( design );
		image = factory.newImage("myimage");
		embeddedimage = StructureFactory.createEmbeddedImage( );
		embeddedimage1 = StructureFactory.createEmbeddedImage( );

	}
	
	public void testImageOperations() throws Exception
	{
		//add a URI image
		image.setURI("http://sprott.physics.wisc.edu/fractals/collect/1999/Wild%20Flower.jpg");
		designHandle.getBody().add(image);
		assertEquals("http://sprott.physics.wisc.edu/fractals/collect/1999/Wild%20Flower.jpg", image.getURI());
		
        //undo it
		assertTrue(designHandle.getCommandStack().canUndo());
	    assertFalse(designHandle.getCommandStack().canRedo());
	    designHandle.getCommandStack().undo();
	    assertEquals(0,designHandle.getBody().getCount());
	    
	    //redo it
	    assertTrue(designHandle.getCommandStack().canRedo());
	    designHandle.getCommandStack().redo();
	    assertEquals("myimage",designHandle.getBody().get(0).getName());
		
        //delete the image
	    designHandle.getBody().drop(image);
	    assertEquals(0,designHandle.getBody().getCount());
	    
	    //undo it
	    assertTrue(designHandle.getCommandStack().canUndo());
	    assertFalse(designHandle.getCommandStack().canRedo());
	    designHandle.getCommandStack().undo();
	    assertEquals("myimage",designHandle.getBody().get(0).getName());
	    
	    //redo it
	    designHandle.getCommandStack().redo();
	    assertEquals(0,designHandle.getBody().getCount());
	    
	    //Edit the image
	    //Change URI
	    designHandle.getCommandStack().undo();
	    image.setURI("http://shanghai/welcome/images/actuatetop.jpg");
	    assertEquals("http://shanghai/welcome/images/actuatetop.jpg",image.getURI());
	    
	    designHandle.getCommandStack().undo();
	    assertEquals("http://sprott.physics.wisc.edu/fractals/collect/1999/Wild%20Flower.jpg", image.getURI());
	    
	    designHandle.getCommandStack().redo();
	    assertEquals("http://shanghai/welcome/images/actuatetop.jpg",image.getURI());
	    
	    //Edit embeddedimage
	    embeddedimage.setName("Sample Pictures.jpg");
	    embeddedimage.setData( "data".getBytes( EmbeddedImage.CHARSET ) );
	    designHandle.addImage(embeddedimage);
	    
	    try
		{
			// we allow empty image.
	    	embeddedimage.setData( null );
		}
		catch ( Throwable e )
		{
			fail( );
		}
	    
	    assertNotNull(designHandle.findImage("Sample Pictures.jpg"));
	    
	    //copy & paste
	    ImageHandle newimage =(ImageHandle)image.copy().getHandle(design);
	    assertNotNull(newimage);
	    assertEquals(image.getName(),newimage.getName());
	    design = (ReportDesign)designHandle.getModule();
	    
	    try{
	    	designHandle.getBody().add(newimage);
		    fail();
	    }
	    catch (NameException e)
	    {
	      assertNotNull(e);
	    }
	    newimage.setName("myimage1");
	    designHandle.getBody().add(newimage);
	    assertEquals("myimage1",newimage.getName());
	    
	    designHandle.getCommandStack().undo();
	    assertEquals(1,designHandle.getBody().getCount());
	    
	    designHandle.getCommandStack().redo();
	    assertEquals(2,designHandle.getBody().getCount());
	    
	    //cut & paste
	    ImageHandle newimage1 = (ImageHandle)image.copy().getHandle(design);
	    assertNotNull(newimage1);
	    assertEquals(image.getName(),newimage1.getName());
	    image.drop();
	    designHandle.getBody().add(newimage1);
	    
	    designHandle.getCommandStack().undo();
	    assertEquals(1,designHandle.getBody().getCount());
	    
	    designHandle.getCommandStack().redo();
	    assertEquals(2,designHandle.getBody().getCount());
		
	    //rename
	    newimage.setName("goldentext");
	    newimage1.setName("silvertext");
	    assertEquals("goldentext",designHandle.getBody().get(0).getName());
	    assertEquals("silvertext",designHandle.getBody().get(1).getName());
	}
	    
	public void testImageProperties() throws Exception
	{
		//general
		image.setProperty(IReportItemModel.HEIGHT_PROP,"10in");
		image.setProperty(IReportItemModel.WIDTH_PROP,"100%");
		SharedStyleHandle style = factory.newStyle("mystyle");
		designHandle.getStyles().add(style);
		image.setStyle(style);
		
		assertEquals("10in", image.getStringProperty(IReportItemModel.HEIGHT_PROP));
		assertEquals("100%", image.getStringProperty(IReportItemModel.WIDTH_PROP));
		assertEquals("mystyle", image.getStyle().getName());
		
		//pereference
		image.setURI("http://www.google.com/images/logo_sm.gif");
		designHandle.getCommandStack().undo();
		assertNull(image.getURI());
		    
		 designHandle.getCommandStack().redo();
		 assertEquals("http://www.google.com/images/logo_sm.gif",image.getURI());
		
		 //hyperlink
		 StructureFactory factory = new StructureFactory();
		    
		 Action action = factory.createAction();
		 image.setAction(action);
		 image.getActionHandle().setLinkType( DesignChoiceConstants.ACTION_LINK_TYPE_HYPERLINK ); 
	     image.getActionHandle().setURI( "http://birt.eclipse.org/" ); 
	     assertEquals("http://birt.eclipse.org/",image.getActionHandle().getURI());
		
	      designHandle.getCommandStack().undo();
		  assertNull(image.getActionHandle().getURI());
		
		  designHandle.getCommandStack().redo();
		  assertEquals("http://birt.eclipse.org/",image.getActionHandle().getURI());
			
		 //alt-text
		  image.setProperty(IImageItemModel.ALT_TEXT_PROP,"goodtime");
		  
		  designHandle.getCommandStack().undo();
		  assertNull(image.getProperty(IImageItemModel.ALT_TEXT_PROP));
		
		  designHandle.getCommandStack().redo();
		  assertEquals("goodtime",image.getProperty(IImageItemModel.ALT_TEXT_PROP));
			
		 //border
			//border-top
			image.setProperty(IStyleModel.BORDER_TOP_STYLE_PROP,"solid");
			image.setProperty(IStyleModel.BORDER_TOP_WIDTH_PROP,"medium");
			image.setProperty(IStyleModel.BORDER_TOP_COLOR_PROP,"red");
			designHandle.getBody().add(image);
			
			assertEquals("solid",designHandle.getBody().get(0).getStringProperty(IStyleModel.BORDER_TOP_STYLE_PROP));
			assertEquals("medium",designHandle.getBody().get(0).getStringProperty(IStyleModel.BORDER_TOP_WIDTH_PROP));
			assertEquals("red",designHandle.getBody().get(0).getStringProperty(IStyleModel.BORDER_TOP_COLOR_PROP));
		
		    //border-left
			image.setProperty(IStyleModel.BORDER_LEFT_STYLE_PROP,"dotted");
			image.setProperty(IStyleModel.BORDER_LEFT_WIDTH_PROP,"5pt");
			image.setProperty(IStyleModel.BORDER_LEFT_COLOR_PROP,"gray");
			
			assertEquals("dotted",designHandle.getBody().get(0).getStringProperty(IStyleModel.BORDER_LEFT_STYLE_PROP));
			assertEquals("5pt",designHandle.getBody().get(0).getStringProperty(IStyleModel.BORDER_LEFT_WIDTH_PROP));
			assertEquals("gray",designHandle.getBody().get(0).getStringProperty(IStyleModel.BORDER_LEFT_COLOR_PROP));
		 
		    //border-bottom
			image.setProperty(IStyleModel.BORDER_BOTTOM_STYLE_PROP,"dashed");
			image.setProperty(IStyleModel.BORDER_BOTTOM_WIDTH_PROP,"thick");
			image.setProperty(IStyleModel.BORDER_BOTTOM_COLOR_PROP,"yellow");
			
			assertEquals("dashed",designHandle.getBody().get(0).getStringProperty(IStyleModel.BORDER_BOTTOM_STYLE_PROP));
			assertEquals("thick",designHandle.getBody().get(0).getStringProperty(IStyleModel.BORDER_BOTTOM_WIDTH_PROP));
			assertEquals("yellow",designHandle.getBody().get(0).getStringProperty(IStyleModel.BORDER_BOTTOM_COLOR_PROP));
		 
			//border-right
			image.setProperty(IStyleModel.BORDER_RIGHT_STYLE_PROP,"none");
			image.setProperty(IStyleModel.BORDER_RIGHT_WIDTH_PROP,"100%");
			image.setProperty(IStyleModel.BORDER_RIGHT_COLOR_PROP,"maroon");
			
			assertEquals("none",designHandle.getBody().get(0).getStringProperty(IStyleModel.BORDER_RIGHT_STYLE_PROP));
			assertEquals("100%",designHandle.getBody().get(0).getStringProperty(IStyleModel.BORDER_RIGHT_WIDTH_PROP));
			assertEquals("maroon",designHandle.getBody().get(0).getStringProperty(IStyleModel.BORDER_RIGHT_COLOR_PROP));
		 
		    //page-break
			image.setProperty(IStyleModel.PAGE_BREAK_AFTER_PROP,"always");
		    
		    designHandle.getCommandStack().undo();
		    assertEquals("auto",image.getProperty(IStyleModel.PAGE_BREAK_AFTER_PROP));
		    
		    designHandle.getCommandStack().redo();
		    assertEquals("always",image.getProperty(IStyleModel.PAGE_BREAK_AFTER_PROP));
		    
		    image.setProperty(IStyleModel.PAGE_BREAK_BEFORE_PROP,"avoid");
		    
		    designHandle.getCommandStack().undo();   
		    designHandle.getCommandStack().redo();
		    assertEquals("avoid",image.getProperty(IStyleModel.PAGE_BREAK_BEFORE_PROP));
		    
			//visibility
		    
	}
	     public void testInsertImage() throws Exception
	    {   	    	 
	    	 
	   	    	 
		    //embedded bmp image
			embeddedimage.setName("new bmp image");
			embeddedimage.setType(DesignChoiceConstants.IMAGE_TYPE_IMAGE_BMP);
			embeddedimage.setData("newdata".getBytes( EmbeddedImage.CHARSET ) );
	 	    designHandle.addImage(embeddedimage);
	    	 
	        assertEquals("new bmp image",embeddedimage.getName());
	        assertEquals(DesignChoiceConstants.IMAGE_TYPE_IMAGE_BMP,embeddedimage.getType(design));
	        
	    	 //embedded gif image
	    	PropertyHandle propHandle = designHandle.getPropertyHandle( ReportDesign.IMAGES_PROP );
	    	Iterator iter = propHandle.iterator( );
	    	EmbeddedImageHandle embeddedimagehandle = (EmbeddedImageHandle) iter.next( );
	    	
	    	embeddedimagehandle.setName("new gif image");
	    	byte[] data = {1, 2, 3, 4};
	    	embeddedimagehandle.setData(data);
	    	embeddedimagehandle.setType(DesignChoiceConstants.IMAGE_TYPE_IMAGE_GIF);
		   
	 	    
	 	    byte[] retData = embeddedimagehandle.getData( );
			assertEquals( 4, retData.length );
			assertEquals( 1, retData[0] );
			assertEquals( 2, retData[1] );
			assertEquals( 3, retData[2] );
			assertEquals( 4, retData[3] );
			
			assertEquals( "new gif image", embeddedimagehandle.getName( ) ); 
			assertEquals( DesignChoiceConstants.IMAGE_TYPE_IMAGE_GIF,embeddedimagehandle.getType( ) );
	        
	        
	        //embedded jpeg image
	        embeddedimage1.setName("new jpeg image");
	    	embeddedimage1.setType(DesignChoiceConstants.IMAGE_TYPE_IMAGE_JPEG);
			embeddedimage1.setData("anotherdata".getBytes( EmbeddedImage.CHARSET ) );
	 	    designHandle.addImage(embeddedimage1);
	 	    
	 	    assertEquals("new jpeg image",embeddedimage1.getName());
	        assertEquals(DesignChoiceConstants.IMAGE_TYPE_IMAGE_JPEG,embeddedimage1.getType(design));
	        
	        //embedded png image
	        EmbeddedImageHandle embeddedimagehandle1 = (EmbeddedImageHandle)designHandle.findImage("new jpeg image").getHandle(propHandle);
	        embeddedimagehandle1.setName("new png image");
	        embeddedimagehandle1.setType(DesignChoiceConstants.IMAGE_TYPE_IMAGE_PNG);
	    	byte[] data1 = {5,6,7};
	    	embeddedimagehandle1.setData( data1 );
	 	    
	    	 
	 	    byte[] retData1 = embeddedimagehandle1.getData( );
			assertEquals( 3, retData1.length );
			assertEquals( 5, retData1[0] );
			assertEquals( 6, retData1[1] );
			assertEquals( 7, retData1[2] );
			
			
			assertEquals( "new png image", embeddedimagehandle1.getName( ) ); 
			assertEquals( DesignChoiceConstants.IMAGE_TYPE_IMAGE_PNG,embeddedimagehandle1.getType( ) );
	    	 
		    //file URI
			image.setURI("\\\\qaant\\QA\\Pictures\\pic\\aa.gif");
			designHandle.getBody().add(image);
			assertEquals("\\\\qaant\\QA\\Pictures\\pic\\aa.gif", image.getURI());
			
			//relative URI
			
			image.setURI("..\\Water lilies.jpg");
			assertEquals("..\\Water lilies.jpg", image.getURI());
			
			image.setURI("\\Water lilies.jpg");
			assertEquals("\\Water lilies.jpg", image.getURI());
	    	 
	    }
	
	
}
