package org.eclipse.birt.report.tests.model.acceptance;

import java.util.Iterator;

import junit.framework.Test;
import junit.framework.TestSuite;

import org.eclipse.birt.report.model.api.DesignEngine;
import org.eclipse.birt.report.model.api.ElementFactory;
import org.eclipse.birt.report.model.api.HideRuleHandle;
import org.eclipse.birt.report.model.api.LabelHandle;
import org.eclipse.birt.report.model.api.PropertyHandle;
import org.eclipse.birt.report.model.api.StructureFactory;
import org.eclipse.birt.report.model.api.activity.SemanticException;
import org.eclipse.birt.report.model.api.command.NameException;
import org.eclipse.birt.report.model.api.elements.DesignChoiceConstants;
import org.eclipse.birt.report.model.api.elements.structures.Action;
import org.eclipse.birt.report.model.api.elements.structures.HideRule;
import org.eclipse.birt.report.model.core.DesignElement;
import org.eclipse.birt.report.model.elements.ReportDesign;
import org.eclipse.birt.report.model.elements.ReportItem;
import org.eclipse.birt.report.model.elements.interfaces.IStyleModel;
import org.eclipse.birt.report.tests.model.BaseTestCase;

import com.ibm.icu.util.ULocale;

public class LabelElementHandleTest extends BaseTestCase {

	private ElementFactory factory = null;

	private LabelHandle label = null;

	public LabelElementHandleTest(String name) {
		super(name);
		// TODO Auto-generated constructor stub
	}

	public static Test suite() {
		return new TestSuite(LabelElementHandleTest.class);
	}

	public void setUp() throws Exception {
		super.setUp();
		sessionHandle = DesignEngine.newSession( ULocale.CHINESE);
		designHandle = sessionHandle.createDesign();
		design = (ReportDesign) designHandle.getModule();
		factory = new ElementFactory(design);
		label = factory.newLabel("myLabel");

	}

	public void testLabelOperations() throws Exception {

		// add a label
		label.setStringProperty(IStyleModel.BACKGROUND_COLOR_PROP, "red");
		designHandle.getBody().add(label);
		assertEquals("red", designHandle.getBody().get(0).getStringProperty(
				IStyleModel.BACKGROUND_COLOR_PROP));

		// undo it
		assertTrue(designHandle.getCommandStack().canUndo());
		assertFalse(designHandle.getCommandStack().canRedo());
		designHandle.getCommandStack().undo();
		assertEquals(0, designHandle.getBody().getCount());

		// redo it
		assertTrue(designHandle.getCommandStack().canRedo());
		designHandle.getCommandStack().redo();
		assertEquals("myLabel", designHandle.getBody().get(0).getName());

		// delete the label
		designHandle.getBody().drop(label);
		assertEquals(0, designHandle.getBody().getCount());

		// undo it
		assertTrue(designHandle.getCommandStack().canUndo());
		assertFalse(designHandle.getCommandStack().canRedo());
		designHandle.getCommandStack().undo();
		assertEquals("myLabel", designHandle.getBody().get(0).getName());

		// redo it
		designHandle.getCommandStack().redo();
		assertEquals(0, designHandle.getBody().getCount());

		// Edit the label
		designHandle.getCommandStack().undo();
		label.setText("a'||'&&");
		assertEquals("a'||'&&", ((LabelHandle) designHandle.getBody().get(0))
				.getText());

		designHandle.getCommandStack().undo();
		assertNull(((LabelHandle) designHandle.getBody().get(0)).getText());

		designHandle.getCommandStack().redo();
		assertEquals("a'||'&&", ((LabelHandle) designHandle.getBody().get(0))
				.getText());

		// test chinese
		label.setText("&#23433;&#35759;&#36719;&#20214;");
		assertEquals("&#23433;&#35759;&#36719;&#20214;", ((LabelHandle) designHandle.getBody().get(0))
				.getText());

		designHandle.getCommandStack().undo();
		assertEquals("a'||'&&", ((LabelHandle) designHandle.getBody().get(0))
				.getText());

		designHandle.getCommandStack().redo();
		assertEquals("&#23433;&#35759;&#36719;&#20214;", ((LabelHandle) designHandle.getBody().get(0))
				.getText());

		// copy and paste the label
		DesignElement newlabel = (DesignElement) label.copy();
		assertNotNull(newlabel);
		assertEquals(label.getName(), newlabel.getName());
		design = (ReportDesign) designHandle.getModule();
		try {

			designHandle.getBody().add(newlabel.getHandle(design));
			fail();
		} catch (NameException e) {
			assertNotNull(e);
		}
		newlabel.setName("mylabel1");
		designHandle.getBody().add(newlabel.getHandle(design));
		assertEquals("mylabel1", designHandle.getBody().get(1).getName());

		designHandle.getCommandStack().undo();
		assertEquals(1, designHandle.getBody().getCount());

		designHandle.getCommandStack().redo();
		assertEquals(2, designHandle.getBody().getCount());

		// rename
		label.setName("goldenlabel");
		newlabel.setName("silverlabel");
		assertEquals("goldenlabel", designHandle.getBody().get(0).getName());
		assertEquals("silverlabel", designHandle.getBody().get(1).getName());
	}

	public void testLabelProperties() throws Exception {
		designHandle.getBody().add(label);

		// font-family
		label.setProperty(IStyleModel.FONT_FAMILY_PROP, "Arial");
		assertEquals("Arial", designHandle.getBody().get(0).getProperty(
				IStyleModel.FONT_FAMILY_PROP));

		// undo it
		assertTrue(designHandle.getCommandStack().canUndo());
		assertFalse(designHandle.getCommandStack().canRedo());
		designHandle.getCommandStack().undo();
		assertEquals("serif", designHandle.getBody().get(0).getProperty(
				IStyleModel.FONT_FAMILY_PROP));

		// redo it
		assertTrue(designHandle.getCommandStack().canRedo());
		designHandle.getCommandStack().redo();
		assertEquals("Arial", designHandle.getBody().get(0).getProperty(
				IStyleModel.FONT_FAMILY_PROP));

		// font-size
		label.setProperty(IStyleModel.FONT_SIZE_PROP, "large");
		assertEquals("large", designHandle.getBody().get(0).getProperty(
				IStyleModel.FONT_SIZE_PROP));

		designHandle.getCommandStack().undo();
		assertEquals("medium", designHandle.getBody().get(0).getProperty(
				IStyleModel.FONT_SIZE_PROP));

		designHandle.getCommandStack().redo();
		assertEquals("large", designHandle.getBody().get(0).getProperty(
				IStyleModel.FONT_SIZE_PROP));

		// wrong font-size value
		try {
			label.setProperty(IStyleModel.FONT_SIZE_PROP, "-1mm");
			fail();
		} catch (SemanticException e) {
			assertNotNull(e);
		}

		try {
			label.setProperty(IStyleModel.FONT_SIZE_PROP, "0mm");
			fail();
		} catch (SemanticException e) {
			assertNotNull(e);
		}

		try {
			label.setProperty(IStyleModel.FONT_SIZE_PROP, "abpoints");
			fail();
		} catch (SemanticException e) {
			assertNotNull(e);
		}

		label.setProperty(IStyleModel.FONT_SIZE_PROP, "50in");
		assertEquals("50in", designHandle.getBody().get(0).getStringProperty(
				IStyleModel.FONT_SIZE_PROP));

		designHandle.getCommandStack().undo();
		assertEquals("large", designHandle.getBody().get(0).getStringProperty(
				IStyleModel.FONT_SIZE_PROP));

		designHandle.getCommandStack().redo();
		assertEquals("50in", designHandle.getBody().get(0).getStringProperty(
				IStyleModel.FONT_SIZE_PROP));

		// font-color

		label.setProperty(IStyleModel.COLOR_PROP, "olive");
		assertEquals("olive", designHandle.getBody().get(0).getStringProperty(
				IStyleModel.COLOR_PROP));

		designHandle.getCommandStack().undo();
		assertEquals("black", designHandle.getBody().get(0).getStringProperty(
				IStyleModel.COLOR_PROP));

		designHandle.getCommandStack().redo();
		assertEquals("olive", designHandle.getBody().get(0).getStringProperty(
				IStyleModel.COLOR_PROP));

		// font-style
		label.setProperty(IStyleModel.FONT_STYLE_PROP, "italic");
		assertEquals("italic", designHandle.getBody().get(0).getStringProperty(
				IStyleModel.FONT_STYLE_PROP));

		designHandle.getCommandStack().undo();
		assertEquals("normal", designHandle.getBody().get(0).getStringProperty(
				IStyleModel.FONT_STYLE_PROP));

		designHandle.getCommandStack().redo();
		assertEquals("italic", designHandle.getBody().get(0).getStringProperty(
				IStyleModel.FONT_STYLE_PROP));

		// text-underline
		label.setProperty(IStyleModel.TEXT_UNDERLINE_PROP, "underline");
		assertEquals("underline", designHandle.getBody().get(0)
				.getStringProperty(IStyleModel.TEXT_UNDERLINE_PROP));

		designHandle.getCommandStack().undo();
		assertEquals("none", designHandle.getBody().get(0).getStringProperty(
				IStyleModel.TEXT_UNDERLINE_PROP));

		designHandle.getCommandStack().redo();
		assertEquals("underline", designHandle.getBody().get(0)
				.getStringProperty(IStyleModel.TEXT_UNDERLINE_PROP));

		// text-align
		label.setProperty(IStyleModel.TEXT_ALIGN_PROP, "center");
		assertEquals("center", designHandle.getBody().get(0).getStringProperty(
				IStyleModel.TEXT_ALIGN_PROP));

		designHandle.getCommandStack().undo();
		assertEquals("left", designHandle.getBody().get(0).getStringProperty(
				IStyleModel.TEXT_ALIGN_PROP));

		designHandle.getCommandStack().redo();
		assertEquals("center", designHandle.getBody().get(0).getStringProperty(
				IStyleModel.TEXT_ALIGN_PROP));

		// background color
		label.setProperty(IStyleModel.BACKGROUND_COLOR_PROP, "silver");
		assertEquals("silver", designHandle.getBody().get(0).getStringProperty(
				IStyleModel.BACKGROUND_COLOR_PROP));

		designHandle.getCommandStack().undo();
		assertNull(designHandle.getBody().get(0).getStringProperty(
				IStyleModel.BACKGROUND_COLOR_PROP));

		designHandle.getCommandStack().redo();
		assertEquals("silver", designHandle.getBody().get(0).getStringProperty(
				IStyleModel.BACKGROUND_COLOR_PROP));

		// padding
		// padding-top
		label.setProperty(IStyleModel.PADDING_TOP_PROP, "2cm");
		assertEquals("2cm", designHandle.getBody().get(0).getStringProperty(
				IStyleModel.PADDING_TOP_PROP));

		designHandle.getCommandStack().undo();
		assertEquals("0pt", designHandle.getBody().get(0).getStringProperty(
				IStyleModel.PADDING_TOP_PROP));

		designHandle.getCommandStack().redo();
		assertEquals("2cm", designHandle.getBody().get(0).getStringProperty(
				IStyleModel.PADDING_TOP_PROP));

		label.setProperty(IStyleModel.PADDING_TOP_PROP, "5000cm");
		assertEquals("5000cm", designHandle.getBody().get(0).getStringProperty(
				IStyleModel.PADDING_TOP_PROP));

		designHandle.getCommandStack().undo();
		assertEquals("2cm", designHandle.getBody().get(0).getStringProperty(
				IStyleModel.PADDING_TOP_PROP));

		designHandle.getCommandStack().redo();
		assertEquals("5000cm", designHandle.getBody().get(0).getStringProperty(
				IStyleModel.PADDING_TOP_PROP));

		// wrong padding value
		try {
			label.setProperty(IStyleModel.PADDING_TOP_PROP, "acm");
			fail();
		} catch (SemanticException e) {
			assertNotNull(e);
		}

		try {
			label.setProperty(IStyleModel.PADDING_TOP_PROP, "-5in");
			fail();
		} catch (SemanticException e) {
			assertNotNull(e);
		}

		// padding-left
		label.setProperty(IStyleModel.PADDING_LEFT_PROP, "2px");
		assertEquals("2px", designHandle.getBody().get(0).getStringProperty(
				IStyleModel.PADDING_LEFT_PROP));

		designHandle.getCommandStack().undo();
		assertEquals("0pt", designHandle.getBody().get(0).getStringProperty(
				IStyleModel.PADDING_LEFT_PROP));

		designHandle.getCommandStack().redo();
		assertEquals("2px", designHandle.getBody().get(0).getStringProperty(
				IStyleModel.PADDING_LEFT_PROP));

		// padding-bottom
		label.setProperty(IStyleModel.PADDING_BOTTOM_PROP, "100%");
		assertEquals("100%", designHandle.getBody().get(0).getStringProperty(
				IStyleModel.PADDING_BOTTOM_PROP));

		designHandle.getCommandStack().undo();
		assertEquals("0pt", designHandle.getBody().get(0).getStringProperty(
				IStyleModel.PADDING_BOTTOM_PROP));

		designHandle.getCommandStack().redo();
		assertEquals("100%", designHandle.getBody().get(0).getStringProperty(
				IStyleModel.PADDING_BOTTOM_PROP));

		// padding-right
		label.setProperty(IStyleModel.PADDING_RIGHT_PROP, "5em");
		assertEquals("5em", designHandle.getBody().get(0).getStringProperty(
				IStyleModel.PADDING_RIGHT_PROP));

		designHandle.getCommandStack().undo();
		assertEquals("0pt", designHandle.getBody().get(0).getStringProperty(
				IStyleModel.PADDING_RIGHT_PROP));

		designHandle.getCommandStack().redo();
		assertEquals("5em", designHandle.getBody().get(0).getStringProperty(
				IStyleModel.PADDING_RIGHT_PROP));

		// border
		// border-top
		label.setProperty(IStyleModel.BORDER_TOP_STYLE_PROP, "solid");
		label.setProperty(IStyleModel.BORDER_TOP_WIDTH_PROP, "medium");
		label.setProperty(IStyleModel.BORDER_TOP_COLOR_PROP, "red");

		assertEquals("solid", designHandle.getBody().get(0).getStringProperty(
				IStyleModel.BORDER_TOP_STYLE_PROP));
		assertEquals("medium", designHandle.getBody().get(0).getStringProperty(
				IStyleModel.BORDER_TOP_WIDTH_PROP));
		assertEquals("red", designHandle.getBody().get(0).getStringProperty(
				IStyleModel.BORDER_TOP_COLOR_PROP));

		// border-left
		label.setProperty(IStyleModel.BORDER_LEFT_STYLE_PROP, "dotted");
		label.setProperty(IStyleModel.BORDER_LEFT_WIDTH_PROP, "5pt");
		label.setProperty(IStyleModel.BORDER_LEFT_COLOR_PROP, "gray");

		assertEquals("dotted", designHandle.getBody().get(0).getStringProperty(
				IStyleModel.BORDER_LEFT_STYLE_PROP));
		assertEquals("5pt", designHandle.getBody().get(0).getStringProperty(
				IStyleModel.BORDER_LEFT_WIDTH_PROP));
		assertEquals("gray", designHandle.getBody().get(0).getStringProperty(
				IStyleModel.BORDER_LEFT_COLOR_PROP));

		// border-bottom
		label.setProperty(IStyleModel.BORDER_BOTTOM_STYLE_PROP, "dashed");
		label.setProperty(IStyleModel.BORDER_BOTTOM_WIDTH_PROP, "thick");
		label.setProperty(IStyleModel.BORDER_BOTTOM_COLOR_PROP, "yellow");

		assertEquals("dashed", designHandle.getBody().get(0).getStringProperty(
				IStyleModel.BORDER_BOTTOM_STYLE_PROP));
		assertEquals("thick", designHandle.getBody().get(0).getStringProperty(
				IStyleModel.BORDER_BOTTOM_WIDTH_PROP));
		assertEquals("yellow", designHandle.getBody().get(0).getStringProperty(
				IStyleModel.BORDER_BOTTOM_COLOR_PROP));

		// border-right
		label.setProperty(IStyleModel.BORDER_RIGHT_STYLE_PROP, "none");
		label.setProperty(IStyleModel.BORDER_RIGHT_WIDTH_PROP, "100%");
		label.setProperty(IStyleModel.BORDER_RIGHT_COLOR_PROP, "maroon");

		assertEquals("none", designHandle.getBody().get(0).getStringProperty(
				IStyleModel.BORDER_RIGHT_STYLE_PROP));
		assertEquals("100%", designHandle.getBody().get(0).getStringProperty(
				IStyleModel.BORDER_RIGHT_WIDTH_PROP));
		assertEquals("maroon", designHandle.getBody().get(0).getStringProperty(
				IStyleModel.BORDER_RIGHT_COLOR_PROP));

		// hyperlink
		StructureFactory factory = new StructureFactory();

		Action action = factory.createAction();
		label.setAction(action);
		label.getActionHandle().setLinkType(
				DesignChoiceConstants.ACTION_LINK_TYPE_HYPERLINK);
		label.getActionHandle().setURI("http://birt.eclipse.org/");
		assertEquals("http://birt.eclipse.org/", label.getActionHandle()
				.getURI());

		designHandle.getCommandStack().undo();
		assertNull(label.getActionHandle().getURI());

		designHandle.getCommandStack().redo();
		assertEquals("http://birt.eclipse.org/", label.getActionHandle()
				.getURI());

		// page-break
		label.setProperty(IStyleModel.PAGE_BREAK_AFTER_PROP, "always");

		designHandle.getCommandStack().undo();
		assertEquals("auto", label
				.getProperty(IStyleModel.PAGE_BREAK_AFTER_PROP));

		designHandle.getCommandStack().redo();
		assertEquals("always", label
				.getProperty(IStyleModel.PAGE_BREAK_AFTER_PROP));

		label.setProperty(IStyleModel.PAGE_BREAK_BEFORE_PROP, "avoid");

		designHandle.getCommandStack().undo();
		designHandle.getCommandStack().redo();
		assertEquals("avoid", label
				.getProperty(IStyleModel.PAGE_BREAK_BEFORE_PROP));

		// visibility
		HideRule hiderule = StructureFactory.createHideRule();
		hiderule.setExpression("new expression");
		hiderule.setFormat(DesignChoiceConstants.FORMAT_TYPE_PDF);
		PropertyHandle propHandle = label
				.getPropertyHandle(ReportItem.VISIBILITY_PROP);
		propHandle.addItem(hiderule);
		Iterator iter = propHandle.iterator();
		HideRuleHandle hideruleHandle = (HideRuleHandle) iter.next();

		assertEquals("new expression", hideruleHandle.getExpression());
		assertEquals(DesignChoiceConstants.FORMAT_TYPE_PDF, hideruleHandle
				.getFormat());

		// localization
		designHandle.addTranslation("actuate", null, "aaa");
		label.setText("abc");
		label.setTextKey("actuate");
		assertEquals("aaa", label.getDisplayText());

		// properties
		label.setProperty(IStyleModel.DISPLAY_PROP, "inline");

		designHandle.getCommandStack().undo();
		assertEquals("block", label.getProperty(IStyleModel.DISPLAY_PROP));
		designHandle.getCommandStack().redo();
		assertEquals("inline", label.getProperty(IStyleModel.DISPLAY_PROP));

	}
	
	public void testLocation() throws Exception 
	{
	   openDesign("labellocation.xml");
	   designHandle.setIncludeResource("a");
	   label = (LabelHandle) designHandle.findElement("laa");
	   label.setTextKey("actuate");
	   assertEquals("aaa",label.getDisplayText());
	   
	} 

}
