package org.eclipse.birt.report.tests.model.acceptance;

import java.util.Iterator;

import junit.framework.Test;
import junit.framework.TestSuite;

import org.eclipse.birt.report.model.api.DataSetHandle;
import org.eclipse.birt.report.model.api.DesignEngine;
import org.eclipse.birt.report.model.api.ElementFactory;
import org.eclipse.birt.report.model.api.HideRuleHandle;
import org.eclipse.birt.report.model.api.LabelHandle;
import org.eclipse.birt.report.model.api.OdaDataSetHandle;
import org.eclipse.birt.report.model.api.PropertyHandle;
import org.eclipse.birt.report.model.api.StructureFactory;
import org.eclipse.birt.report.model.api.TextItemHandle;
import org.eclipse.birt.report.model.elements.ReportDesign;
import org.eclipse.birt.report.model.elements.ReportItem;
import org.eclipse.birt.report.model.elements.TextItem;
import org.eclipse.birt.report.model.elements.interfaces.IStyleModel;
import org.eclipse.birt.report.model.api.command.NameException;
import org.eclipse.birt.report.model.api.core.IDesignElement;
import org.eclipse.birt.report.model.api.elements.DesignChoiceConstants;
import org.eclipse.birt.report.model.api.elements.structures.HideRule;
import org.eclipse.birt.report.model.elements.interfaces.IDesignElementModel;
import org.eclipse.birt.report.model.elements.interfaces.ITextItemModel;
import org.eclipse.birt.report.model.elements.interfaces.IReportItemModel;
import org.eclipse.birt.report.tests.model.BaseTestCase;

import com.ibm.icu.util.ULocale;

public class TextItemHandleTest extends BaseTestCase
{
	private ElementFactory factory = null;
    private TextItemHandle text = null;
	
	public TextItemHandleTest(String name) {
		super(name);
		// TODO Auto-generated constructor stub
	}

	public static Test suite()
	{
		return new TestSuite(TextItemHandleTest.class);
	}
	public void setUp( ) throws Exception
	{
		super.setUp();
		sessionHandle = DesignEngine.newSession(ULocale.ENGLISH);
		designHandle = sessionHandle.createDesign();
		design = (ReportDesign)designHandle.getModule();
		factory = new ElementFactory( design );
		text = factory.newTextItem("mytext");
	}
	
	public void testTextItemOperations() throws Exception
	{
//		add a text
		text.setStringProperty(IStyleModel.BACKGROUND_COLOR_PROP,"red");
		designHandle.getBody().add(text);
		assertEquals("red",designHandle.getBody().get(0).getStringProperty(IStyleModel.BACKGROUND_COLOR_PROP));
		
		//undo it
		assertTrue(designHandle.getCommandStack().canUndo());
	    assertFalse(designHandle.getCommandStack().canRedo());
	    designHandle.getCommandStack().undo();
	    assertEquals(0,designHandle.getBody().getCount());
	    
	    //redo it
	    assertTrue(designHandle.getCommandStack().canRedo());
	    designHandle.getCommandStack().redo();
	    assertEquals("mytext",designHandle.getBody().get(0).getName());
		
	    //delete the text
	    designHandle.getBody().drop(text);
	    assertEquals(0,designHandle.getBody().getCount());
	    
	    //undo it
	    assertTrue(designHandle.getCommandStack().canUndo());
	    assertFalse(designHandle.getCommandStack().canRedo());
	    designHandle.getCommandStack().undo();
	    assertEquals("mytext",designHandle.getBody().get(0).getName());
	    
	    //redo it
	    designHandle.getCommandStack().redo();
	    assertEquals(0,designHandle.getBody().getCount());
		
	    //Edit the text
	    //special characters as content
	    designHandle.getCommandStack().undo();
	    text.setContent("@#$%^&...");
	    assertEquals("@#$%^&...",((TextItemHandle)designHandle.getBody().get(0)).getContent());
	  
	    designHandle.getCommandStack().undo();
	    assertNull(((TextItemHandle)designHandle.getBody().get(0)).getContent());
	    
	    designHandle.getCommandStack().redo();
	    assertEquals("@#$%^&...",((TextItemHandle)designHandle.getBody().get(0)).getContent());
	    
	    //number as content
	    text.setContent("123456");
	    assertEquals("123456",((TextItemHandle)designHandle.getBody().get(0)).getContent());
	  
	    designHandle.getCommandStack().undo();
	    assertEquals("@#$%^&...",((TextItemHandle)designHandle.getBody().get(0)).getContent());
	    
	    designHandle.getCommandStack().redo();
	    assertEquals("123456",((TextItemHandle)designHandle.getBody().get(0)).getContent());
	    
	    //Chinese as content
	    text.setContent("&#20013;&#25991;");
	    assertEquals("&#20013;&#25991;",((TextItemHandle)designHandle.getBody().get(0)).getContent());
	  
	    designHandle.getCommandStack().undo();
	    assertEquals("123456",((TextItemHandle)designHandle.getBody().get(0)).getContent());
	    
	    designHandle.getCommandStack().redo();
	    assertEquals("&#20013;&#25991;",((TextItemHandle)designHandle.getBody().get(0)).getContent());
	    
	    //Japanese as content
	    text.setContent("&#12501;&#12521;&#12531;&#12473;&#35486;");
	    assertEquals("&#12501;&#12521;&#12531;&#12473;&#35486;",((TextItemHandle)designHandle.getBody().get(0)).getContent());
	  
	    designHandle.getCommandStack().undo();
	    assertEquals("&#20013;&#25991;",((TextItemHandle)designHandle.getBody().get(0)).getContent());
	    
	    designHandle.getCommandStack().redo();
	    assertEquals("&#12501;&#12521;&#12531;&#12473;&#35486;",((TextItemHandle)designHandle.getBody().get(0)).getContent());
	    
	    //dynamic text
	    text.setContentType(DesignChoiceConstants.TEXT_DATA_CONTENT_TYPE_HTML);
	    String htmlcontent = "<html><head>" +
	    		"<title>Test Supported Elements</title>" +
       "</head><body><h1>h1:Text Supported Elements</h1>" +
       "<h2>h2: Test h2</h2><h3>h3: Test h3</h3>" +
        "<h4>h4:text h4</h4><h5>h5:test h5</h5>"+
        "<h6>h6:test h6</h6>" +
        "<a href=\"www.actuate.com\">a: test anchor a</a>"+
        "<center><font color=red>font: red words</font><i>i: italic words</i></center>"+
        "<span style=\"font-style:italic\"><i>span: style font-style italic i: italic</i></span>"+
        "<b>b: bold text</b>"+
        "<br>br: line break <em>em: Emphasis</em>"+
        "<strong>strong: strong emphasis</strong>"+
        "<code>code:"+ 
        " private void init( MasterPageDesign masterPage )"+
         "{"+
          "topMargin = masterPage.getTopMargin( );"+
          "rightMargin = new DimensionType( 0.0, DimensionType.UNITS_CM );"+
         "}"+
        "</code>"+
        "<dl>"+
         "<dt>dt: data item1</dt>"+
         "<dd>dd: data description1</dd>"+
         "<dt>dt: data item 2</dt>"+
         "<dd>dd: data description 2</dd>"+
        "</dl>"+
        "<ol>"+
         "<li>li: list item in ol order list</li>"+
         "<ul>"+
          "<li>li : in ul unorder list and ul in ol</li>"+
          "<li>li2 in ul and ul in ol</li>"+
         "</ul>"+
        "<li>li2: in ol</li>"+
        "</ol>"+
        "<div>div1div1div1div1div1div1div1div1div1div1div1div1div1div1div1div1div1div1div1div1div1div1div1div1div1div1div1div1div1div1div1div1div1div1div1div1div1div1div1div1div1div1div1div1div1div1div1div1</div><div>div2div2div2div2div2div2div2div2div2div2div2div2div2div2div2div2div2div2div2div2div2div2div2div2div2div2</div>"+
        "<br><pre>pre: Block of fixed-width text. the text truncates if it is wider than the dynamic text control. the text does notwrap</pre>"+
        "<br>the data number after modified is <del>1234.567</del><ins>7654.321</ins>"+
        "<ins datetime=\"2004-10-10 10:30:30\" cite=\"http://www.actuate.com/Corporate/news/PressRelease.asp?ArticleId=8871\">ins date:2004-10-10 10:30:30 cite:http://www.actuate.com/Corporate/news/PressRelease.asp?ArticleId=8871</ins>so, that's the use of ins"+
        "1<p>p: words in p</p>words out p"+
        "<br>"+
        "Sub like X<sub>2</sub>"+
        "Sup like X<sup>2</sup>"+
        "<br>"+
        "<img src=\"http://www.google.com/intl/zh-CN_ALL/images/logo.gif\" alt=\"the img in google\">img here is a image from google</body></html>";
	  
	    text.setContent(htmlcontent);
	    
	    designHandle.getCommandStack().undo();
	    assertEquals("&#12501;&#12521;&#12531;&#12473;&#35486;",((TextItemHandle)designHandle.getBody().get(0)).getContent());
	    
	    designHandle.getCommandStack().redo();
	    assertEquals(htmlcontent,((TextItemHandle)designHandle.getBody().get(0)).getContent());
	    
	    //auto text
	    text.setContentType(DesignChoiceConstants.TEXT_CONTENT_TYPE_AUTO);
	    text.setContent("<B>bbbbbbbbbbbbbbbb</B>");
	    
	    designHandle.getCommandStack().undo();
	    assertEquals(htmlcontent,text.getContent());
	    
	    designHandle.getCommandStack().redo();
	    assertEquals("<B>bbbbbbbbbbbbbbbb</B>",((TextItemHandle)designHandle.getBody().get(0)).getContent());
	    
	    //content type conversion
	    text.setContentType(DesignChoiceConstants.TEXT_CONTENT_TYPE_PLAIN);
	    
	    designHandle.getCommandStack().undo();
	    assertEquals(DesignChoiceConstants.TEXT_CONTENT_TYPE_AUTO,((TextItemHandle)designHandle.getBody().get(0)).getContentType());
	    
	    designHandle.getCommandStack().redo();
	    assertEquals(DesignChoiceConstants.TEXT_CONTENT_TYPE_PLAIN,((TextItemHandle)designHandle.getBody().get(0)).getContentType());
	    
	    //copy & paste
	    TextItem newtext = (TextItem)text.copy();
	    assertNotNull(newtext);
	    assertEquals(text.getName(),newtext.getName());
	    design = (ReportDesign)designHandle.getModule();
	    
	    try{
	    	designHandle.getBody().add(newtext.getHandle(design));
		    fail();
	    }
	    catch (NameException e)
	    {
	      assertNotNull(e);
	    }
	    newtext.setName("mytext1");
	    designHandle.getBody().add(newtext.getHandle(design));
	    assertEquals("mytext1",designHandle.getBody().get(1).getName());
	    
	    designHandle.getCommandStack().undo();
	    assertEquals(1,designHandle.getBody().getCount());
	    
	    designHandle.getCommandStack().redo();
	    assertEquals(2,designHandle.getBody().getCount());
	    
	    //cut & paste
	    TextItem newtext1 = (TextItem)text.copy();
	    assertNotNull(newtext1);
	    assertEquals(text.getName(),newtext1.getName());
	    text.drop();
	    designHandle.getBody().add(newtext1.getHandle(design));
	    
	    designHandle.getCommandStack().undo();
	    assertEquals(1,designHandle.getBody().getCount());
	    
	    designHandle.getCommandStack().redo();
	    assertEquals(2,designHandle.getBody().getCount());
	    
	    //rename
	    newtext.setName("goldentext");
	    newtext1.setName("silvertext");
	    assertEquals("goldentext",designHandle.getBody().get(0).getName());
	    assertEquals("silvertext",designHandle.getBody().get(1).getName());
	  
	}
	public void testTextProperties() throws Exception
	{
		designHandle.getBody().add(text);
		
		//general
		text.setProperty(IStyleModel.FONT_SIZE_PROP, "10pt");
		text.setProperty(IStyleModel.COLOR_PROP,"blue");
		text.setProperty(IStyleModel.BACKGROUND_COLOR_PROP,"#0000FF");
		text.setProperty(IStyleModel.FONT_STYLE_PROP,"italic");
		text.setProperty(IStyleModel.FONT_WEIGHT_PROP,"bold");
		text.setProperty(IStyleModel.TEXT_LINE_THROUGH_PROP,"line-through");
		text.setProperty(IStyleModel.TEXT_UNDERLINE_PROP,"underline");
		text.setProperty(IStyleModel.TEXT_ALIGN_PROP,"right");
		
		assertEquals("10pt",designHandle.getBody().get(0).getStringProperty(IStyleModel.FONT_SIZE_PROP));
		assertEquals("blue",designHandle.getBody().get(0).getProperty(IStyleModel.COLOR_PROP));
		assertEquals("#0000FF",designHandle.getBody().get(0).getStringProperty(IStyleModel.BACKGROUND_COLOR_PROP));
		assertEquals("italic",designHandle.getBody().get(0).getStringProperty(IStyleModel.FONT_STYLE_PROP));
		assertEquals("bold",designHandle.getBody().get(0).getStringProperty(IStyleModel.FONT_WEIGHT_PROP));
		assertEquals("line-through",designHandle.getBody().get(0).getStringProperty(IStyleModel.TEXT_LINE_THROUGH_PROP));
		assertEquals("underline",designHandle.getBody().get(0).getStringProperty(IStyleModel.TEXT_UNDERLINE_PROP));
		assertEquals("right",designHandle.getBody().get(0).getStringProperty(IStyleModel.TEXT_ALIGN_PROP));
	
		//padding
		text.setProperty(IStyleModel.PADDING_TOP_PROP,"2cm");
		assertEquals("2cm",designHandle.getBody().get(0).getStringProperty(IStyleModel.PADDING_TOP_PROP));
		
		designHandle.getCommandStack().undo();
		assertEquals("0pt", designHandle.getBody().get(0).getStringProperty(IStyleModel.PADDING_TOP_PROP));
		
		designHandle.getCommandStack().redo();
		assertEquals("2cm", designHandle.getBody().get(0).getStringProperty(IStyleModel.PADDING_TOP_PROP));
	
		text.setProperty(IStyleModel.PADDING_LEFT_PROP,"10mm");
		assertEquals("10mm",designHandle.getBody().get(0).getStringProperty(IStyleModel.PADDING_LEFT_PROP));
		
		designHandle.getCommandStack().undo();
		assertEquals("0pt", designHandle.getBody().get(0).getStringProperty(IStyleModel.PADDING_LEFT_PROP));
		
		designHandle.getCommandStack().redo();
		assertEquals("10mm", designHandle.getBody().get(0).getStringProperty(IStyleModel.PADDING_LEFT_PROP));
	
		text.setProperty(IStyleModel.PADDING_BOTTOM_PROP,"50%");
		assertEquals("50%",designHandle.getBody().get(0).getStringProperty(IStyleModel.PADDING_BOTTOM_PROP));
		
		designHandle.getCommandStack().undo();
		assertEquals("0pt", designHandle.getBody().get(0).getStringProperty(IStyleModel.PADDING_BOTTOM_PROP));
		
		designHandle.getCommandStack().redo();
		assertEquals("50%", designHandle.getBody().get(0).getStringProperty(IStyleModel.PADDING_BOTTOM_PROP));
	
		text.setProperty(IStyleModel.PADDING_RIGHT_PROP,"8in");
		assertEquals("8in",designHandle.getBody().get(0).getStringProperty(IStyleModel.PADDING_RIGHT_PROP));
		
		designHandle.getCommandStack().undo();
		assertEquals("0pt", designHandle.getBody().get(0).getStringProperty(IStyleModel.PADDING_RIGHT_PROP));
		
		designHandle.getCommandStack().redo();
		assertEquals("8in", designHandle.getBody().get(0).getStringProperty(IStyleModel.PADDING_RIGHT_PROP));
	
		//border
		text.setProperty(IStyleModel.BORDER_TOP_STYLE_PROP,"solid");
		text.setProperty(IStyleModel.BORDER_TOP_WIDTH_PROP,"medium");
		text.setProperty(IStyleModel.BORDER_TOP_COLOR_PROP,"red");
		
		assertEquals("solid",designHandle.getBody().get(0).getStringProperty(IStyleModel.BORDER_TOP_STYLE_PROP));
		assertEquals("medium",designHandle.getBody().get(0).getStringProperty(IStyleModel.BORDER_TOP_WIDTH_PROP));
		assertEquals("red",designHandle.getBody().get(0).getStringProperty(IStyleModel.BORDER_TOP_COLOR_PROP));
	
		text.setProperty(IStyleModel.BORDER_LEFT_STYLE_PROP,"dotted");
		text.setProperty(IStyleModel.BORDER_LEFT_WIDTH_PROP,"5pt");
		text.setProperty(IStyleModel.BORDER_LEFT_COLOR_PROP,"gray");
		
		assertEquals("dotted",designHandle.getBody().get(0).getStringProperty(IStyleModel.BORDER_LEFT_STYLE_PROP));
		assertEquals("5pt",designHandle.getBody().get(0).getStringProperty(IStyleModel.BORDER_LEFT_WIDTH_PROP));
		assertEquals("gray",designHandle.getBody().get(0).getStringProperty(IStyleModel.BORDER_LEFT_COLOR_PROP));
	 
		text.setProperty(IStyleModel.BORDER_BOTTOM_STYLE_PROP,"dashed");
		text.setProperty(IStyleModel.BORDER_BOTTOM_WIDTH_PROP,"thick");
		text.setProperty(IStyleModel.BORDER_BOTTOM_COLOR_PROP,"yellow");
		
		assertEquals("dashed",designHandle.getBody().get(0).getStringProperty(IStyleModel.BORDER_BOTTOM_STYLE_PROP));
		assertEquals("thick",designHandle.getBody().get(0).getStringProperty(IStyleModel.BORDER_BOTTOM_WIDTH_PROP));
		assertEquals("yellow",designHandle.getBody().get(0).getStringProperty(IStyleModel.BORDER_BOTTOM_COLOR_PROP));
	 
		text.setProperty(IStyleModel.BORDER_RIGHT_STYLE_PROP,"none");
		text.setProperty(IStyleModel.BORDER_RIGHT_WIDTH_PROP,"100%");
		text.setProperty(IStyleModel.BORDER_RIGHT_COLOR_PROP,"maroon");
		
		assertEquals("none",designHandle.getBody().get(0).getStringProperty(IStyleModel.BORDER_RIGHT_STYLE_PROP));
		assertEquals("100%",designHandle.getBody().get(0).getStringProperty(IStyleModel.BORDER_RIGHT_WIDTH_PROP));
		assertEquals("maroon",designHandle.getBody().get(0).getStringProperty(IStyleModel.BORDER_RIGHT_COLOR_PROP));
	 
	    //margin
		text.setProperty(IStyleModel.MARGIN_TOP_PROP,"2cm");
		assertEquals("2cm",designHandle.getBody().get(0).getStringProperty(IStyleModel.MARGIN_TOP_PROP));
		
		designHandle.getCommandStack().undo();
		assertEquals("0pt", designHandle.getBody().get(0).getStringProperty(IStyleModel.MARGIN_TOP_PROP));
		
		designHandle.getCommandStack().redo();
		assertEquals("2cm", designHandle.getBody().get(0).getStringProperty(IStyleModel.MARGIN_TOP_PROP));
	
		text.setProperty(IStyleModel.MARGIN_LEFT_PROP,"10mm");
		assertEquals("10mm",designHandle.getBody().get(0).getStringProperty(IStyleModel.MARGIN_LEFT_PROP));
		
		designHandle.getCommandStack().undo();
		assertEquals("0pt", designHandle.getBody().get(0).getStringProperty(IStyleModel.MARGIN_LEFT_PROP));
		
		designHandle.getCommandStack().redo();
		assertEquals("10mm", designHandle.getBody().get(0).getStringProperty(IStyleModel.MARGIN_LEFT_PROP));
	
		text.setProperty(IStyleModel.MARGIN_BOTTOM_PROP,"50%");
		assertEquals("50%",designHandle.getBody().get(0).getStringProperty(IStyleModel.MARGIN_BOTTOM_PROP));
		
		designHandle.getCommandStack().undo();
		assertEquals("0pt", designHandle.getBody().get(0).getStringProperty(IStyleModel.MARGIN_BOTTOM_PROP));
		
		designHandle.getCommandStack().redo();
		assertEquals("50%", designHandle.getBody().get(0).getStringProperty(IStyleModel.MARGIN_BOTTOM_PROP));
	
		text.setProperty(IStyleModel.MARGIN_RIGHT_PROP,"8in");
		assertEquals("8in",designHandle.getBody().get(0).getStringProperty(IStyleModel.MARGIN_RIGHT_PROP));
		
		designHandle.getCommandStack().undo();
		assertEquals("0pt", designHandle.getBody().get(0).getStringProperty(IStyleModel.MARGIN_RIGHT_PROP));
		
		designHandle.getCommandStack().redo();
		assertEquals("8in", designHandle.getBody().get(0).getStringProperty(IStyleModel.MARGIN_RIGHT_PROP));
	
		//page-break
        text.setProperty(IStyleModel.PAGE_BREAK_AFTER_PROP,"always");
	    
	    designHandle.getCommandStack().undo();
	    assertEquals("auto",text.getProperty(IStyleModel.PAGE_BREAK_AFTER_PROP));
	    
	    designHandle.getCommandStack().redo();
	    assertEquals("always",text.getProperty(IStyleModel.PAGE_BREAK_AFTER_PROP));
	    
	    text.setProperty(IStyleModel.PAGE_BREAK_BEFORE_PROP,"avoid");
	    
	    designHandle.getCommandStack().undo();   
	    designHandle.getCommandStack().redo();
	    assertEquals("avoid",text.getProperty(IStyleModel.PAGE_BREAK_BEFORE_PROP));
	    
	    //visibility
	    HideRule hiderule = StructureFactory.createHideRule( );
	    hiderule.setExpression("new expression");
	    hiderule.setFormat(DesignChoiceConstants.FORMAT_TYPE_PDF );
	    PropertyHandle propHandle = text.getPropertyHandle( ReportItem.VISIBILITY_PROP );
        propHandle.addItem(hiderule);
        Iterator iter = propHandle.iterator( );
		HideRuleHandle hideruleHandle = (HideRuleHandle) iter.next( );

		assertEquals("new expression", hideruleHandle.getExpression());
		assertEquals(DesignChoiceConstants.FORMAT_TYPE_PDF,hideruleHandle.getFormat());
	
	    
	    //background
	    //background-attachment
	    text.setProperty(IStyleModel.BACKGROUND_ATTACHMENT_PROP,"fixed");
	    
	    designHandle.getCommandStack().undo();
	    assertEquals("scroll",text.getProperty(IStyleModel.BACKGROUND_ATTACHMENT_PROP));
	    
	    designHandle.getCommandStack().redo();
	    assertEquals("fixed",text.getProperty(IStyleModel.BACKGROUND_ATTACHMENT_PROP));
	    
	    //position
	    text.setProperty(IStyleModel.BACKGROUND_POSITION_X_PROP,"100%");
	    text.setProperty(IStyleModel.BACKGROUND_POSITION_Y_PROP,"50%");
	    
	    assertEquals("100%",text.getStringProperty(IStyleModel.BACKGROUND_POSITION_X_PROP));
	    assertEquals("50%",text.getStringProperty(IStyleModel.BACKGROUND_POSITION_Y_PROP));
	
	    //repeat
	    text.setProperty(IStyleModel.BACKGROUND_REPEAT_PROP,"repeat-x");
	    
	    designHandle.getCommandStack().undo();
	    assertEquals("repeat",text.getProperty(IStyleModel.BACKGROUND_REPEAT_PROP));
	    
	    designHandle.getCommandStack().redo();
	    assertEquals("repeat-x",text.getProperty(IStyleModel.BACKGROUND_REPEAT_PROP));
	    
	    //comment
	    text.setProperty(IDesignElementModel.COMMENTS_PROP,"my comment");
	    
	    designHandle.getCommandStack().undo();
	    assertNull(text.getProperty(IDesignElementModel.COMMENTS_PROP));
	    
	    designHandle.getCommandStack().redo();
	    assertEquals("my comment",text.getProperty(IDesignElementModel.COMMENTS_PROP));
	    
	    //content
	    text.setProperty(ITextItemModel.CONTENT_PROP,"any content");
	    text.setProperty(ITextItemModel.CONTENT_RESOURCE_KEY_PROP,"key");
	    text.setProperty(ITextItemModel.CONTENT_TYPE_PROP,"plain");
	    
	    assertEquals("any content", text.getProperty(ITextItemModel.CONTENT_PROP));
	    assertEquals("key", text.getProperty(ITextItemModel.CONTENT_RESOURCE_KEY_PROP));
	    assertEquals("plain", text.getProperty(ITextItemModel.CONTENT_TYPE_PROP));
	
	    //binding
	    OdaDataSetHandle dataset = factory.newOdaDataSet("dset",null);
	    designHandle.getDataSets().add(dataset);
	    text.setDataSet(dataset);
	    
	    designHandle.getCommandStack().undo();
	    assertNull(text.getDataSet());
	    
	    designHandle.getCommandStack().redo();
	    assertEquals("dset",text.getDataSet().getName());
	    
	    //others
	    text.setProperty(IReportItemModel.HEIGHT_PROP,"12cm");
	    text.setProperty(IReportItemModel.WIDTH_PROP,"100%");
	    text.setProperty(IStyleModel.CAN_SHRINK_PROP,"false");
	    
	    assertEquals("12cm", text.getStringProperty(IReportItemModel.HEIGHT_PROP));
	    assertEquals("100%", text.getStringProperty(IReportItemModel.WIDTH_PROP));
	   assertEquals("false", text.getStringProperty(IStyleModel.CAN_SHRINK_PROP) );
	}
}
