/*******************************************************************************
 * Copyright (c) 2004 Actuate Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *  Actuate Corporation  - initial API and implementation
 *******************************************************************************/

package org.eclipse.birt.report.tests.model.api;

import java.util.Iterator;
import java.util.List;

import junit.framework.Test;
import junit.framework.TestSuite;

import org.eclipse.birt.report.model.api.ErrorDetail;
import org.eclipse.birt.report.model.api.ScalarParameterHandle;
import org.eclipse.birt.report.model.api.SlotHandle;
import org.eclipse.birt.report.model.api.StructureHandle;
import org.eclipse.birt.report.model.api.elements.DesignChoiceConstants;
import org.eclipse.birt.report.model.api.elements.structures.SelectionChoice;
import org.eclipse.birt.report.model.api.metadata.PropertyValueException;
import org.eclipse.birt.report.tests.model.BaseTestCase;

/**
 * Test ReportItemHandle.
 * 
 * <p>
 * <table border="1" cellpadding="2" cellspacing="2" style="border-collapse:
 * collapse" bordercolor="#111111">
 * <th width="20%">Method</th>
 * <th width="40%">Test Case</th>
 * <th width="40%">Expected</th>
 * 
 * <tr>
 * <td>{@link #testGetsPropertiesOnParameter()}</td>
 * <td>Gets the property values of scalar parameter by the
 * ScalarParameterHandle.</td>
 * <td>The returned value matches with the input file.</td>
 * </tr>
 * 
 * <tr>
 * <td>{@link #testSetsPropertiesOnParameter()}</td>
 * <td>Sets the property values of scalar parameter.</td>
 * <td>Values are set correctly.</td>
 * </tr>
 * 
 * <tr>
 * <td>{@link #testSyntaxCheck()}</td>
 * <td>Checks the semantic error when data set name is missed.</td>
 * <td>Throws <code>PropertyValueException</code> with the error code
 * <code>VALUE_REQUIRED</code>.</td>
 * </tr>
 * 
 * </table>
 * 
 *  
 */

public class ScalarParameterHandleTest extends BaseTestCase
{

	/**
	 * @param name
	 */
	public ScalarParameterHandleTest(String name) {
		super(name);
		// TODO Auto-generated constructor stub
	}
	public static Test suite(){
		return new TestSuite(ScalarParameterHandleTest.class);	
	}
	/*
	 * (non-Javadoc)
	 * 
	 * @see junit.framework.TestCase#setUp()
	 */

	protected void setUp( ) throws Exception
	{
		super.setUp( );

		openDesign( "ScalarParameterHandleTest.xml" ); //$NON-NLS-1$ 
	}

	/**
	 * Tests to read properties on scalar parameters.
	 */

	public void testGetsPropertiesOnParameter( )
	{
		SlotHandle params = designHandle.getParameters( );
		ScalarParameterHandle handle = (ScalarParameterHandle) params.get( 0 );

		assertFalse( handle.isHidden( ) );

		assertEquals( DesignChoiceConstants.PARAM_TYPE_DECIMAL, handle
				.getDataType( ) );
		assertFalse( handle.isConcealValue( ) );
		assertEquals( "State", handle.getDefaultValue( ) ); //$NON-NLS-1$

		assertTrue( handle.allowNull( ) );
		assertFalse( handle.allowBlank( ) );

		assertEquals( "##,###.##", handle.getPattern( ) ); //$NON-NLS-1$

		assertEquals( DesignChoiceConstants.PARAM_CONTROL_CHECK_BOX, handle
				.getControlType( ) );

		assertEquals( DesignChoiceConstants.SCALAR_PARAM_ALIGN_LEFT, handle
				.getAlignment( ) );

		// no list defined, should be null.

		Iterator iter = handle.choiceIterator( );
		int count = 0;
		for ( ; iter.hasNext( ); iter.next( ) )
			count++;
		assertEquals( 0, count );

		assertEquals( "scalar para help", handle.getHelpText( ) ); //$NON-NLS-1$
		assertEquals( "help", handle.getHelpTextKey( ) ); //$NON-NLS-1$

		// tests on the 2nd scalar parameter with the dynamic list.

		handle = (ScalarParameterHandle) params.get( 1 );

		// no default value to be false;

		assertFalse( handle.isHidden( ) );

		// the default value.

		assertEquals( DesignChoiceConstants.PARAM_TYPE_STRING, handle.getDataType( ) );
		assertFalse( handle.isConcealValue( ) );
		assertNull( handle.getDefaultValue( ) );
		assertFalse( handle.allowNull( ) );
		assertTrue( handle.allowBlank( ) );
		assertNull( handle.getPattern( ) );
		assertNull( handle.getControlType( ) );
		assertEquals( DesignChoiceConstants.SCALAR_PARAM_ALIGN_AUTO, handle
				.getAlignment( ) );
		assertNull( handle.getHelpText( ) );

		// no list defined, should be null.

		assertEquals( "dataset 1", handle.getDataSetName( ) ); //$NON-NLS-1$
		assertEquals( "value column", handle.getValueExpr( ) ); //$NON-NLS-1$
		assertEquals( "label column", handle.getLabelExpr( ) ); //$NON-NLS-1$

		// tests on the 3nd scalar parameter with the selection list without
		// choices.

		handle = (ScalarParameterHandle) params.get( 2 );

		assertFalse( handle.isMustMatch( ) ); //$NON-NLS-1$
		assertFalse( handle.isFixedOrder( ) ); //$NON-NLS-1$

		iter = handle.choiceIterator( );
		assertNotNull( iter );
		count = 0;
		for ( ; iter.hasNext( ); iter.next( ) )
			count++;
		assertEquals( 0, count );

		// tests on the 3nd scalar parameter with the selection list with 3
		// choices.

		handle = (ScalarParameterHandle) params.get( 3 );
		assertFalse( handle.isMustMatch( ) ); //$NON-NLS-1$
		assertTrue( handle.isFixedOrder( ) ); //$NON-NLS-1$

		StructureHandle[] choices = new StructureHandle[3];
		count = 0;

		for ( iter = handle.choiceIterator( ); iter.hasNext( ); count++ )
			choices[count] = (StructureHandle) ( iter.next( ) );

		assertEquals( 3, count );

		assertEquals( "option 1", //$NON-NLS-1$
				choices[0].getMember( SelectionChoice.VALUE_MEMBER ).getValue( ) );
		assertEquals( "option 1 label", //$NON-NLS-1$
				choices[0].getMember( SelectionChoice.LABEL_MEMBER ).getValue( ) );
		assertEquals( "key 1 for label 1", //$NON-NLS-1$
				choices[0]
						.getMember( SelectionChoice.LABEL_RESOURCE_KEY_MEMBER )
						.getValue( ) );

		assertEquals( "option 3", //$NON-NLS-1$
				choices[2].getMember( SelectionChoice.VALUE_MEMBER ).getValue( ) );
		assertNull( choices[2].getMember( SelectionChoice.LABEL_MEMBER )
				.getValue( ) );
		assertNull( choices[2].getMember(
				SelectionChoice.LABEL_RESOURCE_KEY_MEMBER ).getValue( ) );

	}

	/**
	 * Sets properties on scalar parameters.
	 * 
	 * @throws Exception
	 */

	public void testSetsPropertiesOnParameter( ) throws Exception
	{
		SlotHandle params = designHandle.getParameters( );
		ScalarParameterHandle handle = (ScalarParameterHandle) params.get( 0 );

		assertFalse( handle.isHidden( ) );
		handle.setHidden( true );
		assertTrue( handle.isHidden( ) );

		handle.setDataType( DesignChoiceConstants.PARAM_TYPE_DATETIME );
		assertEquals( DesignChoiceConstants.PARAM_TYPE_DATETIME, handle
				.getDataType( ) );

		handle.setDataType( DesignChoiceConstants.PARAM_TYPE_FLOAT );
		assertEquals( DesignChoiceConstants.PARAM_TYPE_FLOAT, handle.getDataType( ) );

		handle.setConcealValue( true );
		assertTrue( handle.isConcealValue( ) );

		handle.setDefaultValue( "new default value" ); //$NON-NLS-1$
		assertEquals( "new default value", handle.getDefaultValue( ) ); //$NON-NLS-1$

		handle.setAllowNull( false );
		assertFalse( handle.allowNull( ) );

		handle.setAllowBlank( true );
		assertTrue( handle.allowBlank( ) );

		handle.setPattern( "$***,***.**" ); //$NON-NLS-1$
		assertEquals( "$***,***.**", handle.getPattern( ) ); //$NON-NLS-1$

		handle
				.setControlType( DesignChoiceConstants.PARAM_CONTROL_RADIO_BUTTON );
		assertEquals( DesignChoiceConstants.PARAM_CONTROL_RADIO_BUTTON, handle
				.getControlType( ) );

		handle.setAlignment( DesignChoiceConstants.SCALAR_PARAM_ALIGN_RIGHT );
		assertEquals( DesignChoiceConstants.SCALAR_PARAM_ALIGN_RIGHT, handle
				.getAlignment( ) );

	}

	/**
	 * Tests the syntax check on ScalarParameter.
	 * <p>
	 * If the dynamic list is defined, it must have a data set name property.
	 * 
	 * @throws Exception
	 *             if error occurs during parsing the designf file.
	 */

	public void testSyntaxCheck( ) throws Exception
	{
		openDesign( "ScalarParameterHandleTest_1.xml" ); //$NON-NLS-1$

		List errors = designHandle.getErrorList( );
		assertEquals( 1, errors.size( ) );

		int i = 0;
		assertEquals( PropertyValueException.DESIGN_EXCEPTION_VALUE_REQUIRED,
				( (ErrorDetail) errors.get( i++ ) ).getErrorCode( ) );
	}
}