/*******************************************************************************
 * Copyright (c) 2004, 2005 Actuate Corporation. All rights reserved. This
 * program and the accompanying materials are made available under the terms of
 * the Eclipse Public License v1.0 which accompanies this distribution, and is
 * available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Actuate Corporation - initial API and implementation
 ******************************************************************************/

package org.eclipse.birt.tests.data.engine.api;

import java.util.Collection;
import java.util.Iterator;

import org.eclipse.birt.core.data.DataType;
import org.eclipse.birt.data.engine.api.IBaseExpression;
import org.eclipse.birt.data.engine.api.IParameterMetaData;
import org.eclipse.birt.data.engine.api.IPreparedQuery;
import org.eclipse.birt.data.engine.api.IQueryResults;
import org.eclipse.birt.data.engine.api.IResultIterator;
import org.eclipse.birt.data.engine.api.IResultMetaData;
import org.eclipse.birt.data.engine.api.ISortDefinition;
import org.eclipse.birt.data.engine.api.querydefn.ConditionalExpression;
import org.eclipse.birt.data.engine.api.querydefn.FilterDefinition;
import org.eclipse.birt.data.engine.api.querydefn.GroupDefinition;
import org.eclipse.birt.data.engine.api.querydefn.OdaDataSetDesign;
import org.eclipse.birt.data.engine.api.querydefn.ParameterDefinition;
import org.eclipse.birt.data.engine.api.querydefn.QueryDefinition;
import org.eclipse.birt.data.engine.api.querydefn.ScriptExpression;
import org.eclipse.birt.data.engine.api.querydefn.SortDefinition;
import org.eclipse.birt.data.engine.core.DataException;

import testutil.APITestCase;
import testutil.ConfigText;

/**
 * DtE features test.
 */
public class FeaturesTest extends APITestCase
{

	/*
	 * @see org.eclipse.birt.data.engine.api.APITestCase#getDataSourceInfo()
	 */
	protected DataSourceInfo getDataSourceInfo( )
	{
		return new DataSourceInfo( ConfigText
				.getString( "Api.TestData.TableName" ), ConfigText
				.getString( "Api.TestData.TableSQL" ), ConfigText
				.getString( "Api.TestData.TestDataFileName" ) );
	}

	/**
	 * A standard report, test feature of: group, sort, filter.
	 */
	public void test1( ) throws Exception
	{
		GroupDefinition[] groupDefn = new GroupDefinition[]{
				new GroupDefinition( ), new GroupDefinition( )};
		groupDefn[0].setKeyExpression( "row.COUNTRY" );
		groupDefn[1].setKeyExpression( "row.CITY" );

		SortDefinition[] sortDefn = new SortDefinition[]{new SortDefinition( )};
		sortDefn[0].setColumn( "SALE_DATE" );
		sortDefn[0].setSortDirection( ISortDefinition.SORT_DESC );

		IBaseExpression[] expressions = new IBaseExpression[]{
				new ScriptExpression( "row[0]", 0 ),
				new ScriptExpression( "row._rowPosition ", 0 ),
				new ScriptExpression( "row.COUNTRY", 0 ),
				new ScriptExpression( "row.CITY", 0 ),
				new ScriptExpression( "row.SALE_DATE", 0 ),
				new ScriptExpression( "row.AMOUNT", 0 )};
		FilterDefinition[] filters = new FilterDefinition[]{new FilterDefinition(
				new ScriptExpression( "row.AMOUNT > 100" ) )};

		createAndRunQuery( expressions, groupDefn, sortDefn, filters );
		checkOutputFile( );
	}

	/**
	 * Test feature of group.
	 */
	public void test3( ) throws Exception
	{
		GroupDefinition[] groupDefn = new GroupDefinition[]{
				new GroupDefinition( ), new GroupDefinition( )};
		groupDefn[0].setKeyExpression( "row.COUNTRY" );
		groupDefn[1].setKeyExpression( "row.CITY" );

		IBaseExpression[] expressions = new IBaseExpression[]{
				new ScriptExpression( "row.COUNTRY", 0 ),
				new ScriptExpression( "row.CITY", 0 ),
				new ScriptExpression( "row.SALE_DATE", 0 ),
				new ScriptExpression( "row.AMOUNT", 0 )};
		createAndRunQuery( expressions, groupDefn, null, null );
		checkOutputFile( );
	}

	/**
	 * Test feature of incorrect sort key, throw exception
	 */
	public void test4( ) throws Exception
	{
		GroupDefinition[] groupDefn = new GroupDefinition[]{
				new GroupDefinition( ), new GroupDefinition( )};
		groupDefn[0].setKeyExpression( "row.COUNTRY" );
		groupDefn[1].setKeyExpression( "row.CITY" );

		SortDefinition[] sortDefn = new SortDefinition[]{new SortDefinition( )};
		sortDefn[0].setColumn( "SALE_DA" );
		sortDefn[0].setSortDirection( ISortDefinition.SORT_DESC );

		IBaseExpression[] expressions = new IBaseExpression[]{
				new ScriptExpression( "row.COUNTRY", 0 ),
				new ScriptExpression( "row.CITY", 0 ),
				new ScriptExpression( "row.SALE_DATE", 0 ),
				new ScriptExpression( "row.AMOUNT", 0 )};
		try
		{
			createAndRunQuery( expressions, groupDefn, sortDefn, null );
			fail( "Should throw DteException!" );
		}
		catch ( DataException e )
		{
			// TODO: verify e has expected error code
		}
	}

	/**
	 * Test feature without any group, sort, filter
	 */
	public void test6( ) throws Exception
	{
		IBaseExpression[] expressions = new IBaseExpression[]{
				new ScriptExpression( "row.COUNTRY", 0 ),
				new ScriptExpression( "row.CITY", 0 ),
				new ScriptExpression( "row.SALE_DATE", 0 ),
				new ScriptExpression( "row.AMOUNT", 0 )};
		createAndRunQuery( expressions, null, null, null );
		checkOutputFile( );
	}

	/**
	 * Test feature of incorrect filter expression, throw exception
	 */
	public void test7( ) throws Exception
	{
		GroupDefinition[] groupDefn = new GroupDefinition[]{
				new GroupDefinition( ), new GroupDefinition( )};
		groupDefn[0].setKeyExpression( "row.COUNTRY" );
		groupDefn[1].setKeyExpression( "row.CITY" );

		SortDefinition[] sortDefn = new SortDefinition[]{new SortDefinition( )};
		sortDefn[0].setColumn( "SALE_DATE" );
		sortDefn[0].setSortDirection( ISortDefinition.SORT_DESC );

		IBaseExpression[] expressions = new IBaseExpression[]{
				new ScriptExpression( "row.COUNTRY", 0 ),
				new ScriptExpression( "row.CITY", 0 ),
				new ScriptExpression( "row.SALE_DATE", 0 ),
				new ScriptExpression( "row.AMOUNT", 0 )};

		// wrong operator
		FilterDefinition[] filters = new FilterDefinition[]{new FilterDefinition(
				new ScriptExpression( "row.AMONT && 100" ) )};

		try
		{
			createAndRunQuery( expressions, groupDefn, sortDefn, filters );
			fail( "Should throw DteException!" );
		}
		catch ( DataException e )
		{
			// TODO: verify that e has correct error code
		}
	}

	/**
	 * Test feature of group, sort, ConditionalExpression
	 */
	public void test8( ) throws Exception
	{
		GroupDefinition[] groupDefn = new GroupDefinition[]{
				new GroupDefinition( ), new GroupDefinition( )};
		groupDefn[0].setKeyExpression( "row.COUNTRY" );
		groupDefn[1].setKeyExpression( "row.CITY" );

		SortDefinition[] sortDefn = new SortDefinition[]{new SortDefinition( )};
		sortDefn[0].setColumn( "SALE_DATE" );
		sortDefn[0].setSortDirection( ISortDefinition.SORT_DESC );

		IBaseExpression[] expressions = new IBaseExpression[]{
				new ScriptExpression( "row.COUNTRY", 0 ),
				new ScriptExpression( "row.CITY", 0 ),
				new ScriptExpression( "row.SALE_DATE", 0 ),
				new ScriptExpression( "row.AMOUNT", 0 ),
				new ConditionalExpression( "row.AMOUNT",
						ConditionalExpression.OP_GT, "5" ),
				new ConditionalExpression( "row.CITY",
						ConditionalExpression.OP_EQ, "'Beijing'" ),
				new ConditionalExpression( "row.CITY",
						ConditionalExpression.OP_NE, "'Beijing'" ),
				new ConditionalExpression( "row.SALE_DATE",
						ConditionalExpression.OP_GE, "'01/01/2004'" ),
				new ConditionalExpression( "row.AMOUNT",
						ConditionalExpression.OP_BETWEEN, "5", "100" )};

		createAndRunQuery( expressions, groupDefn, sortDefn, null );
		checkOutputFile( );
	}

	/**
	 * Regression Test for SRC #78568 Filetr on date time doesn't work properly
	 */
	public void regressionTest78568( ) throws Exception
	{
		GroupDefinition[] groupDefn = new GroupDefinition[]{
				new GroupDefinition( ), new GroupDefinition( )};
		groupDefn[0].setKeyExpression( "row.COUNTRY" );
		groupDefn[1].setKeyExpression( "row.CITY" );

		SortDefinition[] sortDefn = new SortDefinition[]{new SortDefinition( )};
		sortDefn[0].setColumn( "SALE_DATE" );
		sortDefn[0].setSortDirection( ISortDefinition.SORT_DESC );

		IBaseExpression[] expressions = new IBaseExpression[]{
				new ScriptExpression( "row.COUNTRY", 0 ),
				new ScriptExpression( "row.CITY", 0 ),
				new ScriptExpression( "row.SALE_DATE", 0 ),
				new ScriptExpression( "row.AMOUNT", 0 )};
		FilterDefinition[] filters = new FilterDefinition[]{new FilterDefinition(
				new ConditionalExpression( "row.SALE_DATE",
						ConditionalExpression.OP_GE, "'2004-03-20 00:00:00.0'" ) )};

		createAndRunQuery( expressions, groupDefn, sortDefn, filters );
		checkOutputFile( );
	}

	/**
	 * Test feature of group, sort, filter
	 */
	public void test9( ) throws Exception
	{
		GroupDefinition[] groupDefn = new GroupDefinition[]{
				new GroupDefinition( ), new GroupDefinition( )};
		groupDefn[0].setKeyExpression( "row.COUNTRY" );
		groupDefn[1].setKeyExpression( "row.CITY" );

		SortDefinition[] sortDefn = new SortDefinition[]{new SortDefinition( )};
		sortDefn[0].setColumn( "SALE_DATE" );
		sortDefn[0].setSortDirection( ISortDefinition.SORT_DESC );

		IBaseExpression[] expressions = new IBaseExpression[]{
				new ScriptExpression( "row.COUNTRY", 0 ),
				new ScriptExpression( "row.CITY", 0 ),
				new ScriptExpression( "row.SALE_DATE", 0 ),
				new ScriptExpression( "row.AMOUNT", 0 )};
		FilterDefinition[] filters = new FilterDefinition[]{new FilterDefinition(
				new ConditionalExpression( "row.AMOUNT",
						ConditionalExpression.OP_GT, "400" ) )};

		createAndRunQuery( expressions, groupDefn, sortDefn, filters );
		checkOutputFile( );
	}

	/**
	 * Test feature of duplicate column name from different tables
	 */
	public void test10_DuplicateColName( ) throws Exception
	{
		// Test a SQL with duplicate column name (quite common with join data
		// sets)
		String testSQL = "select COUNTRY, COUNTRY, CITY from "
				+ getTestTableName( );
		( (OdaDataSetDesign) this.dataSet ).setQueryText( testSQL );

		IBaseExpression[] expressions = new IBaseExpression[]{
				new ScriptExpression( "row.COUNTRY", 0 ),
				new ScriptExpression( "row.CITY", 0 )};

		QueryDefinition queryDefn = createQueryDefn( expressions, null, null,
				null );

		IPreparedQuery preparedQuery = dataEngine.prepare( queryDefn );
		IQueryResults queryResults = preparedQuery.execute( null );
		IResultIterator resultIt = queryResults.getResultIterator( );
		assertTrue( resultIt.next( ) );
	}

	/**
	 * Test feature of GetParameterMetaData
	 * 
	 * @throws Exception
	 */
	public void testBasicGetParameterMetaData( ) throws Exception
	{
		String sql = "select COUNTRY, CITY from " + getTestTableName( )
				+ " where city = ?";
		( (OdaDataSetDesign) this.dataSet ).setQueryText( sql );
		IBaseExpression[] expressions = new IBaseExpression[]{
				new ScriptExpression( "row.COUNTRY", 0 ),
				new ScriptExpression( "row.CITY", 0 )};

		QueryDefinition queryDefn = createQueryDefn( expressions, null, null,
				null );

		IPreparedQuery preparedQuery = dataEngine.prepare( queryDefn );
		Collection parameterMetaData = preparedQuery.getParameterMetaData( );
		assertTrue( parameterMetaData != null && parameterMetaData.size( ) == 1 );

		Iterator iter = parameterMetaData.iterator( );
		while ( iter.hasNext( ) )
		{
			IParameterMetaData paramMd = (IParameterMetaData) iter.next( );
			assertEquals( 1, paramMd.getPosition( ) );
			assertEquals( DataType.STRING_TYPE, paramMd.getDataType( ) );
			assertEquals( null, paramMd.getName( ) );
			assertEquals( "VARCHAR", paramMd.getNativeTypeName( ) );
			assertEquals( 0, paramMd.getScale( ) );
			assertEquals( 10, paramMd.getPrecision( ) );
		}
	}

	/**
	 * Test feature of GetParameterMetaData1
	 * 
	 * @throws Exception
	 */
	public void testBasicGetParameterMetaData1( ) throws Exception
	{
		String sql = "select COUNTRY, CITY from " + getTestTableName( )
				+ " where city = ?";
		( (OdaDataSetDesign) this.dataSet ).setQueryText( sql );

		// add an input parameter hint
		ParameterDefinition inputParamDefn = new ParameterDefinition( "param1",
				DataType.DECIMAL_TYPE, true, false );
		inputParamDefn.setPosition( 1 );
		inputParamDefn.setDefaultInputValue( "0" );
		( (OdaDataSetDesign) this.dataSet ).addParameter( inputParamDefn );

		IBaseExpression[] expressions = new IBaseExpression[]{
				new ScriptExpression( "row.COUNTRY", 0 ),
				new ScriptExpression( "row.CITY", 0 )};

		QueryDefinition queryDefn = createQueryDefn( expressions, null, null,
				null );

		IPreparedQuery preparedQuery = dataEngine.prepare( queryDefn );
		Collection parameterMetaData = preparedQuery.getParameterMetaData( );
		assertTrue( parameterMetaData != null && parameterMetaData.size( ) == 1 );

		Iterator iter = parameterMetaData.iterator( );
		while ( iter.hasNext( ) )
		{
			IParameterMetaData paramMd = (IParameterMetaData) iter.next( );
			assertEquals( 1, paramMd.getPosition( ) );
			assertEquals( DataType.STRING_TYPE, paramMd.getDataType( ) );
			assertEquals( "param1", paramMd.getName( ) );
			assertEquals( "VARCHAR", paramMd.getNativeTypeName( ) );
			assertEquals( 0, paramMd.getScale( ) );
			assertEquals( 10, paramMd.getPrecision( ) );
		}
	}

	/**
	 * Test feature of GetParameterMetaDataDefaultValue
	 * 
	 * @throws Exception
	 */
	public void testBasicGetParameterMetaDataDefaultValue( ) throws Exception
	{
		String sql = "select COUNTRY, CITY from " + getTestTableName( )
				+ " where city = ?";
		( (OdaDataSetDesign) this.dataSet ).setQueryText( sql );

		// add an input parameter hint
		ParameterDefinition inputParamDefn = new ParameterDefinition( "param1",
				DataType.DECIMAL_TYPE );
		inputParamDefn.setInputMode( true );
		inputParamDefn.setDefaultInputValue( "123" );
		inputParamDefn.setPosition( 1 );

		( (OdaDataSetDesign) this.dataSet ).addParameter( inputParamDefn );

		IBaseExpression[] expressions = new IBaseExpression[]{
				new ScriptExpression( "row.COUNTRY", 0 ),
				new ScriptExpression( "row.CITY", 0 )};

		QueryDefinition queryDefn = createQueryDefn( expressions, null, null,
				null );

		IPreparedQuery preparedQuery = dataEngine.prepare( queryDefn );
		Collection parameterMetaData = preparedQuery.getParameterMetaData( );
		assertTrue( parameterMetaData != null && parameterMetaData.size( ) == 1 );

		Iterator iter = parameterMetaData.iterator( );
		while ( iter.hasNext( ) )
		{
			IParameterMetaData paramMd = (IParameterMetaData) iter.next( );
			assertEquals( 1, paramMd.getPosition( ) );
			assertEquals( DataType.STRING_TYPE, paramMd.getDataType( ) );
			assertEquals( "param1", paramMd.getName( ) );
			assertEquals( "VARCHAR", paramMd.getNativeTypeName( ) );
			assertEquals( 0, paramMd.getScale( ) );
			assertEquals( 10, paramMd.getPrecision( ) );
			assertEquals( "123", paramMd.getDefaultInputValue( ) );
		}
	}

	/**
	 * Test feature of NativeColumnTypeName
	 * 
	 * @throws Exception
	 */
	public void testNativeColumnTypeName( ) throws Exception
	{
		String testSQL = "select COUNTRY, CITY from " + getTestTableName( );
		( (OdaDataSetDesign) this.dataSet ).setQueryText( testSQL );

		IBaseExpression[] expressions = new IBaseExpression[]{
				new ScriptExpression( "row.COUNTRY", 0 ),
				new ScriptExpression( "row.CITY", 0 )};

		QueryDefinition queryDefn = createQueryDefn( expressions, null, null,
				null );

		IPreparedQuery preparedQuery = dataEngine.prepare( queryDefn );
		IQueryResults queryResults = preparedQuery.execute( null );
		IResultMetaData metadata = queryResults.getResultMetaData( );

		assertEquals( "VARCHAR", metadata.getColumnNativeTypeName( 1 ) );
		assertEquals( "VARCHAR", metadata.getColumnNativeTypeName( 2 ) );
	}

	/**
	 * Create query definition from passed parameteres and then run it.
	 * 
	 * @param expressions
	 * @param groupDefn
	 * @param sortDefn
	 * @param filters
	 * @throws Exception
	 */
	private void createAndRunQuery( IBaseExpression[] expressions,
			GroupDefinition[] groupDefn, SortDefinition[] sortDefn,
			FilterDefinition[] filters ) throws Exception
	{
		QueryDefinition queryDefn = createQueryDefn( expressions, groupDefn,
				sortDefn, filters );
		
		IPreparedQuery preparedQuery = dataEngine.prepare( queryDefn );
		IQueryResults queryResults = preparedQuery.execute( jsScope );
		IResultIterator resultIt = queryResults.getResultIterator( );
		
		outputQueryResult( resultIt, new String[]{"COL_COUNTRY", "COL_CITY"} );
	}

	/**
	 * Create query definition from passed parameters
	 * 
	 * @param expressions
	 *            row expression
	 * @param groupDefn
	 * @param sortDefn
	 * @param filters
	 * @return QueryDefinition
	 */
	private QueryDefinition createQueryDefn( IBaseExpression[] expressions,
			GroupDefinition[] groupDefn, SortDefinition[] sortDefn,
			FilterDefinition[] filters )
	{
		// define a query design
		QueryDefinition queryDefn = newReportQuery( );

		if ( groupDefn != null )
			for ( int i = 0; i < groupDefn.length; i++ )
				queryDefn.addGroup( groupDefn[i] );
		if ( sortDefn != null )
			for ( int i = 0; i < sortDefn.length; i++ )
				queryDefn.addSort( sortDefn[i] );
		if ( expressions != null )
			for ( int i = 0; i < expressions.length; i++ )
				queryDefn.addResultSetExpression( expressions[i].toString( ),
						expressions[i] );
		if ( filters != null )
			for ( int i = 0; i < filters.length; i++ )
				queryDefn.addFilter( filters[i] );

		return queryDefn;
	}

}