/***********************************************************************
 * Copyright (c) 2004 Actuate Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 * Actuate Corporation - initial API and implementation
 ***********************************************************************/

package org.eclipse.birt.chart.examples.api.interactivity;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.GridLayout;
import java.awt.Rectangle;
import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ComponentEvent;
import java.awt.event.ComponentListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.awt.image.BufferedImage;
import java.util.HashMap;
import java.util.Map;

import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;

import org.eclipse.birt.chart.device.IDeviceRenderer;
import org.eclipse.birt.chart.device.ICallBackNotifier;
import org.eclipse.birt.chart.exception.ChartException;
import org.eclipse.birt.chart.factory.GeneratedChartState;
import org.eclipse.birt.chart.factory.Generator;
import org.eclipse.birt.chart.model.Chart;
import org.eclipse.birt.chart.model.attribute.Bounds;
import org.eclipse.birt.chart.model.attribute.CallBackValue;
import org.eclipse.birt.chart.model.attribute.impl.BoundsImpl;
import org.eclipse.birt.chart.util.PluginSettings;
import org.eclipse.birt.core.exception.BirtException;

/**
 * The selector of interactivity charts in Swing JPanel.
 * 
 * Note: Use an extra off-screen image buffer as the actual drawing canvas instead of 
 * 		 the original chart rendering code in paint( ) method. 
 *   	 Please see [bugzilla] 127615 for more details.
 */
public final class SwingInteractivityViewer extends JPanel implements
        ICallBackNotifier,
		ComponentListener
{

	private static final long serialVersionUID = 1L;
	
	private boolean bNeedsGeneration = true;

	private GeneratedChartState gcs = null;

	private Chart cm = null;

	private IDeviceRenderer idr = null;
	
	private BufferedImage bi = null;

	private Map contextMap;
	
	/**
	 * Contructs the layout with a container for displaying chart and a control
	 * panel for selecting interactivity.
	 * 
	 * @param args
	 */
	public static void main( String[] args )
	{
		final SwingInteractivityViewer siv = new SwingInteractivityViewer( );

		JFrame jf = new JFrame( );
		jf.setDefaultCloseOperation( JFrame.DISPOSE_ON_CLOSE );
		jf.addComponentListener( siv );

		Container co = jf.getContentPane( );
		co.setLayout( new BorderLayout( ) );
		co.add( siv, BorderLayout.CENTER );

		Dimension dScreen = Toolkit.getDefaultToolkit( ).getScreenSize( );
		Dimension dApp = new Dimension( 600, 400 );
		jf.setSize( dApp );
		jf.setLocation( ( dScreen.width - dApp.width ) / 2,
				( dScreen.height - dApp.height ) / 2 );

		jf.setTitle( siv.getClass( ).getName( ) + " [device=" //$NON-NLS-1$
				+ siv.idr.getClass( ).getName( ) + "]" );//$NON-NLS-1$

		ControlPanel cp = siv.new ControlPanel( siv );
		co.add( cp, BorderLayout.SOUTH );
		
		siv.idr.setProperty( IDeviceRenderer.UPDATE_NOTIFIER, siv );

		jf.addWindowListener( new WindowAdapter( ) {

			public void windowClosing( WindowEvent e )
			{
				siv.idr.dispose( );
			}

		} );

		jf.setVisible( true );
	}

	/**
	 * Get the connection with SWING device to render the graphics.
	 */
	SwingInteractivityViewer( )
	{
		contextMap = new HashMap();
		
		final PluginSettings ps = PluginSettings.instance( );
		try
		{
			idr = ps.getDevice( "dv.SWING" );//$NON-NLS-1$
		}
		catch ( ChartException ex )
		{
			ex.printStackTrace( );
		}
		cm = InteractivityCharts.createHSChart( );
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.birt.chart.device.swing.IUpdateNotifier#update()
	 */
	public void regenerateChart( )
	{
		bNeedsGeneration = true;
		updateBuffer( );
		repaint( );
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.birt.chart.device.swing.IUpdateNotifier#update()
	 */
	public void repaintChart( )
	{
		repaint( );
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.birt.chart.device.swing.IUpdateNotifier#peerInstance()
	 */
	public Object peerInstance( )
	{
		return this;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.birt.chart.device.swing.IUpdateNotifier#getDesignTimeModel()
	 */
	public Chart getDesignTimeModel( )
	{
		return cm;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.birt.chart.device.swing.IUpdateNotifier#getRunTimeModel()
	 */
	public Chart getRunTimeModel( )
	{
		return gcs.getChartModel( );
	}
	
	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.birt.chart.device.IUpdateNotifier#getContext(java.lang.Object)
	 */
	public Object getContext( Object key )
	{
		return contextMap.get( key );
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.birt.chart.device.IUpdateNotifier#putContext(java.lang.Object,
	 *      java.lang.Object)
	 */
	public Object putContext( Object key, Object value )
	{
		return contextMap.put( key, value );
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.birt.chart.device.IUpdateNotifier#removeContext(java.lang.Object)
	 */
	public Object removeContext( Object key )
	{
		return contextMap.remove( key );
	}

	public void updateBuffer( )
	{
		Dimension d = getSize( );

		if ( bi == null
				|| bi.getWidth( ) != d.width
				|| bi.getHeight( ) != d.height )
		{
			bi = new BufferedImage( d.width,
					d.height,
					BufferedImage.TYPE_INT_ARGB );
		}

		Graphics2D g2d = (Graphics2D) bi.getGraphics( );

		idr.setProperty( IDeviceRenderer.GRAPHICS_CONTEXT, g2d );
		Bounds bo = BoundsImpl.create( 0, 0, d.width, d.height );
		bo.scale( 72d / idr.getDisplayServer( ).getDpiResolution( ) ); // BOUNDS
		// MUST
		// BE
		// SPECIFIED
		// IN
		// POINTS

		Generator gr = Generator.instance( );
		if ( bNeedsGeneration )
		{
			bNeedsGeneration = false;
			try
			{
				gcs = gr.build( idr.getDisplayServer( ),
						cm,
						bo,
						null,
						null,
						null );
			}
			catch ( ChartException ex )
			{
				showException( g2d, ex );
			}
		}

		try
		{
			gr.render( idr, gcs );
		}
		catch ( ChartException rex )
		{
			showException( g2d, rex );
		}
		finally
		{
			g2d.dispose( );
		}

	}
	
	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.JComponent#paint(java.awt.Graphics)
	 */
	public void paint( Graphics g )
	{
		super.paint( g );

		if ( bi == null )
		{
			updateBuffer( );
		}
		
		g.drawImage( bi, 0, 0, this );
	}

	/**
	 * Presents the Exceptions if the chart cannot be displayed properly.
	 * 
	 * @param g2d
	 * @param ex
	 */
	private final void showException( Graphics2D g2d, Exception ex )
	{
		String sWrappedException = ex.getClass( ).getName( );
		Throwable th = ex;
		while ( ex.getCause( ) != null )
		{
			ex = (Exception) ex.getCause( );
		}
		String sException = ex.getClass( ).getName( );
		if ( sWrappedException.equals( sException ) )
		{
			sWrappedException = null;
		}

		String sMessage = null;
		if ( th instanceof BirtException )
		{
			sMessage = ( (BirtException) th ).getLocalizedMessage( );
		}
		else
		{
			sMessage = ex.getMessage( );
		}

		if ( sMessage == null )
		{
			sMessage = "<null>";//$NON-NLS-1$
		}

		StackTraceElement[] stea = ex.getStackTrace( );
		Dimension d = getSize( );

		Font fo = new Font( "Monospaced", Font.BOLD, 14 );//$NON-NLS-1$
		g2d.setFont( fo );
		FontMetrics fm = g2d.getFontMetrics( );
		g2d.setColor( Color.WHITE );
		g2d.fillRect( 20, 20, d.width - 40, d.height - 40 );
		g2d.setColor( Color.BLACK );
		g2d.drawRect( 20, 20, d.width - 40, d.height - 40 );
		g2d.setClip( 20, 20, d.width - 40, d.height - 40 );
		int x = 25, y = 20 + fm.getHeight( );
		g2d.drawString( "Exception:", x, y );//$NON-NLS-1$
		x += fm.stringWidth( "Exception:" ) + 5;//$NON-NLS-1$
		g2d.setColor( Color.RED );
		g2d.drawString( sException, x, y );
		x = 25;
		y += fm.getHeight( );
		if ( sWrappedException != null )
		{
			g2d.setColor( Color.BLACK );
			g2d.drawString( "Wrapped In:", x, y );//$NON-NLS-1$
			x += fm.stringWidth( "Wrapped In:" ) + 5;//$NON-NLS-1$
			g2d.setColor( Color.RED );
			g2d.drawString( sWrappedException, x, y );
			x = 25;
			y += fm.getHeight( );
		}
		g2d.setColor( Color.BLACK );
		y += 10;
		g2d.drawString( "Message:", x, y );//$NON-NLS-1$
		x += fm.stringWidth( "Message:" ) + 5;//$NON-NLS-1$
		g2d.setColor( Color.BLUE );
		g2d.drawString( sMessage, x, y );
		x = 25;
		y += fm.getHeight( );
		g2d.setColor( Color.BLACK );
		y += 10;
		g2d.drawString( "Trace:", x, y );//$NON-NLS-1$
		x = 40;
		y += fm.getHeight( );
		g2d.setColor( Color.GREEN.darker( ) );
		for ( int i = 0; i < stea.length; i++ )
		{
			g2d.drawString( stea[i].getClassName( ) + ":"//$NON-NLS-1$
					+ stea[i].getMethodName( ) + "(...):"//$NON-NLS-1$
					+ stea[i].getLineNumber( ), x, y );
			x = 40;
			y += fm.getHeight( );
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see java.awt.event.ComponentListener#componentHidden(java.awt.event.ComponentEvent)
	 */
	public void componentHidden( ComponentEvent e )
	{
		// TODO Auto-generated method stub

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see java.awt.event.ComponentListener#componentMoved(java.awt.event.ComponentEvent)
	 */
	public void componentMoved( ComponentEvent e )
	{
		// TODO Auto-generated method stub

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see java.awt.event.ComponentListener#componentResized(java.awt.event.ComponentEvent)
	 */
	public void componentResized( ComponentEvent e )
	{
		bNeedsGeneration = true;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see java.awt.event.ComponentListener#componentShown(java.awt.event.ComponentEvent)
	 */
	public void componentShown( ComponentEvent e )
	{
		// TODO Auto-generated method stub

	}

	/**
	 * An inner class Control Panel, which provides the interactive interface
	 * with the user.
	 */
	private final class ControlPanel extends JPanel implements ActionListener
	{

		private static final long serialVersionUID = 1L;
		
		private JComboBox jcbModels = null;

		private JButton jbUpdate = null;

		private final SwingInteractivityViewer siv;

		ControlPanel( SwingInteractivityViewer siv )
		{
			this.siv = siv;

			setLayout( new GridLayout( 0, 1, 0, 0 ) );

			JPanel jp = new JPanel( );
			jp.setLayout( new FlowLayout( FlowLayout.LEFT, 3, 3 ) );

			jp.add( new JLabel( "Choose:" ) );//$NON-NLS-1$
			jcbModels = new JComboBox( );

			jcbModels.addItem( "Highlight Series" );//$NON-NLS-1$
			jcbModels.addItem( "Show Tooltip" );//$NON-NLS-1$
			jcbModels.addItem( "Toggle Visibility" );//$NON-NLS-1$
			jcbModels.addItem( "URL Redirect" );//$NON-NLS-1$
			jcbModels.addItem( "Call Back" );//$NON-NLS-1$

			jcbModels.setSelectedIndex( 0 );
			jp.add( jcbModels );

			jbUpdate = new JButton( "Update" );//$NON-NLS-1$
			jbUpdate.addActionListener( this );
			jp.add( jbUpdate );

			add( jp );
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see java.awt.event.ComponentListener#componentHidden(java.awt.event.ComponentEvent)
		 */
		public void componentHidden( ComponentEvent cev )
		{
			setVisible( false );
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see java.awt.event.ComponentListener#componentMoved(java.awt.event.ComponentEvent)
		 */
		public void componentMoved( ComponentEvent cev )
		{
			JFrame jf = (JFrame) cev.getComponent( );
			Rectangle r = jf.getBounds( );
			setLocation( r.x, r.y + r.height );
			setSize( r.width, 50 );
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see java.awt.event.ComponentListener#componentResized(java.awt.event.ComponentEvent)
		 */
		public void componentResized( ComponentEvent cev )
		{
			JFrame jf = (JFrame) cev.getComponent( );
			Rectangle r = jf.getBounds( );
			setLocation( r.x, r.y + r.height );
			setSize( r.width, 50 );
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see java.awt.event.ComponentListener#componentShown(java.awt.event.ComponentEvent)
		 */
		public void componentShown( ComponentEvent cev )
		{
			JFrame jf = (JFrame) cev.getComponent( );
			Rectangle r = jf.getBounds( );
			setLocation( r.x, r.y + r.height );
			setSize( r.width, 50 );
			setVisible( true );
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
		 */
		public void actionPerformed( ActionEvent e )
		{
			int i = jcbModels.getSelectedIndex( );
			cm = null;
			switch ( i )
			{
				case 0 :
					cm = InteractivityCharts.createHSChart( );
					break;
				case 1 :
					cm = InteractivityCharts.createSTChart( );
					break;
				case 2 :
					cm = InteractivityCharts.createTVChart( );
					break;
				case 3 :
					cm = InteractivityCharts.createURChart( );
					break;
				case 4 :
					cm = InteractivityCharts.createCBChart( );
					break;
			}

			bNeedsGeneration = true;
			siv.updateBuffer( );
			siv.repaint( );
		}
	}
	
	public void callback( Object event, Object source, CallBackValue value )
	{
		JOptionPane.showMessageDialog( SwingInteractivityViewer.this,
				value.getIdentifier( ) ); 
	}
}